/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "dbl.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

struct dbl_node {
  double ra_rad;
  double dec_rad;
  int wds_number;
  float mag;
  long filepos;
  char spectrum;
};

/* static function prototypes */
static FILE *read_dbl PROTOTYPE((void));
static void calculate_dbl_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_dbl_star PROTOTYPE((char *));
static void get_dbl_pos PROTOTYPE((int,double *,double *));
static void draw_dbl_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_dbl_star PROTOTYPE((struct display *,int,int,int));
static boolean display_dbl PROTOTYPE((FILE *,int));
static void format_dbl_name PROTOTYPE((int,char *,IDfont *));
static int mag_compare PROTOTYPE((struct dbl_node *,struct dbl_node *));
static void fill_dbl_entry PROTOTYPE((char *,int));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern void format_dbl_display PROTOTYPE((char *,int));
extern void update_display_pos PROTOTYPE((double,double));

static struct dbl_node dbl_array[MAX_DBL];

static int dbl_array_max;

extern float mag_limit;

#define DBL_DATA_FILE   "dbl.dat"
#define DBL_BINARY_FILE "dbl.bin"

/* double star record buffer */
static char dbl_rec[DBL_RECLEN + 1 + 1];



/* initialize the catalog data structure */

void init_dbl(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_dbl;
  catalog->calculate_field = calculate_dbl_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_dbl_star;
  catalog->get_object_pos = get_dbl_pos;
  catalog->draw_object = draw_dbl_star;
  catalog->ps_draw_object = ps_draw_dbl_star;
  catalog->display_data = display_dbl;
  catalog->format_name = format_dbl_name;
  catalog->menu_name = DBL_MENU_NAME;
  catalog->catalog_name = DBL_CATALOG_NAME;

  return;
}



/* read in the double star catalog file, extracting needed info */

static FILE *read_dbl()

{
  char *filebuff;
  FILE *dbl_fd;
  char *bin_filebuff;
  int dbl_entry;

/* get the path to the double star data */
  filebuff = build_filespec("DBL_PATH",DBL_DATA_FILE);

/* open the double star data file */
  if ((dbl_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    printf("sky:  can't open double star data file %s\n",filebuff);
    perror("sky");
    exit(1);
  }

/* get the path to the double star binary file */
  bin_filebuff = build_filespec("DBL_BIN_PATH",DBL_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,(void *)dbl_array,&dbl_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* return the fd of the open data file */
      return(dbl_fd);
    }

/* prior to reading the double star catalog, build the rotation matrix for
 * precessing the stellar coordinates to the common equinox from the given
 * epoch */
  if (DBL_EPOCH != COMMON_EQUINOX)
    init_precess(DBL_EPOCH,COMMON_EQUINOX);

/* read the double star catalog, saving information necessary for display */
  dbl_entry = 0;
  while (TRUE) {
    dbl_array[dbl_entry].filepos = ftell(dbl_fd);
    dbl_array[dbl_entry].wds_number = dbl_entry + 1;
    if (fgets(dbl_rec,sizeof(dbl_rec),dbl_fd) == (char *)NULL)
      break;
    fill_dbl_entry(dbl_rec,dbl_entry++);
  }

/* leave the file open for later use */
/*  (void)fclose(dbl_fd);*/

/* remember the last double star entry we filled */
  dbl_array_max = dbl_entry - 1;

/* now sort the double star array in decreasing order of magnitude */
  hs_heapsort((void *)dbl_array,(size_t)dbl_entry,
	                         (size_t)sizeof(struct dbl_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)dbl_array,
				            (size_t)sizeof(struct dbl_node),
				                           (size_t)dbl_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

  return(dbl_fd);
}



/* compare two array entries for magnitude - compare routine for heapsort */

static int mag_compare(dbl_entry_1,dbl_entry_2)

struct dbl_node *dbl_entry_1, *dbl_entry_2;

{
/* major key is the magnitude */
  if (dbl_entry_1->mag < dbl_entry_2->mag)
    return(-1);
  else if (dbl_entry_1->mag > dbl_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  return((boolean)(dbl_array[array_pos].mag <= limit));
}



/* given an object's index, return its right ascension and declination */

static void get_dbl_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = dbl_array[array_pos].ra_rad;
  *dec_rad = dbl_array[array_pos].dec_rad;

  return;
}



/* given a double star record, make a data node for it */

static void fill_dbl_entry(record,dbl_entry)

char *record;
int dbl_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;
  char mag_buff[4 + 1];

/* get the length of the record for GET_FIELD */
  reclen = strlen(record) - 1;

/* get the RA in pieces */
  GET_FIELD(field,record,POS_RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,POS_RA_START + 4,1);
  ra_pos.seconds = (smallint)atoi(field) * 6;

  ra_pos.thousandths = (shortint)0;

/* similarly, get the declination in pieces */
  GET_FIELD(field,record,POS_DEC_START,1);
  dec_pos.sign = (short)1;
  if (field[0] == 'S')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,POS_DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,POS_DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  dec_pos.seconds = (smallint)0;
  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  dbl_array[dbl_entry].ra_rad = ra_pos_to_rad(ra_pos);
  dbl_array[dbl_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (DBL_EPOCH != COMMON_EQUINOX)
    precess(dbl_array[dbl_entry].ra_rad,dbl_array[dbl_entry].dec_rad,
	          &dbl_array[dbl_entry].ra_rad,&dbl_array[dbl_entry].dec_rad);

/* get the magnitude */
  GET_FIELD(field,record,MAG_PRIMARY_START,3);
  if (strcmp(field,"   ") == EQUAL)
    dbl_array[dbl_entry].mag = 99.99;
  else {
    strncpy(mag_buff,field,2);
    mag_buff[2] = '.';
    strcpy(&mag_buff[3],&field[2]);
    dbl_array[dbl_entry].mag = atof(mag_buff);
  }

/* get the spectral class */
  GET_FIELD(field,record,SPECTRUM_START,4);
  dbl_array[dbl_entry].spectrum = field[0];
  if (dbl_array[dbl_entry].spectrum == ' ')
    dbl_array[dbl_entry].spectrum = field[2];
  /* watch out for the one with a spectral class of '0' */
  if (dbl_array[dbl_entry].spectrum == '0')
    /* speculate that O was meant instead */
    dbl_array[dbl_entry].spectrum = 'O';

  return;
}



/* construct a list of those stars which appear in the current display */

static void calculate_dbl_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all double star entries, checking for visibility */
  for (i = 0; i <= dbl_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,dbl_array[i].ra_rad,
                                                       dbl_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_dbl_field returning\n");
#endif
  return;
}



/* draw a double star on the sky */

static void draw_dbl_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(dbl_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
                               ((obj_x - (size + 1) / 2) <= (ul_x + width)) &&
                               ((obj_y + (size + 1) / 2) >= ul_y) &&
                               ((obj_y - (size + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (dbl_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'R':
  case 'N':
  case 'S':
    color = M;
    break;
  default:
    printf("star has unrecognized spectral class %c\n",
	                                      dbl_array[array_pos].spectrum);
  case 'P':        /* some stars are peculiar */
  case 'D':        /* some I have no idea of the meaning */
  case ' ':        /* and some are just not given */
    color = A;     /* make unknowns white */
    break;
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one double star on the PostScript display */

static boolean ps_draw_dbl_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(dbl_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_dbl(dbl_fd,dbl_array_pos)

FILE *dbl_fd;
int dbl_array_pos;

{
  int i;

/* get info from the double star catalog */
  if (fseek(dbl_fd,dbl_array[dbl_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(dbl_rec,sizeof(dbl_rec),dbl_fd) == (char *)NULL)
    return(FALSE);

/* fill out the (possibly short) record with blanks */
  for (i = strlen(dbl_rec) - 1; i < DBL_RECLEN; i++)
    dbl_rec[i] = ' ';

/* format the record into an informational display */
  format_dbl_display(dbl_rec,dbl_array[dbl_array_pos].wds_number);

  return(TRUE);
}



/* find the double star in the buffer and update the display coordinates */

static boolean find_dbl_star(buffer)

char *buffer;

{
  int dbl_no;
  int i;

/* be sure this is a WDS designation */
  if ((strncmp(buffer,"WDS",3) == EQUAL) || strncmp(buffer,"wds",3) == EQUAL)
    buffer = &buffer[3];
  else
    return(FALSE);

/* get the WDS number */
  dbl_no = atoi(buffer);

/* loop through the array looking for this number */
  for (i = 0; i <= dbl_array_max; i++)
    if (dbl_array[i].wds_number == dbl_no)
      break;

/* return failure if we didn't find it */
  if (i > dbl_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(dbl_array[i].ra_rad,dbl_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for a star */

/* ARGSUSED */

static void format_dbl_name(dbl_pos,buffer,font)

int dbl_pos;
char *buffer;
IDfont *font;

{
/* use the WDS number */
  sprintf(buffer,"WDS %d",dbl_array[dbl_pos].wds_number);

  return;
}
