/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-5, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>
#include <ctype.h>

#include <math.h>
#include <stdlib.h>

#include <unistd.h>

#include <string.h>

#include <X11/Intrinsic.h>

#include "bsc5.h"

#include "skydefs.h"

#include "pos.h"
#include "catalog.h"
#include "draw.h"
#include "catreader.h"

#include "starnames.h"

struct bsc5_node {
  double ra_rad;
  double dec_rad;
  float mag;
  long filepos;
  shortint hr_number;
  smallint bayer_index;
  smallint bayer_super;
  shortint flamsteed_number;
  smallint constellation_index;
  char spectrum;
};

/* static function prototypes */
static FILE *read_bsc5 PROTOTYPE((void));
static void calculate_bsc5_field PROTOTYPE((struct display *,
					                struct cat_header *));
static boolean check_mag_limit PROTOTYPE((int,double));
static boolean find_bsc5_star PROTOTYPE((char *));
static void get_bsc5_pos PROTOTYPE((int,double *,double *));
static void draw_bsc5_star PROTOTYPE((Widget,int,int,int,int,int,int,int));
static boolean ps_draw_bsc5_star PROTOTYPE((struct display *,int,int,int));
static boolean display_bsc5 PROTOTYPE((FILE *,int));
static void format_bsc5_name PROTOTYPE((int,char *,IDfont *));
static boolean find_star_by_hr PROTOTYPE((int));
static boolean find_star_by_bayer PROTOTYPE((int,int));
static boolean find_star_by_flamsteed PROTOTYPE((int,int));
static int fill_bsc5_entry PROTOTYPE((char *,int));
static int mag_compare PROTOTYPE((struct bsc5_node *,struct bsc5_node *));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern void hs_heapsort PROTOTYPE((char [],size_t,size_t,int (*)()));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void read_starname_table PROTOTYPE((void));
extern int find_bsc5_bayer PROTOTYPE((char *));
extern void update_display_pos PROTOTYPE((double,double));
extern void format_bsc5_display PROTOTYPE((char *));
extern int find_const_abbrev PROTOTYPE((char *));
extern int find_constellation PROTOTYPE((char *));
extern int find_greek_letter PROTOTYPE((char *));
extern double ra_pos_to_rad PROTOTYPE((struct ra_pos));
extern double dec_pos_to_rad PROTOTYPE((struct dec_pos));
extern void add_to_display_list PROTOTYPE((struct display *,
					           struct cat_header *,
					                  int,double,double));
extern void draw_star PROTOTYPE((Widget,int,int,int,StarColor));
extern boolean ps_draw_star PROTOTYPE((struct display *,double,double,double));
extern find_star_by_name PROTOTYPE((char *));

extern char greek_letter[];

static struct bsc5_node bsc5_array[MAX_BSC5];

static int bsc5_array_max;

/* table of star names */
extern struct starname starnames[];

extern float mag_limit;

/* AFAIK, these are the three possible ways the CD-ROM file name might be
 * represented by the operating system */
#define BSC5_DATA_FILE_1   "BSC5.DAT;1"
#define BSC5_DATA_FILE_2   "BSC5.DAT"
#define BSC5_DATA_FILE_3   "bsc5.dat"

#define BSC5_BINARY_FILE   "bsc5.bin"

static FILE *bsc5_fd;

/* BSC5 record buffer */
static char bsc5_rec[BSC5_RECLEN + 2 + 1];



/* initialize the catalog data structure */

void init_bsc5(catalog)

struct cat_header *catalog;

{
  catalog->read_data = read_bsc5;
  catalog->calculate_field = calculate_bsc5_field;
  catalog->mag_limit_check = check_mag_limit;
  catalog->find_object = find_bsc5_star;
  catalog->get_object_pos = get_bsc5_pos;
  catalog->draw_object = draw_bsc5_star;
  catalog->ps_draw_object = ps_draw_bsc5_star;
  catalog->display_data = display_bsc5;
  catalog->format_name = format_bsc5_name;
  catalog->menu_name = BSC5_MENU_NAME;
  catalog->catalog_name = BSC5_CATALOG_NAME;

  return;
}



/* read in the BSC5 catalog file, extracting needed info */

static FILE *read_bsc5()

{
  char *filebuff;
  char *bin_filebuff;
  int bsc5_entry;

/* build the path to the BSC5 data file and open it */
  filebuff = build_filespec("BSC5_CD_PATH",BSC5_DATA_FILE_1);
  if ((bsc5_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
    free((void *)filebuff);
    filebuff = build_filespec("BSC5_CD_PATH",BSC5_DATA_FILE_2);
    if ((bsc5_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
      free((void *)filebuff);
      filebuff = build_filespec("BSC5_CD_PATH",BSC5_DATA_FILE_3);
      if ((bsc5_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
	printf("sky:  can't open BSC5 data file %s\n",filebuff);
	perror("sky");
	exit(1);
      }
    }
  }

/* get the path to the BSC5 binary file */
  bin_filebuff = build_filespec("BSC5_BIN_PATH",BSC5_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,bsc5_array,&bsc5_array_max)) {
      /* success; free the filename buffers */
      free((void *)filebuff);
      free((void *)bin_filebuff);

      /* read in the star name table */
      read_starname_table();

      /* return the fd of the open data file */
      return(bsc5_fd);
    }

/* prior to reading the BSC5 catalog, build the rotation matrix for precessing
 * the stellar coordinates to the common equinox from the given epoch */
  if (BSC5_EPOCH != COMMON_EQUINOX)
    init_precess(BSC5_EPOCH,COMMON_EQUINOX);

/* read the BSC5 catalog, saving the information necessary for the display */
  bsc5_entry = 0;
  while (TRUE) {
    bsc5_array[bsc5_entry].filepos = ftell(bsc5_fd);
    if (fgets(bsc5_rec,sizeof(bsc5_rec),bsc5_fd) == (char *)NULL)
      break;
    bsc5_entry = fill_bsc5_entry(bsc5_rec,bsc5_entry);
  }

/* leave the file open for later use */
/*  (void)fclose(bsc5_fd);*/

/* remember the last bsc5 entry we filled */
  bsc5_array_max = bsc5_entry - 1;

/* now sort the bsc5 array in decreasing order of magnitude */
  hs_heapsort((void *)bsc5_array,(size_t)bsc5_entry,
	                        (size_t)sizeof(struct bsc5_node),mag_compare);

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)bsc5_array,
				            (size_t)sizeof(struct bsc5_node),
			                                  (size_t)bsc5_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

/* read in the star name table */
  read_starname_table();

  return(bsc5_fd);
}



/* return TRUE if the magnitude of the star meets the limiting magnitude */

static boolean check_mag_limit(array_pos,limit)

int array_pos;
float limit;

{
  return((boolean)(bsc5_array[array_pos].mag <= limit));
}



/* compare two bsc5 array entries for magnitude for heapsort */

static int mag_compare(bsc5_entry_1,bsc5_entry_2)

struct bsc5_node *bsc5_entry_1, *bsc5_entry_2;

{
/* major key is the magnitude */
  if (bsc5_entry_1->mag < bsc5_entry_2->mag)
    return(-1);
  else if (bsc5_entry_1->mag > bsc5_entry_2->mag)
    return(1);
  else
    return(0);

/* NOTREACHED */
}



/* given an object's index, return its right ascension and declination */

static void get_bsc5_pos(array_pos,ra_rad,dec_rad)

int array_pos;
double *ra_rad, *dec_rad;

{
  *ra_rad = bsc5_array[array_pos].ra_rad;
  *dec_rad = bsc5_array[array_pos].dec_rad;

  return;
}



/* given a BSC5 record, make a data node for it */

static int fill_bsc5_entry(record,bsc5_entry)

char *record;
int bsc5_entry;

{
  int reclen;
  char field[11];
  struct ra_pos ra_pos;
  struct dec_pos dec_pos;

/* if no spectrum is given, skip the entry */
  if (strncmp(&bsc5_rec[SPECTRUM],"                    ",SPECTRUM_LENGTH)
	                                                             == EQUAL)
    return(bsc5_entry);

/* get the length of the record for GET_FIELD */
  reclen = BSC5_RECLEN;

/* get the HR number */
  GET_FIELD(field,record,0,HR_NUMBER_LENGTH);
  bsc5_array[bsc5_entry].hr_number = atoi(field);

/* get the Flamsteed number */
  GET_FIELD(field,record,FLAMSTEED_NUMBER,FLAMSTEED_LENGTH);
  bsc5_array[bsc5_entry].flamsteed_number = (shortint)atoi(field);

/* get the Bayer letter index */
  GET_FIELD(field,record,BAYER_DESIGNATION,BAYER_LENGTH);
  if (strncmp(field,"   ",BAYER_LENGTH) == EQUAL) {
    bsc5_array[bsc5_entry].bayer_index = (smallint)-1;
    bsc5_array[bsc5_entry].bayer_super = 0;
  }
  else {
    /* watch out for a trailing blank */
    if (field[BAYER_LENGTH - 1] == ' ')
      field[BAYER_LENGTH - 1] = '\0';
    bsc5_array[bsc5_entry].bayer_index = (smallint)find_bsc5_bayer(field);
    
    /* get the Bayer superscript */
    GET_FIELD(field,record,BAYER_SUPERSCRIPT,BAYER_SUPER_LENGTH);
    bsc5_array[bsc5_entry].bayer_super = (smallint)atoi(field);
  }

/* get the constellation abbreviation */
  GET_FIELD(field,record,CONSTELLATION_NAME,CONSTELLATION_LENGTH);
  if (strcmp(field,"   ") != EQUAL)
    bsc5_array[bsc5_entry].constellation_index =
                                           (smallint)find_const_abbrev(field);
  if (bsc5_array[bsc5_entry].constellation_index == (smallint)-1)
    printf("constellation abbreviation %s was not found for HR %d\n",field,
	                                    bsc5_array[bsc5_entry].hr_number);

/* get the RA in pieces */
  GET_FIELD(field,record,RA_START,2);
  ra_pos.hours = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 2,2);
  ra_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 4,2);
  ra_pos.seconds = (smallint)atoi(field);

  GET_FIELD(field,record,RA_START + 7,1);
  ra_pos.thousandths = (shortint)atoi(field) * 100;

/* similarly, get the declination in pieces */
  dec_pos.sign = (short)1;
  GET_FIELD(field,record,DEC_START,1);
  if (field[0] == '-')
    dec_pos.sign = (short)-1;

  GET_FIELD(field,record,DEC_START + 1,2);
  dec_pos.degrees = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_START + 3,2);
  dec_pos.minutes = (smallint)atoi(field);

  GET_FIELD(field,record,DEC_START + 5,2);
  dec_pos.seconds = (smallint)atoi(field);

  dec_pos.hundredths = (smallint)0;

/* compute the RA and declination in radians */
  bsc5_array[bsc5_entry].ra_rad = ra_pos_to_rad(ra_pos);
  bsc5_array[bsc5_entry].dec_rad = dec_pos_to_rad(dec_pos);

/* precess them to the common equinox */
  if (BSC5_EPOCH != COMMON_EQUINOX)
    precess(bsc5_array[bsc5_entry].ra_rad,bsc5_array[bsc5_entry].dec_rad,
	      &bsc5_array[bsc5_entry].ra_rad,&bsc5_array[bsc5_entry].dec_rad);

/* get the magnitude */
  GET_FIELD(field,record,VISUAL_MAG,VISUAL_MAG_LENGTH);
  bsc5_array[bsc5_entry].mag = atof(field);

/* get the spectral class */
  bsc5_array[bsc5_entry].spectrum = record[SPECTRUM + 2];

  return(++bsc5_entry);
}



/* construct a list of those stars which appear in the current display */

static void calculate_bsc5_field(display,catalog)

struct display *display;
struct cat_header *catalog;

{
  int i;

/* loop through all BSC5 entries, checking for visibility */
  for (i = 0; i <= bsc5_array_max; i++)
    /* check this star and add it to the list if visible */
    add_to_display_list(display,catalog,i,bsc5_array[i].ra_rad,
			                               bsc5_array[i].dec_rad);

#ifdef DEBUG
  printf("calculate_bsc5_field returning\n");
#endif
  return;
}



/* draw a BSC5 star on the sky */

static void draw_bsc5_star(sky_widget,array_pos,obj_x,obj_y,ul_x,ul_y,
		                                                 width,height)

Widget sky_widget;
int array_pos;
int obj_x, obj_y;
int ul_x, ul_y;
int width, height;

{
  int size;
  StarColor color;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return;

  /* calculate size based on magnitude */
  size = STARSIZE(bsc5_array[array_pos].mag);

  /* draw the object even if it is only partially in the box */
  if (((obj_x + (size + 1) / 2) >= ul_x) &&
                               ((obj_x - (size + 1) / 2) <= (ul_x + width)) &&
                               ((obj_y + (size + 1) / 2) >= ul_y) &&
                               ((obj_y - (size + 1) / 2) <= (ul_y + height)))
    ;
  else
    return;

  /* assign a color based on the spectral class */
  switch (bsc5_array[array_pos].spectrum) {
  case 'W':
  case 'O':
    color = O;
    break;
  case 'B':
    color = B;
    break;
  case 'A':
    color = A;
    break;
  case 'F':
    color = F;
    break;
  case 'G':
    color = G;
    break;
  case 'K':
    color = K;
    break;
  case 'M':
  case 'C':
  case 'N':
  case 'S':
    color = M;
    break;
  case 'p':        /* at least one star is just "pec" */
    color = A;     /* make unknowns white */
    break;
  default:
    printf("array pos = %d\n",array_pos);
    printf("star %d has unrecognized spectral class %c\n",
	                                      bsc5_array[array_pos].hr_number,
	                                      bsc5_array[array_pos].spectrum);
  }

/* draw the star with correct location, size, and color */
  draw_star(sky_widget,obj_x,obj_y,size,color);

  return;
}



/* draw one BSC5 star on the PostScript display */

static boolean ps_draw_bsc5_star(display,array_pos,x,y)

struct display *display;
int array_pos;
int x, y;

{
  int size;

/* be sure it is bright enough to display */
  if (! check_mag_limit(array_pos,mag_limit))
    return(TRUE);

/* calculate size based on magnitude */
  size = STARSIZE(bsc5_array[array_pos].mag);

/* draw the star in the correct position with the correct size */
  return(ps_draw_star(display,(double)x,(double)y,(double)size));
}



/* display the catalog data for the given star */

static boolean display_bsc5(bsc5_fd,bsc5_array_pos)

FILE *bsc5_fd;
int bsc5_array_pos;

{
/* get info from the BSC5 catalog */
  if (fseek(bsc5_fd,bsc5_array[bsc5_array_pos].filepos,SEEK_SET) == -1)
    return(FALSE);
  if (fgets(bsc5_rec,sizeof(bsc5_rec),bsc5_fd) == (char *)NULL)
    return(FALSE);

/* format the record into an informational display */
  format_bsc5_display(bsc5_rec);

  return(TRUE);
}



/* if the name in the buffer is a recognizeable format, attempt to find it */

static boolean find_bsc5_star(buffer)

char *buffer;

{
  char *ptr;
  int constellation;
  int hr_number;
  int i;

/*
 * algorithm:
 *
 * 1) if the first two letters are HR or hr, look for HR number
 * 2) if the first three or four letters are BSC, BSC5, bsc,  or bsc5,
 *        look for HR number
 * 3) if the buffer after the first token is a constellation, then (4) or (5)
 * 4) if the first token is a greek letter, look for Bayer designation
 * 5) if the first token is numeric, look for Flamsteed designation
 * 6) check for a name match
 */

  if ((strncmp(buffer,"HR",2) == EQUAL) || strncmp(buffer,"hr",2) == EQUAL)
    /* try the name as a Henry Russell number */
    return(find_star_by_hr(atoi(&buffer[2])));
  else if ((strncmp(buffer,"BSC",3) == EQUAL) ||
	                                     strncmp(buffer,"bsc",3) == EQUAL)
    /* try the name as a Henry Russell number */
    return(find_star_by_hr(atoi(&buffer[3])));
  else if ((strncmp(buffer,"BSC5",4) == EQUAL) ||
	                                    strncmp(buffer,"bsc5",4) == EQUAL)
    /* try the name as a Henry Russell number */
    return(find_star_by_hr(atoi(&buffer[5])));
  else
    ;

/* advance to the second token */
  ptr = buffer;
  while ((! isspace(*ptr)) && (*ptr != '\0'))
    ptr++;
  /* if we found a space, advance to the start of the next token */
  if (isspace(*ptr))
    ptr++;

/* if there is more than one token, check to see if it is a constellation */
  if (*ptr == '\0')
    /* this must be a star name */
    if ((hr_number = find_star_by_name(buffer)) == -1)
      return(FALSE);
    else
      return(find_star_by_hr(hr_number));
  else {
    /* if the rest of the buffer is not a constellation, must be a star name */
    if ((constellation = find_constellation(ptr)) == -1)
      if ((hr_number = find_star_by_name(buffer)) == -1)
	return(FALSE);
      else
	return(find_star_by_hr(hr_number));

    /* plant a nul at the end of the first token */
    *--ptr = '\0';

    /* in case this is a Bayer designation, find the Greek letter index */
    i = find_greek_letter(buffer);

    /* repair the buffer before we forget */
    *ptr = ' ';

    /* was a Greek letter index found? */
    if (i == -1) {
      /* no; is the first token all numeric? */
      ptr = buffer;
      while (isdigit(*ptr))
	ptr++;
      if (*ptr == ' ')
	/* yes; this is a Flamsteed designation */
	return(find_star_by_flamsteed(atoi(buffer),constellation));
      else
	/* no; we don't recognize this type of designation */
	return(FALSE);
    }
    else
      /* yes; this is a Bayer designation */
      return(find_star_by_bayer(i,constellation));
  }

/* NOTREACHED */
}



/* find the HR star in the buffer and update the display coordinates */

static boolean find_star_by_hr(bsc5_no)

int bsc5_no;

{
  int i;

/* loop through the array looking for this number */
  for (i = 0; i <= bsc5_array_max; i++)
    if (bsc5_array[i].hr_number == bsc5_no)
      break;

/* return FALSE if we didn't find it */
  if (i > bsc5_array_max)
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(bsc5_array[i].ra_rad,bsc5_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* find a star given its greek letter and constellation index */

static boolean find_star_by_bayer(greek,constellation)

int greek;
int constellation;

{
  int i;

/* make the Greek letter index 1-relative */
  greek++;

/* search the BSC5 catalog for a matching star */
  for (i = 0; i <= bsc5_array_max; i++)
    if ((bsc5_array[i].bayer_index == greek) &&
	                 (bsc5_array[i].constellation_index == constellation))
      break;

/* if we found a match, move to that star */
  if (i > bsc5_array_max)
    /* no match found; fail */
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(bsc5_array[i].ra_rad,bsc5_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* find a star given its Flamsteed designation */

static boolean find_star_by_flamsteed(flamsteed_no,constellation)

int flamsteed_no;
int constellation;

{
  int i;

/* search the BSC5 catalog for a matching star */
  for (i = 0; i <= bsc5_array_max; i++)
    if ((bsc5_array[i].flamsteed_number == flamsteed_no) &&
	                 (bsc5_array[i].constellation_index == constellation))
      break;

/* if we found a match, move to that star */
  if (i > bsc5_array_max)
    /* no match found; fail */
    return(FALSE);
  else {
    /* update the display coordinates */
    update_display_pos(bsc5_array[i].ra_rad,bsc5_array[i].dec_rad);
    return(TRUE);
  }

/* NOTREACHED */
}



/* format a nice name for a star */

static void format_bsc5_name(bsc5_pos,buffer,font)

int bsc5_pos;
char *buffer;
IDfont *font;

{
  char field[11];
  int reclen;
  int num;
  int i;

/* prefer Bayer designation over Flamsteed */
  if (bsc5_array[bsc5_pos].bayer_index != (smallint)-1) {
    /* change to Greek font and format as a Bayer designation */
    *font = GREEK;
    sprintf(buffer,"%c",greek_letter[bsc5_array[bsc5_pos].bayer_index]);
    /* is there a superscript? */
    if (bsc5_array[bsc5_pos].bayer_super > 0)
      /* yes, add the superscript */
      sprintf(&buffer[strlen(buffer)],"-%d",bsc5_array[bsc5_pos].bayer_super);
  }
  else if (bsc5_array[bsc5_pos].flamsteed_number > 0)
    /* format as a Flamsteed designation */
    sprintf(buffer,"%d",bsc5_array[bsc5_pos].flamsteed_number);
  else {
    /* neither Bayer nor Flamsteed - work harder for other designations */
    if (fseek(bsc5_fd,bsc5_array[bsc5_pos].filepos,SEEK_SET) == -1)
      return;
    if (fgets(bsc5_rec,sizeof(bsc5_rec),bsc5_fd) == (char *)NULL)
      return;

    /* try an HD number next */
    reclen = BSC5_RECLEN;
    GET_FIELD(field,bsc5_rec,HD_NUMBER,HD_NUMBER_LENGTH);
    if ((num = atoi(field)) != 0) {
      if (num >= 100000)
	sprintf(buffer,"HDE %d",num);
      else
	sprintf(buffer,"HD %d",num);
      return;
    }

    /* check Durchmusterung next */
    GET_FIELD(field,bsc5_rec,DM_DEGREES,DM_DEGREES_LENGTH + DM_NUMBER_LENGTH);
    if (strncmp(field,"        ",DM_DEGREES_LENGTH + DM_NUMBER_LENGTH) != 0) {
      for (i = 0; i < DM_NUMBER_LENGTH; i++)
	if (field[DM_DEGREES_LENGTH + i] == ' ')
	  field[DM_DEGREES_LENGTH + i] = '0';
      sprintf(buffer,"%.2d %.3sd %.5s",&bsc5_rec[DM_CATALOG],
	                                                 &field[0],&field[3]);
      return;
    }

    /* finally, use the HR number */
    sprintf(buffer,"HR %d",bsc5_array[bsc5_pos].hr_number);
  }

  return;
}
