/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-4, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

/* this routine formats and displays a BSC5 record */

#include <stdio.h>
#include <string.h>

#include <ctype.h>

#include "bsc5.h"

#include "skydefs.h"

#include "bsc5_greek.h"

#include "catreader.h"

#include "format.h"

/* static function prototypes */
static void display_pos PROTOTYPE((char *));

/* external function prototypes */
extern int find_const_abbrev PROTOTYPE((char *));
extern char *find_starnames PROTOTYPE((int));
extern void format_starnames PROTOTYPE((char *,int));

/* global function prototypes */
int find_bsc5_bayer PROTOTYPE((char *));

extern char *greek[];

extern char *constellations_abbrev[];
extern char *constellations_latin[];

/* table of blank BSC5 entries */
static int bsc5_blank[] = {
                92, 95, 182, 1057, 1841, 2472, 2496, 3515, 3671,
                6309, 6515, 7189, 7539, 8296, 0 };

#define MAX_BSC5_ROW   77
#define NUM_BSC5_ROWS  25



void format_bsc5_display(record)

char *record;

{
  int reclen;
  char field[5];
  int hr_num;
  char const_buffer[CONSTELLATION_LENGTH + 1];
  int con;
  char bayer_buff[BAYER_LENGTH + 1];
  int desig_count;
  int multiple_count;
  char *starlist;
  int i;
  int double_field_count;
  boolean mag_var_flag;
  char var_id_buff[VAR_ID_LENGTH + 1];
  char *tok1, *tok2;
  boolean numeric_flag;
  boolean uppercase_flag;
  int bayer_letter;
  boolean bayer_flag;
  char bayer_super;
  char spectrum_code;
  boolean par_radvel_flag;
  int rad_vel;
  int rot_vel;

/* start at the beginning of the display */
  idx = 0;
  row = 0;
  row_start = 0;
  max_row = MAX_BSC5_ROW;

/* get the record length */
  reclen = strlen(record);

/* display the catalog number */
  GET_FIELD(field,record,HR_NUMBER,4);
  hr_num = atoi(field);
  sprintf(&obj_info[idx],"HR %d",hr_num);
  idx += strlen(&obj_info[idx]);

  /* count the designations to control length of the first line */
  desig_count = 1;

/* see if this is one of the blank records */
  i = 0;
  while (bsc5_blank[i] != 0) {
    if (hr_num == bsc5_blank[i])
      break;
    i++;
  }

/* handle blank records specially */
  if (bsc5_blank[i] != 0) {
    /* print the name */
    advance_to_tab();
    sprintf(&obj_info[idx],"    %.9s",&record[NAME_START]);
    idx += strlen(&obj_info[idx]);

    /* and the variable designation, if any */
    if (strncmp(&record[VAR_ID_START],"         ",VAR_ID_LENGTH) != EQUAL) {
      advance_to_tab();
      sprintf(&obj_info[idx],"    %.9s",&record[VAR_ID_START]);
      idx += strlen(&obj_info[idx]);
    }

    /* fill the display out to the correct number of lines */
    while (row < NUM_BSC5_ROWS)
      next_line();
    obj_info[idx] = '\0';

    return;
  }

/* if there's a constellation name, try for Flamsteed and Bayer designations */
  if (strncmp(&record[CONSTELLATION_NAME],"   ",CONSTELLATION_LENGTH)
                                                                   != EQUAL) {
    /* get the constellation abbreviation index */
    strncpy(const_buffer,&record[CONSTELLATION_NAME],CONSTELLATION_LENGTH);
    const_buffer[CONSTELLATION_LENGTH] = '\0';
    con = find_const_abbrev(const_buffer);

    if (con == -1)
      printf("no match for constellation found in record %s\n",record);
    else {
      /* display Flamsteed designation, if provided */
      if (strncmp(&record[FLAMSTEED_NUMBER],"   ",FLAMSTEED_LENGTH) != EQUAL) {
	add_space(4);
	sprintf(&obj_info[idx],"%.3s %s",&record[FLAMSTEED_NUMBER],
		                                   constellations_latin[con]);
	idx += strlen(&obj_info[idx]);

	/* count the designations */
	desig_count++;
      }

      /* display Bayer designation, if provided */
      if (strncmp(&record[BAYER_DESIGNATION],"   ",BAYER_LENGTH) != EQUAL) {
	/* grab the Bayer letter index */
	strncpy(bayer_buff,&record[BAYER_DESIGNATION],BAYER_LENGTH);
	bayer_buff[BAYER_LENGTH] = '\0';

	/* watch out for two-character Greek letters */
	if (bayer_buff[BAYER_LENGTH - 1] == ' ')
	  bayer_buff[BAYER_LENGTH - 1] = '\0';

	/* print the Bayer designation */
	add_space(4);
	sprintf(&obj_info[idx],"%s",greek[find_bsc5_bayer(bayer_buff)]);
	idx += strlen(&obj_info[idx]);

	/* get the Bayer superscript number */
	if (record[BAYER_SUPERSCRIPT] != ' ') {
	  sprintf(&obj_info[idx],"-%c",record[BAYER_SUPERSCRIPT]);
	  idx += strlen(&obj_info[idx]);
	}

	/* finally, tack on the constellation */
	sprintf(&obj_info[idx]," %s",constellations_latin[con]);
	idx += strlen(&obj_info[idx]);

	/* count the designations */
	desig_count++;
      }
    }
  }

/* display the BD number */
  if (record[DM_DEGREES + 1] == ' ')
    record[DM_DEGREES + 1] = '0';
  for (i = 0; i < DM_NUMBER_LENGTH; i++)
    if (record[DM_NUMBER + i] == ' ')
      record[DM_NUMBER + i] = '0';
  add_space(4);
  sprintf(&obj_info[idx],"%.2s %.3sd %.5s",&record[DM_CATALOG],
	                              &record[DM_DEGREES],&record[DM_NUMBER]);
  idx += strlen(&obj_info[idx]);

  /* count the designations */
  if (++desig_count == 4) {
    next_line();
    advance_to_tab();
  }

/* display the HD number */
  ljust(&record[HD_NUMBER],HD_NUMBER_LENGTH);
  if (desig_count < 4)
    add_space(4);
  if (record[HD_NUMBER + HD_NUMBER_LENGTH - 1] == ' ')
    sprintf(&obj_info[idx],"HD %.5s",&record[HD_NUMBER]);
  else
    sprintf(&obj_info[idx],"HDE %.6s",&record[HD_NUMBER]);
  idx += strlen(&obj_info[idx]);

/* check to see if we have star names for this star */
  if ((starlist = find_starnames(hr_num)) != (char *)NULL) {
    /* space down a little */
    blank_line();

    /* put out the star name list header */
    add_string("Common names:  ");

    /* put in the formatted star name list */
    format_starnames(starlist,MAX_BSC5_ROW);
  }

/* space down some */
  blank_line();

/* identify infrared sources */
  if (record[INFRARED_FLAG] == 'I') {
    add_string("Infrared source");
    if (record[INFRARED_ID] == ' ')
      add_string(" - NASA");
    else if (record[INFRARED_ID] == '\'')
      add_string(" - Engles");
    else if (record[INFRARED_ID] == ':')
      add_string(" - uncertain ID");
    else
      printf("unrecognized infrared ID code %c\n",record[INFRARED_ID]);

    blank_line();
  }

/* display multiple-component data */
  if (record[DOUBLE_CODE] != ' ') {
    double_field_count = 0;

    /* do a bunch of stuff for multiple-star systems */
    if (record[DOUBLE_CODE] == 'W')
      add_string("Binary - Worley");
    else if (record[DOUBLE_CODE] == 'D')
      add_string("Occultation binary");
    else if (record[DOUBLE_CODE] == 'S')
      add_string("Speckle binary");
    else if (record[DOUBLE_CODE] == 'A')
      /* this is undocumented; I've chosen to display the BSC4 meaning */
      add_string("Astrometric binary");
    else if (record[DOUBLE_CODE] == 'I')
      /* this appears several times without explanation */
      add_string("Multiple - type I");
    else if (record[DOUBLE_CODE] == 'R')
      /* this appears several times without explanation */
      add_string("Multiple - type R");
    else
      ;
    double_field_count++;

    advance_to_tab();

/* output the ADS number, if any */
    if (strncmp(&record[ADS],"     ",ADS_LENGTH) != EQUAL) {
      ljust(&record[ADS],ADS_LENGTH);
      sprintf(&obj_info[idx],"    ADS %.4s",&record[ADS]);
      idx += strlen(&obj_info[idx]);

      double_field_count++;
    }

/* output the identified components */
    if (strncmp(&record[ADS_COMPONENT_START],"  ",ADS_COMPONENT_LENGTH)
	                                                           != EQUAL) {
      advance_to_tab();
      add_string("component(s) ");
      if (record[ADS_COMPONENT_START] != ' ')
	add_char(record[ADS_COMPONENT_START]);
      if (record[ADS_COMPONENT_START + 1] != ' ') {
	add_char(' ');
	add_char(record[ADS_COMPONENT_START + 1]);
      }

      double_field_count++;
    }

/* give the magnitude difference and separation, if specified */
    if (strncmp(&record[DELTA_MAG],"    ",DELTA_MAG_LENGTH) != EQUAL) {
      if (double_field_count++ >= 3)
	next_line();

      advance_to_tab();
      sprintf(&obj_info[idx],"Magnitude difference:  %.4s",&record[DELTA_MAG]);
      idx += strlen(&obj_info[idx]);
    }

    if (strncmp(&record[SEPARATION],"      ",SEPARATION_LENGTH) != EQUAL) {
      if (double_field_count++ >= 3)
	next_line();

      advance_to_tab();
      sprintf(&obj_info[idx],"Separation:  %.4s\"%.2s",&record[SEPARATION],
	                                              &record[SEPARATION + 4]);
      idx += strlen(&obj_info[idx]);
   }

/* show the total component count */
    GET_FIELD(field,record,TOTAL_COMPONENTS,2);
    multiple_count = atoi(field);
    if (multiple_count > 0) {
      if (double_field_count++ >= 3)
	next_line();

      advance_to_tab();
      sprintf(&obj_info[idx],"Total of %d components",multiple_count);
      idx += strlen(&obj_info[idx]);
    }

/* put in some blankness */
    blank_line();
  }

/* display the position */
  display_pos(&record[RA_START]);

/* now display the magnitude */
  mag_var_flag = FALSE;
  if (strncmp(&record[VISUAL_MAG],"     ",VISUAL_MAG_LENGTH) != EQUAL) {
    mag_var_flag = TRUE;
    next_line();
    add_string("Mag:  ");
    sprintf(&obj_info[idx],"%.5s",&record[VISUAL_MAG]);
    idx += strlen(&obj_info[idx]);
    if (record[VMAG_UNCORR_CODE] == ':')
      add_string(" (uncertain)");
    else if (record[VMAG_UNCORR_CODE] == '?')
      add_char('?');
    else
      ;

    /* add the magnitude system information */
    if (record[VMAG_CODE] == ' ')
      add_string(" - UBV system");
    else if (record[VMAG_CODE] == 'H')
      add_string(" - HR system");
    else if (record[VMAG_CODE] == 'R')
      add_string(" - HR mag reduced to UBV system");
    else
      ;

    advance_tabs(2);
  }

/* display variability information */
  if (strncmp(&record[VAR_ID_START],"         ",VAR_ID_LENGTH) != EQUAL) {
    mag_var_flag = TRUE;

    /* copy out the variable ID field */
    strncpy(var_id_buff,&record[VAR_ID_START],VAR_ID_LENGTH);
    var_id_buff[VAR_ID_LENGTH] = '\0';

/* there are basically 6 types of designation here:  two or three mixed-case
 * letters followed by a possible numeric subscript followed by a constella-
 * tion are a Bayer designation, V folowed by all numeric followed by a con-
 * stellation is an ordinal designation, all numeric is an NSV designation,
 * one or two letters followed by a constellation are an Argelander designa-
 * tion, and Var or Var? for unnamed (suspected) variables.  additionally,
 * there are some which are a lower-case letter followed by a constellation */

    /* skip leading blanks */
    i = 0;
    if (var_id_buff[i] == ' ')
	i++;
    tok1 = &var_id_buff[i];

    /* find the end of the first token */
    while (i < VAR_ID_LENGTH)
      if (var_id_buff[i] == ' ')
	break;
      else
	i++;
    var_id_buff[i++] = '\0';

    /* find the start of the second token, if any */
    while (i < VAR_ID_LENGTH)
      if (var_id_buff[i] != ' ')
	break;
      else
	i++;

    if (i < VAR_ID_LENGTH) {
      /* second token starts here */
      tok2 = &var_id_buff[i++];

      /* find the end of the second token */
      while (i < VAR_ID_LENGTH)
	if (var_id_buff[i] == ' ')
	  break;
	else
	  i++;
      var_id_buff[i++] = '\0';
    }
    else
      /* no second token */
      tok2 = (char *)NULL;

    /* check to see if the first token is all numeric */
    i = 0;
    do {
      if (! isdigit(tok1[i]))
	break;
      else
	i++;
    } while (tok1[i] != '\0');

    /* if all numeric, set a flag */
    if (tok1[i] == '\0')
      numeric_flag = TRUE;
    else
      numeric_flag = FALSE;

    /* check to see if the first token is all upper case */
    i = 0;
    do {
      if (! isupper(tok1[i]))
	break;
      else
	i++;
    } while (tok1[i] != '\0');

    /* if all upper case, set a flag */
    if (tok1[i] == '\0')
      uppercase_flag = TRUE;
    else
      uppercase_flag = FALSE;

    /* if the second token exists, it is a constellation abbreviation */
    if (tok2 != (char *)NULL) {
      con = find_const_abbrev(tok2);
      if (con == -1)
	printf("no match for constellation found in record %s\n",record);
    }

    /* if the first token is non-numeric, not all upper-case, and longer than
     * 1 character, it must be a Bayer designation */
    if (numeric_flag || uppercase_flag || (strlen(tok1) <= 1))
      bayer_flag = FALSE;
    else {
      /* save any superscript */
      bayer_super = tok1[strlen(tok1) - 1];
      if (isdigit(bayer_super))
	tok1[strlen(tok1) - 1] = '\0';
      else
	bayer_super = ' ';

      /* get the index of this Greek letter */
      bayer_letter = find_bsc5_bayer(tok1);

      bayer_flag = TRUE;
    }

    /* put out a field ID */
    add_string("Variable:  ");

    if (tok2 == (char *)NULL)
      /* no second token; either an NSV number or an unnamed variable */
      if (numeric_flag)
	/* this is an NSV number */
	sprintf(&obj_info[idx],"NSV %s",tok1);
      else
	if (strcmp(tok1,"Var") == EQUAL)
	  sprintf(&obj_info[idx],"unnamed");
        else if (strcmp(tok1,"Var?") == EQUAL)
	  sprintf(&obj_info[idx],"unnamed (suspected)");
        else
	  printf("Unrecognized variable ID %s\n",tok1);
    else
      /* all other types of designation require a constellation index */
      if (con != -1)
	/* handle Bayer designations */
	if (bayer_flag)
	  if (bayer_letter == -1)
	    /* invalid letter */
	    ;
	  else if (bayer_super == ' ')
	    sprintf(&obj_info[idx],"%s %s",greek[bayer_letter],
		                                   constellations_latin[con]);
	  else
	    sprintf(&obj_info[idx],"%s-%c %s",greek[bayer_letter],bayer_super,
		                                   constellations_latin[con]);
	else
	  /* all other designation formats are simple */
	    sprintf(&obj_info[idx],"%s %s",tok1,constellations_latin[con]);
      else
	;

    idx += strlen(&obj_info[idx]);
  }

/* space down some more, if necessary */
  if (mag_var_flag)
    blank_line();

/* display color indices, if any */
  if (strncmp(&record[COLOR_INDICES_START],"     ",COLOR_INDICES_LENGTH)
                                                                   != EQUAL) {
    add_string("Color indices:");
    next_line();
    advance_to_tab();

    if (strncmp(&record[B_V_INDEX],"     ",B_V_LENGTH) != EQUAL) {
      sprintf(&obj_info[idx],"B - V:  %.5s",&record[B_V_INDEX]);
      idx += strlen(&obj_info[idx]);

      if (record[B_V_CODE] == ':')
	add_string(" (uncertain)");
      else if (record[B_V_CODE] == '?')
	add_char('?');
      else
	;

      advance_to_tab();
    }

    if (strncmp(&record[U_B_INDEX],"     ",U_B_LENGTH) != EQUAL) {
      sprintf(&obj_info[idx],"U - B:  %.5s",&record[U_B_INDEX]);
      idx += strlen(&obj_info[idx]);
      if (record[U_B_CODE] == ':')
	add_string(" (uncertain)");
      else if (record[U_B_CODE] == '?')
	add_char('?');
      else
	;

      advance_to_tab();
    }

    if (strncmp(&record[R_I_INDEX],"     ",R_I_LENGTH) != EQUAL) {
      sprintf(&obj_info[idx],"R - I (Johnson):  %.5s",&record[R_I_INDEX]);
      idx += strlen(&obj_info[idx]);
      if (record[U_B_CODE] == ':')
	add_string(" (uncertain)");
      else if (record[U_B_CODE] == '?')
	add_char('?');
      else if (record[U_B_CODE] == 'C')
	add_string(" - Cousins");
      else if (record[U_B_CODE] == 'E')
	add_string(" - Eggen");
      else
	;
    }

/* space down some more */
    blank_line();
  }

/* save the spectrum code so we can nul-terminate the spectrum */
  spectrum_code = record[SPECTRUM_CODE];
  i = SPECTRUM + SPECTRUM_LENGTH - 1;
  while (isspace(record[i]))
    i--;
  record[++i] = '\0';

/* add the spectral class and code */
  sprintf(&obj_info[idx],"Spectral classification (MK):  %s",
	                                               &record[SPECTRUM]);
  idx += strlen(&obj_info[idx]);

  if (spectrum_code != ' ') {
    /* the meaning of this code is undocumented */
    sprintf(&obj_info[idx]," - type code %c",spectrum_code);
    idx += strlen(&obj_info[idx]);
  }

  blank_line();

/* list the proper motions */
  sprintf(&obj_info[idx],"J2000 proper motion (FK5) - RA:  %.2s\"%.4s",
	                                       &record[PROPER_MOTION_RA],
	                                       &record[PROPER_MOTION_RA + 2]);
  idx += strlen(&obj_info[idx]);
  advance_to_tab();
  sprintf(&obj_info[idx],"Dec:  %.2s\"%.4s",&record[PROPER_MOTION_DEC],
	                                      &record[PROPER_MOTION_DEC + 2]);
  idx += strlen(&obj_info[idx]);
  blank_line();

/* output parallax and radial velocity, if any */
  par_radvel_flag = FALSE;
  if (strncmp(&record[PARALLAX],"         ",PARALLAX_LENGTH) != EQUAL) {
    par_radvel_flag = TRUE;
    if (record[PARALLAX_CODE] == ' ')
      add_string("Trigonometric");
    else if (record[PARALLAX_CODE] == 'D')
      add_string("Dynamical");
    else
      ;

    sprintf(&obj_info[idx]," parallax:  %.1s0\"%.4s",&record[PARALLAX],
	                                               &record[PARALLAX + 1]);
    idx += strlen(&obj_info[idx]);
    advance_tabs(2);
  }

  if (strncmp(&record[RADIAL_VELOCITY],"    ",RADIAL_VEL_LENGTH) != EQUAL) {
    par_radvel_flag = TRUE;

    /* scan the radial velocity magnitude */
    GET_FIELD(field,record,RADIAL_VELOCITY + 1,3);
    rad_vel = atoi(field);
    sprintf(&obj_info[idx],"Radial velocity:  %.1s%d km/sec",
	                                    &record[RADIAL_VELOCITY],rad_vel);
    idx += strlen(&obj_info[idx]);
  }

/* add the rotational velocity (v sin i) information */
  if (strncmp(&record[ROTATIONAL_VELOCITY],"   ",ROTATIONAL_VEL_LENGTH)
                                                                   != EQUAL) {
    /* space down a bit */
    if (par_radvel_flag)
      blank_line();

    add_string("Rotational velocity (v sin i) - Slettebak system:  ");
    GET_FIELD(field,record,ROTATIONAL_VELOCITY,3);
    rot_vel = atoi(field);

    if (record[ROTATIONAL_VEL_CODE] != ' ')
      add_char(record[ROTATIONAL_VEL_CODE]);
    sprintf(&obj_info[idx],"%d km/sec",rot_vel);
    idx += strlen(&obj_info[idx]);

    if (record[ROTATIONAL_VEL_LIMIT] == '=')
      add_string(" (limit)");

    if (record[ROT_VEL_UNCERTAIN] == ':')
      add_string(" (uncertain)");
    else if (record[ROT_VEL_UNCERTAIN] == 'v')
      /* one 'v' occurs in this field; meaning undocumented */
      add_string(" v");
    else
      ;
  }  

/* fill the display out to the correct number of lines */
  while (row < NUM_BSC5_ROWS)
    next_line();
  obj_info[idx] = '\0';

/* and go back */
  return;
}



static void display_pos(posptr)

char *posptr;

{
/* identify the epoch */
  add_string("Position (2000.0)");

/* display the right ascension */
  advance_to_tab();
  sprintf(&obj_info[idx]," RA:   %.2sh %.2sm %.2ss%.2s",&posptr[0],&posptr[2],
	                                               &posptr[4],&posptr[6]);
  idx += strlen(&obj_info[idx]);
  next_line();

/* and display the declination */
  advance_tabs(3);
  sprintf(&obj_info[idx],"Dec:  %.3sd %.2s' %.2s\"",&posptr[8],&posptr[11],
	                                             &posptr[13],&posptr[15]);
  idx += strlen(&obj_info[idx]);
  next_line();

  return;
}



/* find this Bayer designation in the BSC5 Greek-letter table, and return a
 * zero-relative index */

int find_bsc5_bayer(buff)

char *buff;

{
  int greek_idx;

/* loop through the BSC5 Greek-letter table, looking for a match */
  greek_idx = 0;
  while (strcmp(buff,bsc5_greek[greek_idx]) != EQUAL)
    if (++greek_idx >= NUM_GREEK_LETTERS)
      break;

/* if we didn't find it, complain and return a bogus value */
  if (greek_idx >= NUM_GREEK_LETTERS) {
    printf("no Greek letter found to match %s in BSC5\n",buff);
    greek_idx = -1;
  }

/* return zero-relative, as promised */
  return(greek_idx);
}
