/*
 * xsky - an interactive sky atlas
 *
 * Copyright 1992-6, Terry R. Friedrichsen
 *
 * This program may be copied and redistributed, in whole or in part,
 * as long as you don't try to make any money from the sale or redis-
 * tribution of the program or any part of the program, or pretend
 * that you wrote the program or any of its parts unless specifically
 * credited by the original author.
 *
 * You are free to make use of this software in your own programs, as
 * long as you credit the original author where it is due.
 */

/*
 * WARRANTY:
 * xsky was written as a learning project and as a demonstration of
 * X Window System programming.  xsky doesn't do anything; it is not
 * merchantable, and it is not fit for any purpose whatsoever.  In
 * fact, don't use xsky at all; it's free, and you're getting what
 * you paid for.
 */

#include <stdio.h>

#include <stdlib.h>

#include <unistd.h>
#include <limits.h>

#include <string.h>

#include <math.h>

#include <X11/Intrinsic.h>

#include "skydefs.h"

#include "pos.h"
#include "catreader.h"
#include "con-bounds.h"

/* static function prototypes */
static int fill_bound_entry PROTOTYPE((char *,int));
static int calculate_constellation PROTOTYPE((struct display *,int));
static void draw_ps_boundary_line PROTOTYPE((struct display *,int *,int));
static struct point *get_point PROTOTYPE((void));
static void save_point PROTOTYPE((int,int,int));
static struct arraynode *get_arraynode PROTOTYPE((void));
static void save_arrays PROTOTYPE((XPoint *,int *,int));
static void save_point_list PROTOTYPE((void));
static void herget_precession PROTOTYPE((double,double,double *,double *));
static void init_herget_precession PROTOTYPE((double,double));

/* external function prototypes */
extern char *build_filespec PROTOTYPE((char *,char *));
extern boolean read_binary_database PROTOTYPE((char *,void *,int *));
extern void init_precess PROTOTYPE((double,double));
extern boolean write_binary_database PROTOTYPE((char *,void *,size_t,size_t));
extern void precess PROTOTYPE((double,double,double *,double *));
extern void draw_boundary_line PROTOTYPE((Widget,XPoint *,int));
extern void pos_rad_to_X PROTOTYPE((struct display *,double,double,
				                                int *,int *));
extern boolean within_display PROTOTYPE((struct display *,double,double,
					                        int *,int *));
extern void ps_newpath PROTOTYPE((void));
extern void pos_rad_to_X_double PROTOTYPE((struct display *,double,double,
				                          double *,double *));
extern void ps_draw_point PROTOTYPE((struct display *,double,double));
extern void ps_endpath PROTOTYPE((void));
extern void X_to_pos_rad PROTOTYPE((struct display *,int,int,
				                          double *,double *));

#define BOUNDARY_DATA_FILE      "boundaries.dat"
#define BOUNDARY_BINARY_FILE    "boundaries.bin"

/* description of EQ2000 boundary database */
#define BOUNDARY_EPOCH   2000.0

#define BOUND_RA        0     /* nn.nnnnnnn; RA in decimal hours */
#define BOUND_DEC      11     /* szz.nnnnnn; declination in decimal degrees */
#define CONSTELLATION  23     /* xxxx; constellation abbreviation */
#define FLAG           28     /* x; 'O' original point; 'I' interpolated */

#define CONSTELLATION_LENGTH   4

/* a full circle in radians */
#define TWOPI   (2 * PI)

/* conversion factors for hours to radians and degrees to radians */
#define CONVH   (TWOPI / 24)
#define CONVD   (TWOPI / 360)

#define CDR     CONVD

/* epoch of constellation boundaries */
#define B1875   1875.0

/* list of constellation names */
extern char *constellations[];

#define SERPENS   73

/* Herget precession rotation matrix */
static double r[3][3];

/* flag to indicate calculation of rotation matrix */
static boolean herget_init_flag = FALSE;

/* flag that we're drawing in PostScript, not on the display */
extern boolean ps_draw_flag;

struct bound_node {
  double ra_rad;
  double dec_rad;
};

/* array to hold vertex points plus end-of-constellation flags */
#define MAX_BOUND   13422
static struct bound_node bound_array[MAX_BOUND + NUM_CONSTELLATIONS];

static int bound_array_max;

/* boundary record buffer */
#define BOUND_RECLEN   29
static char bound_rec[BOUND_RECLEN + 2 + 1];

struct arraynode {
  struct arraynode *next;
  XPoint *pointarray;
  int *posarray;
  int arraysize;
};

struct point {
  struct point *next;
  XPoint point;
  int bound_array_pos;
};

static struct arraynode *array_list_head = (struct arraynode *)NULL;
static struct arraynode *array_list_rear = (struct arraynode *)NULL;

static struct arraynode *free_arraynode_list = (struct arraynode *)NULL;

static struct point *point_list_head = (struct point *)NULL;
static struct point *point_list_rear = (struct point *)NULL;

static int point_count = 0;

static struct point *free_point_list = (struct point *)NULL;



/* draw constellation boundary lines in the exposed area of the display */

/* ARGSUSED */

void draw_boundary_lines(sky_widget,display,x,y,width,height)

Widget sky_widget;
struct display *display;
int x, y;
int width, height;

{
  struct arraynode *this_array;

#ifdef BOUNDARY_DEBUG
  printf("drawing boundary lines from %d,%d for width,height = %d,%d\n",
                                                            x,y,width,height);

/* print the array list */
  this_array = array_list_head;
  while (this_array != (struct arraynode *)NULL) {
    printf("arraynode at %X; arrays %X and %X; numpoints = %d\n",this_array,
	                         this_array->pointarray,this_array->posarray,
	                         this_array->arraysize);
    this_array = this_array->next;
  }
#endif

/* loop through the array list, drawing each chunk of line segments */
  this_array = array_list_head;
  while (this_array != (struct arraynode *)NULL) {
    if (ps_draw_flag)
      draw_ps_boundary_line(display,this_array->posarray,
			                               this_array->arraysize);
    else
      draw_boundary_line(sky_widget,this_array->pointarray,
		                                       this_array->arraysize);

    /* step to the next array node */
    this_array = this_array->next;
  }
    
  return;
}



/* read the constellation boundary database */

boolean init_boundaries()

{
  char *bin_filebuff;
  char *filebuff;
  FILE *bound_fd;
  int bound_entry;

/* get the path to the boundary binary file */
  bin_filebuff = build_filespec("BOUNDARY_BIN_PATH",BOUNDARY_BINARY_FILE);

/* see if we can read the binary file */
  if (access(bin_filebuff,R_OK) == 0)
    /* we can; try to use the pre-built database */
    if (read_binary_database(bin_filebuff,bound_array,&bound_array_max)) {
      /* success; free the filename buffer */
      free((void *)bin_filebuff);

      /* return success */
      return(TRUE);
    }

/* get the path to the boundary data */
  filebuff = build_filespec("BOUNDARY_PATH",BOUNDARY_DATA_FILE);

/* open the boundary data file */
  if ((bound_fd = fopen(filebuff,"r")) == (FILE *)NULL) {
#ifdef DEBUG
    printf("sky:  can't open boundary data file %s\n",filebuff);
    perror("sky");
    exit(1);
#else
    return(FALSE);
#endif
  }

/* prior to reading the boundary file, build the rotation matrix for precessing
 * the polygon coordinates to the common equinox from the given epoch */
  if (BOUNDARY_EPOCH != COMMON_EQUINOX)
    init_precess(BOUNDARY_EPOCH,COMMON_EQUINOX);

/* read the boundary catalog, saving the information necessary */
  bound_entry = 0;
  while (TRUE) {
    if (fgets(bound_rec,sizeof(bound_rec),bound_fd) == (char *)NULL)
      break;
    bound_entry = fill_bound_entry(bound_rec,bound_entry);
  }

/* add a constellation boundary marker at the end */
  bound_array[bound_entry++].ra_rad = -1.0;

/* close the boundary file */
  (void)fclose(bound_fd);

/* remember the last boundary entry we filled */
  bound_array_max = bound_entry - 1;

/* attempt to save the binary database */
  if (access(bin_filebuff,W_OK) == 0)
    /* it's writeable, so let's try */
    if (! write_binary_database(bin_filebuff,(void *)bound_array,
                                           (size_t)sizeof(struct bound_node),
                                                         (size_t)bound_entry))
      /* dump of binary database failed; be sure the file is gone */
      (void)unlink(bin_filebuff);

/* free the filename buffers */
  free((void *)filebuff);
  free((void *)bin_filebuff);

/* return success */
  return(TRUE);
}



/* given a boundary record, make a data node for it */

static int fill_bound_entry(record,bound_entry)

char *record;
int bound_entry;

{
/* get the decimal RA and convert to radians */
  bound_array[bound_entry].ra_rad = atof(&record[BOUND_RA]);
  /* watch out for the constellation-boundary records */
  if (bound_array[bound_entry].ra_rad != -1.0)
    bound_array[bound_entry].ra_rad *= RAD_PER_HOUR;

/* get the decimal declination and convert to radians */
  bound_array[bound_entry].dec_rad = atof(&record[BOUND_DEC]) * RAD_PER_DEGREE;

/* precess them to the common equinox */
  if (BOUNDARY_EPOCH != COMMON_EQUINOX)
    precess(bound_array[bound_entry].ra_rad,bound_array[bound_entry].dec_rad,
	                                    &bound_array[bound_entry].ra_rad,
	                                    &bound_array[bound_entry].dec_rad);

  return(++bound_entry);
}



/*
 * The logic here is complex enough to merit some discussion.
 *
 * The constellation boundary file defines a series of closed paths, one per
 * constellation.  No attempt is made to avoid duplicating segments around
 * adjoining constellations.  The intent of the code below is to build a
 * linked list of XPoint arrays, each of which describes all or part of a
 * constellation boundary within the display.
 *
 * This is done in two stages.  First, within each constellation, we construct
 * one or more linked lists of points describing line segments which appear
 * completely or partially within the display.  Next, each of these linked
 * lists is converted into an XPoint array and added to the array list.
 *
 * Thus, when it comes time to draw the constellation boundaries, it is only
 * necessary to traverse the array list and call the appropriate X routine
 * to draw the array of points as line segments on the display.
 *
 * In order to reduce the overhead of this somewhat, we only free the arbi-
 * trarily-sized arrays themselves.  The nodes linking the arrays, and the
 * nodes linking the points in the first stage of the build, are retained
 * on our own free list for re-use.
 */

/* construct a list of arrays of sets of points describing those boundary
 * segments which will appear in the current display */

void calculate_boundary_field(display)

struct display *display;

{
  int i;

/* free any previous array list */
  if (array_list_head != (struct arraynode *)NULL) {
    array_list_rear->next = free_arraynode_list;
    free_arraynode_list = array_list_head;
    array_list_head = array_list_rear = (struct arraynode *)NULL;
  }

/* loop through all constellations in the boundary array */
  i = 0;
  do {
    i = calculate_constellation(display,i);
  } while (i < bound_array_max);

  return;
}



/* add point arrays for this constellation to the point array list */

static int calculate_constellation(display,i)

struct display *display;
int i;

{
  int first_i;
  double first_point_ra, first_point_dec;
  boolean start_within_flag, end_within_flag;
  int x1, y1;
  int x2, y2;
  boolean first_segment = TRUE;

/* save the the first point */
  first_i = i;
  first_point_ra = bound_array[i].ra_rad;
  first_point_dec = bound_array[i].dec_rad;

/* see if the first point is within the display */
  start_within_flag = within_display(display,first_point_ra,
				                     first_point_dec,&x1,&y1);

  /* step to the second point */
  i++;

/* loop through the entire constellation, examining line segments */
  do {
    /* see if the line segment ends within the display */
    end_within_flag = within_display(display,bound_array[i].ra_rad,
				              bound_array[i].dec_rad,&x2,&y2);

/* if either end of the line segment falls within the display, save the line */
    if (start_within_flag || end_within_flag) {

      /* if this is the first segment, save the starting point */
      if (first_segment) {
	/* if the coordinates are nonsense, calculate the correct ones */
	if (x1 == INT_MAX)
	  pos_rad_to_X(display,bound_array[i].ra_rad,
		                          bound_array[i - 1].dec_rad,&x1,&y1);

	/* save the start point of the segment */
	if (display->downunder_flag)
	  save_point(display->width - x1,display->height - y1,i - 1);
	else
	  save_point(x1,y1,i - 1);
	first_segment = FALSE;
      }

      /* if the coordinates are nonsense, calculate the correct ones */
      if (x2 == INT_MAX)
	pos_rad_to_X(display,bound_array[i].ra_rad,bound_array[i].dec_rad,
		                                                     &x2,&y2);

      /* save the end point of the segment */
      if (display->downunder_flag)
	save_point(display->width - x2,display->height - y2,i);
      else
	save_point(x2,y2,i);
    }
    else {
      /* the line segment is entirely outside the display - end this array */
      if (point_list_head != (struct point *)NULL)
	save_point_list();

      /* revert to hunting for an initial displayable segment again */
      first_segment = TRUE;
    }

    /* step to the next point */
    start_within_flag = end_within_flag;
    x1 = x2;
    y1 = y2;
    i++;
  } while (bound_array[i].ra_rad != -1.0);

/* use the first point as the end of the final segment */
  /* see if the line segment ends within the display */
  end_within_flag = within_display(display,first_point_ra,
				                     first_point_dec,&x2,&y2);

/* if either end of the line segment falls within the display, save the line */
  if (start_within_flag || end_within_flag) {

    /* if this is the first segment, save the starting point */
    if (first_segment) {
      /* if the coordinates are nonsense, calculate the correct ones */
      if (x1 == INT_MAX)
	pos_rad_to_X(display,bound_array[i - 1].ra_rad,
		                          bound_array[i - 1].dec_rad,&x1,&y1);

      /* save the start point of the segment */
      if (display->downunder_flag)
	save_point(display->width - x1,display->height - y1,i - 1);
      else
	save_point(x1,y1,i - 1);
    }

    /* if the coordinates are nonsense, calculate the correct ones */
    if (x2 == INT_MAX)
      pos_rad_to_X(display,first_point_ra,first_point_dec,&x2,&y2);

    /* save the end point of the segment */
    if (display->downunder_flag)
      save_point(display->width - x2,display->height - y2,first_i);
    else
      save_point(x2,y2,first_i);
  }

/* save any remaining list of points */
  if (point_list_head != (struct point *)NULL)
    save_point_list();

/* return the index of the start of the next constellation */
  return(++i);
}



/* get a free point structure */

static struct point *get_point()

{
  struct point *point;

/* pick one off the free list if possible */
  if (free_point_list != (struct point *)NULL) {
    point = free_point_list;
    free_point_list = free_point_list->next;
  }
  else
    /* otherwise, get a new one */
    point = (struct point *)malloc(sizeof(struct point));

  return(point);
}



/* get a free array structure */

static struct arraynode *get_arraynode()

{
  struct arraynode *arraynode;

/* pick one off the free list if possible */
  if (free_arraynode_list != (struct arraynode *)NULL) {
    arraynode = free_arraynode_list;
    free_arraynode_list = free_arraynode_list->next;

    /* free the old arrays attached to it */
    free((void *)arraynode->pointarray);
    free((void *)arraynode->posarray);
  }
  else
    /* otherwise, get a new one */
    arraynode = (struct arraynode *)malloc(sizeof(struct arraynode));

/* initialize the array pointers */
  arraynode->pointarray = (XPoint *)NULL;
  arraynode->posarray = (int *)NULL;

  return(arraynode);
}



/* save a point on the point list */

static void save_point(x,y,i)

int x, y;
int i;

{
  struct point *this_point;

/* get a new point structure and initialize it */
  this_point = get_point();
  this_point->next = (struct point *)NULL;
  this_point->point.x = (short)x;
  this_point->point.y = (short)y;
  this_point->bound_array_pos = i;

/* link it into the point list */
  if (point_list_head == (struct point *)NULL)
    /* empty list */
    point_list_head = point_list_rear = this_point;
  else {
    point_list_rear->next = this_point;
    point_list_rear = this_point;
  }

/* keep count of the points */
  point_count++;

  return;
}



/* save an array on the array list */

static void save_arrays(arrayptr,posptr,size)

XPoint *arrayptr;
int *posptr;
int size;

{
  struct arraynode *this_arraynode;

/* get a new arraynode structure and initialize it */
  this_arraynode = get_arraynode();
  this_arraynode->next = (struct arraynode *)NULL;
  this_arraynode->pointarray = arrayptr;
  this_arraynode->posarray = posptr;
  this_arraynode->arraysize = size;

/* link it into the array list */
  if (array_list_head == (struct arraynode *)NULL)
    /* empty list */
    array_list_head = array_list_rear = this_arraynode;
  else {
    array_list_rear->next = this_arraynode;
    array_list_rear = this_arraynode;
  }

#ifdef BOUNDARY_DEBUG
/* print the array list */
  this_arraynode = array_list_head;
  while (this_arraynode != (struct arraynode *)NULL) {
    printf("arraynode at %X; arrays %X and %X; array size = %d\n",
	                                           this_arraynode,
	                                           this_arraynode->pointarray,
	                                           this_arraynode->posarray,
	                                           this_arraynode->arraysize);
    this_arraynode = this_arraynode->next;
}
#endif

  return;
}



/* save a point list as an array list entry */

static void save_point_list()

{
  XPoint *point_array;
  int *pos_array;
  struct point *this_point;
  int i;

/* grab an array big enough to hold all the XPoint structures */
  point_array = (XPoint *)malloc(sizeof(XPoint) * point_count);

/* grab an array big enough to hold all the bound array indexes */
  pos_array = (int *)malloc(sizeof(int) * point_count);

/* fill the array from the point list */
  this_point = point_list_head;
  i = 0;
  while (this_point != (struct point *)NULL) {
    point_array[i].x = this_point->point.x;
    point_array[i].y = this_point->point.y;
    pos_array[i++] = this_point->bound_array_pos;
    this_point = this_point->next;
  }

/* save the arrays on the array list */
  save_arrays(point_array,pos_array,point_count);

/* free the point list */
  point_list_rear->next = free_point_list;
  free_point_list = point_list_head;
  point_list_head = point_list_rear = (struct point *)NULL;
  point_count = 0;

  return;
}



/* draw a constellation boundary on the PostScript display */

static void draw_ps_boundary_line(display,arraypos,numpoints)

struct display *display;
int *arraypos;
int numpoints;

{
  int i;
  double ps_x, ps_y;

/* start a new path for this line */
  ps_newpath();

/* draw a line through all the points */
  for (i = 0; i < numpoints; i++) {
    /* convert the position from radians to floating X coordinates */
    pos_rad_to_X_double(display,bound_array[arraypos[i]].ra_rad,
		                          bound_array[arraypos[i]].dec_rad,
		                                                 &ps_x,&ps_y);

    /* draw a line to this point */
    ps_draw_point(display,ps_x,ps_y);
  }

/* end the path for this line and draw it */
  ps_endpath();

  return;
}



/* find the constellation containing the given position - this is adapted
 * from FORTRAN-77 code distributed by NASA's Astronomical Data Center along
 * with catalog 6/6042; the code is due to Nancy G. Roman as a companion to
 * the catalog, described in 1987, PASP, 99, 695.  she credits the Herget
 * precession routine is credited to Dr. Wayne H. Warren, Jr.; note, however,
 * that I've hacked up the precession routine somewhat.
 *
 * thanks to Bill Owen at JPL and Mike Delevoryas for pointing me to Dr.
 * Roman's paper and this clever method of identifying the constellation
 * containing a given point on the sky */

char *identify_constellation(display,x,y)

struct display *display;
int x, y;

{
  double arad, drad;
  double a, d;
  double ra, dec;
  int i;
  int con;

/* convert the X position to RA and declination in radians */
  X_to_pos_rad(display,x,y,&arad,&drad);

/* precess the position to the constellation boundary epoch */
  herget_precession(arad,drad,&a,&d);

/* normalize the RA */
  if (a < 0)
    a += TWOPI;
  if (a >= TWOPI)
    a -= TWOPI;

/* convert to decimal RA and declination */
  ra = a / CONVH;
  dec = d / CONVD;

/*
 * find constellation such that the declination entered is higher than
 *  the lower boundary of the constellation when the upper and lower
 *  right ascensions for the constellation bound the entered right
 *  ascension
 */

/* first, find the declination */
  i = 0;
  while (bound_list[i].decl > dec)
    i++;

  while (TRUE) {
/* next, find an upper RA that fits */
    while (bound_list[i].rau <= ra)
      i++;

/* then look for a lower RA that fits */
    while (bound_list[i].ral > ra)
      i++;

/* if this boundary line fits, this is the right constellation */
    if ((ra >= bound_list[i].ral) && (ra < bound_list[i].rau) &&
	                                        (bound_list[i].decl <= dec)) {
      con = bound_list[i].conidx;
      break;
    }
    else if (bound_list[i].rau < ra)
      /* this line was too short; go back to the upper-limit search */
      continue;
    else
      /* otherwise, we mysteriously failed to find a constellation */
      return((char *)NULL);
  }

/* if the constellation is "Serpens", separate into Caput and Cauda */
  if (con == SERPENS)
    if ((bound_list[i].decl == 22.0000) || (bound_list[i].decl == -3.2500))
      return("Serpens Caput");
    else
      return("Serpens Cauda");
  else
    return(constellations[con]);

/* NOTREACHED */
}



/*
 * Herget precession, see p. 9 of Publ. Cincinnati Obs. no. 24
 *
 * input:  ra1 and dec1 in radians
 * output: ra2 and dec2 in radians
 */

static void herget_precession(ra1,dec1,ra2,dec2)

double ra1, dec1;
double *ra2, *dec2;

{
  double a;
  double x1[3], x2[3];
  int i, j;

/* if we haven't calculated the rotation matrix yet, do it now */
  if (! herget_init_flag)
    init_herget_precession(COMMON_EQUINOX,B1875);

/* compute input direction cosines */
  a = cos(dec1);
  x1[0] = a * cos(ra1);
  x1[1] = a * sin(ra1);
  x1[2] = sin(dec1);

/* perform the rotation to get the direction cosines at epoch2 */
  for (i = 0; i < 3; i++) {
    x2[i] = 0;
    for (j = 0; j < 3; j++)
      x2[i] += r[i][j] * x1[j];
  }

/* calculate the new RA and declination from the new direction cosines */
  *ra2 = atan2(x2[1],x2[0]);
  if (*ra2 < 0)
    *ra2 += TWOPI;
  *dec2 = asin(x2[2]);

  return;
}



/* initialize the Herget precession routine */

static void init_herget_precession(epoch1,epoch2)

double epoch1, epoch2;

{
  double t, st, csr;
  double a, b, c;
  double sina, sinb, sinc;
  double cosa, cosb, cosc;

/* set up the rotation matrix (r) */
  csr = CDR / 3600;
  t = 0.001 * (epoch2 - epoch1);
  st = 0.001 * (epoch1 - 1900);
  a = csr * t * (23042.53 + st * (139.75 + 0.06 * st) + t *
		                              (30.23 - 0.27 * st + 18.0 * t));
  b = csr * t * t * (79.27 + 0.66 * st + 0.32 * t) + a;
  c = csr * t * (20046.85 - st * (85.33 + 0.37 * st) + t *
		                             (-42.67 - 0.37 * st - 41.8 * t));

  sina = sin(a);
  sinb = sin(b);
  sinc = sin(c);
  cosa = cos(a);
  cosb = cos(b);
  cosc = cos(c);

  r[0][0] = cosa * cosb * cosc - sina * sinb;
  r[0][1] = -cosa * sinb - sina * cosb * cosc;
  r[0][2] = -cosb * sinc;
  r[1][0] = sina * cosb + cosa * sinb * cosc;
  r[1][1] = cosa * cosb - sina * sinb * cosc;
  r[1][2] = -sinb * sinc;
  r[2][0] = cosa * sinc;
  r[2][1] = -sina * sinc;
  r[2][2] = cosc;

/* flag that the Herget precession rotation matrix is initialized */
  herget_init_flag = TRUE;

  return;
}
