/*
 * $Header: /home/orchestra5/davy/stuff/misc/xsat/RCS/maps.c,v 1.1 92/04/10 14:08:11 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Routines for dealing with the "maps" menu, drawing maps on the screen and
 * in PostScript, etc.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	maps.c,v $
 * Revision 1.1  92/04/10  14:08:11  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/Xaw/Cardinals.h>
#include <stdio.h>

#include "xsat.h"

static MapInfo mapInfo[] = {
	{ "Africa",		"map.africa",
	    -29.4440,	-54.7660,	 74.0970,	 41.5830,
	   1 },
	{ "Antarctica",		"map.antarctica",
	   -179.1379,	-85.5060,	179.2119,	-56.1500,
	   1 },
	{ "Asia",		"map.asia",
	     32.6750,	-11.1440,	190.3679,	 79.6949,
	   1 },
	{ "Australia",		"map.australia",
	    112.9570,	-50.9990,	184.5289,	-10.7920,
	   1 },
	{ "Europe",		"map.europe",
	    -25.0520,	 31.0960,	105.0430,	 81.4910,
	   1 },
	{ "Greenland",		"map.greenland",
	    -72.7720,	 59.8480,	-11.9840,	 83.7540,
	   1 },
	{ "North America",	"map.namerica",
	   -178.0709,	 25.2050,	-52.8140,	 83.1390,
	   1 },
	{ "Pacific",		"map.pacific",
	   -177.7920,	-27.5060,	-91.1169,	 28.3850,
	   1 },
	{ "South America",	"map.samerica",
	   -177.7920,	-55.7040,	-34.7290,	 32.4820,
	   1 },
	{ "United States",	"map.usa",
	   -124.6833,	 25.1326,	-67.0096,	 49.3810,
	   0 },
	{ "World",		"",
	   -179.1379,	-85.5060,	190.3679,	 83.7540,
	   0 }
};

#define MINX   -1000.0
#define MINY   -1000.0

static void	displayMapX();
static void	displayMapPS();

/*
 * Select a map and put it on the screen.
 */
void
mapsSelectMap(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	int i;
	char *mname;
	char mfname[BUFSIZ];

	if (w != NULL)
		mname = XtName(w);
	else
		mname = (char *) clientData;

	if ((currentMap = findMap(mname)) == NULL) {
		fprintf(stderr, "Unknown map name: %s\n", mname);
		exit(1);
	}

	canvasBusy(mapCanvas);
	canvasClearPaper(mapCanvas);
	canvasClearScratch(mapCanvas);

	/*
	 * World map is done by concatenating all the others.
	 */
	if (strcmp(mname, "World") == 0) {
		for (i=0; i < XtNumber(mapInfo); i++) {
			if (mapInfo[i].m_partofworld == 0)
				continue;

			sprintf(mfname, "%s/%s", XSATLIBDIR,
				mapInfo[i].m_file);
			displayMapX(mfname);
		}
	}
	else {
		sprintf(mfname, "%s/%s", XSATLIBDIR, currentMap->m_file);
		displayMapX(mfname);
	}

	canvasUpdateWindow(mapCanvas);
	canvasIdle(mapCanvas);
}

/*
 * Draw a map in PostScript.
 */
void
mapsDrawMapPS()
{
	int i;
	char mfname[BUFSIZ];

	/*
	 * World is done by concatenating all the others.
	 */
	if (strcmp(currentMap->m_name, "World") == 0) {
		for (i=0; i < XtNumber(mapInfo); i++) {
			if (mapInfo[i].m_partofworld == 0)
				continue;

			sprintf(mfname, "%s/%s", XSATLIBDIR,
				mapInfo[i].m_file);
			displayMapPS(mfname);
		}
	}
	else {
		sprintf(mfname, "%s/%s", XSATLIBDIR, currentMap->m_file);
		displayMapPS(mfname);
	}
}

/*
 * Draw an X line.
 */
void
mapsDrawLineX(x1, y1, x2, y2)
double x1, y1, x2, y2;
{
	float xscale, yscale;

	xscale = (MAPWIN_WIDTH - 10) /
	                    (currentMap->m_maxx - currentMap->m_minx);
	yscale = (MAPWIN_HEIGHT - 10) /
			    (currentMap->m_maxy - currentMap->m_miny);

	x1 = (x1 - currentMap->m_minx) * xscale + 0.5 + 5;
	x2 = (x2 - currentMap->m_minx) * xscale + 0.5 + 5;

	y1 = MAPWIN_HEIGHT - ((y1 - currentMap->m_miny) * yscale + 0.5 + 5);
	y2 = MAPWIN_HEIGHT - ((y2 - currentMap->m_miny) * yscale + 0.5 + 5);

	canvasDrawLineScratch(mapCanvas, (int) x1, (int) y1, (int) x2,
			     (int) y2);
}

/*
 * Draw some X text.
 */
void
mapsDrawTextX(x, y, str)
double x, y;
char *str;
{
	float xscale, yscale;

	xscale = (MAPWIN_WIDTH - 10) /
	                    (currentMap->m_maxx - currentMap->m_minx);
	yscale = (MAPWIN_HEIGHT - 10) /
	                    (currentMap->m_maxy - currentMap->m_miny);

	x = (x - currentMap->m_minx) * xscale + 0.5 + 5;
	y = MAPWIN_HEIGHT - ((y - currentMap->m_miny) * yscale + 0.5 + 5);

	canvasDrawTextScratch(mapCanvas, (int) x, (int) y, str);
}

/*
 * Draw a PostScript line.
 */
static double endx = MINX, endy = MINY;

void
mapsDrawLinePS(x1, y1, x2, y2)
double x1, y1, x2, y2;
{
	x1 -= currentMap->m_minx;
	x2 -= currentMap->m_minx;

	y1 -= currentMap->m_miny;
	y2 -= currentMap->m_miny;

	if ((x1 == endx) && (y1 == endy))
		fprintf(psFP, "%.4lf %.4lf L\n",x2,y2);
	else
		fprintf(psFP, "E %.4lf %.4lf %.4lf %.4lf S\n", x2, y2, x1, y1);

	endx = x2;
	endy = y2;
}

/*
 * Draw some PostScript text.
 */
void
mapsDrawTextPS(x, y, str)
double x, y;
char *str;
{
	x -= currentMap->m_minx;
	y -= currentMap->m_miny;

	fprintf(psFP, "(%s) %.4lf %.4lf T\n", str, x, y);
}

/*
 * Find the named map in the MapInfo structures.
 */
MapInfo *
findMap(mname)
char *mname;
{
	int i;
	MapInfo *mi;

	mi = NULL;

	for (i=0; i < XtNumber(mapInfo); i++) {
		if (strcmp(mapInfo[i].m_name, mname) == 0) {
			mi = &mapInfo[i];
			break;
		}
	}

	return(mi);
}

/*
 * Display a map in the X window.
 */
static void
displayMapX(fname)
char *fname;
{
	int n;
	FILE *fp;
	float xscale, yscale;
	float x1, y1, x2, y2;
	XSegment segments[MAP_NSEGMENTS];

	if ((fp = fopen(fname, "r")) == NULL) {
		perror(fname);
		return;
	}
	
	xscale = (MAPWIN_WIDTH - 10) /
			    (currentMap->m_maxx - currentMap->m_minx);
	yscale = (MAPWIN_HEIGHT - 10) /
			    (currentMap->m_maxy - currentMap->m_miny);

	n = -1;
	while (fscanf(fp, "%f%f%f%f", &x1, &y1, &x2, &y2) != EOF) {
		n++;

		segments[n].x1 = (short) ((x1 - currentMap->m_minx) *
						  xscale + 0.5) + 5;
		segments[n].x2 = (short) ((x2 - currentMap->m_minx) *
						  xscale + 0.5) + 5;
		segments[n].y1 = MAPWIN_HEIGHT -
					 (short) (((y1 - currentMap->m_miny) *
						   yscale + 0.5) + 5);
		segments[n].y2 = MAPWIN_HEIGHT -
					 (short) (((y2 - currentMap->m_miny) *
						   yscale + 0.5) + 5);

		if (n == (MAP_NSEGMENTS-1)) {
			canvasDrawSegmentsPaper(mapCanvas, segments, n);
			n = -1;
		}
	}

	if (n >= 0)
		canvasDrawSegmentsPaper(mapCanvas, segments, n);

	fclose(fp);
}

/*
 * Display a map in PostScript.
 */
static void
displayMapPS(fname)
char *fname;
{
	FILE *fp;
	float x1, y1, x2, y2;

	if ((fp = fopen(fname, "r")) == NULL) {
		perror(fname);
		return;
	}
	
	while (fscanf(fp, "%f%f%f%f", &x1, &y1, &x2, &y2) != EOF)
		mapsDrawLinePS(x1, y1, x2, y2);

	fclose(fp);
}
