/*
 * $Header: /home/orchestra5/davy/stuff/misc/xsat/RCS/commands.c,v 1.1 92/04/10 14:08:09 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Routines for dealing with the "commands" menu.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	commands.c,v $
 * Revision 1.1  92/04/10  14:08:09  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Dialog.h>
#include <stdio.h>

#include "xsat.h"

#define MAXLINES     38
#define LINEHEIGHT   14

static void	doPostScript();
static void	setStart();
static void	setStop();
static void	setStep();
static void	setElev();
static void	setTime();
static void	doPrint();
static void	doSave();

/*
 * Prompt the user for a new start time.
 */
void
commandsSetStartTime(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	prompt("Enter start time:", startTimeStr, setStart);
}

/*
 * Prompt the user for a new stop time.
 */
void
commandsSetStopTime(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	prompt("Enter stop time:", stopTimeStr, setStop);
}

/*
 * Prompt the user for a new step time.
 */
void
commandsSetStepTime(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[16];

	sprintf(buf, "%d", stepTime);
	prompt("Enter step time (seconds):", buf, setStep);
}

/*
 * Prompt the user for a new minimum elevation.
 */
void
commandsSetMinElev(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[16];

	sprintf(buf, "%d", minElevation);
	prompt("Enter minimum elevation (degrees):", buf, setElev);
}

/*
 * Prompt the user for a new time label frequency.
 */
void
commandsSetTimeLabels(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[16];

	sprintf(buf, "%d", timeLabelFreq);
	prompt("Enter time label frequency (minutes):", buf, setTime);
}

/*
 * Run the orbit prediction.
 */
void
commandsPredict(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	Arg argList[2];
	char buf[BUFSIZ];

	if (currentCity == NULL) {
		prompt("Must select a city first.", NULL, promptOkay);
		return;
	}

	if (currentSat == NULL) {
		prompt("Must select a satellite first.", NULL, promptOkay);
		return;
	}

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	/*
	 * Clear the scratch area if we're drawing.
	 */
	if (currentMap != NULL) {
		canvasClearScratch(mapCanvas);
		canvasUpdateWindow(mapCanvas);
	}

	canvasBusy(mapCanvas);

	/*
	 * Run the simulation.
	 */
	orbitPredict(mapsDrawLineX, mapsDrawTextX);

	/*
	 * Update the window if we drew.
	 */
	if (currentMap != NULL)
		canvasUpdateWindow(mapCanvas);

	fflush(tmpFP);

	/*
	 * Label stuff and reset the text widget.
	 */
	(void)changeTextSource(orbitText,tmpFile);

	sprintf(buf, "Orbit Information for %s from %s, %s",
		currentSat->s_name, currentCity->c_city,
		currentCity->c_statecountry);
	XtSetArg(argList[0], XtNlabel, buf);
	XtSetValues(botLabel, argList, ONE);

	canvasIdle(mapCanvas);
}

/*
 * Print the output.
 */
void
commandsPrint(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	if (currentCity == NULL) {
		prompt("Must select a city first.", NULL, promptOkay);
		return;
	}

	if (currentSat == NULL) {
		prompt("Must select a satellite first.", NULL, promptOkay);
		return;
	}

	prompt("Print command:", printCmd, doPrint);
}

/*
 * Save the output to a file.
 */
void
commandsSave(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	if (currentCity == NULL) {
		prompt("Must select a city first.", NULL, promptOkay);
		return;
	}

	if (currentSat == NULL) {
		prompt("Must select a satellite first.", NULL, promptOkay);
		return;
	}

	prompt("File name:", saveFile, doSave);
}

/*
 * Quit.
 */
void
commandsQuit(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	XtUnmapWidget(topLevel);
	XCloseDisplay(XtDisplay(topLevel));

	unlink(tmpFile);
	exit(0);
}

/*
 * Callback for setting start time.
 */
static void
setStart(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	Arg argList[2];

	new = XawDialogGetValueString(XtParent(w));

	startTime = parsedate(new);
	new = topLabelStr(startTime, stopTime, stepTime, minElevation);

	XtSetArg(argList[0], XtNlabel, new);
	XtSetValues(topLabel, argList, ONE);
	XtDestroyWidget(clientData);
}

/*
 * Callback for setting stop time.
 */
static void
setStop(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	Arg argList[2];

	new = XawDialogGetValueString(XtParent(w));

	stopTime = parsedate(new);
	new = topLabelStr(startTime, stopTime, stepTime, minElevation);

	XtSetArg(argList[0], XtNlabel, new);
	XtSetValues(topLabel, argList, ONE);
	XtDestroyWidget(clientData);
}

/*
 * Callback for setting step time.
 */
static void
setStep(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	Arg argList[2];

	new = XawDialogGetValueString(XtParent(w));

	stepTime = atoi(new);
	new = topLabelStr(startTime, stopTime, stepTime, minElevation);

	XtSetArg(argList[0], XtNlabel, new);
	XtSetValues(topLabel, argList, ONE);
	XtDestroyWidget(clientData);
}

/*
 * Callback for setting minimum elevation.
 */
static void
setElev(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	Arg argList[2];

	new = XawDialogGetValueString(XtParent(w));

	minElevation = atoi(new);
	new = topLabelStr(startTime, stopTime, stepTime, minElevation);

	XtSetArg(argList[0], XtNlabel, new);
	XtSetValues(topLabel, argList, ONE);
	XtDestroyWidget(clientData);
}

/*
 * Callback for setting time label frequency.
 */
static void
setTime(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	Arg argList[2];

	new = XawDialogGetValueString(XtParent(w));

	timeLabelFreq = atoi(new);
	XtDestroyWidget(clientData);
}

/*
 * Callback for printing the file.
 */
static void
doPrint(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	strcpy(printCmd, XawDialogGetValueString(XtParent(w)));

	if ((psFP = popen(printCmd, "w")) == NULL) {
		prompt("Cannot open pipe to print command.", NULL, promptOkay);
		return;
	}

	canvasBusy(mapCanvas);

	doPostScript();
	pclose(psFP);

	XtDestroyWidget(clientData);
	canvasIdle(mapCanvas);
}

/*
 * Callback for saving the file.
 */
static void
doSave(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	strcpy(saveFile, XawDialogGetValueString(XtParent(w)));

	if ((psFP = fopen(saveFile, "w")) == NULL) {
		prompt("Cannot open save file.", NULL, promptOkay);
		return;
	}

	canvasBusy(mapCanvas);

	doPostScript();
	fclose(psFP);

	XtDestroyWidget(clientData);
	canvasIdle(mapCanvas);
}

/*
 * Generate PostScript output.
 */
static void
doPostScript()
{
	int nlines;
	char buf[BUFSIZ];
	float mapwidth, mapheight;

	fprintf(psFP, "%%!\n0 setgray\n0 setlinewidth\n");

	fprintf(psFP, "/S { moveto lineto } bind def\n");
	fprintf(psFP, "/L { lineto } bind def\n");
	fprintf(psFP, "/E { stroke newpath } bind def\n");
	fprintf(psFP, "/T { gsave moveto show grestore } bind def\n");

	fprintf(psFP, "gsave initclip clippath pathbbox grestore\n");
	fprintf(psFP, "/ury exch def /urx exch def\n");
	fprintf(psFP, "/lly exch def /llx exch def\n");

	/* factor in the PostScript margins (remember that the maps are
	 * drawn with the short edge of the paper as the y coordinate) */
	if (appResources.mapTopMargin != 0)
	  fprintf(psFP, "/llx llx %f add def\n",
		                        mmtopt(appResources.mapTopMargin));
	if (appResources.mapLeftMargin != 0)
	  fprintf(psFP, "/lly lly %f add def\n",
		                        mmtopt(appResources.mapLeftMargin));
	if (appResources.mapBottomMargin != 0)
	  fprintf(psFP, "/urx urx %f sub def\n",
		                        mmtopt(appResources.mapBottomMargin));
	if (appResources.mapRightMargin != 0)
	  fprintf(psFP, "/ury ury %f sub def\n",
		                        mmtopt(appResources.mapRightMargin));

	fprintf(psFP, "urx lly translate 90 rotate\n");
	fprintf(psFP, "/yhgt urx llx sub def\n");
	fprintf(psFP, "/xwid ury lly sub def\n");

	/*
	 * If there's a map selected, print it first.
	 */
	if (currentMap != NULL) {
		mapwidth = currentMap->m_maxx - currentMap->m_minx;
		mapheight = currentMap->m_maxy - currentMap->m_miny;

		fprintf(psFP, "0 36 translate\n");
		fprintf(psFP, "gsave\n");

		fprintf(psFP, "/xscale xwid %.4f div def\n", mapwidth);
		fprintf(psFP, "/yscale yhgt 36 sub %.4f div def\n", mapheight);
		fprintf(psFP, "xscale yscale scale\n");

		fprintf(psFP, "/Helvetica findfont ");
		fprintf(psFP, "[ 8 xscale div 0 0 8 yscale div 0 0 ] ");
		fprintf(psFP, "makefont setfont\n");
		fprintf(psFP, "newpath\n");

		mapsDrawMapPS();
	}

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	/*
	 * Run the orbit simulation.
	 */
	orbitPredict(mapsDrawLinePS, mapsDrawTextPS);

	fflush(tmpFP);

	/*
	 * Finish up the map, if there was one.
	 */
	if (currentMap != NULL) {
		fprintf(psFP, "stroke grestore\n");

		fprintf(psFP, "/Helvetica findfont 10 scalefont setfont\n");

		fprintf(psFP, "0 -16 moveto (%s) show\n",
			currentSat->s_name);
		fprintf(psFP, "0 -28 moveto (NORAD catalog #%lu) show\n",
			currentSat->s_satnum);

		fprintf(psFP, "/str (Start Time: %s) def\n", startTimeStr);
		fprintf(psFP, "xwid str stringwidth pop sub -16 moveto\n");
		fprintf(psFP, "str show\n");

		fprintf(psFP, "/str (Stop Time: %s) def\n", stopTimeStr);
		fprintf(psFP, "xwid str stringwidth pop sub -28 moveto\n");
		fprintf(psFP, "str show\n");

		fprintf(psFP, "showpage\n");
		fprintf(psFP, "urx lly translate 90 rotate\n");
	}

	fprintf(psFP, "/Courier findfont 12 scalefont setfont\n");

	rewind(tmpFP);

	/*
	 * Emit the header.
	 */
	fprintf(psFP, "/top yhgt yhgt %d sub 2 div sub %d sub def\n",
                                           MAXLINES * LINEHEIGHT, LINEHEIGHT);
	fprintf(psFP, "72 top moveto\n");

	fprintf(psFP, "(Orbit Information for %s from %s, %s) show\n",
		currentSat->s_name, currentCity->c_city,
		currentCity->c_statecountry);

	nlines = 2;

	/*
	 * Copy in the orbit data.
	 */
	while (fgets(buf, sizeof(buf), tmpFP) != NULL) {
		buf[strlen(buf)-1] = '\0';
		fprintf(psFP, "72 top %d sub moveto\n", nlines * LINEHEIGHT);
		fprintf(psFP, "(%s) show\n", buf);

		if (++nlines == MAXLINES) {
			fprintf(psFP, "showpage\n");
			fprintf(psFP, "urx lly translate 90 rotate\n");

			nlines = 0;
		}
	}

	if (nlines > 0)
		fprintf(psFP, "showpage\n");
}
