/*
 * $Header: /tmp_mnt/net/orchestra5/davy/stuff/misc/xsat/RCS/cities.c,v 1.1 92/04/10 14:27:24 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Routines for dealing with the "cities" menu, reading the city information
 * file, etc.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	cities.c,v $
 * Revision 1.1  92/04/10  14:27:24  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Dialog.h>
#include <stdio.h>
#include <string.h>

#include "xsat.h"

#define EQUAL   0

static CityInfo	*cityInfo;

static void	setCity();

/*
 * Prompt the user for a new city name.
 */
void
citiesSelectCity(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[BUFSIZ];

	bzero(buf, sizeof(buf));

	if (currentCity != NULL) {
		sprintf(buf, "%s, %s", currentCity->c_city,
			currentCity->c_statecountry);
	}

	prompt("Enter city name:", buf, setCity);
}

/*
 * Display information about the current city.
 */
void
citiesShowCity(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	Arg argList[2];
	char buf[BUFSIZ];

	if (currentCity == NULL) {
		prompt("No city selected.", NULL, promptOkay);
		return;
	}

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	fprintf(tmpFP, "City Name:     %s\n", currentCity->c_city);
	fprintf(tmpFP, "State/Country: %s\n", currentCity->c_statecountry);
	fprintf(tmpFP, "Latitude:      %.4f deg %c\n",
		abs(currentCity->c_latitude),
		currentCity->c_latitude < 0 ? 'S' : 'N');
	fprintf(tmpFP, "Longitude:     %.4f deg %c\n",
		abs(currentCity->c_longitude),
		currentCity->c_longitude < 0 ? 'W' : 'E');
	fprintf(tmpFP, "Elevation:     %.2f m\n", currentCity->c_elevation);

	fflush(tmpFP);

	/* reset the text source to get the new file contents read in */
	(void)changeTextSource(orbitText,tmpFile);

	sprintf(buf, "Information for Current City");
	XtSetArg(argList[0], XtNlabel, buf);
	XtSetValues(botLabel, argList, ONE);
}

/*
 * List all known cities.
 */
void
citiesListCities(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	CityInfo *ci;
	Arg argList[2];
	char buf[BUFSIZ];

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	for (ci = cityInfo; ci != NULL; ci = ci->c_next)
		fprintf(tmpFP, "%s, %s\n", ci->c_city, ci->c_statecountry);

	fflush(tmpFP);

	/* reset the text source to get the new file contents read in */
	(void)changeTextSource(orbitText,tmpFile);

	sprintf(buf, "List of Known Cities");
	XtSetArg(argList[0], XtNlabel, buf);
	XtSetValues(botLabel, argList, ONE);
}

/*
 * Read in the CITYFILE.
 */
void
readCityList()
{
	char c;
	FILE *fp;
	int d, m, s;
	CityInfo *ci;
	char line[BUFSIZ];
	char *p, *city, *stcntry, *lat, *lon, *elev;

	sprintf(line, "%s/%s", XSATLIBDIR, CITYFILE);

	if ((fp = fopen(line, "r")) == NULL) {
		perror(line);
		exit(1);
	}

	/*
	 * Should probably do a better job of checking the
	 * input format.
	 */
	while (fgets(line, sizeof(line), fp) != NULL) {
		if ((*line == '#') || (*line == '\n'))
			continue;

		line[strlen(line)-1] = '\0';

		p = city = line;

		while ((*p != ',') && (*p != '\0'))
			p++;

		*p++ = '\0';
		while ((*p == ' ') || (*p == '\t'))
			p++;

		stcntry = p;
		while ((*p != ';') && (*p != '\0'))
			p++;

		*p++ = '\0';
		while ((*p == ' ') || (*p == '\t'))
			p++;

		lat = p;
		while ((*p != ';') && (*p != '\0'))
			p++;

		*p++ = '\0';
		while ((*p == ' ') || (*p == '\t'))
			p++;

		lon = p;
		while ((*p != ';') && (*p != '\0'))
			p++;

		*p++ = '\0';
		while ((*p == ' ') || (*p == '\t'))
			p++;

		elev = p;

		stripSpace(city);
		stripSpace(stcntry);

		if (cityInfo == NULL) {
			cityInfo = (CityInfo *) safeAlloc(sizeof(CityInfo));
			ci = cityInfo;
		}
		else {
			ci->c_next = (CityInfo *) safeAlloc(sizeof(CityInfo));
			ci = ci->c_next;
		}

		ci->c_city = saveString(city);
		ci->c_statecountry = saveString(stcntry);

		sscanf(lat, "%d %d %d %c", &d, &m, &s, &c);

		ci->c_latitude = (double) d + (double) m / 60.0 +
				 (double) s / 3600.0;

		if ((c == 'S') || (c == 's'))
			ci->c_latitude = -ci->c_latitude;

		sscanf(lon, "%d %d %d %c", &d, &m, &s, &c);

		ci->c_longitude = (double) d + (double) m / 60.0 +
				  (double) s / 3600.0;

		if ((c == 'W') || (c == 'w'))
			ci->c_longitude = -ci->c_longitude;

		sscanf(elev, "%lf", &ci->c_elevation);
		ci->c_next = NULL;
	}

	fclose(fp);
}

/*
 * Find the named city in the CityInfo structures.
 */
CityInfo *
findCity(cname)
char *cname;
{
	CityInfo *ci;
	char *city, *stcntry;
	Boolean ambig;
	CityInfo *match_ci;
	char *strcasestr();

	city = stcntry = cname;

	while ((*stcntry != ',') && (*stcntry != '\0'))
		stcntry++;

	*stcntry++ = '\0';

	while ((*stcntry == ' ') || (*stcntry == '\t'))
		stcntry++;

	stripSpace(city);
	stripSpace(stcntry);

	ambig = False;
	match_ci = (CityInfo *)NULL;
	for (ci = cityInfo; ci != (CityInfo *)NULL; ci = ci->c_next)
		if (strncasecmp(ci->c_city, city, strlen(city)) == EQUAL)
			if (match_ci != (CityInfo *)NULL) {
				ambig = True;
				break;
			}
			else
				match_ci = ci;

	/* if we weren't ambiguous, we're done */
	if (! ambig)
		return(match_ci);

	/* we were ambiguous; see if state/country info eliminates ambiguity */
	ambig = False;
	match_ci = (CityInfo *)NULL;
	for (ci = cityInfo; ci != (CityInfo *)NULL; ci = ci->c_next)
		if ((strncasecmp(ci->c_city, city, strlen(city)) == EQUAL) &&
		    (strncasecmp(ci->c_statecountry, stcntry, strlen(stcntry))
		                                                     == EQUAL))
			if (match_ci != (CityInfo *)NULL) {
				ambig = True;
				break;
			}
			else
				match_ci = ci;

	/* if we weren't ambiguous, we're done */
	if (! ambig)
		return(match_ci);

	return(NULL);
}

/*
 * Callback from the new city prompt.
 */
static void
setCity(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	CityInfo *ci;
	char buf[BUFSIZ];

	new = XawDialogGetValueString(XtParent(w));

	if ((ci = findCity(new)) == NULL) {
		XBell(XtDisplay(w),0);
		sprintf(buf, "Unknown or ambiguous city: %s", new);
		prompt(buf, NULL, promptOkay);
	}
	else {
		currentCity = ci;
	}

	XtDestroyWidget(clientData);
}
