/*
 * $Header: /home/orchestra5/davy/stuff/misc/xsat/RCS/Canvas.c,v 1.1 92/04/10 14:08:02 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Code for the Canvas widget.  Based in part on the Template widget from
 * the X11R5 distribution; I also looked quite a bit at the code from the
 * "xproof" program to figure out how to set things up.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	Canvas.c,v $
 * Revision 1.1  92/04/10  14:08:02  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Xlib.h>
#include <stdio.h>

#include "CanvasP.h"
#include "xsat.h"

static void	Destroy();
static void	Realize();
static void	Redisplay();
static void	Initialize();
static void	setup_gcs();

/*
 * Canvas widget resources.
 */
static XtResource resources[] = {
#define offset(field) XtOffsetOf(CanvasRec,canvas.field)
#define goffset(field) XtOffsetOf(CanvasRec,core.field)
	{ XtNforeground,	XtCForeground,	XtRPixel,	sizeof(Pixel),
		offset(foreground_pixel),	XtRString,	"black" },
	{ XtNbackground,	XtCBackground,	XtRPixel,	sizeof(Pixel),
		goffset(background_pixel),	XtRString,	"white" },
	{ XtNidleCursor,	XtCCursor,	XtRCursor,	sizeof(Cursor),
		offset(idle_cursor),		XtRString,	"left_ptr" },
	{ XtNbusyCursor,	XtCCursor,	XtRCursor,	sizeof(Cursor),
		offset(busy_cursor),		XtRString,	"watch" },
	{ XtNfont,		XtCFont,	XtRFontStruct,	sizeof(XFontStruct *),
		offset(font),			XtRString,	XtDefaultFont },
#undef offset
#undef goffset
};

CanvasClassRec canvasClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Canvas",
    /* widget_size		*/	sizeof(CanvasRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	NULL,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* canvas fields */
    /* empty			*/	0
  }
};

WidgetClass canvasWidgetClass = (WidgetClass)&canvasClassRec;

/*
 * Initialize a widget instance by allocating graphics contexts.
 */
static void
Initialize(request, new)
Widget request, new;
{
/* nothing to do here */
}

/*
 * Realize a widget instance.  Allocate pixmaps, create the window, etc.
 */
static void
Realize(gw, valueMask, attr)
XSetWindowAttributes *attr;
XtValueMask *valueMask;
Widget gw;
{
	Display *d;
	CanvasWidget w = (CanvasWidget) gw;

	if ((attr->cursor = w->canvas.idle_cursor) != None)
		*valueMask |= CWCursor;

	XtCreateWindow(gw, InputOutput, (Visual *) CopyFromParent,
		       *valueMask, attr);

	d = XtDisplay(w);

	w->canvas.paper = XCreatePixmap(d, DefaultRootWindow(d),
					w->core.width, w->core.height,
					DefaultDepth(d, DefaultScreen(d)));

	w->canvas.scratch = XCreatePixmap(d, DefaultRootWindow(d),
					  w->core.width, w->core.height,
					  DefaultDepth(d, DefaultScreen(d)));

	/* set up the graphics contexts */
	setup_gcs(w,d);

	XSetLineAttributes(d, w->canvas.drawGC, 0, LineSolid,
			   CapButt, JoinMiter);

	canvasClearPaper(w);
	canvasClearScratch(w);
}

/*
 * Destroy a widget instance.
 */
static void
Destroy(gw)
Widget gw;
{
	CanvasWidget w = (CanvasWidget) gw;

	XtReleaseGC((Widget) w, w->canvas.drawGC);
	XtReleaseGC((Widget) w, w->canvas.copyGC);
	XtReleaseGC((Widget) w, w->canvas.clearGC);
	XtReleaseGC((Widget) w, w->canvas.mergeGC);

	XFreePixmap(XtDisplay(w), w->canvas.paper);
	XFreePixmap(XtDisplay(w), w->canvas.scratch);
}

/*
 * Redisplay a widget after an expose event by copying data from the pixmaps.
 */
static void
Redisplay(gw, event, region)
Region region;
XEvent *event;
Widget gw;
{
	CanvasWidget w = (CanvasWidget) gw;
	XExposeEvent *e = (XExposeEvent *) event;
	
	canvasBusy(w);

	XCopyArea(XtDisplay(gw), w->canvas.paper, XtWindow(gw),
		  w->canvas.copyGC, e->x, e->y,
		  (Dimension) e->width, (Dimension) e->height,
		  e->x, e->y);
	XCopyArea(XtDisplay(gw), w->canvas.scratch, XtWindow(gw),
		  w->canvas.mergeGC, e->x, e->y,
		  (Dimension) e->width, (Dimension) e->height,
		  e->x, e->y);
		  
	canvasIdle(w);
}

/*
 * Draw line segments on the "paper" pixmap of the widget.
 */
void
canvasDrawSegmentsPaper(w, segments, nsegments)
XSegment *segments;
CanvasWidget w;
int nsegments;
{
	XDrawSegments(XtDisplay(w), w->canvas.paper, w->canvas.drawGC,
		      segments, nsegments);
}

/*
 * Draw line segments on the "scratch" pixmap of the widget.
 */
void
canvasDrawSegmentsScratch(w, segments, nsegments)
XSegment *segments;
CanvasWidget w;
int nsegments;
{
	XDrawSegments(XtDisplay(w), w->canvas.scratch, w->canvas.drawGC,
		      segments, nsegments);
}

/*
 * Draw a line on the "paper" pixmap of the widget.
 */
void
canvasDrawLinePaper(w, x1, y1, x2, y2)
int x1, y1, x2, y2;
CanvasWidget w;
{
	XDrawLine(XtDisplay(w), w->canvas.paper, w->canvas.drawGC,
		  x1, y1, x2, y2);
}

/*
 * Draw a line on the "scratch" pixmap of the widget.
 */
void
canvasDrawLineScratch(w, x1, y1, x2, y2)
int x1, y1, x2, y2;
CanvasWidget w;
{
	XDrawLine(XtDisplay(w), w->canvas.scratch, w->canvas.drawGC,
		  x1, y1, x2, y2);
}

/*
 * Draw text on the "paper" pixmap of the widget.
 */
void
canvasDrawTextPaper(w, x, y, str)
CanvasWidget w;
char *str;
int x, y;
{
	XDrawString(XtDisplay(w), w->canvas.paper, w->canvas.drawGC,
		    x, y, str, strlen(str));
}

/*
 * Draw text on the "scratch" pixmap of the widget.
 */
void
canvasDrawTextScratch(w, x, y, str)
CanvasWidget w;
char *str;
int x, y;
{
	XDrawString(XtDisplay(w), w->canvas.scratch, w->canvas.drawGC,
		    x, y, str, strlen(str));
}

/*
 * Clear the "paper" pixmap of the widget.
 */
void
canvasClearPaper(w)
CanvasWidget w;
{
	XFillRectangle(XtDisplay(w), w->canvas.paper, w->canvas.clearGC,
		       0, 0, w->core.width, w->core.height);
}

/*
 * Clear the "scratch" pixmap of the widget.
 */
void
canvasClearScratch(w)
CanvasWidget w;
{
	XFillRectangle(XtDisplay(w), w->canvas.scratch, w->canvas.clearGC,
		       0, 0, w->core.width, w->core.height);
}

/*
 * Update the window of the widget with the "paper" and "scratch" pixmaps.
 */
void
canvasUpdateWindow(w)
CanvasWidget w;
{
	XCopyArea(XtDisplay(w), w->canvas.paper, XtWindow(w),
		  w->canvas.copyGC, 0, 0, w->core.width, w->core.height,
		  0, 0);
	XCopyArea(XtDisplay(w), w->canvas.scratch, XtWindow(w),
		  w->canvas.mergeGC, 0, 0, w->core.width, w->core.height,
		  0, 0);
}

/*
 * Set the cursor to the busy cursor.
 */
void
canvasBusy(w)
CanvasWidget w;
{
	XSetWindowAttributes attr;

	if ((attr.cursor = w->canvas.busy_cursor) != None) {
		XChangeWindowAttributes(XtDisplay(w), XtWindow(w),
					CWCursor, &attr);
		XFlush(XtDisplay(w));
	}
}

/*
 * Set the cursor to the idle cursor.
 */
void
canvasIdle(w)
CanvasWidget w;
{
	XSetWindowAttributes attr;

	if ((attr.cursor = w->canvas.idle_cursor) != None) {
		XChangeWindowAttributes(XtDisplay(w), XtWindow(w),
					CWCursor, &attr);
		XFlush(XtDisplay(w));
	}
}

/*
 * Set up the graphics contexts for the Canvas.
 */
static void setup_gcs(w,dpy)
CanvasWidget w;
Display *dpy;
{
	XGCValues gcv;
	int screen;
	XWindowAttributes attr;
	Colormap colormap;
	unsigned long plane_mask[1];
	unsigned long pixels[1];
	XColor color[2];

/* get the screen of the display */
	screen = DefaultScreen(dpy);

/* get the attributes of the map window */
	XGetWindowAttributes(dpy,XtWindow(w),&attr);
	if (((attr.visual)->class == GrayScale) ||
                                       ((attr.visual)->class == StaticGray)) {
	  /* just stick with black on white */
	  w->core.background_pixel = WhitePixel(dpy,screen);
	  w->canvas.foreground_pixel = BlackPixel(dpy,screen);
	  plane_mask[0] = AllPlanes;
	}
	else {
/* allocate a couple of color cells which differ in just one plane */
	  colormap = DefaultColormap(dpy,screen);
	  if (XAllocColorCells(dpy,colormap,False,plane_mask,1,pixels,1)) {
/* get colors at the current foreground and background pixels */
	    color[0].pixel = w->core.background_pixel;
	    color[1].pixel = w->canvas.foreground_pixel;
	    XQueryColors(dpy,colormap,color,2);

/* use the allocated pixel as the background, and OR it
 * with the plane mask to get the foreground pixel */
	    color[0].pixel = pixels[0];
	    color[1].pixel = pixels[0] | plane_mask[0];

/* make the two new pixels have the foreground and background RGB values */
	    XStoreColors(dpy,colormap,color,2);

	    /* change the widget's idea of foreground and background */
	    w->core.background_pixel = color[0].pixel;
	    w->canvas.foreground_pixel = color[1].pixel;
	  }
	  else {
	    /* fall back on black and white */
	    w->core.background_pixel = WhitePixel(dpy,screen);
	    w->canvas.foreground_pixel = BlackPixel(dpy,screen);
	    plane_mask[0] = AllPlanes;
	  }
	}

	w->canvas.copyGC = XtGetGC((Widget) w, 0, &gcv);

	gcv.foreground = w->canvas.foreground_pixel;
	gcv.background = w->core.background_pixel;
	gcv.font = w->canvas.font->fid;

	w->canvas.drawGC = XtGetGC((Widget) w,
				   GCForeground | GCBackground | GCFont,
				   &gcv);

	gcv.function = GXor;
	gcv.plane_mask = plane_mask[0];
	w->canvas.mergeGC = XtGetGC((Widget) w,
				    GCFunction | GCForeground |
				                 GCBackground | GCPlaneMask,
				    &gcv);

	gcv.foreground = gcv.background;
	w->canvas.clearGC = XtGetGC((Widget) w, GCForeground, &gcv);
}
