/****************************************/
/* sunsat.c				*/
/*					*/
/* finds near approaches of satellites	*/
/* to the sun				*/
/****************************************/

/***** description
 *
 *	$Id: sunsat.c,v 1.3 1993/05/18 16:50:28 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: sunsat.c,v $
 * Revision 1.3  1993/05/18  16:50:28  craig
 * added a plotfile FILE for search.c
 *
 * Revision 1.2  1993/05/12  19:01:53  craig
 * Now has a better searching algorithm.  Avoids multiple searches
 * for a given satellite at the same near approach.
 *
 * Revision 1.1  1993/05/05  16:39:22  craig
 * Initial revision
 *
 */

/***** include files *****/

#include <ctype.h>
#include <string.h>
#include <math.h>
#include "satellite.h"
#include "satproto.h"
#include "aaproto.h"

/***** global variables *****/

/* from search.c */

extern FILE *elefile;
extern FILE *plotfile;

extern struct star fstar;

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double JD;

extern struct PCONSTANTS pcnsts;

/**********/
/* sunsat */
/**********/

int    main (int argc, char *argv[])
{
    /***** Local variables *****/

    int    mon, year;

    char   sname[32], osname[32];

    FILE   *outputfile, *scrfile1, *scrfile2, *scrfile3;

    double day, srate, ojd;
    double JDTIME, JDSTOP, JDSTART, DTIME;

    register i;

    static char pfname[] = "sunplot";


    if (argc < 5)
    {
	fprintf (stderr, "Usage: sunsat day mon year tlefile [outfile]\n");
	fprintf (stderr, "\tday is the day of the month\n");
	fprintf (stderr, "\tmon is the month [1-12] \n");
	fprintf (stderr, "\tyear is the year [e.g. 1993] \n");
	fprintf (stderr, "\ttlefile is the 2-line element set file.\n");
	fprintf (stderr, "\toutfile is the optional output file name.\n");

	exit (-1);
    }

    /***** initialize constants *****/

    prtflg = 0;
    cnstinit ();

    if (kinit () == -1)
    {
	fprintf (stderr, "Unable to find a ./.site.ini or a ");
	fprintf (stderr, "~/.site.init site initialization file.\n");
	exit (-1);
    }

    sname[0] = '\0';
    strcpy (osname, "crud");

    /***** get inputs *****/

    /* get the date for the epheremus */

    if (sscanf (argv[1], "%lf", &day) == 0)
    {
	fprintf (stderr, "Error in reading the command line date.\n");
	exit (-1);
    }

    if (sscanf (argv[2], "%d", &mon) == 0)
    {
	fprintf (stderr, "Error in reading the command line date.\n");
	exit (-1);
    }

    if (sscanf (argv[3], "%d", &year) == 0)
    {
	fprintf (stderr, "Error in reading the command line date.\n");
	exit (-1);
    }

    JD = mjd (year, mon, day);
    JDSTART = JD + .5;			/* 12 hours after UT midnight */
    JDSTOP = JD + 1.125;		/* 3 hours into next day */

    /* open the element file */

    elefile = fopen (argv[4], "r");

    if (elefile == NULL)
    {
	fprintf (stderr, "Error in opening element file: %s\n", argv[4]);
	exit (-1);
    }

    /* open the output file if a name is given */

    if (argc == 6)
    {
	outputfile = fopen (argv[5], "w+");

	if (outputfile == NULL)
	{
	    fprintf (stderr, "Error in opening output file: %s\n",
		argv[5]);
	    exit (-1);
	}
    }
    else
    {
        outputfile = stdout;
    }

    /* open the plot file */

    plotfile = fopen (pfname, "a+");

    if (plotfile == NULL)
    {
	fprintf (stderr, "Error in opening plot data file: %s\n", pfname);
	exit (-1);
    }

    /* open the scratch files */

    scrfile1 = tmpfile ();
    scrfile2 = tmpfile ();
    scrfile3 = tmpfile ();

    if (scrfile1 == NULL || scrfile2 == NULL || scrfile3 == NULL)
    {
	fprintf (stderr, "Error in opening scratch files.\n");
	exit (-1);
    }

    /***** do the searching *****/

    /* first coarse search */

    outfile = scrfile1;
    DTIME = 2.5;

    search (JDSTART, JDSTOP, DTIME, sname, 15., 30., -1);

    /* now redo the search at intervals that avoid duplicate entries */

    rewind (scrfile1);
    outfile = scrfile2;
    /* fprintf (stdout, "\n 30 degree \n"); */

    while (fscanf (scrfile1, "%lf%lf", &JDTIME, &srate) == 2)
    {
	if (fgets (sname, 31, scrfile1) == NULL)
	{
	    fprintf (stderr, "Error in reading scratch file 1.\n");
	    exit (-1);
	}

        /* remove the carriage return from the name */

	for (i = 0; i < strlen (sname); i++)
	{
	    if (iscntrl (sname[i]))
	    {
		sname[i] = '\0';		/* null */
	    }
	}

	if (strstr (sname, osname) != NULL)
	{
	    continue;
	}

	/* fprintf (stdout, "%s ", sname); */

	strcpy (osname, sname);
        DTIME = 30. / srate / 2.;
	JDSTART = JDTIME - 5. / pcnsts.xmnpda;

	/*
	fprintf (stdout, "%f %f %f %f\n", JDSTART, JDSTOP, DTIME,
	    (JDSTOP - JDSTART) / DTIME * pcnsts.xmnpda);
	*/

        search (JDSTART, JDSTOP, DTIME, sname, 15., 30., 2);
    }

    /* now refine the search with a 5 degree box */

    rewind (scrfile2);
    outfile = scrfile3;
    strcpy (osname, "crud");
    ojd = 0.;

    /* fprintf (stdout, "\n 5 degree \n"); */

    while (fscanf (scrfile2, "%lf%lf", &JDTIME, &srate) == 2)
    {
	if (fgets (sname, 31, scrfile2) == NULL)
	{
	    fprintf (stderr, "Error in reading scratch file 2.\n");
	    exit (-1);
	}

        /* remove the carriage return from the name */

	for (i = 0; i < strlen (sname); i++)
	{
	    if (iscntrl (sname[i]))
	    {
		sname[i] = '\0';		/* null */
	    }
	}

	if (strstr (sname, osname) != NULL && JDTIME - ojd < 30. / srate)
	{
	    ojd = JDTIME;
	    continue;
	}

	ojd = JDTIME;
	strcpy (osname, sname);

	/* fprintf (stdout, "%f %f %s", JDTIME, srate, sname); */

        JDSTART = JDTIME - 30. / srate * 2. / pcnsts.xmnpda;
        JDSTOP = JDTIME + 30. / srate * 2. / pcnsts.xmnpda;
	DTIME = 6. / srate / 2.;

	/*
	fprintf (stdout, "%f %f %f %f\n", JDSTART, JDSTOP, DTIME,
	    (JDSTOP - JDSTART) / DTIME * pcnsts.xmnpda);
	*/

        search (JDSTART, JDSTOP, DTIME, sname, 15., 5., 2);
    }

    /* final epheremus at 1 second intervals */

    rewind (scrfile3);
    outfile = outputfile;
    strcpy (osname, "crud");
    ojd = 0;
    /* fprintf (stdout, "\n 1 second \n"); */

    while (fscanf (scrfile3, "%lf%lf", &JDTIME, &srate) == 2)
    {
	if (fgets (sname, 31, scrfile3) == NULL)
	{
	    fprintf (stderr, "Error in reading scratch file 3.\n");
	    exit (-1);
	}

        /* remove the carriage return from the name */

	for (i = 0; i < strlen (sname); i++)
	{
	    if (iscntrl (sname[i]))
	    {
		sname[i] = '\0';		/* null */
	    }
	}

	if (strstr (sname, osname) != NULL && JDTIME - ojd < 6. / srate)
	{
	    ojd = JDTIME;
	    continue;
	}

	ojd = JDTIME;
	strcpy (osname, sname);

	/* fprintf (stdout, "%f %f %s", JDTIME, srate, sname); */

        JDSTART = JDTIME - 5. / srate * 2. / pcnsts.xmnpda;
        JDSTOP = JDTIME + 5. / srate * 2. / pcnsts.xmnpda;
	DTIME = 1. / 60.;

	/*
	fprintf (stdout, "%f %f %f %f\n", JDSTART, JDSTOP, DTIME,
	    (JDSTOP - JDSTART) / DTIME * pcnsts.xmnpda);
	*/

        search (JDSTART, JDSTOP, DTIME, sname, 15., 1.5, 1);
    }

    /***** cleanup *****/

    if (argc == 6)
    {
	/* close the output file */

	fclose (outputfile);
    }

    /* close the scratch files */

    fclose (scrfile1);
    fclose (scrfile2);
    fclose (scrfile3);

    /* close the element file */

    fclose (elefile);

    /* close the plot file */

    fclose (plotfile);

    exit (0);
}
