/****************************************/
/* search.c				*/
/*					*/
/* search through a list of satellites	*/
/* in within a two line element file	*/
/* and generate orbits and output for	*/
/* matching entries			*/
/****************************************/

/***** description
 *
 *	$Id: search.c,v 1.3 1993/05/18 16:46:33 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: search.c,v $
 * Revision 1.3  1993/05/18  16:46:33  craig
 * added some printing statements for outputting information
 * to the sunplot file.
 *
 * Revision 1.2  1993/05/12  19:00:01  craig
 * Added the decision making stuff from rsat.c.  Changed how the
 * rates are updated using a second call to do_orbit and satreduce
 * at a future time.
 *
 * Revision 1.1  1993/05/06  18:35:12  craig
 * Initial revision
 *
 */

/***** include files *****/

#include <string.h>
#include <math.h>
#include "satellite.h"
#include "satproto.h"
#include "aaproto.h"

/***** global variables *****/

FILE  *elefile;
FILE  *plotfile;

struct star fstar;
struct ELEMENT element;

/* from cnstinit.c */

extern FILE *outfile;

extern double JD;
extern double UT;

extern struct PCONSTANTS pcnsts;
extern struct MCONSTANTS mcnsts;


/**********/
/* search */
/**********/

/* jdstart is the starting time 				   */
/* jdstop  is the ending time 					   */
/* jddelta is the interval time in minutes			   */
/* satname is the satellite name				   */
/* horiz   is the minimum altitude of the satellite		   */
/* sdist   is the max distance between satellite and sun or area   */
/* opflag  is a option flag for satreduce (controls printout)	   */

int    search (double jdstart, double jdstop, double jddelta, 
	   char *satname, double horiz, double sdist, int opflag)
{
    /***** Local variables *****/

    int    iept, iflag;

    char   whois[6];

    double delo, del1, temp;
    double a1, xnodp, ao;
    double sdelt, x, N, D, srate, jdtime;
    double delra = 0, deldec = 0;

    double salt, saz, sra, sdec;	/* search area position */ 
    double oalt, oaz, ora, odec;	/* object position */
    double nalt, naz, nra, ndec;	/* object position in future */
    double sdiam;

    /* sort through the element files */

    rewind (elefile); 

    while (rdelement () == 0)
    {
	/* check the name of the satellite with that just read */


	if (strlen (satname) != 0 && strstr (satname, element.name) == NULL)
	{
	    continue;
	}

	/***** INPUT CHECK FOR PERIOD VS EPHEMERIS SELECTED *****/
	/***** PERIOD GE 225 MINUTES  IS DEEP SPACE *****/

	a1 = pow (pcnsts.xke / element.xno, mcnsts.tothrd);
	temp = pcnsts.ck2 * 1.5 *
	    (cos (element.xincl) * cos (element.xincl) * 3. - 1.) /
	    pow (1. - element.eo * element.eo, 1.5);

	del1 = temp / (a1 * a1);
	ao = a1 * (1. - del1 * (mcnsts.tothrd * .5 + del1 *
		(del1 * 134. / 81. + 1.)));
	delo = temp / (ao * ao);
	xnodp = element.xno / (delo + 1.);

	if (mcnsts.twopi / xnodp / pcnsts.xmnpda >= .15625)
	{
	    iept = 3;
	}
	else
	{
	    iept = 2;
	}

	iflag = 1;

	/***** do the ephemeris *****/

	jdtime = jdstart;

	/* initial plotfile information */

	if (opflag == 1)
	{
	    JD = jdstart;
	    update (outfile);
	    dosun (&salt, &saz, &sra, &sdec, &sdiam);

	    sdiam *= mcnsts.ra2de;

	    fprintf (plotfile, "SUNDATA %f %f %s\n", JD, sdiam, element.name);
	}

	while (jdtime <= jdstop)
	{
	    /* update the time */

	    JD = jdtime;

            do_orbit (iflag, iept);

	    satreduce (&oalt, &oaz, &ora, &odec);

            if (oalt > horiz)
            {
	        switch (opflag)
	        {
		  case -1:
	          case 0:

		    salt = horiz;
                    sdelt = 0.;
	            break;
        
	          case 1:
	          case 2:
	    
	            dosun (&salt, &saz, &sra, &sdec, &sdiam);
	            break;

	          case 3:
	          case 4:

	            rstar (&fstar, &salt, &saz, &sra, &sdec);
	            break;
        	}

		if (salt < horiz)
		{
	            jdtime += jddelta / pcnsts.xmnpda;
		    continue;
		}

                if (opflag > 0)
	        {
	            /* calculate the distance between the satellite 
	               and the sun or search coordinate */

	            delra = (ora - sra);

	            if (delra > mcnsts.pi)
	            {
		        delra -= mcnsts.twopi;
	            }
	            else if (delra < -mcnsts.pi)
	            {
		        delra += mcnsts.twopi;
	            }
        
	            sdelt = delra * delra;
        
	            deldec = (odec - sdec);
        
	            sdelt += deldec * deldec;
        
 	            sdelt = sqrt (sdelt);
        
	            sdelt *= mcnsts.ra2de;
		    delra *= mcnsts.ra2de;
		    deldec *= mcnsts.ra2de;
	        }
        
	        /* find the rate at which ra and dec are changing */
	        /* using the position at time + jddelta */
        
		JD = jdtime + jddelta / pcnsts.xmnpda;
        
                do_orbit (iflag, iept);

	        satreduce (&nalt, &naz, &nra, &ndec);

		JD = jdtime;
		update (outfile);

                x = mcnsts.ra2de / jddelta;

		N = nra - ora;

		if (N > mcnsts.pi)
		{
		    N -= mcnsts.twopi;
		}
		else if (N < -mcnsts.pi)
		{
		    N += mcnsts.twopi;
		}
		    
                N = x * N;
                D = x * (ndec - odec);
        
	        srate = N * N + D * D;
	        srate = sqrt (srate);
        
	        /* print out */
        
                if (sdelt < sdist)
	        {
	            switch (opflag)
	            {
	              case 0:
        
	                fprintf (outfile, "\n");
	                jtocal (outfile, UT);
	                fprintf (outfile, "\t%s Position: \n", element.name);
	                fprintf (outfile, 
			    "\t\tAltitude %.3f deg Azimuth %.3f deg\n", 
	                    oalt, oaz);
	                fprintf (outfile, "\t\tR.A.");
	                hms (outfile, ora);
	                fprintf (outfile, " Dec.");
	                dms (outfile, odec);
	                fprintf (outfile, "\n");
                        fprintf (outfile, 
		            "\t\tdRA/dt %.2f deg/min, dDec/dt %.2f deg/min\n", 
		            N, D );
	                fprintf (outfile, "\n");
        
		        break;
        
	              case 1:
	              case 3:
        
	                fprintf (outfile, "\n");
	                jtocal (outfile, UT);
	                fprintf (outfile, "\t%s Position: \n", element.name);
	                fprintf (outfile, 
			   "\t\tAltitude %.3f deg Azimuth %.3f deg\n", 
	                    oalt, oaz);
	                fprintf (outfile, "\t\tR.A.");
	                hms (outfile, ora);
	                fprintf (outfile, " Dec.");
	                dms (outfile, odec);
	                fprintf (outfile, "\n");
                        fprintf (outfile, 
		            "\t\tdRA/dt %.2f deg/min, dDec/dt %.2f deg/min\n", 
		            N, D );
        
		        if (opflag == 1)
		        {
		            strcpy (whois, "Solar");
			    fprintf (plotfile, "%f %f %f\n", JD, delra,
				deldec);
		        }
		        else
		        {
		            strcpy (whois, "Area");
		        }
        
	                fprintf (outfile, "\tSatellite - %s distance %f\n", 
		            whois, sdelt);
	                fprintf (outfile, "\t%s Position: \n", whois);
	                fprintf (outfile, 
			    "\t\tAltitude %.3f deg Azimuth %.3f deg\n", 
	                    salt, saz);
	                fprintf (outfile, "\t\tR.A.");
	                hms (outfile, sra);
	                fprintf (outfile, " Dec.");
	                dms (outfile, sdec);
	                fprintf (outfile, "\n");
	            
	                break;
        
		      case -1:
	              case 2:
	              case 4:
        
	                fprintf (outfile, "%f %f %s\n", 
			    jdtime, srate, element.name); 
	                break;
	            }
	        }
            }

	    jdtime += jddelta / pcnsts.xmnpda;
	}
    }
        
    return (0);
}
