/****************************************/
/* sawsat.c				*/
/*					*/
/* finds near approaches of satellites	*/
/* to a given coordinate at a given time*/
/****************************************/

/***** description
 *
 *	$Id: sawsat.c,v 1.2 1993/05/18 16:52:10 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: sawsat.c,v $
 * Revision 1.2  1993/05/18  16:52:10  craig
 * added a declariation of the plotfile FILE for search.c
 *
 * Revision 1.1  1993/05/12  19:03:19  craig
 * Initial revision
 *
 */

/***** include files *****/

#include <ctype.h>
#include <string.h>
#include <math.h>
#include "satellite.h"
#include "satproto.h"
#include "aaproto.h"

/***** global variables *****/

/* from search.c */

extern FILE *elefile;
extern FILE *plotfile;

extern struct star fstar;

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double JD;

extern struct MCONSTANTS mcnsts;
extern struct PCONSTANTS pcnsts;

/**********/
/* sawsat */
/**********/

int    main (void)
{
    /***** Local variables *****/

    int    rhrs, rmin, ddeg, dmin;

    char   dsign[2], tfname[32], ofname[32], sname[32], osname[32];

    FILE   *outputfile, *scrfile1, *scrfile2;

    double srate, rsec, dsec, delt, ojd;
    double JDTIME, JDSTOP, JDSTART, DTIME;

    register i;



    /***** initialize constants *****/

    prtflg = 0;
    cnstinit ();

    if (kinit () == -1)
    {
	fprintf (stderr, "Unable to find a ./.site.ini or a ");
	fprintf (stderr, "~/.site.init site initialization file.\n");
	exit (-1);
    }

    /***** initializations *****/

    strcpy (tfname, "visual.tle");
    strcpy (ofname, "stdout");
    sname[0] = '\0';
    strcpy (osname, "crud");
    rhrs = 0;
    rmin = 0;
    strcpy (dsign, "+");
    ddeg = 0;
    dmin = 0;
    rsec = 0.;
    dsec = 0.;
    delt = 1.5;
    outfile = stdout;
    plotfile = stdout;		/* not generating plots here anyway */

    fstar.obname[0] = '\0';
    fstar.epoch = 2000.0;
    fstar.px = 0.;
    fstar.mura = 0.;
    fstar.mudec = 0.;
    fstar.v = 0.;
    fstar.equinox = 0.;

    /***** get inputs *****/

    getstring ("Element file name: ", tfname, 32);

    /* open the element file */

    elefile = fopen (tfname, "r");

    if (elefile == NULL)
    {
	fprintf (stderr, "Error in opening element file: %s\n", tfname);
	exit (-1);
    }

    /* open the output file */

    getstring ("Output file name: ", ofname, 32);

    if (strstr (ofname, "stdout") == NULL)
    {
	outputfile = fopen (ofname, "w+");

	if (outputfile == NULL)
	{
	    fprintf (stderr, "Error in opening output file: %s\n",
		ofname);
	    exit (-1);
	}
    }
    else
    {
        outputfile = stdout;
    }

    /* get the starting and stoping times */

    fprintf (stdout, "\nEnter search starting time: \n\n");
    JDSTART = getdate ();
    JDSTART += gethms ();
    JD = JDSTART;
    update (stdout);

    fprintf (stdout, "\nEnter search ending time: \n\n");
    JDSTOP = getdate ();
    JDSTOP += gethms ();

    /* get the position and search box size */

    fprintf (stdout, "\nEnter the R.A of the search area coordinates: \n\n");
    getint ("Hours", &rhrs);
    getint ("Minutes", &rmin);
    getdouble ("Seconds", &rsec);
    fstar.ra = mcnsts.twopi * (3600.0 * rhrs + 60.0 * rmin + rsec) /
	       pcnsts.secpda;

    fprintf (stdout, "\nEnter the Dec. of the search area coordinates: \n\n");
    getstring ("sign", dsign, 2);
    getint ("Degrees", &ddeg);
    getint ("Minutes", &dmin);
    getdouble ("Seconds", &dsec);
    fstar.dec = (3600.0 * ddeg + 60.0 * dmin + dsec) / mcnsts.ra2sec;

    if (strstr (dsign, "-") != NULL)
    {
	fstar.dec = -fstar.dec;
    }

    getdouble ("\nEpoch of the coordinates", &fstar.epoch);

    if (fstar.epoch == 2000.0)
    {
	fstar.epoch = pcnsts.J2000;
    }
    else if (fstar.epoch == 1950.0)
    {
	fstar.epoch = pcnsts.B1950;
    }
    else if (fstar.epoch == 1900)
    {
	fstar.epoch = pcnsts.J1900;
    }
    else
    {
	fstar.epoch = pcnsts.J2000 + (pcnsts.dapcen / 100.) *
		      (fstar.epoch - 2000.0);
    }
	     
    getdouble ("Size of the search area (degrees)", &delt);

    /* open the scratch files */

    scrfile1 = tmpfile ();
    scrfile2 = tmpfile ();

    if (scrfile1 == NULL || scrfile2 == NULL)
    {
	fprintf (stderr, "Error in opening scratch files.\n");
	exit (-1);
    }

    /***** do the searching *****/

    /* first coarse search */

    outfile = scrfile1;
    DTIME = 2.5;

    search (JDSTART, JDSTOP, DTIME, sname, 15., 30., -1);

    /* now redo the search at intervals that avoid duplicate entries */

    rewind (scrfile1);
    outfile = scrfile2;

    while (fscanf (scrfile1, "%lf%lf", &JDTIME, &srate) == 2)
    {
        if (fgets (sname, 31, scrfile1) == NULL)
	{
	    fprintf (stderr, "Error in reading scratch file 1.\n");
	    exit (-1);
	}

        /* remove the carriage return from the name */

	for (i = 0; i < strlen (sname); i++)
	{
	    if (iscntrl (sname[i]))
	    {
		sname[i] = '\0';		/* null */
	    }
	}

	if (strstr (sname, osname) != NULL)
	{
	    continue;
	}

        strcpy (osname, sname);
	DTIME = delt * 20. / srate / 2.;
        JDSTART = JDTIME - 5. / pcnsts.xmnpda;

        search (JDSTART, JDSTOP, DTIME, sname, 15., delt * 20., 4);
    }

    /* final search */

    rewind (scrfile2);
    outfile = outputfile;
    strcpy (osname, "crud");
    ojd = 0.;

    while (fscanf (scrfile2, "%lf%lf", &JDTIME, &srate) == 2)
    {
        if (fgets (sname, 31, scrfile2) == NULL)
	{
	    fprintf (stderr, "Error in reading scratch file 2.\n");
	    exit (-1);
	}

        /* remove the carriage return from the name */

	for (i = 0; i < strlen (sname); i++)
	{
	    if (iscntrl (sname[i]))
	    {
		sname[i] = '\0';		/* null */
	    }
	}

	if (strstr (sname, osname) != NULL && 
	    JDTIME - ojd < delt * 20. / srate)
	{
	    ojd = JDTIME;
	    continue;
	}

	ojd = JDTIME;
	strcpy (osname, sname);

        JDSTART = JDTIME - delt * 20. / srate * 2. / pcnsts.xmnpda;
        JDSTOP = JDTIME + delt * 20. / srate * 2. / pcnsts.xmnpda;
	DTIME = (6. * delt / 5.) / srate / 2.;

        search (JDSTART, JDSTOP, DTIME, sname, 15., delt, 3);
    }

    /***** cleanup *****/

    if (strstr (ofname, "stdout") == NULL)
    {
	/* close the output file */

	fclose (outputfile);
    }

    /* close the scratch files */

    fclose (scrfile1);
    fclose (scrfile2);

    /*** close the element file ***/

    fclose (elefile);

    exit (0);
}
