/****************************************/
/* plotgen.c				*/
/*					*/
/* generate gnuplot plot commands for   */
/* a satellite - sun near approach.	*/
/****************************************/

/***** description
 *
 *	$Id: plotgen.c,v 1.1 1993/05/18 16:48:46 craig Exp craig $
 *
 */

/***** modification history
 *
 *	$Log: plotgen.c,v $
 * Revision 1.1  1993/05/18  16:48:46  craig
 * Initial revision
 *
 */

/***** include files *****/

#include <ctype.h>
#include <string.h>
#include <math.h>
#include "satellite.h"
#include "satproto.h"
#include "aaproto.h"

/***** defines *****/

#define BASEL 32

/***** global variables *****/

/* from cnstinit.c */

extern double UT;

extern struct PCONSTANTS pcnsts;
extern struct MCONSTANTS mcnsts;

/***** local global variables *****/

static char *months[12] = {
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December"
};

/***** local function prototypes *****/

int    append000 (char *string, int maxlength);

void   appendstr (char *string, int maxlength, char *ext);
void   str_jtocal (char *str, double J);
void   increment000 (char *string, int offset);

/***********/
/* plotgen */
/***********/

int    main (int argc, char *argv[])
{
    /***** Local variables *****/

    int    addspot, dflag = -1;

    char   basename[32], sunname[32], outname[32], drvname[32], sname[40];
    char   cdummy[40], titstr[80];

    FILE  *drvfile, *sunfile, *plotfile, *outfile;

    double date, th, fdummy, srad;
    double ra = 0., dec = 0., ora = 0., odec = 0.;

    register i;

    static char sext[] = "sun";
    static char dext[] = "drv";

    /***** help *****/

    if (argc < 3)
    {
	fprintf (stderr, "Usage: plotgen plotfile basename \n");
	fprintf (stderr, "\tplotfile is the sunsat plot file name. \n");
	fprintf (stderr, "\tbasename is the basename of the output files.\n");

	exit (-1);
    }

    /***** initialization *****/

    cnstinit();

    /***** get inputs *****/

    /* open the plot file */

    plotfile = fopen (argv[1], "r");

    if (plotfile == NULL)
    {
	fprintf (stderr, "Error in opening plot file: %s\n", argv[1]);
	exit (-1);
    }

    /* get the basename for the output files */

    strcpy (basename, argv[2]);
    strcpy (outname, basename);
    addspot = append000 (outname, BASEL);

    /***** generate the plotting information for the sun *****/

    /* open the sunplot file */

    strcpy (sunname, basename);
    appendstr (sunname, BASEL, sext);

    sunfile = fopen (sunname, "w");

    if (sunfile == NULL)
    {
	fprintf (stderr, "Error in opening sun data file: %s\n", sunname);
	exit (-1);
    }

    if (fscanf (plotfile, "%s %lf %lf", sname, &fdummy, &srad) != 3)
    {
	fprintf (stderr, "Error in reading first line of plot file %s\n",
	    argv[1]);
	exit (-1);
    }

    if (strstr (sname, "SUNDATA") == NULL)
    {
	fprintf (stderr, "First line of plot file %s is not expected line.\n",
	    argv[1]);
	exit (-1);
    }

    for (th = 0; th <= mcnsts.twopi; th += (mcnsts.twopi / 1000.))
    {
	fprintf (sunfile, "%f %f\n", srad * cos (th), srad * sin (th));
    }

    fclose (sunfile);

    /***** generate plot files for each of the satellites found *****/

    /* open the plot driver file */

    strcpy (drvname, basename);
    appendstr (drvname, BASEL, dext);

    drvfile = fopen (drvname, "w");

    if (drvfile == NULL)
    {
	fprintf (stderr, "Error in opening gnuplot driver file: %s\n", 
	    drvname);
	exit (-1);
    }

    /* preliminary commands for gnuplot */

    fprintf (drvfile, "set term postscript\n");
    fprintf (drvfile, "set output \"|lpr -Psp2\"\n");
    fprintf (drvfile, "set nokey\n");
    fprintf (drvfile, "set xrange [-2.15:2.15]\n");
    fprintf (drvfile, "set yrange [-1.5:1.5]\n");
    fprintf (drvfile, "set label 1 \"E\" at -1.7,1\n");
    fprintf (drvfile, "set label 2 \"N\" at -1.99,1.33 center\n");


    rewind (plotfile);

    while (fscanf (plotfile, "%s", cdummy) == 1)
    {
	if (strstr (cdummy, "SUNDATA") != NULL)
	{
	    if (dflag > 0)
	    {
		dflag = 0;
	    }

	    strcpy (sname, cdummy);

	    /* get the intervening numbers */
		
	    if (fscanf (plotfile, "%lf %lf", &fdummy, &srad) != 2)
    	    {
	        fprintf (stderr, "Error in reading plot file.\n");
	        exit (-1);
            }

            /* get the satellite name */

	    if (fgets (sname, 31, plotfile) == NULL)
    	    {
	        fprintf (stderr, "Error in reading plot file.\n");
	        exit (-1);
            }
		    
	    /* remove the carriage return from the name */

	    for (i = 0; i < strlen (sname); i++)
	    {
		if (iscntrl (sname[i]))
		{
		    sname[i] = '\0';		/* null */
		}
	    }
	}
	else
	{
	    if (dflag < 0)
	    {
		dflag = 1;

		/* open the first output file */

		outfile = fopen (outname, "w");

		if (drvfile == NULL)
		{
		    fprintf (stderr, "Error in opening output file: %s\n", 
	    		outname);
		    exit (-1);
		}

		/* update the title for the driver */

		sscanf (cdummy,"%lf", &date);

                str_jtocal (titstr, date);
		strcat (titstr, sname);

		fprintf (drvfile, "set title \"%s\"\n", titstr);
	    }
	    else if (dflag == 0)
	    {
		dflag = 1;

		/* put together the direction arrow */

		fprintf (drvfile, "set arrow 1 from %f,%f to %f,%f\n",
		    ora + .25, odec, ra + .25, dec);

		/* write the plot command to the driver file */

		fprintf (drvfile, "plot \"%s\" with line 1, \"%s\", \\\n", 
		    sunname, outname);
		fprintf (drvfile, "\"mac1.field\" with line 2,\\\n");
		fprintf (drvfile, "\"cardinal\" with line 1\n");

		/* close the open output file */

		fclose (outfile);

		/* open the next output file */

		increment000 (outname, addspot);

		outfile = fopen (outname, "w");

		if (outfile == NULL)
		{
		    fprintf (stderr, "Error in opening output file: %s\n", 
	    		outname);
		    exit (-1);
		}

		/* update the title for the driver */

		sscanf (cdummy,"%lf", &date);

                str_jtocal (titstr, date);
		strcat (titstr, sname);

		fprintf (drvfile, "set title \"%s\"\n", titstr);
	    }

	    /* read the delta ra and delta dec */

            ora = ra;
	    odec = dec;

	    if (fscanf (plotfile, "%lf %lf", &ra, &dec) != 2)
    	    {
	        fprintf (stderr, "Error in reading plot file.\n");
	        exit (-1);
            }

	    /* put the points into the output file */

	    fprintf (outfile, "%f %f\n", ra, dec);
	}
    }

    /***** cleanup *****/

    /* put together the direction arrow */

    fprintf (drvfile, "set arrow 1 from %f,%f to %f,%f\n",
	ora + .25, odec, ra + .25, dec);

    /* write the plot command to the driver file */

    fprintf (drvfile, "plot \"%s\" with line 1, \"%s\", \\\n", 
	sunname, outname);
    fprintf (drvfile, "\"mac1.field\" with line 2,\\\n");
    fprintf (drvfile, "\"cardinal\" with line 1\n");

    /* close the open output file */

    fclose (outfile);

    /* write the exit command to the driver file */

    fprintf (drvfile, "exit\n");

    /* close the driver file */

    fclose (drvfile);

    /* close the plot file */

    fclose (plotfile);

    exit (0);
}

/************************************************************************/
/* 	append000 	July 25 1986 nh	 adds .00 to string		*/
/*			returns the offset into the string where 1 	*/
/*			should be added to increment the filename	*/
/*			3 apr 92 CAG modified to add .000		*/
/************************************************************************/

int    append000 (char *string, int maxlength)
{
    /* local variables:	 */

    int    offset = -1, i;

    for (i = 0; i < maxlength - 4; i++)
    {
	if (string[i] == '\0')
	{
	    string[i] = '.';
	    string[i + 1] = '0';
	    string[i + 2] = '0';
	    string[i + 3] = '0';
	    string[i + 4] = '\0';	/* NULL ends a string	 */
	    offset = i + 3;		/* where to add 1 for the next
					   name */
	    break;
	}
    }

    return (offset);
}

/************************************************************************/
/* 	increment000 	July 25 1986 nh add one to he number		*/
/*			appended to the string				*/
/*			3 apr 92 CAG modified to handle .000		*/
/************************************************************************/

void   increment000 (char *string, int offset)
{
    string[offset]++;

    if (string[offset] == ':')		/* ':' is = 10	 */
    {
	string[offset] = '0';
	string[offset - 1]++;
	if (string[offset - 1] == ':')
	{
	    string[offset - 1] = '0';
	    string[offset - 2]++;
	}
    }
}

/************************************************************************/
/* 	appendstr 	Craig Gullixson 6 Dec 91			*/
/*			Appends an extention to a base filename		*/
/************************************************************************/

void   appendstr (char *string, int maxlength, char *ext)
{
    /* local variables */
    int    i, j;

    for (i = 0; i < maxlength; i++)
    {
	if (string[i] == NULL)
	{
	    string[i] = '.';
	    for (j = 0; j < strlen (ext); j++)
	    {
		string[i + 1 + j] = ext[j];
	    }
	    string[i + 1 + j] = NULL;	/* NULL ends a string	 */
	    break;
	}
    }
}

/***************************************************/
/* str_jtocal					   */
/*						   */
/* Calculate month, day, and year from Julian date */
/* and stuff the result into a string.		   */
/***************************************************/

void   str_jtocal (char *str, double J)
{
    int    month, day;
    int    BC;
    int    h, m;

    char   tmpstr[80];

    long   year, a, c, d, x, y, jd;

    double dd;
    double s;

    if (J < 1721425.5)			/* January 1.0, 1 A.D. */
    {
	BC = 1;
    }
    else
    {
	BC = 0;
    }

    jd = J + 0.5;			/* round Julian date up to integer */

    /* Find the number of Gregorian centuries since March 1, 4801 B.C. */

    a = (100 * jd + 3204500L) / 3652425L;

    /* Transform to Julian calendar by adding in Gregorian century years
       that are not leap years. Subtract 97 days to shift origin of JD to
       March 1. Add 122 days for magic arithmetic algorithm. Add four
       years to ensure the first leap year is detected. */

    c = jd + 1486;

    if (jd >= 2299160.5)
    {
	c += a - a / 4;
    }
    else
    {
	c += 38;
    }

    /* Offset 122 days, which is where the magic arithmetic month formula
       sequence starts (March 1 = 4 * 30.6 = 122.4). */

    d = (100 * c - 12210L) / (long) pcnsts.dapcen;

    /* Days in that many whole Julian years */

    x = ((long) pcnsts.dapcen * d) / 100L;

    /* Find month and day. */

    y = ((c - x) * 100L) / 3061L;
    day = c - x - ((306L * y) / 10L);
    month = y - 1;

    if (y > 13)
    {
	month -= 12;
    }

    /* Get the year right. */

    year = d - 4715;

    if (month > 2)
    {
	year -= 1;
    }

    /* Day of the week. */

    a = (jd + 1) % 7;

    /* Fractional part of day. */

    dd = day + J - jd + 0.5;

    /* post the year. */

    if (BC)
    {
	year = -year + 1;
	sprintf (str, "%ld B.C. ", year);
    }
    else
    {
	sprintf (str, "%ld ", year);
    }

    day = dd;

    sprintf (tmpstr, "%d %s", day, months[month - 1]);
    strcat (str, tmpstr);

    /* Display fraction of calendar day as clock time. */

    a = dd;
    dd = dd - a;

    s = dd * 24.;

    if (s < 0.0)
    {
	s += 24.0;
    }

    h = (int) s;
    s -= (double) h;
    s *= 60.;
    m = (int) s;
    s -= (double) m;
    s *= 60.;

    /* correct for rounding */

    if (60. - s <= .0005)
    {
	m++;
	s = 0.;

	if (m == 60)
	{
	    h++;
	    m = 0;
	}
    }

    sprintf (tmpstr, "%3dh %02dm %04.1fs UT\t", h, m, s);
    strcat (str, tmpstr);

    return;
}
