/*****************************************/
/* trnsit.c				 */
/*					 */
/* Calculate time of transit assuming	 */
/* RA and Dec change uniformly with time */
/*					 */
/* -- S. L. Moshier			 */
/*****************************************/

/***** description
 *
 *	$Id: trnsit.c,v 1.5 1993/04/30 18:11:31 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: trnsit.c,v $
 * Revision 1.5  1993/04/30  18:11:31  craig
 * Changed the output statements to output to a output file
 * (which can be stdout) instead of stdout.
 *
 * Revision 1.4  1993/04/22  19:51:54  craig
 * Minor changes.
 *
 * Revision 1.3  1993/04/21  21:59:47  craig
 * Changed the path of the satellite.h include.
 * Changed ecnsts to pcnsts.
 *
 * Revision 1.2  1993/04/21  15:55:38  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs
 *
 *
 */

#define COSSUN -0.014543897651582657	/* cosine of 90 degrees 50
					   minutes: */
#define COSZEN -9.8900378587411476e-3	/* cosine of 90 degrees 34
					   minutes: */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int objnum;

extern FILE *outfile;

extern double dradt;
extern double ddecdt;
extern double obpolar[3];

extern struct MCONSTANTS mcnsts;
extern struct PCONSTANTS pcnsts;

/* from kfiles.c */

extern double glat;

/**********/
/* trnsit */
/**********/

/* J = Julian date */
/* lha = local hour angle, radians */
/* dec = declination, radians */

int    trnsit (double J, double lha, double dec)
{
    double x, y, z, N, D;
    double lhay, cosdec, sindec, coslat, sinlat;

    /* observer's geodetic latitude, in radians */

    x = glat * mcnsts.de2ra;
    coslat = cos (x);
    sinlat = sin (x);

    cosdec = cos (dec);
    sindec = sin (dec);

    x = (J - floor (J - 0.5) - 0.5) * mcnsts.twopi;

    /* adjust local hour angle */

    y = lha;

    while (y < -mcnsts.pi)
    {
	y += mcnsts.twopi;
    }

    while (y > mcnsts.pi)
    {
	y -= mcnsts.twopi;
    }

    lhay = y;
    y = y / (-dradt / mcnsts.twopi + 1.00273790934);
    N = x - y;
    fprintf (outfile, "approx local meridian transit");
    hms (outfile, N);
    fprintf (outfile, "UT  (date + %.5f)\n", N / mcnsts.twopi);

    if ((coslat == 0.0) || (cosdec == 0.0))
    {
	return (0);
    }

    /* The time at which the upper limb of the body meets the horizon
       depends on the body's angular diameter. */

    switch (objnum)
    {
      case 0:

	/* Sun */

	N = COSSUN;
	break;

      case 3:

	/* Moon, elevation = -34' - semidiameter + parallax semidiameter =
	   0.272453 * parallax + 0.0799" */

	N = 1.0 / (pcnsts.rapau * obpolar[2]);
	D = asin (N);			/* the parallax */
	N = -9.890199094634534e-3
	    + (1.0 - 0.2725076) * D
	    - 3.874e-7;
	N = sin (N);
	break;

      default:

	/* Other object */

	N = COSZEN;
	break;
    }

    y = (N - sinlat * sindec) / (coslat * cosdec);

    if ((y < 1.0) && (y > -1.0))
    {
	/* Derivative of y with respect to declination times rate of
	   change of declination: */

	z = -ddecdt * (sinlat + COSZEN * sindec);
	z /= mcnsts.twopi * coslat * cosdec * cosdec;

	/* Derivative of acos(y): */

	z /= sqrt (1.0 - y * y);
	y = acos (y);
	fprintf (outfile, "rises approx ");
	D = -dradt / mcnsts.twopi + 1.00273790934;
	N = x - (lhay + y) * (1.0 + z) / D;
	hms (outfile, N);
	fprintf (outfile, "UT,  sets approx ");
	N = x - (lhay - y) * (1.0 - z) / D;
	hms (outfile, N);
	fprintf (outfile, "UT\n");
    }

    return (0);
}
