/*********************************************************/
/* sun.c						 */
/*							 */
/* Calculate and display apparent coordinates of the sun */
/* at the time given by the external variables TDT, UT.	 */
/* Before calling this routine, the geometric position	 */
/* of the earth must be calculated and put in rearth[].	 */
/*********************************************************/

/***** description
 *
 *	$Id: sun.c,v 1.9 1993/05/18 16:42:54 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: sun.c,v $
 * Revision 1.9  1993/05/18  16:42:54  craig
 * added calculation of solar semi-diameter.  The function
 * dosun now requires a semi-diameter input variable.
 *
 * Revision 1.8  1993/04/30  18:13:02  craig
 * Change the output calls from sending the output to stdout to
 * sending the output to the file outfile (which can be stdout).
 *
 * Revision 1.7  1993/04/29  16:03:21  craig
 * Changed dosun so that it returns the topocentric alt, az,
 * ra, and dec of the sun.  Prints the alt, az, ra, and dec
 * if the prtflg is on.  Changed the call to altaz since
 * it was changed to return the alt, az, ra, and dec.
 *
 * Revision 1.6  1993/04/27  16:46:51  craig
 * Took the aberration showcor out of a prtflg if so the abberation
 * is applied weather or not the prtflg is on.
 *
 * Revision 1.5  1993/04/23  19:01:38  craig
 * Changed 173.1446 ... to pcnsts.caupda.
 *
 * Revision 1.4  1993/04/22  19:51:07  craig
 * Minor changes
 *
 * Revision 1.3  1993/04/21  21:58:07  craig
 * Changed the path of the satellite.h include.
 *
 * Revision 1.2  1993/04/21  15:54:52  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double UT;
extern double TDT;
extern double dradt;
extern double ddecdt;
extern double rearth[3];
extern double eapolar[3];

extern struct orbit earth;
extern struct MCONSTANTS mcnsts;
extern struct PCONSTANTS pcnsts;

/* from epsiln.c */

extern double coseps, sineps;

/*********/
/* dosun */
/*********/

int    dosun (double *salt, double *saz, double *sra, double *sdec,
           double *semidiam)
{
    int    i;
    double r, x, y, t;
    double ecr[3], rec[3], pol[3];


    /* Display ecliptic longitude and latitude. */

    for (i = 0; i < 3; i++)
    {
	ecr[i] = -rearth[i];
    }

    r = eapolar[2];

    if (prtflg)
    {
	lonlat (ecr, TDT, pol);
    }

    /* Philosophical note: the light time correction really affects only
       the Sun's barymetric position; aberration is due to the speed of
       the Earth.  In Newtonian terms the aberration is the same if the
       Earth is standing still and the Sun moving or vice versa.  Thus the
       following is actually wrong, but it differs from relativity only in
       about the 8th decimal. It should be done the same way as the
       corresponding planetary correction, however. */

    pol[2] = r;
    for (i = 0; i < 2; i++)
    {
	t = pol[2] / pcnsts.caupda;
	/* Find the earth at time TDT - t */
	kepler (TDT - t, &earth, ecr, pol);
    }

    r = pol[2];

    *semidiam = 696000. / (r * pcnsts.kmpau);
    *semidiam = asin (*semidiam);

    for (i = 0; i < 3; i++)
    {
	x = -ecr[i];
	y = -rearth[i];
	ecr[i] = x;			/* position t days ago */
	rec[i] = y;			/* position now */
	pol[i] = y - x;			/* change in position */
    }

    if (prtflg)
    {
	fprintf (outfile, "semi-diameter");
        dms (outfile, *semidiam);
	fprintf (outfile, "\nlight time %.4fm,  ", pcnsts.xmnpda * t);
    }
    showcor ("aberration", ecr, pol);

    /* Estimate rate of change of RA and Dec for use by altaz(). */

    deltap (ecr, rec, &dradt, &ddecdt);	/* see dms.c */
    dradt /= t;
    ddecdt /= t;


    /* There is no light deflection effect. AA page B39. */

    /* precess to equinox of date */

    precess (ecr, TDT, -1);

    /* Nutation. */

    epsiln (TDT);
    nutate (TDT, ecr);

    showrd ("    Apparent:", ecr, pol);

    /* Show it in ecliptic coordinates */

    if (prtflg)
    {
	y = coseps * ecr[1] + sineps * ecr[2];
	y = matan2 (y, ecr[0]);
	fprintf (outfile, "Apparent longitude %.3f deg\n", mcnsts.ra2de * y);
    }

    /* Report altitude and azimuth */

    altaz (pol, UT, salt, saz, sra, sdec);

    if (prtflg)
    {
        fprintf (outfile, "Topocentric Coordinates:\n");
        fprintf (outfile, "\tAltitude: %.3f deg, Azimuth %.3f deg\n", 
	    *salt, *saz);
        fprintf (outfile, "\tR.A.");
        hms (outfile, *sra);
        fprintf (outfile, "Dec.");
        dms (outfile, *sdec);
        fprintf (outfile, "\n");
    }

    return (0);
}
