/****************************************************************/
/* sidrlt.c							*/
/*								*/
/* Local Apparent Sidereal Time with equation of the equinoxes	*/
/* AA page B6							*/
/*								*/
/* Caution. At epoch J2000.0, the 16 decimal precision		*/
/* of IEEE double precision numbers				*/
/* limits time resolution measured by Julian date		*/
/* to approximately 24 microseconds.				*/
/****************************************************************/

/***** description
 *
 *	$Id: sidrlt.c,v 1.4 1993/04/22 19:50:17 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: sidrlt.c,v $
 * Revision 1.4  1993/04/22  19:50:17  craig
 * minor changes.
 *
 * Revision 1.3  1993/04/21  21:55:48  craig
 * Changed the path for the satellite.h include.
 * Changed ecnsts to pcnsts.
 *
 * Revision 1.2  1993/04/21  15:37:48  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern double TDT;

extern struct PCONSTANTS pcnsts;
extern struct MCONSTANTS mcnsts;

/* from nutate.c */

extern double nutl;

/* from   epsiln.c */

extern double coseps;

/************************************************************/
/* sidrlt						    */
/*							    */
/* program returns sidereal seconds since sidereal midnight */
/************************************************************/

/* j = Julian date and fraction */
/* tlong = East longitude of observer, degrees */

double sidrlt (double j, double tlong)
{
    double jd0;				/* Julian day at midnight
					   Universal Time */
    double secs;			/* Time of day, UT seconds since
					   UT midnight */
    double eqeq, gmst, jd, T0, T, msday;


    /* Julian day at given UT */

    jd = j;
    jd0 = floor (jd);
    secs = j - jd0;

    if (secs < 0.5)
    {
	jd0 -= 0.5;
	secs += 0.5;
    }
    else
    {
	jd0 += 0.5;
	secs -= 0.5;
    }

    secs *= pcnsts.secpda;

    /* Julian centuries from standard epoch J2000.0 */

    T = (jd - pcnsts.J2000) / pcnsts.dapcen;

    /* Same but at 0h Universal Time of date */

    T0 = (jd0 - pcnsts.J2000) / pcnsts.dapcen;

    /* The equation of the equinoxes is the nutation in longitude times
       the cosine of the obliquity of the ecliptic. We already have
       routines for these.  Using the nutation formula given by Meeus, the
       peak error in 1986 is 5 milliseconds. */

    nutlo (TDT);
    epsiln (TDT);

    /* nutl is in radians; convert to seconds of time at 240 seconds per
       degree */

    eqeq = 240.0 * mcnsts.ra2de * nutl * coseps;

    /* Greenwich Mean Sidereal Time at 0h UT of date */

    gmst = ((-6.2e-6 * T0 + 9.3104e-2) * T0 + 8640184.812866) * T0
	+ 24110.54841;

    /* mean solar days per sidereal day at date T0, = 1.00273790934 in
       1986 */

    msday = 1.0 + ((-1.86e-5 * T0 + 0.186208) * T0 + 8640184.812866)
	/ (pcnsts.secpda * pcnsts.dapcen);

    /* Local apparent sidereal time at given UT */

    gmst = gmst + msday * secs + eqeq + 240.0 * tlong;

    /* Sidereal seconds modulo 1 sidereal day */

    gmst = gmst - pcnsts.secpda * floor (gmst / pcnsts.secpda);

    return (gmst);
}
