/********************************************************/
/* rstar.c						*/
/*							*/
/* This program reduces catalogue coordinates of a star	*/
/* to apparent place.					*/
/*							*/
/* - S. L. Moshier, November, 1987			*/
/********************************************************/

/***** description
 *
 *	$Id: rstar.c,v 1.7 1993/05/12 19:09:59 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: rstar.c,v $
 * Revision 1.7  1993/05/12  19:09:59  craig
 * rstar now returns the alt, az, ra, and dec to the calling program.
 *
 * The printing of the topocentric coordinates is now optional using
 * the prtflg flag.
 *
 * Revision 1.6  1993/04/30  18:14:36  craig
 * Changed the output calls from sending the output to stdout to
 * sending the output to the file outfile (which can be stdout).
 *
 * Revision 1.5  1993/04/29  16:11:14  craig
 * Changed call to altaz as it now returns the topocentric alt,
 * az, ra, and dec of the star.  The topocentric coordinates
 * are now printed here.
 *
 * Revision 1.4  1993/04/22  19:49:32  craig
 * Minor changes.
 *
 * Revision 1.3  1993/04/21  21:52:59  craig
 * Changed the path for the satellite.h include.
 * Changed ecnsts to pcnsts.
 *
 * Revision 1.2  1993/04/21  15:37:06  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double TDT;
extern double UT;
extern double rearth[3];
extern double dradt;
extern double ddecdt;

extern struct PCONSTANTS pcnsts;

/* from angles.c */

extern double EO;

/*********/
/* rstar */
/*********/

int    rstar (struct star *el, double *salt, double *saz, double *sra,
	      double *sdec)
{
    int    i;
    double p[3], q[3], e[3], m[3], temp[3], polar[3];
    double T, vpi, epoch;
    double cosdec, sindec, cosra, sinra;


    /* Convert from RA and Dec to equatorial rectangular direction */

loop:

    cosdec = cos (el->dec);
    sindec = sin (el->dec);
    cosra = cos (el->ra);
    sinra = sin (el->ra);
    q[0] = cosra * cosdec;
    q[1] = sinra * cosdec;
    q[2] = sindec;

    /* space motion */

    vpi = 21.094952663 * el->v * el->px;

    m[0] = -el->mura * cosdec * sinra
	- el->mudec * sindec * cosra
	+ vpi * q[0];

    m[1] = el->mura * cosdec * cosra
	- el->mudec * sindec * sinra
	+ vpi * q[1];

    m[2] = el->mudec * cosdec
	+ vpi * q[2];

    epoch = el->epoch;

    /* Convert FK4 to FK5 catalogue */

    if (epoch == pcnsts.B1950)
    {
	fk4fk5 (q, m, el);
	goto loop;
    }

    for (i = 0; i < 3; i++)
    {
	e[i] = rearth[i];
    }

    /* precess the earth to the star epoch */

    precess (e, epoch, -1);

    /* Correct for proper motion and parallax */

    T = (TDT - epoch) / pcnsts.dapcen;

    for (i = 0; i < 3; i++)
    {
	p[i] = q[i] + T * m[i] - el->px * e[i];
    }

    /* precess the star to J2000 */

    precess (p, epoch, 1);

    /* reset the earth to J2000 */

    for (i = 0; i < 3; i++)
    {
	e[i] = rearth[i];
    }

    /* Find Euclidean vectors between earth, object, and the sun angles(
       p, q, e ); */

    angles (p, p, e);

    /* Find unit vector from earth in direction of object */

    for (i = 0; i < 3; i++)
    {
	p[i] /= EO;
	temp[i] = p[i];
    }

    if (prtflg)
    {
	/* Report astrometric position */

	showrd ("Astrometric J2000.0:", p, polar);

	/* Also in 1950 coordinates */

	precess (temp, pcnsts.B1950, -1);
	showrd ("Astrometric B1950.0:", temp, polar);

	/* For equinox of date: */

	for (i = 0; i < 3; i++)
	{
	    temp[i] = p[i];
	}

	precess (temp, TDT, -1);
	showrd ("Astrometric of date:", temp, polar);
    }

    /* Correct position for light deflection relativity( p, q, e ); */

    relativity (p, p, e);

    /* Correct for annual aberration */

    annuab (p);

    /* Precession of the equinox and ecliptic from J2000.0 to ephemeris
       date */

    precess (p, TDT, -1);

    /* Ajust for nutation at current ecliptic. */

    epsiln (TDT);
    nutate (TDT, p);

    /* Display the final apparent R.A. and Dec. for equinox of date. */

    showrd ("    Apparent:", p, polar);

    /* Go do topocentric reductions. */

    dradt = 0.0;
    ddecdt = 0.0;
    polar[2] = 1.0e38;			/* make it ignore diurnal parallax */
    altaz (polar, UT, salt, saz, sra, sdec);

    if (prtflg)
    {
        fprintf (outfile, "Topocentric Coordinates:\n");
        fprintf (outfile, "\tAltitude: %.3f deg, Azimuth %.3f deg\n", 
	    *salt, *saz);
        fprintf (outfile, "\tR.A.");
        hms (outfile, *sra);
        fprintf (outfile, "Dec.");
        dms (outfile, *sdec);
        fprintf (outfile, "\n");
    }

    return (0);
}
