/****************************************************************/
/* rplanet.c							*/
/*								*/
/* The following program reduces the heliocentric equatorial	*/
/* rectangular coordinates of the earth and object that		*/
/* were computed by kepler() and produces apparent geocentric	*/
/* right ascension and declination.				*/
/****************************************************************/

/***** description
 *
 *	$Id: rplanet.c,v 1.6 1993/04/30 18:15:49 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: rplanet.c,v $
 * Revision 1.6  1993/04/30  18:15:49  craig
 * Changed the output calls from sending the output to stdout to
 * sending the output to the file outfile (which can be stdout).
 *
 * Revision 1.5  1993/04/29  16:09:37  craig
 * Changed call to altaz as it now returns the topocentric alt,
 * az, ra, and dec of the planet.  The topocentric coordinates
 * are now printed here.
 *
 * Revision 1.4  1993/04/22  19:48:43  craig
 * Minor changes.
 *
 * Revision 1.3  1993/04/21  21:51:13  craig
 * Changed the path of the satellite.h include.
 * Changed ecnsts to pcnsts.
 *
 * Revision 1.2  1993/04/21  15:36:16  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double TDT;
extern double UT;

extern struct PCONSTANTS pcnsts;

/* from angles.c */

extern double EO;
extern double SO;
extern double pq;

/**********/
/* reduce */
/**********/

/* elemnt = orbital elements of q */
/* q[], e[] = heliocentric coordinates */

int    reduce (struct orbit *elemnt, double q[3], double e[3])
{
    int    i;
    double p[3], temp[3], polar[3];
    double a, b, s;
    double oalt, oaz, ora, odec;

    /* Save the geometric coordinates at TDT */

    for (i = 0; i < 3; i++)
    {
	temp[i] = q[i];
    }	
    
    /* Display ecliptic longitude and latitude */

    if (prtflg)
    {
	lonlat (q, TDT, polar);
    }

    /* Adjust for light time (planetary aberration) */

    lightt (elemnt, q, e);

    /* Find Euclidean vectors between earth, object, and the sun */

    for (i = 0; i < 3; i++)
    {
	p[i] = q[i] - e[i];
    }

    angles (p, q, e);

    if (prtflg)
    {
	a = 0.0;

	for (i = 0; i < 3; i++)
	{
	    b = temp[i] - e[i];
	    a += b * b;
	}

	a = sqrt (a);

	fprintf (outfile, "true geocentric distance %.7f au    ", a);
		/* was EO */
	fprintf (outfile, "equatorial diameter %.2f\"\n", 
	    2.0 * elemnt->sdiam / EO);

	/* Calculate visual magnitude. "Visual" refers to the spectrum of
	   visible light. Phase = 0.5(1+pq) = geometric fraction of disc
	   illuminated. where pq = cos( sun-object-earth angle ) The
	   magnitude is V(1,0) + 2.5 log10( SE^2 SO^2 / Phase) where
	   V(1,0) = elemnt->mag is the magnitude at 1au from both earth
	   and sun and 100% illumination. */

	a = 0.5 * (1.0 + pq);

	/* Fudge the phase for light leakage in magnitude estimation. Note
	   this phase term estimate does not reflect reality well.
	   Calculated magnitudes of Mercury and Venus are inaccurate. */

	b = 0.5 * (1.01 + 0.99 * pq);
	s = elemnt->mag + 2.1715 * log (EO * SO) - 1.085 * log (b);

	fprintf (outfile, "approx. visual magnitude %.1f, phase %.3f\n", 
	    s, a);
    }

    /* Find unit vector from earth in direction of object */

    for (i = 0; i < 3; i++)
    {
	p[i] /= EO;
	temp[i] = p[i];
    }

    if (prtflg)
    {
	/* Report astrometric position */

	showrd ("Astrometric J2000.0:", p, polar);

	/* Also in 1950 coordinates */

	precess (temp, pcnsts.B1950, -1);
	showrd ("Astrometric B1950.0:", temp, polar);
    }

    /* Correct position for light deflection */

    relativity (p, q, e);

    /* Correct for annual aberration */

    annuab (p);

    /* Precession of the equinox and ecliptic from J2000.0 to ephemeris
       date */

    precess (p, TDT, -1);

    /* Ajust for nutation at current ecliptic. */

    epsiln (TDT);
    nutate (TDT, p);

    /* Display the final apparent R.A. and Dec. for equinox of date. */

    showrd ("    Apparent:", p, polar);

    /* Go do topocentric reductions. */

    polar[2] = EO;
    altaz (polar, UT, &oalt, &oaz, &ora, &odec);

    fprintf (outfile, "Topocentric Coordinates:\n");
    fprintf (outfile, "\tAltitude: %.3f deg, Azimuth %.3f deg\n", oalt, oaz);
    fprintf (outfile, "\tR.A.");
    hms (outfile, ora);
    fprintf (outfile, "Dec.");
    dms (outfile, odec);
    fprintf (outfile, "\n");


    return (0);
}
