/*************************************************/
/* refrac.c					 */
/*						 */
/* Atmospheric refraction			 */
/*						 */
/* Returns correction in degrees to be added 	 */
/* to true altitude to obtain apparent altitude. */
/*						 */
/* -- S. L. Moshier				 */
/*************************************************/

/***** description
 *
 *	$Id: refrac.c,v 1.3 1993/04/21 21:43:57 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: refrac.c,v $
 * Revision 1.3  1993/04/21  21:43:57  craig
 * Changed the path of the satellite.h include.
 *
 * Revision 1.2  1993/04/21  15:33:55  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

extern struct MCONSTANTS mcnsts;

/* from kfiles.c */

extern double atpress;			/* millibars */
extern double attemp;			/* degrees C */

/**********/
/* refrac */
/**********/

/* alt = altitude in degrees */

double refrac (double alt)
{
    int    i;
    double y, y0, D0, N, D, P, Q;

    if ((alt < -2.0) || (alt >= 90.0))
    {
	return (0.0);
    }

    /* For high altitude angle, AA page B61 Accuracy usually about 0.1' */

    if (alt > 15.0)
    {
	D = 0.00452 * atpress / ((273.0 + attemp) * tan (mcnsts.de2ra * alt));
	return (D);
    }

    /* Formula for low altitude is from the Almanac for Computers. It
       gives the correction for observed altitude, so has to be inverted
       numerically to get the observed from the true. Accuracy about 0.2'
       for -20C < T < +40C and 970mb < P < 1050mb. */

    /* Start iteration assuming correction = 0 */

    y = alt;
    D = 0.0;

    /* Invert Almanac for Computers formula numerically */

    P = (atpress - 80.0) / 930.0;
    Q = 4.8e-3 * (attemp - 10.0);
    y0 = y;
    D0 = D;

    for (i = 0; i < 4; i++)
    {
	N = y + (7.31 / (y + 4.4));
	N = 1.0 / tan (mcnsts.de2ra * N);
	D = N * P / (60.0 + Q * (N + 39.0));
	N = y - y0;
	y0 = D - D0 - N;		/* denominator of derivative */

	if ((N != 0.0) && (y0 != 0.0))
	{
	    /* Newton iteration with numerically estimated derivative */
	    N = y - N * (alt + D - y) / y0;
	}
	else
	{
	    /* Can't do it on first pass */
	    N = alt + D;
	}

	y0 = y;
	D0 = D;
	y = N;
    }

    return (D);
}
