/********************************************************/
/* oearth.c						*/
/*							*/
/* Orbital elements and perturbations for the earth	*/
/* Meeus, chapter 18					*/
/********************************************************/

/***** description
 *
 *	$Id: oearth.c,v 1.3 1993/04/21 21:28:24 craig Exp $
 *
 */

/***** modification history
 *
 *	$Id: oearth.c,v 1.3 1993/04/21 21:28:24 craig Exp $
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

extern struct MCONSTANTS mcnsts;
extern struct PCONSTANTS pcnsts;

double M;				/* Mean anomaly of the earth (and
					   sun) */
double T;				/* centuries from 1900.0 */

/**********/
/* oearth */
/**********/

int    oearth (struct orbit * e, double J)
{
    double f;

    e->epoch = J;
    T = (J - pcnsts.J1900) / pcnsts.dapcen; /* centuries from 1900.0 */

    /* mean anomaly of the earth (and sun) */

    f = ((-3.3e-6 * T - 0.000150) * T + 35999.04975) * T + 358.47583;
    f = mod360 (f);
    e->M = f;
    M = f;				/* save in global place */

    /* eccentricity of the earth's orbit */

    e->ecc = (-0.000000126 * T - 0.0000418) * T + 0.01675104;

    /* semimajor axis */

    e->a = 1.0000002;

    e->equinox = pcnsts.J2000;
    e->i = ((1.4e-8 * T - 9.27e-6) * T + 0.0130855) * T - 0.0130762;
    f = ((3.333e-6 * T + 0.00013610) * T + 0.5647920) * T + 287.511505;
    e->w = mod360 (f);
    f = ((-2.8e-8 * T + 7.94e-6) * T - 0.2416582) * T + 175.105679;
    e->W = mod360 (f);
    f = e->w + e->M + e->W;
    e->L = mod360 (f);

    return (0);
}

/************************************************/
/* cearth 					*/
/*						*/
/* perturbations of the earth's orbit added in 	*/
/* after solving Kepler's equation		*/
/************************************************/

int    cearth (struct orbit *e)
{
    double A, B, C, D, E, H, f;

    /* perturbations due to Venus: */

    A = (22518.7541 * T + 153.23) * mcnsts.de2ra;
    B = (45037.5082 * T + 216.57) * mcnsts.de2ra;

    /* due to Jupiter: */

    C = (32964.3577 * T + 312.69) * mcnsts.de2ra;
    H = (65928.7155 * T + 353.40) * mcnsts.de2ra;

    /* due to the moon: */

    D = ((-0.00144 * T + 445267.1142) * T + 350.74) * mcnsts.de2ra;

    /* long period perturbation: */

    E = (20.20 * T + 231.19) * mcnsts.de2ra;

    /* corrections to earth's longitude */

    f = 0.00134 * cos (A)
	+ 0.00154 * cos (B)
	+ 0.00200 * cos (C)
	+ 0.00179 * sin (D)
	+ 0.00178 * sin (E);
    e->L += f * mcnsts.de2ra;

    /* corrections to earth's radius vector */

    f = 0.00000543 * sin (A)
	+ 0.00001575 * sin (B)
	+ 0.00001627 * sin (C)
	+ 0.00003076 * cos (D)
	+ 0.00000927 * sin (H);
    e->r += f;

    return (0);
}
