/********************************************************/
/* lightt.c						*/
/*							*/
/* Correction for light time from object to earth	*/
/* including gravitational retardation due to the Sun.	*/
/* AA page B36.						*/
/********************************************************/

/***** description
 *
 *	$Id: lightt.c,v 1.6 1993/04/30 18:20:53 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: lightt.c,v $
 * Revision 1.6  1993/04/30  18:20:53  craig
 * Changed the output call from sending the output to stdout to
 * sending the output to the file outfile (which can be stdout).
 *
 * Revision 1.5  1993/04/27  16:58:49  craig
 * changed the calling of deltap so that the calculated motions
 * have the correct sign.
 *
 * Revision 1.4  1993/04/23  19:00:43  craig
 * replaced 173.1446... with pcnsts.caupda.
 *
 * Revision 1.3  1993/04/22  19:46:49  craig
 * minor changes.
 *
 * Revision 1.2  1993/04/21  15:18:33  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double TDT;
extern double dp[3];
extern double dradt;
extern double ddecdt;

extern struct PCONSTANTS pcnsts;

/* from vearth.c */

extern double vearth[3];

/**********/
/* lightt */
/**********/

/* e[], q[] = rectangular position vectors */
/* elemnt = orbital elements of object q */

int    lightt (struct orbit *elemnt, double *q, double *e)
{
    int    i, k;
    double p[3], p0[3], ptemp[3];
    double P, Q, E, t, x, y;


    /* save initial q-e vector for display */

    for (i = 0; i < 3; i++)
    {
	p0[i] = q[i] - e[i];
    }

    E = 0.0;

    for (i = 0; i < 3; i++)
    {
	E += e[i] * e[i];
    }

    E = sqrt (E);

    for (k = 0; k < 2; k++)
    {
	P = 0.0;
	Q = 0.0;

	for (i = 0; i < 3; i++)
	{
	    y = q[i];
	    x = y - e[i];
	    p[i] = x;
	    Q += y * y;
	    P += x * x;
	}

	P = sqrt (P);
	Q = sqrt (Q);

	/* Note the following blows up if object equals sun. */

	t = (P + 1.97e-8 * log ((E + P + Q) / (E - P + Q))) / 
	    pcnsts.caupda;
	kepler (TDT - t, elemnt, q, ptemp);
    }

    if (prtflg)
    {
	fprintf (outfile, "light time %.4fm,  ", pcnsts.xmnpda * t);
    }

    /* Final object-earth vector and the amount by which it changed.  */

    for (i = 0; i < 3; i++)
    {
	x = q[i] - e[i];
	p[i] = x;
	dp[i] = x - p0[i];
    }

    showcor ("aberration", p0, dp);

    /* Calculate dRA/dt and dDec/dt.
     * The desired correction of apparent coordinates is relative
     * to the equinox of date, but the coordinates here are
     * for J2000.  This introduces a slight error.
     *
     * Estimate object-earth vector t days ago.  We have
     * p(?) = q(J-t) - e(J), and must adjust to
     * p(J-t)  =  q(J-t) - e(J-t)  =  q(J-t) - (e(J) - Vearth * t)
     *         =  p(?) + Vearth * t.
     */

    velearth (TDT);

    for (i = 0; i < 3; i++)
    {
	p[i] += vearth[i] * t;
    }

    deltap (p0, p, &dradt, &ddecdt);	/* see dms.c */
    dradt /= t;
    ddecdt /= t;

    return (0);
}
