/****************************************************************/
/* kfiles.c							*/
/*								*/
/* Disc file input routines to read initialization parameters	*/
/* parameters or file containing orbital elements.		*/
/****************************************************************/

/***** description
 *
 *	$Id: kfiles.c,v 1.7 1993/05/12 19:07:25 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: kfiles.c,v $
 * Revision 1.7  1993/05/12  19:07:25  craig
 * kinit now searches for a .site.ini file in the current directory or
 * the home directory.
 *
 * The printing of the site information is now optional using prtflg.
 *
 * The call to fincat in the getstar routine now includes a dummy
 * second char string to fit the prototype for fincat.
 *
 * Revision 1.6  1993/04/30  18:10:49  craig
 * Minor cosmetic changes.
 *
 * Revision 1.5  1993/04/22  21:04:36  craig
 * Added filename and flag option to kinit.  Kinit will now try to
 * open the file filename and print Moshier's banner if flag == 1.
 *
 * Revision 1.4  1993/04/22  19:45:09  craig
 * Minor changes.
 *
 * Revision 1.3  1993/04/21  21:22:42  craig
 * Changed the path of the satellite.h include.
 * Changed ecnsts to pcnsts.
 *
 * Revision 1.2  1993/04/21  15:17:42  craig
 * First working version.  Ran through indent and converted to ansi.
 * Added hooks for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "aaproto.h"
#include "satellite.h"

/***** global variables *****/

/* from cnstinit.c */

extern int jdflag;
extern int prtflg;

extern struct orbit earth;
extern struct MCONSTANTS mcnsts;
extern struct PCONSTANTS pcnsts;

/* from deltat.c */

extern double dtgiven;

/* The following items will be read in automatically from the disc file
 * named aa.ini, if one is provided.  The file contains one ASCII
 * string number per line so is easily edited.
 *
 * Terrestrial geocentric latitude and longitude of observer
 * in degrees East of Greenwich, North of equator
 * (kfiles.c converts from geodetic latitude input.)
 */

double tlong = -71.13;			/* Cambridge, Massachusetts */
double tlat = 42.38;			/* geocentric */
double glat = 42.17;			/* geodetic */

/* Parameters for calculation of azimuth and elevation */

double attemp = 20.0;			/* atmospheric temperature,
					   degrees Centigrade */
double atpress = 1013.0;		/* atmospheric pressure, millibars */

/* Distance from observer to center of earth, in earth radii */

double trho = 0.9985;

/***** local global variables *****/

static int linenum = 1;

static char starnam[80] = {'s', 't', 'a', 'r', '.', 'c', 'a', 't', '\0'};
static char orbnam[80] = {'o', 'r', 'b', 'i', 't', '.', 'c', 'a', 't', '\0'};

static double flat = 298.257222;
static double aearth = 6378137.;
static double height = 0.0;

/************************************************/
/* kinit					*/
/*						*/
/* Read initialization file aa.ini and 		*/
/* adjust topocentric coordinates of observer.	*/
/************************************************/

int    kinit (void)
{
    char   s[84];
    char  *homedir, home[160];
    FILE  *f;
    double a, b, fl, co, si, u;

    /*
    if (prtflg)
    {
	printf ("\n\tSteve Moshier's Ephemeris Program v5.1\n\n");
	printf ("Planetary perturbations are from Meeus.\n");
	printf ("Extended Chapront Lunar theory.\n");
    }
    */


    f = fopen ("./.site.ini", "r");	/* primary site file */

    if (f == NULL)
    {
	homedir = getenv ("HOME");
	strcpy (home, homedir);
	strcat (home, "/.site.ini");
	f = fopen (home, "r");		/* alternate site file */

	if (f == NULL)
	{
	    return (-1);
	}
    }

    fgets (s, 80, f);
    sscanf (s, "%lf", &tlong);
    fgets (s, 80, f);
    sscanf (s, "%lf", &glat);
    fgets (s, 80, f);
    sscanf (s, "%lf", &height);
    u = glat * mcnsts.de2ra;

    /* Reduction from geodetic latitude to geocentric latitude AA page K5 */

    co = cos (u);
    si = sin (u);
    fl = 1.0 - 1.0 / flat;
    fl = fl * fl;
    si = si * si;
    u = 1.0 / sqrt (co * co + fl * si);
    a = aearth * u + height;
    b = aearth * fl * u + height;
    trho = sqrt (a * a * co * co + b * b * si);
    tlat = mcnsts.ra2de * acos (a * co / trho);

    if (glat < 0.0)
    {
	tlat = -tlat;
    }

    trho /= aearth;

    /* Reduction from geodetic latitude to geocentric latitude AA page K5 */

    /* tlat = glat - 0.19242861 * sin(2.0*u) + 0.00032314 * sin(4.0*u) -
       0.00000072 * sin(6.0*u);
    
    trho = 0.998327073 + 0.001676438 * cos(2.0*u) - 0.000003519 *
       cos(4.0*u) + 0.000000008 * cos(6.0*u); trho += height/6378160.; */

    fgets (s, 80, f);
    sscanf (s, "%lf", &attemp);
    fgets (s, 80, f);
    sscanf (s, "%lf", &atpress);
    fgets (s, 80, f);
    sscanf (s, "%d", &jdflag);
    fgets (s, 80, f);
    sscanf (s, "%lf", &dtgiven);

    if (prtflg)
    {
	printf ("\nSite parameters:\n\n");
	printf ("Terrestrial east longitude %.4f deg\n", tlong);
	printf ("geocentric latitude %.4f deg\n", tlat);
	printf ("Earth radius %.5f\n", trho);
	printf ("temperature %.1f C\n", attemp);
	printf ("pressure %.0f mb\n", atpress);

	switch (jdflag)
	{
	  case 0:
	    printf ("TDT and UT assumed equal.\n");
	    break;
	  case 1:
	    printf ("Input time is TDT.\n");
	    break;
	  case 2:
	    printf ("Input time is UT.\n");
	    break;
	  default:
	    printf ("Illegal jdflag\n");
	    exit (0);
	}

	if (dtgiven != 0.0)
	{
	    printf ("Using deltaT = %.2fs.\n", dtgiven);
	}

	printf ("\n");
    }

    fclose (f);

    return (0);
}

/***********************************************************/
/* getorbit						   */
/*							   */
/* Program to read in a file containing orbital parameters */
/***********************************************************/

int    getorbit (struct orbit * el)
{
    int    i;
    FILE  *f;
    char   s1[100], s2[100], *u, *v;


    getstring ("Name of orbit catalogue file: ", orbnam, 80);

    f = fincat (orbnam, 2, s1, s2);

    if (f == 0)
    {
	return (-1);			/* failure flag */
    }

    printf ("%s\n", s1);
    printf ("%s\n", s2);

    /* Read in ASCII floating point numbers */

    sscanf (s1, "%lf %lf %lf %lf %lf %lf",
	&el->epoch, &el->i, &el->W, &el->w, &el->a, &el->dm);

    sscanf (s2, "%lf %lf %lf %lf %lf %15s", &el->ecc, &el->M,
	&el->equinox, &el->mag, &el->sdiam, &el->obname[0]);

    el->obname[15] = '\0';

    /* Clear out the rest of the data structure */

    el->oelmnt = 0;
    el->celmnt = 0;
    el->L = 0.0;
    el->r = 0.0;
    el->plat = 0.0;

    if (strcmp (&el->obname[0], "Earth"))
    {
	return (0);
    }
    else
    {
	u = (char *) &earth;
	v = (char *) el;

	for (i = 0; i < sizeof (struct orbit); i++)
	{
	    *u++ = *v++;
	}

	printf ("Read in earth orbit\n");

	return (1);
    }
}

/***********/
/* getstar */
/***********/

int    getstar (struct star * el)
{
    int    i;
    int    sign;
    char  *p;
    char   s1[100], s2[100];
    FILE  *f;
    double rh, rm, rs, dd, dm, ds, x, z;

    getstring ("Name of star catalogue file: ", starnam, 80);
    f = fincat (starnam, 1, s1, s2);

    if (f == 0)
    {
	return (-1);			/* failure flag */
    }

    printf ("%s\n", s1);

    /* Read in the ASCII string data and name of the object */

    sscanf (s1, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %s",
	&el->epoch, &rh, &rm, &rs, &dd, &dm, &ds,
	&el->mura, &el->mudec, &el->v, &el->px, &x, &el->obname[0]);

    x = el->epoch;

    if (x == 2000.0)
    {
	x = pcnsts.J2000;
    }
    else if (x == 1950.0)
    {
	x = pcnsts.B1950;
    }
    else if (x == 1900.0)
    {
	x = pcnsts.J1900;
    }
    else
    {
	x = pcnsts.J2000 + (pcnsts.dapcen / 100.) * (x - 2000.0);
    }

    el->epoch = x;

    /* read the right ascension */

    el->ra = 2.0 * mcnsts.pi * (3600.0 * rh + 60.0 * rm + rs) /
	pcnsts.secpda;


    /* read the declination */

    sign = 1;

    if ((dd < 0.0) || (dm < 0.0) || (ds < 0.0))
    {
	sign = -1;
    }

    z = (3600.0 * fabs (dd) + 60.0 * fabs (dm) + fabs (ds)) / mcnsts.ra2sec;

    if (dd == 0.0)
    {
	/* Scan the text for possible minus sign in front of declination 0 */

	p = s1;

	/* skip over 4 fields */

	for (i = 0; i < 4; i++)
	{
	    while (*p++ == ' ');

	    while (*p++ != ' ');
	}

	while (*p++ == ' ');

	--p;

	if (*p == '-')
	{
	    sign = -1;
	}
    }

    if (sign < 0)
    {
	z = -z;
    }

    el->dec = z;

/*
    printf ("%.2lf\n", el->epoch);
    printf ("%.0lf %.0lf %.3lf\n", rh, rm, rs);
    printf ("%.8lf\n", el->ra);
    printf ("%.0lf %.0lf %.3lf\n", dd, dm, ds);
    printf ("%.8lf\n", el->dec);
    printf ("d %.3lf mua %.3lf mud %.3lf v %.3lf\n",
	el->px, el->mura, el->mudec, el->v);
*/

    el->mura *= 15.0 / mcnsts.ra2sec;	/* s/century -> "/century ->
					   rad/century */
    el->mudec /= mcnsts.ra2sec;
    z = el->px;

    if (z < 1.0)
    {
	if (z <= 0.0)
	{
	    el->px = 0.0;
	}
	else
	{
	    el->px = mcnsts.sec2ra * z;	/* assume px in arc seconds */
	}
    }
    else
    {
	el->px = 1.0 / (mcnsts.ra2sec * z);	/* parsecs -> radians */
    }
    return (0);
}

/****************************************/
/* fincat				*/
/*					*/
/* Open catalogue and find line number	*/
/****************************************/

FILE  *fincat (char *name, int n, char *str1, char *str2)
{
    int    i;
    FILE  *f;

    f = fopen (name, "r");

    if (f == 0)
    {
	printf ("Can't find file %s\n", name);
	return (0);			/* failure flag */
    }

    getint ("Line number", &linenum);

    if (linenum <= 0)
    {
	goto failure;
    }

    for (i = 0; i < linenum; i++)
    {
	fgets (str1, 98, f);

	if (*str1 == '-')
	{
	    goto endf;
	}

	if (n > 1)
	{
	    fgets (str2, 98, f);

	    if (*str2 == '-')
	    {
		goto endf;
	    }
	}
    }

    fclose (f);

    return (f);

endf:

    printf ("End of file reached.\n");

failure:

    fclose (f);

    return (0);
}
