/********************************************************/
/* cnstinit.c						*/
/*							*/
/* initialize math constants and physical constants	*/
/********************************************************/

/***** description
 *
 *	$Id: cnstinit.c,v 1.8 1993/05/12 19:12:57 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: cnstinit.c,v $
 * Revision 1.8  1993/05/12  19:12:57  craig
 * updated the orbital elements for the planets.
 *
 * Revision 1.7  1993/04/30  18:04:57  craig
 * Changed the earth's orbital elements to the elements for
 * 13 Jan 93.
 *
 * Added the global variable outfile for use in directing the
 * output.
 *
 * Revision 1.6  1993/04/27  17:00:16  craig
 * added initialization of pcnsts.caupda --- the speed of light
 * in au per day.
 *
 * Revision 1.5  1993/04/22  19:02:29  craig
 * Moved some statics back to aa.c
 *
 * Revision 1.4  1993/04/22  18:31:41  craig
 * Moved the global variables from aa.c to this file.
 *
 * Revision 1.3  1993/04/21  21:00:24  craig
 * added initialization of pcnsts.c and pcnsts.k
 *
 * Revision 1.2  1993/04/21  20:52:11  craig
 * changed path for satellite.h include.  changed ecnsts to
 * pcnsts. added initialization for pcnsts.secpda.
 *
 * Revision 1.1  1993/04/21  15:02:19  craig
 * Initial revision
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "satellite.h"
#include "aaproto.h"

/***** global variables *****/

struct PCONSTANTS pcnsts;
struct MCONSTANTS mcnsts;

/* approximate motion of right ascension and declination of object,
   in radians per day */

double dradt = 0.0;
double ddecdt = 0.0;

/* Orbits for each planet.  The indicated orbital elements are not
   actually used, since they are now calculated from the Meeus
   expansions.  Magnitude and semidiameter are still used. */

struct orbit mercury = {
    "Mercury        ",
    2449000.5,		/* epoch */
    7.00539,		/* inclination */
    48.3392,		/* Longitude of ascending node */
    29.105,		/* arg of perihelion (Long of perh - long asc node */
    0.3870989,		/* mean distance */
    4.092336,		/* daily motion */
    0.2056201,		/* eccentricity */
    201.85228,		/* mean anomaly (Mean long - long of perh) */
    2451545.0,		/* equiniox (J2000) */
    -0.42,		/* vis mag at 1 AU from earth and sun */
    3.36,		/* equitorial semi-diameter at 1AU, arcsec */
    omercury,		/* address of program to compute elements */
    cmercury,		/* program to correct longitude and radius */
    0.0,		/* computed mean longitude */
    0.0,		/* computed radius vector */
    0.0			/* perturbation in ecliptic latitude */
};

struct orbit venus = {
    "Venus          ",
    2449000.5,
    3.39460,
    76.6998,
    54.9962,
    0.7233249,
    1.602155,
    0.0067491,
    293.66487,
    2451545.0,
    /* Note the calculated apparent visual magnitude for Venus is not very
       accurate. */
    -4.40,
    8.34,
    ovenus,
    cvenus,
    0.0,
    0.0,
    0.0
};

struct orbit earth = {
    "Earth          ",
    2449000.5,
    0.00096,
    0.6,
    102.247,
    1.0000029,
    0.9856049,
    0.0167052,
    9.73412,
    2451545.0,
    -3.86,
    0.0,
    0,
    0,
    0.0,
    0.0,
    0.0
};

struct orbit mars = {
    "Mars           ",
    2449000.5,
    1.85033,
    49.5753,
    286.4424,
    1.5236263,
    0.5240666,
    0.0934810,
    126.03036,
    2451545.0,
    -1.52,
    4.68,
    omars,
    cmars,
    0.0,
    0.0,
    0.0
};

struct orbit jupiter = {
    "Jupiter        ",
    2449000.5,
    1.30464,
    100.4696,
    275.2175,
    5.202988,
    0.08308681,
    0.0482999,
    167.26314,
    2451545.0,
    -9.40,
    98.44,
    ojupiter,
    0,
    0.0,
    0.0,
    0.0
};

struct orbit saturn = {
    "Saturn         ",
    2449000.5,
    2.48682,
    113.6735,
    339.8235,
    9.525356,
    0.03353083,
    0.0538115,
    231.30317,
    2451545.0,
    -8.88,
    82.73,
    osaturn,
    csaturn,
    0.0,
    0.0,
    0.0
};

struct orbit uranus = {
    "Uranus         ",
    2449000.5,
    0.77226,
    74.0219,
    98.8178,
    19.21680,
    0.01170016,
    0.0477621,
    110.40368,
    2451545.0,
    -7.19,
    35.02,
    ouranus,
    curanus,
    0.0,
    0.0,
    0.0
};

struct orbit neptune = {
    "Neptune        ",
    2449000.5,
    1.77202,
    131.7531,
    272.0569,
    30.10411,
    0.005967282,
    0.0066367,
    245.64979,
    2451545.0,
    -6.87,
    33.50,
    oneptune,
    cneptune,
    0.0,
    0.0,
    0.0
};

/* Note there are no perturbation formulas for Pluto. The program
   automatically uses the given numerical values since the pointers to
   perturbation subroutines are null. For some reason the J2000 orbit
   given for Pluto in the AA does not give the same results as the "of
   date" orbit.  Yet, results are the same for the other planets. */

struct orbit pluto = {
    "Pluto          ",
    2449000.5,
    17.14012,
    110.2901,
    113.6581,
    39.77157,
    0.003929569,
    0.2542840,
    4.75105,
    2451545.0,
    -1.0,
    2.07,
    0,
    0,
    0.0,
    0.0,
    0.0
};

/* coordinates of object */

int    objnum = 0;			/* I.D. number of object */

/* ecliptic polar coordinates: longitude, latitude in radians radius in
   au */

double obpolar[3];

/* coordinates of Earth */

/* Heliocentric rectangular equatorial position of the earth at time TDT
   re equinox J2000 */

double rearth[3];

/* Corresponding polar coordinates of earth: longitude and latitude in
   radians, radius in au */

double eapolar[3];

/* Julian date of ephemeris */

double JD;
double TDT;
double UT;

int    jdflag = 0;

/* flag = 0 if TDT assumed = UT,
	= 1 if input time is TDT,
	= 2 if input time is UT. */

/* correction vector, saved for display  */

double dp[3];

/* display enable flag */

int    prtflg = 1;

/* the output file */

FILE  *outfile;

/****************/
/* cnstinit	*/
/****************/

void cnstinit (void)
{
    double qo, so, xj2, xj3, xj4;

    /***** initialize constants *****/

    mcnsts.pi = 3.14159265358979323846;
    mcnsts.pio2 = mcnsts.pi / 2.;
    mcnsts.twopi = mcnsts.pi * 2.;
    mcnsts.x3pio2 = mcnsts.pi * 3. / 2.;
    mcnsts.tothrd = 2. / 3.;
    mcnsts.e6a = 1.e-6;
    mcnsts.de2ra = 2. * mcnsts.pi / 360.;
    mcnsts.ra2de = 1. / mcnsts.de2ra;
    mcnsts.ra2sec = 3600. * mcnsts.ra2de;
    mcnsts.sec2ra = 1. / mcnsts.ra2sec;

    qo = 120.0;
    so = 78.0;
    xj2 = .001082616;			/* second grav. zonal harmonic */
    xj3 = -.253881e-5;			/* third grav. zonal harmonic */
    xj4 = -1.65597e-6;			/* fourth grav. zonal harmonic */

    pcnsts.c = 299792458.;		/* speed of light in m/s */
    pcnsts.k = 0.01720209895;		/* gaussian grav. constant */
    pcnsts.ae = 1.;			/* distance units = 1 earth radii */
    pcnsts.kmpau = 1.4959787066e8;	/* km per au */
    pcnsts.xkmper = 6378.135;		/* earth radius in km */
    pcnsts.xmnpda = 1440.;		/* minutes per day */
    pcnsts.secpda = 86400.;		/* seconds per day */
    pcnsts.dapcen = 36525.0;		/* days per julian century */
    pcnsts.J2000 = 2451545.0;		/* julian day for 2000 Jan 1.5 */
    pcnsts.B1950 = 2433282.423;		/* julian day for 1950 Jan 0.923 */
    pcnsts.J1900 = 2415020.0;		/* julian day for 1900 Jan 0.5 */

    pcnsts.caupda = pcnsts.c * pcnsts.secpda / 1000. / pcnsts.kmpau;
    pcnsts.xauper = pcnsts.xkmper / pcnsts.kmpau; /* earth radius in au */
    pcnsts.rapau = 1. / pcnsts.xauper;	/* # earth radii per au */

    pcnsts.ck2 = xj2 * pcnsts.ae * pcnsts.ae / 2.;
    pcnsts.ck4 = (-3. / 8.) * xj4 * pow (pcnsts.ae, 4.);
    pcnsts.xj3 = xj3;
    pcnsts.xke = .0743669161;
    pcnsts.qoms2t = pow (qo - so, 4.) * pow (pcnsts.ae / pcnsts.xkmper, 4.);
    pcnsts.s = pcnsts.ae * ((so / pcnsts.xkmper) + 1.);

    return;
}
