/************************************************************************/
/* altaz.c								*/
/*									*/
/* Converts apparent geocentric coordinates to local tropocentric place */
/************************************************************************/

/***** description
 *
 *	$Id: altaz.c,v 1.7 1993/04/30 18:17:51 craig Exp $
 *
 * Apply diurnal aberrations and calculate topocentric
 * altitude and azimuth, given the geocentric apparent
 * right ascension and declination.
 *
 * Ephemeris transit times can be obtained by modifying
 * aa.ini to declare TDT = UT.
 *
 * This program assumes that deltaT, the difference
 * between Ephemeris Time (TDT) and Universal Time (UT),
 * has already been calculated.  The global variables
 * TDT and UT must already have been loaded with the
 * correct values of TDT and UT, respectively.  See deltat.c.
 *
 * Inputs are polar coordinates:
 * dist is the true geocentric distance in au.
 * ra and dec are in radians.
 *
 * J is the Julian date in UT measure.
 *
 * AA page B60 and D3.
 */

/***** modification history
 *
 *	$Log: altaz.c,v $
 * Revision 1.7  1993/04/30  18:17:51  craig
 * Changed the output calls from sending the output to stdout to
 * sending the output to the file outfile (which can be stdout).
 *
 * Revision 1.6  1993/04/29  16:07:51  craig
 * changed altaz so that it returns the topocentric alt, az,
 * ra and dec of the object.  altaz no longer prints the topocentric
 * coordinates.
 *
 * Revision 1.5  1993/04/27  16:56:24  craig
 * Made the printing of the siderial time and the calculation of the
 * transit times optional using prtflg.
 *
 * Revision 1.4  1993/04/22  19:22:57  craig
 * minor change.
 *
 * Revision 1.3  1993/04/21  20:25:06  craig
 * changed the path of the satellite.h include.
 *
 * Revision 1.2  1993/04/21  14:58:55  craig
 * first working version.  Ran through indent. Converted to ansi.  Added
 * stuff for working with the satellite programs.
 *
 *
 */

/***** include files *****/

#include <math.h>
#include "satellite.h"
#include "aaproto.h"

/***** global variables *****/

/* from cnstinit.c */

extern int prtflg;

extern FILE *outfile;

extern double dradt;
extern double ddecdt;

extern struct MCONSTANTS mcnsts;

/* from kfiles.c */

extern double tlong, tlat, glat;

/*********/
/* altaz */
/*********/

int    altaz (double *pol, double J, double *objalt, double *objaz, 
	   double *objra, double *objdec)
{
    double dec, cosdec, sindec, lha, coslha, sinlha;
    double ra, dist, last, alt, az, coslat, sinlat;
    double N, D, x, y, z;

    ra = pol[0];
    dec = pol[1];
    dist = pol[2];

    /* local apparent sidereal time, seconds converted to radians */

    last = sidrlt (J, tlong) * mcnsts.de2ra / 240.0;
    lha = last - ra;			/* local hour angle, radians */

    if (prtflg)
    {
        fprintf (outfile, "Local apparent sidereal time ");
        hms (outfile, last);
        fprintf (outfile, "\n");
    }

    /* Display rate at which ra and dec are changing */
    
    if (prtflg)
    {
    	x = mcnsts.ra2sec / 24.0;
    	N = x * dradt;
    	D = x * ddecdt;
    	if( N != 0.0 )
	{
    	    fprintf (outfile, "dRA/dt %.2f\"/h, dDec/dt %.2f\"/h\n", N, D );
	}
    }

    /* Do rise, set, and transit times */

    if (prtflg)
    {
        trnsit (J, lha, dec);
    }

    /* Diurnal parallax */

    diurpx (last, &ra, &dec, dist);

    /* Diurnal aberration */

    diurab (last, &ra, &dec);

    /* Convert ra and dec to altitude and azimuth */

    cosdec = cos (dec);
    sindec = sin (dec);
    lha = last - ra;
    coslha = cos (lha);
    sinlha = sin (lha);

    /* Use the geodetic latitude for altitude and azimuth */

    x = mcnsts.de2ra * glat;
    coslat = cos (x);
    sinlat = sin (x);

    N = -cosdec * sinlha;
    D = sindec * coslat - cosdec * coslha * sinlat;
    az = matan2 (N, D) * mcnsts.ra2de;
    alt = sindec * sinlat + cosdec * coslha * coslat;
    alt = asin (alt) * mcnsts.ra2de;

    /* Correction for atmospheric refraction (unit = degrees) */

    D = refrac (alt);
    alt += D;

    /* Convert back to R.A. and Dec.  */
 
    y = sin (mcnsts.de2ra * alt);
    x = cos (mcnsts.de2ra * alt);
    z = cos (mcnsts.de2ra * az);
    sinlha = -x * sin (mcnsts.de2ra * az);
    coslha = y * coslat - x * z * sinlat;
    sindec = y * sinlat + x * z * coslat;
    lha = matan2 (sinlha, coslha);

    /* save previous values, before refrac() */

    y = ra;
    z = dec;
    dec = asin (sindec);
    ra = last - lha;
    y = ra - y;				/* change in ra */

    while (y < -mcnsts.pi)
    {
	y += mcnsts.twopi;
    }

    while (y > mcnsts.pi)
    {
	y -= mcnsts.twopi;
    }

    y = mcnsts.ra2sec * y / 15.0;
    z = mcnsts.ra2sec * (dec - z);

    if (prtflg)
    {
	fprintf (outfile, "refraction %.3f deg  dRA %.3fs dDec %.2f\"\n",
	    D, y, z);
    }

    *objalt = alt;
    *objaz = az;
    *objra = ra;
    *objdec = dec;

    return (0);
}
