/************************************************************************/
/* aa.c --- driver for astronomical almanac programs			*/
/*									*/
/* This program calculates orbits of planetary bodies and reduces	*/
/* the coordinates of planets or stars to geocentric and topocentric	*/
/* place.  An effort has been made to use rigorous methods throughout.	*/
/*									*/
/* References to AA page numbers are to The Astronomical Almanac, 1986	*/
/* published by the U.S. Government Printing Office.			*/
/*									*/
/* The program uses as a default the orbital perturbations given in	*/
/* Jean Meeus, "Astronomical Formulae for Calculators", 3rd ed.,	*/
/* Willmann-Bell, Inc., 1985.						*/
/*									*/
/* Warning! Your atan2() function may not work the same as the one	*/
/* assumed by this program.						*/
/* atan2(x,y) computes atan(y/x), result between 0 and 2pi.		*/
/*									*/
/* S. L. Moshier, November, 1987					*/
/************************************************************************/

/***** description
 *
 *	$Id: aa.c,v 1.9 1993/05/18 16:44:15 craig Exp $
 *
 */

/***** modification history
 *
 *	$Log: aa.c,v $
 * Revision 1.9  1993/05/18  16:44:15  craig
 * modified call of dosun to add a semi-diameter input variable.
 *
 * Revision 1.8  1993/05/12  19:11:58  craig
 * changed the usage of the kinit routine to reflect modifications
 * with that routine.   The same for the rstar routine.
 *
 * Revision 1.7  1993/04/30  18:08:11  craig
 * Put in the option to send the output to somewhere different
 * than stdout.  Also some cosmetic changes in the output
 * and looping through the program.
 *
 * Revision 1.6  1993/04/29  16:05:26  craig
 * Changed the call to dosun as dosun now returns the alt, az
 * ra, and dec of the sun.
 *
 * Revision 1.5  1993/04/22  21:06:33  craig
 * Changed the call to kinit to a call with a filename and a flag
 * to reflect the change in kinit.
 *
 * Revision 1.4  1993/04/22  19:06:13  craig
 * Moved global variable declarations to cnstinit.c
 *
 * Revision 1.3  1993/04/21  20:24:03  craig
 * Removed the satellite.h include and the ecnsts and
 * mcnsts global structures.
 *
 * Revision 1.2  1993/04/21  14:57:10  craig
 * Ran through indent.  Converted to ansi.  Incorporated stuff for
 * use with the satellite program.
 *
 *
 */

/***** include files *****/

#include <string.h>
#include "aaproto.h"

/***** global variables *****/

/* from cnstinit.c */

extern int objnum;
extern int prtflg;

extern FILE *outfile;

extern double JD;
extern double TDT;
extern double rearth[3];
extern double eapolar[3];
extern double obpolar[3];

extern struct orbit mars;
extern struct orbit earth;
extern struct orbit pluto;
extern struct orbit venus;
extern struct orbit saturn;
extern struct orbit uranus;
extern struct orbit jupiter;
extern struct orbit mercury;
extern struct orbit neptune;

/***** local global variables *****/

/* Space for star description read from a disc file.  */

static struct star fstar;

/* Space for orbit read from a disc file. Entering 99 for the planet
   number yields a prompt for a file name containg ASCII strings
   specifying the elements. */

static struct orbit forbit;

/* coordinates of object */

static double robject[3] = {0.0};

/* Tabulation parameters */

static double djd = 1.0;
static int ntab = 1;


/********/
/* Main */
/********/

void   main (void)
{
    int    i;

    char   fname[40];

    double oalt, oaz, ora, odec, sdiam;

    struct orbit *elobject;		/* pointer to orbital elements of
					   object */
    cnstinit ();
    if (kinit () == -1)
    {
	fprintf (stderr, "Unable to find a ./.site.ini or a ");
	fprintf (stderr, "~/.site.ini site initialization file.\n");
	exit (-1);
    }

    outfile = stdout;

    printf ("Output file name: ( stdout ) ? ");

    if (strlen (gets (fname)))
    {
	/* open the output file */

	outfile = fopen (fname, "w");

	if (outfile == NULL)
	{
	    fprintf (stderr, "Error in opening output file: %s\n",
		fname);

	    exit (-1);
	}
    }

    printf ("\n");

    prtflg = 1;

loop:

    getint ("Planet number 0-9, 88 to read star, 99 to read orbit, -1 to exit",
	&objnum);
    
    if (objnum == -1)
    {
    	if (strlen (fname))
    	{
	    /* close the output file */

            fclose (outfile);
    	}

	exit (0);
    }

    printf ("\nEnter starting date of tabulation\n");
    JD = getdate ();			/* date */
    JD += gethms ();			/* time of day */
    update (stdout);			/* find UT and ET */
    fprintf ("Julian day %.7f\n", JD);

    getdouble ("\nEnter interval between tabulations in days", &djd);
    getint ("Number of tabulations to display", &ntab);

    if (ntab <= 0)
    {
	ntab = 1;
    }

    switch (objnum)
    {
      case 0:

	elobject = 0;
	fprintf (outfile, "\n                   The Sun\n");
	break;

      case 1:

	elobject = &mercury;
	break;

      case 2:

	elobject = &venus;
	break;

      case 3:

	elobject = 0;
	fprintf (outfile, "\n                   The Moon\n");
	break;

      case 4:

	elobject = &mars;
	break;

      case 5:

	elobject = &jupiter;
	break;

      case 6:

	elobject = &saturn;
	break;

      case 7:

	elobject = &uranus;
	break;

      case 8:

	elobject = &neptune;
	break;

      case 9:

	elobject = &pluto;
	break;

      case 88:

morstar:

        elobject = (struct orbit *) & fstar;
	i = getstar (&fstar);

	if (i == 1)
	{
	    goto loop;
	}

	if (i == 0)
	{
	    break;
	}

	goto operr;

      case 99:

	elobject = &forbit;
	i = getorbit (elobject);

	if (i == 1)
	{
	    goto loop;
	}

	if (i == 0)
	{
	    break;
	}

      default:

operr:

	printf ("Operator error.\n");

	goto loop;
    }

    if (elobject == (struct orbit *) & fstar)
    {
	showcname (&elobject->obname[0]);
    }
    else if (elobject)
    {
	fprintf (outfile, "\n                  %s\n", &elobject->obname[0]);
    }

    for (i = 0; i < ntab; i++)
    {
	/* print Julian date */

	fprintf (outfile, "\nJD %.2f,  ", JD);
	update (outfile);

	/* Always calculate heliocentric position of the earth */

	kepler (TDT, &earth, rearth, eapolar);

	switch (objnum)
	{
	  case 0:

	    dosun (&oalt, &oaz, &ora, &odec, &sdiam);
	    break;

	  case 3:

	    domoon ();
	    break;

	  case 88:

	    rstar (&fstar, &oalt, &oaz, &ora, &odec);
	    goto morstar;

	  default:

	    /* calculate heliocentric position of the object */

	    kepler (TDT, elobject, robject, obpolar);

	    /* apply correction factors and print apparent place */

	    reduce (elobject, robject, rearth);
	    break;
	}

	fprintf (outfile, "\n");

	if (outfile != stdout)
	{
	    printf ("\n");
	}

	JD += djd;
    }
    goto loop;
}
