/******************************************************************************/
/*                                                                            */
/*  Title       : satfilter.c                                                 */
/*  Author      : Manfred Bester                                              */
/*  Date        : 24Oct94                                                     */
/*  Last change : 01Jan94                                                     */
/*                                                                            */
/*  Synopsis    : This program filters satellites for specified inclination   */
/*                and orbital period out of the TLE file 'tlex.dat' and       */
/*                creates a listfile of the type 'satlist_nnnn.dat'.          */
/*                                                                            */
/*  Input file  : tlex.dat                                                    */
/*  Output file : satlist_nnnn.dat (where 'nnnn' needs to be specified)       */
/*                                                                            */
/*                                                                            */
/*  Copyright (c) 1992-1995 Manfred Bester. All Rights Reserved.              */
/*                                                                            */
/*  Permission to use, copy, modify, and distribute this software and its     */
/*  documentation for educational, research and non-profit purposes, without  */
/*  fee, and without a written agreement is hereby granted, provided that the */
/*  above copyright notice and the following three paragraphs appear in all   */
/*  copies.                                                                   */
/*                                                                            */
/*  Permission to incorporate this software into commercial products may be   */
/*  obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,    */
/*  Berkeley, CA 94709, USA.                                                  */
/*                                                                            */
/*  IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, */
/*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF   */
/*  THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED  */
/*  OF THE POSSIBILITY OF SUCH DAMAGE.                                        */
/*                                                                            */
/*  THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT      */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A   */
/*  PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"      */
/*  BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, */
/*  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                  */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef STDLIB
#include <stdlib.h>
#endif

#include "sattrack.h"

#ifdef HPTERM
#include "hpterm.h"
#else
#include "vt100.h"
#endif

/******************************************************************************/
/*                                                                            */
/* external functions                                                         */
/*                                                                            */
/******************************************************************************/

extern double getElement();
extern void   mGets();

/******************************************************************************/
/*                                                                            */
/* definitions                                                                */
/*                                                                            */
/******************************************************************************/

#define INCLINLOW        0.0
#define INCLINHIGH      35.0
#define PERIODLOW        0.0
#define PERIODHIGH     225.0

/******************************************************************************/
/*                                                                            */
/* main program                                                               */
/*                                                                            */
/******************************************************************************/

int main(argc,argv)

int  argc;
char *argv[];

{
    double inclin, inclinLow, inclinHigh, period, periodLow, periodHigh;
    long   satNum;
    int    rangeFlag, usageFlag;
    char   data[100], tle[100], tleLine0[100], tleLine1[100], tleLine2[100];
    char   group[20], str[100];
    char   satlistFile[100], tleFile[100];
    char   *strpHome, *getenv();

    FILE   *InputFile, *OutputFile;

#ifdef HOMEDIR
    strpHome = getenv("HOME");
#else
    strpHome = SATDIR;
#endif

    if (argc >= 2)
    {
        usageFlag = TRUE;

        if (argc == 2)
        {
            if (!strncmp(argv[1],"leoe",4))
            {
                inclinLow  =   0.0;
                inclinHigh =  35.0;
                periodLow  =   0.0;
                periodHigh = 225.0;
                sprintf(group,"leoe");
                usageFlag  = FALSE;
            }

            if (!strncmp(argv[1],"leom",4))
            {
                inclinLow  =  35.0;
                inclinHigh =  65.0;
                periodLow  =   0.0;
                periodHigh = 225.0;
                sprintf(group,"leom");
                usageFlag  = FALSE;
            }

            if (!strncmp(argv[1],"leop",4))
            {
                inclinLow  =  65.0;
                inclinHigh = 180.0;
                periodLow  =  60.0;
                periodHigh = 225.0;
                sprintf(group,"leop");
                usageFlag  = FALSE;
            }
        }

        if (usageFlag)
        {
            printf("\nUsage: satfilter [leoe] [leom] [leop]\n\n");
            exit(-1);
        }
    }

    else
    {
        nl();
        printf("enter range of inclination    [deg] <%5.1f %5.1f> : ",
            INCLINLOW,INCLINHIGH);
        mGets(str);

        if (!strlen(str))
        {
            inclinLow  = INCLINLOW;
            inclinHigh = INCLINHIGH;
        }

        else
            sscanf(str,"%lf %lf",&inclinLow,&inclinHigh);

        printf("enter range of orbital period [min] <%5.1f %5.1f> : ",
            PERIODLOW,PERIODHIGH);
        mGets(str);

        if (!strlen(str))
        {
            periodLow  = PERIODLOW;
            periodHigh = PERIODHIGH;
        }

        else
            sscanf(str,"%lf %lf",&periodLow,&periodHigh);

        printf("\n'satfilter' will filter out satellites with inclinations ");
        printf("between \n");
        printf("%.1f and %.1f deg ",inclinLow,inclinHigh);
        printf("and orbital periods between ");
        printf("%.1f and %.1f minutes.\n\n",periodLow,periodHigh);

        printf("enter name of satellite group            <filter> : ");
        mGets(str); 

        if (!strlen(str))
            sprintf(group,"filter");
        else
            sprintf(group,"%s",str);

        printf("writing to output file 'satlist_%s.dat' ...\n",group);
    }

    sprintf(tle,"%s/%s",strpHome,TLE);
    sprintf(tleFile,"%s/tlex.dat",tle);

    sprintf(data,"%s/%s",strpHome,DATA);
    sprintf(satlistFile,"%s/satlist_%s.dat",data,group);

    if ((InputFile = fopen(tleFile,"r")) == 0)
    {
        alarm(); nl(); reverseBlink();
        printf("'%s' not found\n",tleFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(InputFile);

    if ((OutputFile = fopen(satlistFile,"w")) == 0)
    {
        alarm(); nl(); reverseBlink();
        printf("can't write '%s'\n",satlistFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(OutputFile);

    InputFile  = fopen(tleFile, "r");
    OutputFile = fopen(satlistFile,"a");

    while (fgets(tleLine0,80,InputFile))
    {
        if (!strncmp(tleLine0,TLEHEADER,10))           /* discard header line */
            fgets(tleLine0,80,InputFile);

        fgets(tleLine1,80,InputFile);
        fgets(tleLine2,80,InputFile);

        satNum    = (long) (getElement(tleLine1,3,7) + ONEPPM);
        inclin    = getElement(tleLine2,9,16);
        period    = MPD / getElement(tleLine2,53,63);

        rangeFlag = (inclin >= inclinLow  && 
                     inclin <  inclinHigh && 
                     period >= periodLow  &&
                     period <  periodHigh) ? TRUE : FALSE;

        if (rangeFlag)
            fprintf(OutputFile,"%ld\n",satNum);
    }

    fclose(InputFile);
    fclose(OutputFile);
}

/******************************************************************************/
/*                                                                            */
/* getElement: returns double of orbital element out of ASCII string          */
/*                                                                            */
/******************************************************************************/

double getElement(gstr,gstart,gstop)

int  gstart, gstop;
char gstr[80];

{
    int  k, glength;
    char gestr[80];

    glength = gstop - gstart + 1;

    for (k = 0; k <= glength; k++)
        gestr[k] = gstr[gstart+k-1];

    gestr[glength] = '\0';

    return((double) atof(gestr));
}

/******************************************************************************/
/*                                                                            */
/* mGets: Manfred's version of fgets (wipes out newline character)            */
/*                                                                            */
/******************************************************************************/

void mGets(string)

char *string;

{
    int i;
    fgets(string,80,stdin);
    i = (int) strlen(string);

    if (i > 0)
        string[i-1] = '\0';
    else
        string[0]   = '\0';

    return;
}

/******************************************************************************/
/*                                                                            */
/* End of program satfilter.c                                                 */
/*                                                                            */
/******************************************************************************/
