/*
 * SACDRT Version 3.1 SAC-Database-Reader for DeepSky Objects
 * Module LAYOUT.C is written by Sven van der Meer for the SACDRT.
 *
 * You need the NCURSES library to use SAC, and with MSDOS you need a VGA-
 * VIDEOCARD and a VGA-MONITOR (we use vga interrupts)
 *
 *
 *  Copyright (C) 1994 Lars Hagen & Sven van der Meer
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Problems or comments to hagenl@cs.tu-berlin.de
 *
 */
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "sacdrt.h"     /* includes stdio.h                              */
#include "database.h"   /* main data structures                          */
#include "layout.h"     /* includes ncurses.h                            */



/*
 *
 *  INIT_SCREEN()  set VGA Card (MSDOS) to 28 row mode and initialize curses
 *                 if possible. When the terminal has not 28 or more rows SAC
 *                 ends with an error. If colors are supported, the colors for
 *                 SAC will be initialized otherwise black and white mode will
 *                 be invoked. SAC is also terminated, if not 80 or more columns
 *                 in the terminal.
 */
void init_screen()
{
 #ifdef MSDOS
   Register.x.ax=0x0043;            /* mov ax,0043h - screen mode 03          */
   int86(0x10,&Register,&Register); /* int 10h      - without cls             */

   Register.x.ax=0x1111;            /* mov ax,1111h - 8*14 ROM character      */
   Register.x.bx=0x0000;            /* mov bx,0000h - set with font 00        */
   int86(0x10,&Register,&Register); /* int 10h      - both int only with VGA! */
                                    /* YOU NEED VGA TO RUN SAC ON DOS.        */
 #endif

 if (!(stdscr=initscr()))           /* major init for ncurses                 */
   sac_exit(0,"");
 if (LINES<28)                      /* sac needs 28 lines on screen           */
   sac_exit(1,"");
 if (COLS<80)                       /* sac needs 80 colums on screen          */
   sac_exit(4,"");

 start_color();
 if (has_colors()){                                /* colordefinitions        */
   init_pair(1,FG_COLOR,BG_COLOR);                 /* for sac_[sub]_win       */
   init_pair(2,SCREEN_FG_COLOR,SCREEN_BG_COLOR);   /* for screen_win          */
   init_pair(3,INFO_FG_COLOR,INFO_BG_COLOR);       /* for info_win            */
   init_pair(4,INPUT_FG_COLOR,INPUT_BG_COLOR);     /* for find_win            */
   init_pair(5,DATA_FG_COLOR,DATA_BG_COLOR);       /* colors for data-output  */
   init_pair(6,INPUT_BG_COLOR,INPUT_FG_COLOR);
   init_pair(7,CS_FG_COLOR,CS_BG_COLOR);
   status.colors=TRUE;
 }
 else                                             /* black&white definitions  */
   status.colors=FALSE;                           /* in the function          */
                                                  /* cs_set_color() below     */
 noecho();
 nodelay(stdscr,FALSE);
 curs_set(FALSE);                                 /* cursor invisible         */
 cbreak();
 keypad(stdscr,TRUE);
 nonl();
 stdscr->_use_idc=TRUE;

return;
}



/*
 *
 *  OPEN_LAYOUT()  is the main function for the screen initialization.
 *                 init_screen is called at first, then a couple of
 *                 windows are made and the windows are initialized with
 *                 colors (when supported). Then the layout of SAC is
 *                 printed on screen.
 *
 */
void open_layout()
{
 init_screen();                           /* screen init for sac / curses     */

 if (!(screen_win=newwin(28,80,SAC_START_ROW,SAC_START_COL)))   /* background                       */
   sac_exit(2,"screen_win in function open_layout");            /* (only for colormanipulation)     */
 if (!(sac_win=newwin(26,78,SAC_START_ROW+1,SAC_START_COL+1)))  /* sac main window                  */
   sac_exit(2,"sac_win in function open_layout");
 if (!(sac_sub_win=subwin(sac_win,10,66,SAC_START_ROW+16,SAC_START_COL+11)))  /* sub window for descr notes class */
   sac_exit(2,"sac_sub_win in function open_layout");

 if (status.colors){
   wattrset(screen_win,COLOR_PAIR(2));    /* init colors for background frame */
   wattrset(sac_win,COLOR_PAIR(1));       /* init colors for layout           */
   wattrset(sac_sub_win,COLOR_PAIR(1));   /* init colors for layout           */
 }
 else{
   wattrset(screen_win,A_REVERSE);        /* init for b&w screens             */
   wattrset(sac_win,A_BOLD);
   wattrset(sac_sub_win,A_BOLD);
 }
 sacdrt_werase(screen_win);
 sacdrt_werase(sac_sub_win);
 sacdrt_werase(sac_win);
 wborder(screen_win,' ',' ',' ',' ',' ',' ',' ',' ');
 wborder(sac_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(sac_win,1,20,"SAC Database Reader and Translater V ");
 mvwaddstr(sac_win,1,55,VERSION);
 mvwaddch(sac_win,2,0,SAC_VLEFT);
 line(sac_win,76);
 mvwaddch(sac_win,2,77,SAC_VRIGHT);

 mvwaddch(sac_win,8,0,SAC_VLEFT);
 line(sac_win,76);
 mvwaddch(sac_win,8,77,SAC_VRIGHT);

 mvwaddch(sac_win,10,0,SAC_VLEFT);
 line(sac_win,76);
 mvwaddch(sac_win,10,77,SAC_VRIGHT);

 mvwaddch(sac_win,20,0,SAC_VLEFT);
 line(sac_win,76);
 mvwaddch(sac_win,20,77,SAC_VRIGHT);

 mvwaddstr(sac_win,4,2,"Name  :");
 mvwaddstr(sac_win,5,2,"Info  :");
 mvwaddstr(sac_win,6,2,"Type  :");
 mvwaddstr(sac_win,7,2,"Const :");
 mvwaddstr(sac_win,9,2,"R.A.  :          DEC:           Bright:");
 if (config.base_version==v52){
   mvwaddstr(sac_win,11,2,"Tirion:                         Bright.Star:");
   mvwaddstr(sac_win,12,2,"U2000 :                         No.of Stars:");
   mvwaddstr(sac_win,13,2,"Size  :");
   mvwaddstr(sac_win,14,2,"Source:");
 }
 else{
   mvwaddstr(sac_win,11,2,"Tirion:                         Surf.Bright:");
   mvwaddstr(sac_win,12,2,"U2000 :                         Bright.Star:");
   mvwaddstr(sac_win,13,2,"Size  :                         No.of Stars:");
   mvwaddch(sac_win,14,0,SAC_VLEFT);
   line(sac_win,76);
   mvwaddch(sac_win,14,77,SAC_VRIGHT);   
 }
 mvwaddstr(sac_win,15,2,"Class :");
 mvwaddstr(sac_win,17,2,"Notes :");
 mvwaddstr(sac_win,21,2,"Descr :");        /* now the layout is done     */

 if (status.colors){
   wattrset(sac_win,COLOR_PAIR(5));        /* init colors for dataoutput */
   wattrset(sac_sub_win,COLOR_PAIR(5));    /* init colors for dataoutput */
 }
 else{
   wattrset(sac_win,A_NORMAL);             /* the same for b&w screens   */
   wattrset(sac_sub_win,A_NORMAL);
 }

 mvwaddstr(screen_win,27,1,"Help  Info  Search  Complex  Mark  Treat  Quit  ");
 waddch(screen_win,ACS_DARROW);
 waddstr(screen_win,"[d]Down  ");
 waddch(screen_win,ACS_UARROW);
 waddstr(screen_win,"[u]Up");
 waddstr(screen_win,"   Filter: OFF");              /* statusline done      */

 wnoutrefresh(screen_win);            /* refresh win in mem (not on screen) */
 wnoutrefresh(sac_win);               /* refresh win in mem (not on screen) */
 wnoutrefresh(sac_sub_win);           /* refresh win in mem (not on screen) */
 doupdate();                          /* ready, everything on screen now    */
return;
}



/*
 *
 *  CLOSE_LAYOUT()  is called when the program terminats normaly. all
 *                  windows are deleted and the sources from these windows
 *                  are given free then. The original screen is recreated
 *                  (not implemented in MSDOS) and curses and sac ends
 *
 */
void close_layout()
{
 keypad(stdscr,FALSE);
 delwin(sac_sub_win);
 delwin(sac_win);
 delwin(screen_win);          /* all win deleted                 */
 curs_set(TRUE);
 clear();
 doupdate();
 endwin();                    /* end curses - freeup all sources */
return;
}



/*
 *
 *  OUTPUT(...)  prints the data generated from input to the screen
 *
 */
void output(char *name, char *name_info, char *other, char *type, char *con, 
            char *ra, char *dec, char *mag, char *u2000, char *tirion,
            char *descr, char *size, char *clas, char *nsts, char *brstr, 
            char *notes, char *source, char *subr)
{

 mvwaddstr(sac_win,4,10,name);
 if (*other != '\0'){
   waddstr(sac_win,"   ");
   waddch(sac_win,'(');
   waddstr(sac_win,other);
   waddch(sac_win,')');
 }
 clear_line(0,0,77,sac_win);

 mvwaddstr(sac_win,5,10,name_info);  clear_line(0,0,77,sac_win);
 mvwaddstr(sac_win,6,10,type);       clear_line(0,0,77,sac_win);
 mvwaddstr(sac_win,7,10,con);        clear_line(0,0,77,sac_win);
 mvwaddstr(sac_win,9,10,ra);         clear_line(0,0,16,sac_win);
 mvwaddstr(sac_win,9,25,dec);        clear_line(0,0,32,sac_win);
 mvwaddch(sac_win,9,28,ACS_DEGREE);
 mvwaddstr(sac_win,9,43,mag);        clear_line(0,0,77,sac_win);
 mvwaddstr(sac_win,11,10,tirion);    clear_line(0,0,32,sac_win);
 mvwaddstr(sac_win,12,10,u2000);     clear_line(0,0,32,sac_win);

 if(config.base_version==v52){
   mvwaddstr(sac_win,11,48,brstr);     clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,12,48,nsts);      clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,14,10,source);    clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,13,10,size);      clear_line(0,0,77,sac_win);
 }
 else{
   mvwaddstr(sac_win,11,48,subr);      clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,12,48,brstr);     clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,13,48,nsts);      clear_line(0,0,77,sac_win);
   mvwaddstr(sac_win,13,10,size);      clear_line(0,0,32,sac_win);
 }

 clear_line(15,10,78,sac_win);
 clear_line(16,10,78,sac_win);
 wmove(sac_sub_win,0,0);
 break_line(clas);
 status.clas=FALSE;

 clear_line(17,10,78,sac_win);
 clear_line(18,10,78,sac_win);
 clear_line(19,10,78,sac_win);
 wmove(sac_sub_win,2,0);
 break_line(notes);

 clear_line(21,10,78,sac_win);
 clear_line(22,10,78,sac_win);
 clear_line(23,10,78,sac_win);
 clear_line(24,10,78,sac_win);
 wmove(sac_sub_win,6,0);
 break_line(descr);

 wnoutrefresh(sac_sub_win);
 wnoutrefresh(sac_win);                   /* win refreshed in memory */
 doupdate();                              /* update physical screen  */

return;
}



/*
 *
 *  LINE(...)  prints a line with the character SAC_HLINE (layout.h)
 *             on the screen. the original cursorposition is used.
 *
 */
void line(WINDOW *win,int length)
{
static int i;
  for (i=0;i<length;i++)
    waddch(win,SAC_HLINE);          /* horizontal lines in the layout */
return;
}



/*
 *
 *  CLEAR_LINE  sends blank characters to the screen. begin at the position
 *              marked with x_start and y_start. if x_start and y_start are
 *              are 0, the original cursorposition is used. the function
 *              send blanks up to length
 *
 */
void clear_line(int x_start, int y_start, int length,WINDOW *win)
{
static int i;

 if (x_start+y_start==0){
   getyx(win,y_start,x_start);
   for(i=x_start+1; i<=length;i++)
     waddch(win,' ');
 }
 else{
   for (i=y_start; i<length-1; i++){
      mvwaddch(win,x_start,y_start,' ');
      y_start++;
   }
 }
return;
}



/*
 *
 *  INFO()  info opens a info-window, prints a message in it and waits for
 *          a key. when a key is pressed, the window will be closed.
 *
 */
void info()
{
WINDOW *info_win;

 if(!(info_win = newwin(10,37,SAC_START_ROW+9,SAC_START_COL+21)))
   sac_exit(2,"info_win in function info");
 touchwin(sac_win);

 cs_set_color(normal,info_win);
 
 sacdrt_werase(info_win);
 wborder(info_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(info_win,1,6,"SAC-Database-Reader V ");
 mvwaddstr(info_win,1,27,VERSION);
 mvwaddstr(info_win,3,6,"Idea & Parser: Lars Hagen");
 mvwaddstr(info_win,4,6,"Layout & More: Sven v.d.M.");
 mvwaddstr(info_win,6,3,"If you have problems, find bugs");
 mvwaddstr(info_win,7,3,"or have suggestions please send");
 mvwaddstr(info_win,8,3,"mail to: hagenl@cs.tu-berlin.de");
 wrefresh(info_win);

 getch();

 delwin(info_win);
 wrefresh(sac_win);
return;
}



/*
 *
 *  MESSAGE_WINDOW(char *,int) is called to open the window and print the
 *                             string STRING in it (used for searching and
 *                             not found etc).
 *                             int is one of the enums normal or inverse.
 *  CLOSE_MESSAGE_WINDOW()     is called to close the win if this was opened.
 *
 */
void message_window(char *string,int color_test)
{
 if(!(message_win=newwin(3,20,SAC_START_ROW+9,SAC_START_COL+30)))
    sac_exit(2,"messagge_win in function message_window");
 touchwin(sac_win);

 switch (color_test){
   case normal:  cs_set_color(normal,message_win);
		 break;
   case inverse: cs_set_color(edit,message_win);
                 break;
 }
 
 sacdrt_werase(message_win);
 wborder(message_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);
 mvwaddstr(message_win,1,1,string);

 wrefresh(message_win);
return;
}

void close_message_window(int color_test)
{
 if (message_win != FALSE){
   delwin(message_win);
   if (color_test==normal)
     wrefresh(sac_win);
   if (color_test==inverse)
     wrefresh(find_win);
   message_win=NULL;
 }
return;
}



/*
 *
 *  char *FINDWINDOW()   opens a window for the functions, which are searching
 *                       for an object and waits for the input.
 *
 */
char *findwindow()
{
char *tofindstring;
WINDOW *tmp_win;

 tofindstring=(char *)malloc(30);

 if(!(find_win = newwin(3,37,SAC_START_ROW+9,SAC_START_COL+21)))
   sac_exit(2,"find_win in function findwindow");
 if(!(tmp_win = newwin(1,21,SAC_START_ROW+10,SAC_START_COL+35)))
   sac_exit(2,"tmp_win in function findwindow");
 touchwin(sac_win);

 cs_set_color(normal,find_win);
 cs_set_color(edit,tmp_win);
 
 sacdrt_werase(find_win);
 wborder(find_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);
 mvwaddstr(find_win,1,2,"Objectname: ");

 mvwaddstr(tmp_win,0,0,"123456789012345678901");
 sacdrt_werase(tmp_win);
 wrefresh(find_win);
 wrefresh(tmp_win);

 strcpy(tofindstring,cs_input(0,0,21,22,22,tmp_win,cs_inp_search));

 delwin(tmp_win);
 delwin(find_win);
 wrefresh(sac_win);
 find_win=FALSE;

return (tofindstring);
}



/*
 *
 *  ENDING_MESSAGE()  prints the last message from the program to the screen
 *                    when sac terminats normaly.
 *
 */
void ending_message()
{
 printf("\n\n");
 printf("\nCopyright (C) %s by Lars Hagen & Sven van der Meer",YEAR);
 printf("\n                      special thanks to Sven Mathing");
 printf("\nBerlin, Germany");
 printf("\nFeedbacks please via e-mail to hagenl@cs.tu-berlin.de");
 printf("\n\n");
return;
}



/*
 *
 *  SAC_EXIT  is called, when a function has an error detected (i.e.
 *            when a window couldn't be opened)
 *            SAC now terminates abnormaly with the errormessage storred
 *            in sac_err[] and a message from the calling function.
 *
 */
void sac_exit(int sac_error,char *string)
{
char *sac_err[] = {
  "sac: error creating initscr",
  "sac: unable to start sac, term has less than 28 rows",
  "sac: unable to create ",
  "sac: unknown error abnormal program termination",
  "sac: unable to start sac, term has less then 80 colums",
  "sac: unble to open file for output ",
  "sac: error calculating time in function print_page"
  "sac: not enough memory for malloc"
  "sac: "
};

 if(head_core.next!=&tail_core)
   delete_core_list();
 if(head_mark.next_mark!=&tail_mark)
   delete_all_marked();

 endwin();
 fprintf(stderr,"%s %s\n",sac_err[sac_error],string);
 exit(1);
return;
}



/*
 *
 *  HELP()  help opens a help-window, print a message in it and wait for
 *          a key. when a key is pressed, the window will be closed.
 *
 */
void help()
{
WINDOW *info_win;

 if(!(info_win = newwin(19,60,SAC_START_ROW+5,SAC_START_COL+10)))
   sac_exit(2,"info_win in function help");
 touchwin(sac_win);

 cs_set_color(normal,info_win);

 sacdrt_werase(info_win);
 wborder(info_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(info_win,1,15,"Help for SAC-Database-Reader");
 mvwaddstr(info_win,3,2," (H)elp    -  shows this information");
 mvwaddstr(info_win,4,2," (I)nfo    -  some general informations");
 mvwaddstr(info_win,5,2," (S)earch  -  search an object");
 mvwaddstr(info_win,6,2," (C)omplex -  complex search");
 mvwaddstr(info_win,7,2," (M)ark    -  mark the current object for further treat");
 mvwaddstr(info_win,8,2," (T)reat   -  sort/print/delete marked objects");
 mvwaddstr(info_win,9,2," (W)rite   -  write startup values to $HOME/.sacdrtrc");
 mvwaddstr(info_win,10,2," (F)ilter  -  turns filter for objects, found with");
 mvwaddstr(info_win,11,2,"              complex search, on or off");
 mvwaddstr(info_win,12,2," (O)rig.   -  turns parse mode on or off");
 mvwaddstr(info_win,13,2," (Q)uit    -  quit the program");
 mvwaddstr(info_win,15,2," Moveing   -     ");
 waddch(info_win,ACS_DARROW);
 waddstr(info_win,"[d]   Next      ");
 waddch(info_win,ACS_UARROW);
 waddstr(info_win,"[u]  Previous");
 mvwaddstr(info_win,16,16,"PgUp[+]   +100   PgDn[-]  -100     ");
 mvwaddstr(info_win,17,16,"Home[b]   Begin  End [e]  End");

 wrefresh(info_win);

 getch();

 delwin(info_win);
 wrefresh(sac_win);
return;
}



/*
 * 
 *  SHOW_MARKED  will show the marked objects when the user pressed p.
 *               the objects are printed in a scroll region and some action
 *               to manipulate them are possible (delete, print, sort etc).
 *
 */
int show_marked()
{
int ret;
 scale=11;
 current=1;
 ende=status.max_mark-1;
 mcount=1;

 if(!(find_win = newwin(13,40,SAC_START_ROW+8,SAC_START_COL+20)))
   sac_exit(2,"find_win in function show_marked");
 touchwin(sac_win);

 cs_set_color(normal,find_win); 
 
 sacdrt_werase(find_win);
 wborder(find_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);
 mvwaddstr(find_win,0,10,"[ Marked Objects ]");

 mvwaddstr(find_win,2,24,"One Delete");
 mvwaddstr(find_win,3,24,"All Delete");
 mvwaddstr(find_win,4,24,"Sort Window");
 mvwaddstr(find_win,5,24,"Print Window");
 mvwaddstr(find_win,6,24,"Quit");
 mvwaddstr(find_win,8,22,"[Enter]/[Space]");
 mvwaddstr(find_win,9,22,"Show the Object");

 display_marked(find_win);

 okloop=YES;
 while (okloop){
   switch(getch()){
     case 'U':
     case 'u':
     case KEY_UP   :  scroll_marked(mcount-1);
                      break;
     case 'D':
     case 'd':
     case KEY_DOWN :  scroll_marked(mcount+1);
                      break;
     case '-':
     case KEY_PPAGE:  scroll_marked(mcount-10);
		      break;
     case '+':
     case KEY_NPAGE:  scroll_marked(mcount+10);
		      break;
     case KEY_A1:
     case 'b':
     case 'B':
     case KEY_HOME:   scroll_marked(0);
                      break;
     case KEY_C1:
     case 'e':
     case 'E':
     case KEY_END:    scroll_marked(ende+1);
                      break;
     case 'S':
     case 's':        sort_window();
                      break;
     case 'Q':
     case 'q':
     case ' ':
     case LF :
     case CR :	      message_window("   searching...",inverse);
 		      if ((scale-10)<1)
                        handle_objects(search_mark,current-1,search_found,"");
		      else
		        handle_objects(search_mark,scale-11+current,search_found,"");
		      okloop=NO;
                      break;
     case 'F' :
     case 'f' :
     case 'P' :
     case 'p' :       print_window();
	              break;
     case 'O'       :
     case 'o'       : if ((scale-10)<1)
		        delete_one_marked(current-1); 
		      else
		        delete_one_marked(scale-11+current);
		      ende=status.max_mark-1;
                      if (!ende)
			okloop=NO;
		      else
			scroll_marked(mcount);
                      break;
     case 'A'       :
     case 'a'       : delete_all_marked();
                      okloop=NO;
                      break;
     default        : break;
   }
   if (okloop)
     display_marked(find_win);
 }
 delwin(find_win);
 wrefresh(sac_win);
 find_win=FALSE;
 okloop=YES;

 if (status.max_mark<2)
   ret=FALSE;
 else
  ret=TRUE;
return ret;
}



/*
 *
 *  DISPLAY_MARKED(WINDOW *win)  display the marked objects in the scrollregion
 *                               from SHOW_MARKED(). the values are calculated
 *                               from SCROLL_MARKED().
 *                               these function is used from SHOW_MARKED() and
 *                               cs_search().
 *
 */
int display_marked(WINDOW *win)
{
struct mark_list *sac_mark=head_mark.next_mark;
int i,count;

 if ((scale-10)<1)
   count=1;
 else
   count=scale-10;

 while (count!=sac_mark->mark_count)
   sac_mark=sac_mark->next_mark;

 for (i=1;i<12;i++){
   if (current==i){
     cs_set_color(scroll_mark,win);
     mvwaddstr(win,current,3," ");
     mvwaddstr(win,current,4,(char *)sac_mark->mark_name);
     clear_line(0,0,21,win);
     cs_set_color(scroll_normal,win);
   }
   else{
     if (sac_mark->mark_count>0){
       mvwaddstr(win,i,3," ");
       mvwaddstr(win,i,4,(char *)sac_mark->mark_name);
       clear_line(0,0,21,win);
     }
     else{
       mvwaddstr(win,i,3,"                  ");
     }
   }
   if (sac_mark!=&tail_mark)
   sac_mark=sac_mark->next_mark;
 }
 cs_set_color(normal,win);
 mvwprintw(find_win,12,3,"%i Objects marked",status.max_mark-1);
 cs_set_color(scroll_normal,win);
 wrefresh(win);

return 0;
}



/*
 *
 * SCROLL_MARKED(int toline)  these function will change the values used
 *                            for the scrolling in SHOW_MARKED() and cs_search.
 *                            
 */
int scroll_marked(int toline)
{
 if ((toline<=1)||(toline>=ende)){
   if ((toline<1)||(toline==1)){
     scale=11;
     mcount=1;
     current=1;
   }
   else{
     if (ende<11){
       scale=11;
       current=ende;
     }
     else{
       scale=ende;
       current=11;
     }
     mcount=ende;
   }
 }
 else{
   if ((toline>=(scale-10))&&(toline<scale)){
     if (mcount>toline){
       current=current-(mcount-toline);
     }
     else{
       current=current+(toline-mcount);
     }
   }
   else{
     if (toline<mcount){
       scale=toline+10;
       current=1;
     }
     else{
       scale=toline;
       current=11;
     }
   }
   mcount=toline;
 }
return 0;
}



/*
 *
 * STR2LOWER(char *1,char *2)  copy *2 to *1 and converts all characters
 *                             to lowercase. these function is used for
 *                             searching non case sensitive. 
 *                             this function eleminates also all spaces
 *                             so that you don't have to type space when
 *                             searching for an object.
 *
 */
int str2lower(char *string1,char *string2)
{
 while ((*string1=tolower(*string2))!='\0'){
   if (!isspace(*string1))
     string1++;
   string2++;
 }
return TRUE;
}



/*
 * void BREAK_LINE (char*,char*,int) this function will break the lines for
 *				     the pointer descr, clas and notes. 
 *			             called from output. does not work 100%
 *                                   correct.
 *
 */
void break_line(char ptr[])
{
int i=0,j,lines=0;

 while(ptr[i]){
   j=i;
   while(!isspace(ptr[j])&&ptr[j])
     j++;
   j=j-lines*65;
   if(j>66){
     wmove(sac_sub_win,sac_sub_win->_cury+1,0);
     lines++;
     while(isspace(ptr[i]))
       i++;
     if(status.clas&&lines==1)
       wmove(sac_sub_win,sac_sub_win->_cury,status.clas);
   }
   j=sac_sub_win->_curx;
   j=j-lines*65;
   waddch(sac_sub_win,ptr[i++]);
   if(sac_sub_win->_curx<j){
     sac_sub_win->_curx=++j;
     sac_sub_win->_cury--;
   }
 }
 
return;
}



/*
 * int COMPLEX_SEARCH ()  functin for the complex search. a window will be
 *			  opened and the user can input some values which
 *			  the program will search for.
 *			  the following code is not tricky but dirty i
 *                        think. when i have time it will be changed.
 *                        (when you think this is more basic then c you are
 *                        right - this was former an editor written in power
 *                        basic)
 *
 */
int complex_search()
{
WINDOW *cs_win;
int ret=FALSE,next_tab=cs_ra_min,old_tab=nothing;
int mag_row=4,decl_row=3,ra_row=2,all_col=3,oth_row=6;
char *tmp_ptr;
int i;

 tmp_ptr=(char *)malloc(9);

 scale=11;
 current=1;
 mcount=1;

 if(!(cs_win = newwin(23,60,SAC_START_ROW+3,SAC_START_COL+10)))
   sac_exit(2,"cs_win in function complex_search");
 touchwin(sac_win);

 cs_set_color(normal,cs_win);

 sacdrt_werase(cs_win);
 wborder(cs_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(cs_win,0,21,"[ Complex Search ]");

 mvwaddstr(cs_win,oth_row+13,all_col+11,"Press [s] for search and [h] for help");
 if(cs_values.cs_default_val)
   mvwaddstr(cs_win,oth_row+15,all_col+17,"   Using default values   ");
 else
   mvwaddstr(cs_win,oth_row+15,all_col+17,"Using user specific values");

 cs_set_color(edit,cs_win);

 mvwaddstr(cs_win,ra_row,23,cs_values.min_ra);
 mvwaddch(cs_win,ra_row,25,'h');
 mvwaddstr(cs_win,ra_row,41,cs_values.max_ra);
 mvwaddch(cs_win,ra_row,43,'h');
 mvwaddstr(cs_win,decl_row,23,cs_values.min_decl);
 mvwaddch(cs_win,decl_row,26,ACS_DEGREE);
 mvwaddch(cs_win,decl_row,29,'\'');
 mvwaddstr(cs_win,decl_row,41,cs_values.max_decl);
 mvwaddch(cs_win,decl_row,44,ACS_DEGREE);
 mvwaddch(cs_win,decl_row,47,'\'');
 mvwaddstr(cs_win,mag_row,25,cs_values.min_mag);
 mvwaddstr(cs_win,mag_row,43,cs_values.max_mag);

 cs_set_color(normal,cs_win);

 ende=max_type-1;
 cs_show(cs_win,cs_type);
 ende=max_con-1;
 cs_show(cs_win,cs_con);
 ende=max_name-1;
 cs_show(cs_win,cs_name);
 cs_show(cs_win,next_tab);

 wrefresh(cs_win);

 while(okloop){
   switch (next_tab){
     case cs_ra_min:   switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,ra_row,23,"  h  . ");
					strcpy(cs_values.min_ra,cs_input(all_col+20,ra_row,7,2,5,cs_win,cs_inp_ra));
					if (!strlen(cs_values.min_ra)){
					  strcpy(cs_values.min_ra,CS_MIN_RA_VAL);
					  mvwaddstr(cs_win,ra_row,23,CS_MIN_RA);
					}
                                        else{
                                          mvwaddstr(cs_win,ra_row,23,cs_values.min_ra);
                                          mvwaddch(cs_win,ra_row,25,'h');
					  cs_values.cs_default_val=FALSE;
                                        }
					cs_set_color(normal,cs_win);
                                        break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_ra_max;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_con;
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_ra_max:   switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,ra_row,41,"  h  . ");
					strcpy(cs_values.max_ra,cs_input(all_col+38,ra_row,7,2,5,cs_win,cs_inp_ra));
					if(!strlen(cs_values.max_ra)){
					  strcpy(cs_values.max_ra,CS_MAX_RA_VAL);
					  mvwaddstr(cs_win,ra_row,41,CS_MAX_RA);
					}
					else{
					  mvwaddstr(cs_win,ra_row,41,cs_values.max_ra);
					  mvwaddch(cs_win,ra_row,43,'h');
					  cs_values.cs_default_val=FALSE;
					}
                                        cs_set_color(normal,cs_win);
                                        break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_decl_min;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_ra_min;
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_decl_min: switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,decl_row,23,"   ");
					mvwaddstr(cs_win,decl_row,27,"  ");
					strcpy(cs_values.min_decl,cs_input(all_col+20,decl_row,6,3,7,cs_win,cs_inp_decl));
					if(!strlen(cs_values.min_decl)){
					  strcpy(cs_values.min_decl,CS_MIN_DECL_VAL);
					  mvwaddstr(cs_win,decl_row,23,CS_MIN_DECL);
					  mvwaddstr(cs_win,decl_row,27,CS_NOR_DECL);
					}
					else{
					  mvwaddstr(cs_win,decl_row,23,cs_values.min_decl);
					  cs_values.cs_default_val=FALSE;
					}
					mvwaddch(cs_win,decl_row,26,ACS_DEGREE);
                                        cs_set_color(normal,cs_win);
                                        break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_decl_max;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_ra_max;
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_decl_max: switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,decl_row,41,"   ");
					mvwaddstr(cs_win,decl_row,45,"  ");
					strcpy(cs_values.max_decl,cs_input(all_col+38,decl_row,6,3,7,cs_win,cs_inp_decl));
					if(!strlen(cs_values.max_decl)){
					  strcpy(cs_values.max_decl,CS_MAX_DECL_VAL);
					  mvwaddstr(cs_win,decl_row,41,CS_MAX_DECL);
					  mvwaddstr(cs_win,decl_row,45,CS_NOR_DECL);
					}
					else{
					  mvwaddstr(cs_win,decl_row,41,cs_values.max_decl);
					  cs_values.cs_default_val=FALSE;
					}
					mvwaddch(cs_win,decl_row,44,ACS_DEGREE);
                                        cs_set_color(normal,cs_win);
                                        curs_set(FALSE);
                                        break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_mag_min;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_decl_min;
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_mag_min:  switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,mag_row,25,"   . ");
					strcpy(cs_values.min_mag,cs_input(all_col+22,mag_row,5,3,6,cs_win,cs_inp_mag));
					if(!strlen(cs_values.min_mag)){
					  strcpy(cs_values.min_mag,CS_MIN_MAG);
					  mvwaddstr(cs_win,mag_row,25,CS_MIN_MAG);
					}
					else{
					  mvwaddstr(cs_win,mag_row,25,cs_values.min_mag);
					  cs_values.cs_default_val=FALSE;
					}
                                        cs_set_color(normal,cs_win);
                                        break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_mag_max;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_decl_max;
                                        cs_set_color(normal,cs_win);

                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_mag_max:  switch (getch()){
			 case LF:
                         case CR:       cs_set_color(edit,cs_win);
					mvwaddstr(cs_win,mag_row,43,"   . ");
					strcpy(cs_values.max_mag,cs_input(all_col+40,mag_row,5,3,6,cs_win,cs_inp_mag));
					if(!strlen(cs_values.max_mag)){
					  strcpy(cs_values.max_mag,CS_MAX_MAG);
					  mvwaddstr(cs_win,mag_row,43,CS_MAX_MAG);
					}
					else{
					  mvwaddstr(cs_win,mag_row,43,cs_values.max_mag);
					  cs_values.cs_default_val=FALSE;
					}
					cs_set_color(normal,cs_win);
					break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_name;
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_mag_min;
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_name:     ende=max_name-1;
                       switch (getch()){
                         case KEY_UP:
			 case 'U':
			 case 'u':	scroll_marked(mcount-1);
                                        break;
                         case KEY_DOWN: 
			 case 'D':
			 case 'd':      scroll_marked(mcount+1);
                                        break;
                         case 'M':
                         case 'm':
                         case ' ':
			 case LF :
                         case CR :      if (!object_data[mcount].marked){
                                          object_data[mcount].marked=TRUE;
					  cs_values.mark_name++;
					}
                                        else{
                                          object_data[mcount].marked=FALSE;
					  cs_values.mark_name--;
					}
					cs_values.cs_default_val=FALSE;
					break;
			 case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_type;
                                        mcount=1;
                                        scroll_marked(mcount);
					cs_show(cs_win,cs_name);
                                        break;
			 case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_mag_max;
                                        mcount=1;
                                        scroll_marked(mcount);
					cs_show(cs_win,cs_name);
                                        break;
			 case '-':
                         case KEY_PPAGE:scroll_marked(mcount-10);
                                        break;
			 case '+':
                         case KEY_NPAGE:scroll_marked(mcount+10);
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'I':
                         case 'i':      old_tab=next_tab;
                                        next_tab=cs_invert;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
			 case 'C':
			 case 'c':      if(cs_values.name_to_use==use_short)
					  cs_values.name_to_use=use_middle;
					else
					  cs_values.name_to_use=use_short;
					break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_type:     ende=max_type-1;
		       switch (getch()){
                         case KEY_UP:   
			 case 'U':
			 case 'u':	scroll_marked(mcount-1);
                                        break;
                         case KEY_DOWN: 
			 case 'D':
			 case 'd':	scroll_marked(mcount+1);
                                        break;
                         case 'M':
                         case 'm':      
                         case ' ':
			 case LF :
                         case CR :      if (!type_data[mcount].marked){
					  type_data[mcount].marked=TRUE;
					  cs_values.mark_type++;
					}
					else{
					  type_data[mcount].marked=FALSE;
					  cs_values.mark_type--;
					}
 					cs_values.cs_default_val=FALSE;
					break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_con;
                                        mcount=1;
                                        scroll_marked(mcount);
					cs_show(cs_win,cs_type);
                                        break;
                         case KEY_LEFT:
			 case 'P':
                         case 'p':      next_tab=cs_name;
                                        mcount=1;
                                        scroll_marked(mcount);
					cs_show(cs_win,cs_type);
                                        break;
			 case '-':
                         case KEY_PPAGE:scroll_marked(mcount-10);
                                        break;
			 case '+':
                         case KEY_NPAGE:scroll_marked(mcount+10);
                                        break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'I':
                         case 'i':      old_tab=next_tab;
                                        next_tab=cs_invert;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
  		       break;
     case cs_con:      ende=max_con-1;
		       switch (getch()){
                         case KEY_UP:   
			 case 'U':
			 case 'u':	scroll_marked(mcount-1);
                                        break;
                         case KEY_DOWN: 
			 case 'D':
			 case 'd':	scroll_marked(mcount+1);
                                        break;
                         case 'M':
                         case 'm':
                         case ' ':
			 case LF :
                         case CR :      if (!const_data[mcount].marked){
                                          const_data[mcount].marked=TRUE;
					  cs_values.mark_con++;
					}
                                        else{
                                          const_data[mcount].marked=FALSE;
					  cs_values.mark_con--;
					}
					cs_values.cs_default_val=FALSE;
					break;
                         case KEY_RIGHT:
                         case 'N':
                         case 'n':      next_tab=cs_ra_min;
                                        mcount=1;
					scroll_marked(mcount);
					cs_show(cs_win,cs_con);
                                        break;
                         case KEY_LEFT:
                         case 'P':
                         case 'p':      next_tab=cs_type;
                                        mcount=1;
                                        scroll_marked(mcount);
					cs_show(cs_win,cs_con);
                                        break;
			 case '-':
			 case KEY_PPAGE:scroll_marked(mcount-10);
                       			break;
			 case '+':
       			 case KEY_NPAGE:scroll_marked(mcount+10);
					break;
                         case 'S':
                         case 's':      old_tab=next_tab;
					next_tab=cs_search;
                                        break;
                         case 'I':
                         case 'i':      old_tab=next_tab;
                                        next_tab=cs_invert;
                                        break;
                         case 'O':
                         case 'o':      old_tab=next_tab;
                                        next_tab=cs_default;
                                        break;
                         case 'H':
                         case 'h':      cs_help(cs_win);
                                        break;
                         case 'Q':
                         case 'q':      okloop=NO;
                                        break;
                       }
                       break;
     case cs_search:   touchwin(cs_win);
		       if (atof(cs_values.min_mag)<atof(CS_MIN_MAG)){
			 message_window("   Mag < -26.9 ???",inverse);
                         getch();
                         close_message_window(nothing);
		       }
		       else if (atof(cs_values.max_mag)>atof(CS_MAX_MAG)){
			 message_window("  Mag > 99.9 ??? ",inverse);
                         getch();
                         close_message_window(nothing);
		       }
		       else if (atof(cs_values.min_decl)<atof(CS_MIN_DECL_VAL)){
			 message_window("  Decl. < -90 ??? ",inverse);
                         getch();
                         close_message_window(nothing);
		       }
		       else if (atof(cs_values.max_decl)>atof(CS_MAX_DECL_VAL)){
			 message_window("  Decl. > +90 ??? ",inverse);
                         getch();
                         close_message_window(nothing);
		       }
		       else{
  		         message_window(" searching: ",inverse);
	     	         if ((ret=cs_do_search())){
                           message_window("   marked: ",inverse);
			   mvwprintw(message_win,1,12,"%i",status.max_mark-1);
			   wrefresh(message_win);
                           getch();
                           close_message_window(nothing);
                           okloop=NO;
			 }
		         else{
			   message_window(" no Objects found",inverse);
			   getch();
			   close_message_window(nothing);
			   next_tab=cs_mag_min;
		         }
		         close_message_window(nothing);
		       }
		       next_tab=old_tab;
		       break;
     case cs_invert:   switch(old_tab){
                         case cs_name: for (i=1;i<max_name;i++){
                                         if(!object_data[i].marked){
                                           object_data[i].marked=TRUE;
                                           cs_values.mark_name++;
                                         }
                                         else{
                                           object_data[i].marked=FALSE;
                   			   cs_values.mark_name--;
                                         }
                                       }
                                       break;
                         case cs_type: for (i=1;i<max_type;i++){ 
                                         if(!type_data[i].marked){
                                           type_data[i].marked=TRUE;
			                   cs_values.mark_type++;
                                         }
                                         else{
                                           type_data[i].marked=FALSE;
                                           cs_values.mark_type--;
                                         }
                                       }
                                       break;
                         case cs_con:  for (i=1;i<max_con;i++){
                                         if(!const_data[i].marked){
                                           const_data[i].marked=TRUE;
                                           cs_values.mark_con++;
                                         }
                                         else{
                                           const_data[i].marked=FALSE;
                                           cs_values.mark_con--;
                                         }
                                       }
                                       break;
                       }
                       next_tab=old_tab;
                       break;
     case cs_default:  strcpy(cs_values.min_mag,CS_MIN_MAG);
                       strcpy(cs_values.max_mag,CS_MAX_MAG);
                       strcpy(cs_values.min_ra,CS_MIN_RA_VAL);
                       strcpy(cs_values.max_ra,CS_MAX_RA_VAL);
                       strcpy(cs_values.min_decl,CS_MIN_DECL_VAL);
                       strcpy(cs_values.max_decl,CS_MAX_DECL_VAL);

                       cs_set_color(edit,cs_win);
		       mvwaddstr(cs_win,ra_row,23,CS_MIN_RA);
                       mvwaddstr(cs_win,ra_row,41,CS_MAX_RA);
                       mvwaddstr(cs_win,decl_row,23,CS_MIN_DECL);
                       mvwaddstr(cs_win,decl_row,27,CS_NOR_DECL);
                       mvwaddstr(cs_win,decl_row,41,CS_MAX_DECL);
                       mvwaddstr(cs_win,decl_row,45,CS_NOR_DECL);
                       mvwaddstr(cs_win,mag_row,25,CS_MIN_MAG);
                       mvwaddstr(cs_win,mag_row,43,CS_MAX_MAG);
                       cs_set_color(normal,cs_win);

                       for (i=1;i<max_name;i++)
                         object_data[i].marked=FALSE;
                       for (i=1;i<max_type;i++)
                         type_data[i].marked=FALSE;
                       for (i=1;i<max_con;i++)
                         const_data[i].marked=FALSE;

                       cs_values.mark_name=0;
                       cs_values.mark_type=0;
                       cs_values.mark_con=0;

                       mcount=1;
                       scroll_marked(mcount);
                       ende=max_con-1;
                       cs_show(cs_win,cs_con);
                       ende=max_type-1;
                       cs_show(cs_win,cs_type);
                       ende=max_name-1;
                       cs_show(cs_win,cs_name);
		       wrefresh(cs_win);

		       cs_values.cs_default_val=TRUE;
                       next_tab=old_tab;
		       break;
   }
   if(cs_values.cs_default_val)
     mvwaddstr(cs_win,oth_row+15,all_col+17,"   Using default values   ");
   else
     mvwaddstr(cs_win,oth_row+15,all_col+17,"Using user specific values"); 
   cs_show(cs_win,next_tab);
   wrefresh(cs_win);
 }
 okloop=YES;

 delwin(cs_win);
 wrefresh(sac_win);

 free(tmp_ptr);
return ret;
}



/*
 *  void CS_SHOW(int,window*)  prints the names, types and const scroll regions
 *                             to the cs_search window.
 *                             it's nearly the same function as in show_marked.
 *
 */
void cs_show(WINDOW *win,int next_tab)
{
int count,i;
int pos_name=3;    /* column for object_data                     */
int pos_type=14;   /* column for type_data   (15 short, 15 long) */
int pos_con=52;    /* column for const_data  (24 short, 44 long) */
int offs_row=6;    /* row for all datas                          */
int mag_row=4,decl_row=3,ra_row=2,all_col=3,oth_row=6; 

 if((next_tab!=cs_default)&&(next_tab!=cs_search)){
   cs_set_color(normal,win);
   mvwaddstr(win,ra_row,all_col,"Rektaszension  min:");
   mvwaddstr(win,ra_row,all_col+30,"to max:");
   mvwaddstr(win,decl_row,all_col,"Declination    min:");
   mvwaddstr(win,decl_row,all_col+30,"to max:");
   mvwaddstr(win,mag_row,all_col,"Magnitude      min:");
   mvwaddstr(win,mag_row,all_col+30,"to max:");
   mvwaddstr(win,oth_row,all_col,"Name:      Type:                                 Const");

   cs_set_color(active,win); 
   switch(next_tab){
     case cs_ra_min:   mvwaddstr(win,ra_row,all_col,"Rektaszension  min:");
                       break;
     case cs_ra_max:   mvwaddstr(win,ra_row,all_col+30,"to max:");
                       break;
     case cs_decl_min: mvwaddstr(win,decl_row,all_col,"Declination    min:");
                       break;
     case cs_decl_max: mvwaddstr(win,decl_row,all_col+30,"to max:");
                       break;
     case cs_mag_min:  mvwaddstr(win,mag_row,all_col,"Magnitude      min:");
                       break;
     case cs_mag_max:  mvwaddstr(win,mag_row,all_col+30,"to max:");
                       break;
     case cs_name:     mvwaddstr(win,oth_row,all_col,"Name:");
                       break;
     case cs_type:     mvwaddstr(win,oth_row,all_col+11,"Type:");
                       break;
     case cs_con:      mvwaddstr(win,oth_row,all_col+49,"Const");
                       break;
   }
 
   cs_set_color(normal,win);
   if((next_tab==cs_name)||(next_tab==cs_type)||(next_tab==cs_con)){
     if ((scale-10)<1)
       count=1;
     else
       count=scale-10;
  
     for (i=1;i<12;i++){
       if (current==i){
         cs_set_color(scroll_mark,win);
         switch(next_tab){
           case cs_name: if(object_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_name,">");
                         else
                           mvwaddstr(win,i+offs_row,pos_name," ");
                         if(cs_values.name_to_use==use_short)
    		           mvwaddstr(win,i+offs_row,pos_name+1,(char *)object_data[count].short_name);
                         else
                           mvwaddstr(win,i+offs_row,pos_name+1,(char *)object_data[count].middle_name);
		         clear_line(0,0,pos_name+10,win);
                         if(object_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_name+9,"<");
		         break;
           case cs_type: if(type_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_type,">");
  	  	         else
		           mvwaddstr(win,i+offs_row,pos_type," ");
                         mvwaddstr(win,i+offs_row,pos_type+1,(char *)type_data[count].long_type);
                         clear_line(0,0,pos_type+37,win);
		         if(type_data[count].marked==1)
                           mvwaddstr(win,i+offs_row,pos_type+36,"<");
                         break;
           case cs_con:  if(const_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_con,">");
                         else
                           mvwaddstr(win,i+offs_row,pos_con," ");
 	  	         mvwaddstr(win,i+offs_row,pos_con+1,(char *)const_data[count].short_con);
                         clear_line(0,0,pos_con+5,win);
                         if(const_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_con+4,"<");
                         break;
         }
       }
       else{
         cs_set_color(scroll_normal,win);
         switch(next_tab){
           case cs_name: if(object_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_name,">");
  		         else
		           mvwaddstr(win,i+offs_row,pos_name," ");
                         if(cs_values.name_to_use==use_short)
                           mvwaddstr(win,i+offs_row,pos_name+1,(char *)object_data[count].short_name);
                         else
                           mvwaddstr(win,i+offs_row,pos_name+1,(char *)object_data[count].middle_name);
                         clear_line(0,0,pos_name+10,win);
		         if(object_data[count].marked)
		           mvwaddstr(win,i+offs_row,pos_name+9,"<");
                         break;
           case cs_type: if(type_data[count].marked)
                           mvwaddstr(win,i+offs_row,pos_type,">");
                         else
                           mvwaddstr(win,i+offs_row,pos_type," ");
                         mvwaddstr(win,i+offs_row,pos_type+1,(char *)type_data[count].long_type);
                         clear_line(0,0,pos_type+37,win);
                         if(type_data[count].marked==1)
                           mvwaddstr(win,i+offs_row,pos_type+36,"<");
                         break;
           case cs_con:  if(const_data[count].marked)
  	  	           mvwaddstr(win,i+offs_row,pos_con,">");
		         else
		           mvwaddstr(win,i+offs_row,pos_con," ");
                         mvwaddstr(win,i+offs_row,pos_con+1,(char *)const_data[count].short_con);
                         clear_line(0,0,pos_con+5,win);
		         if(const_data[count].marked)
		           mvwaddstr(win,i+offs_row,pos_con+4,"<");
                         break;
           }
       }
       count++;
     }
   }
   cs_set_color(normal,win);
 } 
return;
}



/*
 *
 * void CS_SET_COLOR(int,window*)  set's color or b&w definitions for the 
 *                                 windows
 *
 */
void cs_set_color(int to_do,WINDOW *win)
{
 if (to_do==edit){
   if (status.colors)
     wattrset(win,COLOR_PAIR(6));
   else
     wattrset(win,BW_EDIT);
 }
 if(to_do==active){
   if (status.colors)
     wattrset(win,COLOR_PAIR(7));
   else
     wattrset(win,BW_ACTIVE);
 }
 if(to_do==normal){
   if (status.colors)
     wattrset(win,COLOR_PAIR(3));
   else
     wattrset(win,BW_NORMAL);
 }
 if(to_do==scroll_mark){
   if (status.colors)
     wattrset(win,COLOR_PAIR(6));
   else
     wattrset(win,BW_SCROLL_MARK);
 }
 if(to_do==scroll_normal){
   if (status.colors)
     wattrset(win,COLOR_PAIR(4));
   else
     wattrset(win,BW_SCROLL_NORMAL);
 }
return;
}



/*
 *
 *  char *CS_INPUT(3x int,WINDOW*)  this is our input function for findwindow
 * 				    and cs_search. called from findwindow,
 *				    nothing special happens. called from cs_
 *				    search, this function will check the typed
 *				    char for digit and add '0' if nessesary.
 *                                  This is also very dirty code and must be
 *                                  changed (newwritten) vsn.
 *
 */
char *cs_input(int x_start, int y_start, int length, int ignore_1,
	       int ignore_2, WINDOW *win, int todo)
{
char tmp_ptr[25]="xxxxxxxxxxx";
int i=0;
unsigned char c;

 curs_set(TRUE);
 okloop=YES;
 wmove(win,y_start,x_start);
 wrefresh(win);

 while(okloop){
   if ((i==ignore_1)||(i==ignore_2)){
     switch(todo){
       case cs_inp_mag:  tmp_ptr[i++]='.';
		  	 break;
       case cs_inp_ra:   if(i==ignore_1)
                    	 tmp_ptr[i++]=' ';
                  	 else
                  	   tmp_ptr[i++]='.';
		  	 break;
       case cs_inp_decl: tmp_ptr[i++]=' ';
		  	 break;
     }
     wmove(win,y_start,x_start+i);
   }
   if (i==length)
     i--;
   else{
     switch(c=getch()){
       case KEY_BACKSPACE:
       case KEY_DC:
       case DEL:
       case BS:    if(((i==length-1)&&(ignore_1+ignore_2==0)) || 
                     ((i==length-1)&&(ignore_1+ignore_2>0)&&(tmp_ptr[i]!='\0'))){
                     if(tmp_ptr[i]=='\0')
                       i--;
                   }
                   else if((i-1==ignore_1)||(i-1==ignore_2))
		     i-=2;
                   else
		     i--;
                   if(i<0)
                     i=0;
                   mvwaddch(win,y_start,x_start+i,' ');
		   tmp_ptr[i]='\0';
                   break;
       case LF:
       case CR:    okloop=NO;
                   if((!isdigit(tmp_ptr[i]))&&(i==length-1))
		     i--;
                   break;
       default:    
                   if (todo!=cs_inp_search){
                     if ((i==0)&&(length<7)){
                       if((c=='+')||(c=='-')||(c==' ')){
                         tmp_ptr[i]=c;
                         mvwaddch(win,y_start,x_start+i,c);
                         i++;
                       }
                       else if(isdigit(c)){
			 tmp_ptr[0]='+';
                         tmp_ptr[1]=c;
			 mvwaddch(win,y_start,x_start,'+');
                         mvwaddch(win,y_start,x_start+1,c);
                         i=2;
                       }
                     }
                     else{
                       if (isdigit(c)){
		         tmp_ptr[i]=c;
		         mvwaddch(win,y_start,x_start+i,c);
		         i++;
                       }
                     }
                   }
                   else{
                     if ((isalpha(c))||(isdigit(c))||(c==' ')||
			 (c=='+')    ||(c=='-')    ||(c=='.')||
                         (c==',')    ||(c=='\'')){
                       tmp_ptr[i]=c;
                       mvwaddch(win,y_start,x_start+i,c);
                       i++;
                     }
                   }
	           break;
     } 
   }
   if (i<1)
     i=0;
   wmove(win,y_start,x_start+i);
   wrefresh(win);
 }

 okloop=YES;
 curs_set(FALSE);

 if((todo!=cs_inp_search)&&(i)){
   if((i==2)&&(todo!=cs_inp_ra)){
     tmp_ptr[2]=tmp_ptr[1];
     tmp_ptr[1]='0';
     i=3;
   }
   if((i==1)&&(todo==cs_inp_ra)){
     tmp_ptr[1]=tmp_ptr[0];
     tmp_ptr[0]='0';
     i=2;
   }
   if((i==4)&&(todo==cs_inp_decl)){
     if(isdigit(tmp_ptr[i]))
       i++;
   }
   while(i<length-1){
     if((i!=ignore_1)&&(i!=ignore_2))
       tmp_ptr[i++]='0';
     else{
       switch(todo){
         case cs_inp_mag:  tmp_ptr[i++]='.';
                           break;
         case cs_inp_ra:   if(i==ignore_1)
                             tmp_ptr[i++]=' ';
                           else
                             tmp_ptr[i++]='.';
                           break;
         case cs_inp_decl: tmp_ptr[i++]=' ';
                           break;
       }
     }
   }
   if(!isdigit(tmp_ptr[i++]))
     tmp_ptr[i-1]='0';
   tmp_ptr[i]='\0';
 }
 else
   tmp_ptr[i]='\0';
return tmp_ptr;
}



/*
 *
 *  void CS_HELP()  help windows for cs_search
 *
 */
void cs_help(WINDOW *win)
{
WINDOW *help_win;
int all_col=3,i;

 if(!(help_win = newwin(23,60,SAC_START_ROW+3,SAC_START_COL+10)))
   sac_exit(2,"help_win in function complex_search");
 touchwin(win);

 cs_set_color(normal,help_win);
 sacdrt_werase(help_win);
 wborder(help_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(help_win,0,17,"[ Help for Complex Search ]");
 mvwaddstr(help_win,2,all_col,"Key(s)     in R.A./Decl/Mag.       in Name/Type/Con");
 wmove(help_win,3,2);
 for (i=0;i<55;i++)
   waddch(help_win,SAC_HLINE);
 mvwaddch (help_win,4,all_col,ACS_UARROW);
 mvwaddstr(help_win,4,all_col+1,"[u]         no function       scroll to prev object");
 mvwaddch (help_win,5,all_col,ACS_DARROW);
 mvwaddstr(help_win,5,all_col+1,"[d]         no function       scroll to next object");
 mvwaddch (help_win,6,all_col,ACS_RARROW);
 mvwaddstr(help_win,6,all_col+1,"[p]          prev item               prev item");
 mvwaddch (help_win,7,all_col,ACS_LARROW);
 mvwaddstr(help_win,7,all_col+1,"[n]          next item               next item");
 mvwaddstr(help_win,8,all_col,"m[M]/Return   edit item          mark/unmark object");
 mvwaddstr(help_win,9,all_col,"c[C]          no function      change display (name)");
 mvwaddstr(help_win,10,all_col,"s[S]         begin search          begin search");
 mvwaddstr(help_win,11,all_col,"o[O]     set default values     set default values");
 mvwaddstr(help_win,12,all_col,"h[H]         this window            this window");
 mvwaddstr(help_win,13,all_col,"i[I]         no function            invert marks");
 mvwaddstr(help_win,14,all_col,"q[Q]         quit search            quit search");
 mvwaddstr(help_win,15,all_col,"Next Page    no function        scroll to next page");
 mvwaddstr(help_win,16,all_col,"Prev Page    no function        scroll to prev page");

 mvwaddstr(help_win,18,all_col+1,"When there objects found, they whill be inserted in");
 mvwaddstr(help_win,19,all_col+1,"the marked list and the Filter will be invoked. Use");
 mvwaddstr(help_win,20,all_col+1,"(f) to turn the Filter on/off. The objects will be");
 mvwaddstr(help_win,21,all_col+1,"   in the Filter up to the next complex search");

 wrefresh(help_win);
 getch();
 delwin(help_win);
 wrefresh(win);
return;
}



/*
 *
 *  SAC_FILTER(int)  invoke the filter or not.
 *
 */
void sac_filter (int todo)
{
 if(todo){
   mvwaddstr(screen_win,27,76," ON");
   config.filter=TRUE;
 }
 else{
   mvwaddstr(screen_win,27,76,"OFF");
   config.filter=FALSE;
 }
 wrefresh(screen_win);
 status.mark_current=1;
return;
}



/*
 *
 * void SORT_WINDOW()  when objects marked and youa re in print mode you can
 *                     press 's' to get this window opened. You can specifie
 *                     what the function should sort for (mag, ra, decl, name,
 *                     type or con and ascending or descending) and what algo-
 *                     rythm you would use (shell sort or quick sort).
 *
 */
void sort_window()
{
WINDOW *sort_win;
int next_tab=sort_ra;

 if(!(sort_win = newwin(13,40,SAC_START_ROW+8,SAC_START_COL+20)))
   sac_exit(2,"sort_win in function sort_window");
 touchwin(find_win);

 cs_set_color(normal,sort_win);

 sacdrt_werase(sort_win);
 wborder(sort_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);

 mvwaddstr(sort_win,0,8,"[ Sort Marked Objects ]");

 mvwaddstr(sort_win,8,3,"  [u] one Up      [n] Next active");
 mvwaddch (sort_win,8,4,ACS_UARROW);
 mvwaddch (sort_win,8,20,ACS_RARROW);

 mvwaddstr(sort_win,9,3,"  [d] one Down    [p] Prev active");
 mvwaddch (sort_win,9,4,ACS_DARROW);
 mvwaddch (sort_win,9,20,ACS_LARROW);

 mvwaddstr(sort_win,10,2,"   [q] Quit        [s] begin Sort");
 mvwaddstr(sort_win,11,2,"       [Enter]/[Space] change item");

 mvwaddch(sort_win,7,0,SAC_VLEFT);
 line(sort_win,38);
 mvwaddch(sort_win,7,39,SAC_VRIGHT);

 sort_show(sort_win,next_tab);

 while(okloop){
   switch(getch()){
     case 'S':
     case 's':        if(sort.ra)
			sort.sort_to=sort_ra;
 		      if(sort.decl)
			sort.sort_to=sort_decl;
		      if(sort.mag)
			sort.sort_to=sort_mag;
                      if(sort.name)
			sort.sort_to=sort_name;
                      if(sort.type)
			sort.sort_to=sort_type;
                      if(sort.con)
			sort.sort_to=sort_con;

                      if(config.sort_mode)
			message_window("  sorting:",inverse);
                      else
                        message_window("    sorting...",inverse);
                      wrefresh(sort_win);
                      sort_marked();
                      close_message_window(nothing);
		      message_window("  Objects sorted",inverse);
                      getch();
                      close_message_window(nothing);
                      okloop=NO;
                      break;
     case ' ':
     case LF:
     case CR:         switch(next_tab){
                        case sort_name:
                        case sort_decl:
                        case sort_type:
                        case sort_con:
                        case sort_ra:
                        case sort_mag:        sort.ra=FALSE;
					      sort.mag=FALSE;
					      sort.decl=FALSE;
					      sort.name=FALSE;
				              sort.type=FALSE;
					      sort.con=FALSE;
                                              break;
                        case sort_ascending:
                        case sort_descending: if(sort.up_down)
						sort.up_down=FALSE;
					      else
						sort.up_down=TRUE;
                                              break;
                        case sort_shell:
                        case sort_quick:      if(config.sort_mode)
						config.sort_mode=FALSE;
					      else
						config.sort_mode=TRUE;
                                              break;
                      }
                      switch(next_tab){
                        case sort_name:       sort.name=TRUE;
                                              break;
                        case sort_decl:       sort.decl=TRUE;
                                              break;
                        case sort_type:       sort.type=TRUE;
                                              break;
                        case sort_con:        sort.con=TRUE;
                                              break;
                        case sort_ra:         sort.ra=TRUE;
                                              break;
                        case sort_mag:        sort.mag=TRUE;
                                              break;
                      }
                      break;
     case KEY_DOWN:     
     case 'D':
     case 'd':        switch(next_tab){
			case sort_ra:         next_tab=sort_decl;
					      break;
			case sort_mag:        next_tab=sort_ascending;
                                              break;
			case sort_shell:      next_tab=sort_ra;
                                              break;
			case sort_decl:       next_tab=sort_mag;
                                              break;
			case sort_ascending:  next_tab=sort_shell;
                                              break;
			case sort_name:       next_tab=sort_type;
                                              break;
			case sort_type:       next_tab=sort_con;
                                              break;
			case sort_con:        next_tab=sort_descending;
                                              break;
			case sort_quick:      next_tab=sort_name;
                                              break;
			case sort_descending: next_tab=sort_quick;
                                              break;
                      }
                      break;
     case KEY_UP:
     case 'U':
     case 'u':        switch(next_tab){
                        case sort_ra:         next_tab=sort_shell;
                                              break;
                        case sort_mag:        next_tab=sort_decl;
                                              break;
                        case sort_decl:       next_tab=sort_ra;
                                              break;
                        case sort_ascending:  next_tab=sort_mag;
                                              break;
			case sort_shell:      next_tab=sort_ascending;
                                              break;
                        case sort_name:       next_tab=sort_quick;
                                              break;
                        case sort_type:       next_tab=sort_name;
                                              break;
                        case sort_con:        next_tab=sort_type;
                                              break;
                        case sort_descending: next_tab=sort_con;
                                              break;
			case sort_quick:      next_tab=sort_descending;
                                              break;
                      }
                      break;
     case KEY_RIGHT:
     case 'N':
     case 'n':        switch(next_tab){
                        case sort_ra:         next_tab=sort_name;
                                              break;
                        case sort_mag:        next_tab=sort_con;
                                              break;
                        case sort_decl:       next_tab=sort_type;
                                              break;
                        case sort_ascending:  next_tab=sort_descending;
                                              break;
                        case sort_name:       next_tab=sort_decl;
                                              break;
                        case sort_type:       next_tab=sort_mag;
                                              break;
                        case sort_con:        next_tab=sort_ascending;
                                              break;
			case sort_shell:      next_tab=sort_quick;
                                              break;
			case sort_quick:      next_tab=sort_ra;
					      break;
                        case sort_descending: next_tab=sort_shell;
                                              break;
                      }
                      break;
     case KEY_LEFT:
     case 'p':
     case 'P':        switch(next_tab){
                        case sort_ra:         next_tab=sort_quick;
                                              break;
                        case sort_mag:        next_tab=sort_type;
                                              break;
                        case sort_decl:       next_tab=sort_name;
                                              break;
                        case sort_ascending:  next_tab=sort_con;
                                              break;
                        case sort_shell:      next_tab=sort_descending;
                                              break;
                        case sort_quick:      next_tab=sort_shell;
                                              break;
                        case sort_name:       next_tab=sort_ra;
                                              break;
                        case sort_type:       next_tab=sort_decl;
                                              break;
                        case sort_con:        next_tab=sort_mag;
                                              break;
                        case sort_descending: next_tab=sort_ascending;
                                              break;
                      }
                      break;
     case 27:
     case 'Q':
     case 'q':        okloop=NO;
                      break;
   } 
   if(okloop)
     sort_show(sort_win,next_tab);
 }
 delwin(sort_win);
 wrefresh(find_win);
 okloop=YES;

return;
}



/*
 *
 * void SORT_SHOW(...) will print the "YES" and "NO" strings in the window
 *
 */
void sort_show(WINDOW *win, int to_do)
{
 cs_set_color(edit,win);
 if(sort.ra)
   mvwaddstr(win,2,17,SORT_YES);
 else
   mvwaddstr(win,2,17,SORT_NO);

 if(sort.decl)
   mvwaddstr(win,3,17,SORT_YES);
 else
   mvwaddstr(win,3,17,SORT_NO);

 if(sort.mag)
   mvwaddstr(win,4,17,SORT_YES);
 else
   mvwaddstr(win,4,17,SORT_NO);

 if(sort.name)
   mvwaddstr(win,2,35,SORT_YES);
     else
   mvwaddstr(win,2,35,SORT_NO);

 if(sort.type)
   mvwaddstr(win,3,35,SORT_YES);
 else
   mvwaddstr(win,3,35,SORT_NO);

 if(sort.con)
   mvwaddstr(win,4,35,SORT_YES);
    else
   mvwaddstr(win,4,35,SORT_NO);

 if(sort.up_down){
   mvwaddstr(win,5,17,SORT_YES);
   mvwaddstr(win,5,35,SORT_NO);
 }
 else{
   mvwaddstr(win,5,17,SORT_NO);
   mvwaddstr(win,5,35,SORT_YES);
 }

 if(config.sort_mode){
   mvwaddstr(win,6,17,SORT_YES);
   mvwaddstr(win,6,35,SORT_NO);
 }
 else{
   mvwaddstr(win,6,17,SORT_NO);
   mvwaddstr(win,6,35,SORT_YES);
 }

 cs_set_color(normal,win);
 mvwaddstr(win,2,2,"Rektaszension:");
 mvwaddstr(win,2,23,"Name:");
 mvwaddstr(win,3,2,"Declination:");
 mvwaddstr(win,3,23,"Type:");
 mvwaddstr(win,4,2,"Magnitude:");
 mvwaddstr(win,4,23,"Const:");
 mvwaddstr(win,5,2,"Ascending:");
 mvwaddstr(win,5,23,"Descending:");
 mvwaddstr(win,6,2,"Shell sort:");
 mvwaddstr(win,6,23,"Quick sort:");

 cs_set_color(active,win);
 switch(to_do){
   case sort_name:       mvwaddstr(win,2,23,"Name:");
                         break;
   case sort_type:       mvwaddstr(win,3,23,"Type:");
                         break;
   case sort_con:        mvwaddstr(win,4,23,"Const:");
                         break;
   case sort_mag:        mvwaddstr(win,4,2,"Magnitude:");
                         break;
   case sort_ra:         mvwaddstr(win,2,2,"Rektaszension:");
                         break;
   case sort_decl:       mvwaddstr(win,3,2,"Declination:");
                         break;
   case sort_ascending:  mvwaddstr(win,5,2,"Ascending:");
                         break;
   case sort_descending: mvwaddstr(win,5,23,"Descending:");
                         break;
   case sort_shell:      mvwaddstr(win,6,2,"Shell sort:");
                         break;
   case sort_quick:      mvwaddstr(win,6,23,"Quick sort:");
                         break;
 }
 cs_set_color(normal,win);
 wrefresh(win);

return;
}



/*
 *
 * SACDRT_WERASE(...)  this is because ncurses will no longer use the
 *                     current attribut of the window when erasing it
 *                     (from version 1.8.5). ncurses must do this,
 *                     because otherwise occure problmes with svr4.
 *                     (that would make ncurses incompatible with SVR4)
 *
 *                     the origimel idea of these function is of course
 *                     from Zeyd. i've changed his code only a little bit.
 *                     THANKS TO HIME FOR HIS HELP AND FOR THE NCURSES.
 */
void sacdrt_werase(WINDOW *win)
{
int  y,x;

 for (y=0;y<=win->_maxy;y++)
   for (x=0;x<=win->_maxx;x++)
     (chtype *)win->_line[y][x]=' '|win->_attrs;
 win->_curx = win->_cury = 0;
 touchwin(win);
return;
}



/*
 *
 * PRINT_WINDOW(...)
 *
 */
void print_window()
{
WINDOW *print_win;
int next_tab=print_printer;

 if(!(print_win = newwin(13,40,SAC_START_ROW+8,SAC_START_COL+20)))
   sac_exit(2,"print_win in function print_window");
 touchwin(find_win);

 cs_set_color(normal,print_win);

 sacdrt_werase(print_win);
 wborder(print_win,SAC_VLINE,SAC_VLINE,SAC_HLINE,SAC_HLINE,SAC_ULCORNER,SAC_URCORNER,SAC_LLCORNER,SAC_LRCORNER);
 mvwaddstr(print_win,0,8,"[ Print Marked Objects ]");

 mvwaddstr(print_win,9,3,"  [u] one Up      [d] one Down");
 mvwaddch (print_win,9,4,ACS_UARROW);
 mvwaddch (print_win,9,20,ACS_DARROW);
 mvwaddstr(print_win,10,2,"   [q] Quit        [w] begin Write");
 mvwaddstr(print_win,11,2,"       [Enter]/[Space] change item");

 mvwaddch(print_win,4,0,SAC_VLEFT);
 line(print_win,38);
 mvwaddch(print_win,4,39,SAC_VRIGHT);

 mvwaddch(print_win,8,0,SAC_VLEFT);
 line(print_win,38);
 mvwaddch(print_win,8,39,SAC_VRIGHT);

 print_show(print_win,next_tab);

 while(okloop){
   switch(getch()){
     case 'W':
     case 'w':        if(config.print_fop){
                        message_window("    writing...",inverse);
                        touchwin(find_win);
                        nodelay (stdscr,TRUE);
                        handle_objects(file_current,nothing,nothing,"");
                        close_message_window(inverse);
                        nodelay (stdscr,FALSE);
                      }
                      else{
                        message_window("   printing...",inverse);
                        touchwin(find_win);
                        nodelay (stdscr,TRUE);
                        handle_objects(print_current,nothing,nothing,"");
                        nodelay (stdscr,FALSE);
                        close_message_window(inverse);
                      }
                      okloop=NO;
                      break;
     case ' ':
     case LF :
     case CR :        switch(next_tab){
                        case print_file:
                        case print_printer: if(config.print_fop)
                                              config.print_fop=FALSE;
                                            else
                                              config.print_fop=TRUE;
                                            break;
                        case print_all:
                        case print_tab:     if(config.print_mode)
                                              config.print_mode=FALSE;
                                            else
                                              config.print_mode=TRUE;
                                            break;
                      }
                      break;
     case KEY_DOWN:
     case 'D':
     case 'd':        switch(next_tab){
                        case print_printer: next_tab=print_file;
			                    break;
			case print_file:    next_tab=print_tab;
                                            break;
                        case print_tab:     next_tab=print_all;
                                            break;
			case print_all:     next_tab=print_printer;
                                            break;
                                         }
                      break;
     case KEY_UP:
     case 'U':
     case 'u':        switch(next_tab){
                        case print_printer: next_tab=print_all;
                                            break;
                        case print_file:    next_tab=print_printer;
                                            break;
                        case print_tab:     next_tab=print_file;
                                            break;
                        case print_all:     next_tab=print_tab;
                                            break;
                      }
                      break;
     case 27:
     case 'Q':
     case 'q':        okloop=NO;
                      break;
   }
   if(okloop)
     print_show(print_win,next_tab);
 }
 delwin(print_win);
 wrefresh(find_win);
 okloop=YES;

return;
}



/*
 *
 *  void SHOW_PRINT(...)
 *
 */
void print_show(WINDOW *win, int to_do)
{
 cs_set_color(edit,win);
 if(!config.print_fop){
   mvwaddstr(win,2,33,SORT_YES);
   mvwaddstr(win,3,33,SORT_NO);
 }
 else{
   mvwaddstr(win,2,33,SORT_NO);
   mvwaddstr(win,3,33,SORT_YES);
 }
 if(config.print_mode){
   mvwaddstr(win,7,33,SORT_YES);
   mvwaddstr(win,6,33,SORT_NO);
 }
 else{
   mvwaddstr(win,7,33,SORT_NO);
   mvwaddstr(win,6,33,SORT_YES);
 }

 cs_set_color(normal,win);
 mvwaddstr(win,2,4,"Write to Printer:");
 mvwaddstr(win,3,4,"Write to File:");
 mvwaddstr(win,7,4,"Write All (screendump):");
 mvwaddstr(win,5,4,"Write Array");
 mvwaddstr(win,6,4,"(Name,CON,RA,DEC,Mag):");

 cs_set_color(active,win);
 switch(to_do){
   case print_printer: mvwaddstr(win,2,4,"Write to Printer:");
                       break;
   case print_file:    mvwaddstr(win,3,4,"Write to File:");
                       break;
   case print_all:     mvwaddstr(win,7,4,"Write All (screendump):"); 
                       break;
   case print_tab:     mvwaddstr(win,5,4,"Write Array");
                       mvwaddstr(win,6,4,"(Name,CON,RA,DEC,Mag):");
                       break;
 }

 cs_set_color(normal,win); 
 wrefresh(win);
return;
}
