/*
 * roche.c -- Program to generate a grid of potential and effective gravity
 * -------    as a function of xy-position in the 2-body Roche problem. This
 *            treatment does not account for any relativistic effects.
 *
 * Written by Derek C. Richardson (richards@cita.utoronto.ca) Jun 11/95.
 *
 * To compile: cc -o roche roche.c -lm
 * Then just type "roche" (or roche > roche.dat) to run.
 *
 * This program may be distributed freely according to the terms and
 * conditions of the GNU Public License (a copy of which is included
 * in the original source distribution).
 *
 */

#include <stdio.h>
#include <math.h>

/*
 * Units: In the following treatment, the gravitational constant G = 1,
 * ------ so a suitable system might be masses in units of one solar mass,
 *        lengths in units of one astronomical unit, and times in units of
 *        (1 / 2 pi) sidereal years.
 *
 */

#define M1  (1.0)	/* Mass of body 1 in mass units (on -ve x-axis) */
#define M2  (1.0)	/* Mass of body 2 in mass units (on +ve x-axis) */
#define SEP (2.0)	/* Separation of bodies 1 and 2 in length units */
#define NXY (21)	/* # grid points in either x or y (square grid) */
#define LXY (3.5 * SEP) /* Grid size in either x or y (in length units) */

/* Useful macros */

#define SQ(x) ((x) * (x))
#define CUBE(x) ((x) * SQ(x))

/* Everything happens in main() */

int main(void)
{
  /*
   * Outputs to stdout the potential and effective gravity (both x and y
   * component) as a function of grid position for the 2-body Roche problem.
   * The format of the output is:
   *
   *    x y p gx gy
   *
   * The mass points are fixed on the x-axis in such a way that the centre
   * of mass (barycentre) is at the origin (0,0).
   *
   */

  int ix, iy;		/* Grid indices */

  double x1, x2,	/* Position of mass 1 & 2 */
         w,		/* Angular frequency of rotating frame */
         x, y,          /* Grid coordinates */
         r1, r2,        /* Distance from grid coordinate to body 1 & 2 */
         p1, p2,        /* Potential due to body 1 & 2 */
         pc,		/* Potential giving rise to centrifugal force */
         p,		/* Total potential */
         gx1, gy1,	/* Effective gravity due to body 1 (2 components) */
         gx2, gy2,	/* Ditto for body 2 */
         gxc, gyc,      /* Components of centrifugal force */
         gx, gy;        /* Total effective gravity, both components */

  /* Determine x-coordinates of masses, requiring x1 to be negative */

  x1 = - (M2 / (M1 + M2)) * SEP;

  x2 = x1 + SEP;

  /* Determine angular frequency needed for circular motion */

  w = sqrt(M1 / x2) / SEP;

  /* Loop over grid points (-LXY/2 to LXY/2 in both directions) */

  for (ix = 0; ix < NXY; ix++)
    for (iy = 0; iy < NXY; iy++) {

      /* Determine grid coordinates */

      x = LXY * ((double) ix / (NXY - 1) - 0.5);
      y = LXY * ((double) iy / (NXY - 1) - 0.5);

      /* Get distance to mass points */

      r1 = sqrt(SQ(x1 - x) + SQ(y));
      r2 = sqrt(SQ(x2 - x) + SQ(y));

      /* Calculate potential */

      p1 = M1 / r1;
      p2 = M2 / r2;
      pc = 0.5 * SQ(w) * (SQ(x) + SQ(y));

      p = p1 + p2 + pc;

      /* Calculate effective gravity */

      gx1 = M1 * (x1 - x) / CUBE(r1);
      gy1 = - M1 * y / CUBE(r1);

      gx2 = M2 * (x2 - x) / CUBE(r2);
      gy2 = - M2 * y / CUBE(r2);

      gxc = SQ(w) * x;
      gyc = SQ(w) * y;

      gx = gx1 + gx2 + gxc;
      gy = gy1 + gy2 + gyc;

      /* Output data */

      (void) printf("%e %e %e %e %e\n", x, y, p, gx, gy);
    }

  return 0;
}

/* roche.c */
