/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include "Light.h"



/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Attach the throat of star a to star b
 @tip       This is to verify eclipse by 'own' star
 @param     (void) 
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/
void LightCopyThroat()
{
  long   j;                                 /* loop counter                 */
  long  Nplus = 0;                          /* surface elements + throat    */
  long  NP = 0;                             /* surface elements primary     */
  long NS = 0;                              /* surface elements secondary   */

  /* --------  attach Secondary throat to Primary ----------------------    */

  Nplus = Binary[Secondary].N_PhiStep[StepsPerPi-1]
    + Binary[Secondary].N_PhiStep[StepsPerPi-2];
  NP    = Binary[Primary].NumElem;
  NS    = Binary[Secondary].NumElem;


  Binary[Primary].NumPlus = NP + Nplus;
  if (Binary[Primary].NumPlus > (long) MaximumElements) 
    nf_error(_(errmsg[10]));

  for (j = 0; j < Nplus; ++j) {

    Surface[Primary][NP+j] 
      = Surface[Secondary][NS-Nplus+j];

    /* ---------------  INVERT X ---------------------------------------   */

    Surface[Primary][NP+j].lambda = (float)
      (1.0 - Surface[Primary][NP+j].lambda);

    Surface[Primary][NP+j].l =
      (-Surface[Primary][NP+j].l);

  }

  /* ---------------- attach Primary throat to Secondary ----------------  */

  Nplus = Binary[Primary].N_PhiStep[StepsPerPi-1]
    + Binary[Primary].N_PhiStep[StepsPerPi-2];

  Binary[Secondary].NumPlus = NS + Nplus;

  if (Binary[Primary].NumPlus > (long) MaximumElements) 
    nf_error(_(errmsg[10]));

  for (j = 0; j < Nplus; ++j) {

    Surface[Secondary][NS+j] 
      = Surface[Primary][NP-Nplus+j]; 

    /* ---------------------  INVERT X  ---------------------------------  */

    Surface[Secondary][NS+j].lambda = (float)
      (1.0 - Surface[Secondary][NS+j].lambda);

    Surface[Secondary][NS+j].l =
      (-Surface[Secondary][NS+j].l);

  }

  return;
}                            

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Copy the throat back
 @param     (void) 
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/
void LightCopyThroatBack()
{
  long   j;                                  /* loop counter                 */
  long  Nplus = 0;                           /* surface elements + throat    */
  long  NP = 0;                              /* surface elements primary     */
  long  NS = 0;                              /* surface elements secondary   */

  NP    = Binary[Primary].NumElem;
  NS    = Binary[Secondary].NumElem;

  /* -----------  copy back Primary throat to Primary -------------------   */

  Nplus = Binary[Secondary].NumPlus - NS;

  for (j = 0; j < Nplus; ++j) {

    if(Surface[Secondary][NS+j].EclipseFlag == ON){

      Surface[Primary][NP-Nplus+j].EclipseFlag 
	= Surface[Secondary][NS+j].EclipseFlag;

      Surface[Primary][NP-Nplus+j].visibility 
	= Surface[Secondary][NS+j].visibility;

    }
    
  }

  /* -------------  copy back Secondary throat to Secondary -------------   */

  Nplus = Binary[Primary].NumPlus - NP;
  
  for (j = 0; j < Nplus; ++j) {

    if (Surface[Primary][NP+j].EclipseFlag == ON){

      Surface[Secondary][NS-Nplus+j].EclipseFlag 
	= Surface[Primary][NP+j].EclipseFlag;

      Surface[Secondary][NS-Nplus+j].visibility 
	= Surface[Primary][NP+j].visibility;

    }
  }

  return;
}

/********************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Calculate various quantities for simple geometric tests
 @param     (void) 
 @return    (int)  Exit status 
 @heading   Light Curve
 ********************************************************************/
int LightSetupTests()
{

  int      testerr = 0;     /* Exit status RootFind                 */
  double   RLag1;           /* Lagrange One                         */
  /* double   Xl, h1, h2; */      /* Xl lagrange one                      */
  double   Xp1, Xp2;        /* max. distance surface-center of star */
  /* double   PhiLagrange; */     /* tangent angle of roche lobes at L1   */
  double   CosPhiLagrange;  /* tangent angle at Lagrange One (cos)  */
  double   CosPhiOne = 0.0; /* perhaps more stringent if Roche lobe */
                            /* not filled                           */

  /* ----------  simple geometric tests for eclipse possible ------ */ 


  /* osculating cone angle at L1                                    */
  /*
  Xl    = Binary[Primary].RLag1;
  h1    = 1.0/(Xl*Xl*Xl);
  h2    = 1.0/((1.0-Xl)*(1.0-Xl)*(1.0-Xl));
  PhiLagrange = sqrt((2*h1 + 2*Mq*h2 + (1.0 - Mq))/(h1 + Mq*h2 - (1.0 - Mq)));
  PhiLagrange = atan(PhiLagrange);
  CosPhiLagrange = cos(PhiLagrange);
  */

  Xp1 = 1.0; Xp2 = 1.0;

  /* calculate Xp == distance from center to surface towards L1     */

  /* lambda, nu global; potential on x-axis                         */

  lambda     = 1.0; nu = 0.0;  
  RochePot   = Binary[Primary].RochePot;
  Mq         = Binary[Primary].Mq;
  if (Flags.asynchron1 == ON)
    F        = Binary[Primary].Fratio;
  else
    F        = 1.0;
  RLag1      = Binary[Primary].RLag1;

  if (fabs(Binary[Primary].RocheFill - 1.0) <= FLT_EPSILON) { 
    Xp1     = Binary[Primary].RXCrit;
  } else {
    if (Flags.fill == OFF) {
      Xp1     = RootFind(RocheSurface, 0.00001, Binary[Primary].RXCrit, 1.0e-8,
			 "LightSetupTests", &testerr);
      if (testerr == 1) return(8);
    } else {
      Xp1     = Binary[Primary].LimRad;
    }
  }
  Binary[Primary].Xp = Xp1;

  RochePot   = Binary[Secondary].RochePot;
  Mq         = Binary[Secondary].Mq;
  if (Flags.asynchron2 == ON)
    F        = Binary[Secondary].Fratio;
  else
    F        = 1.0;
  RLag1      = Binary[Secondary].RLag1;

  if (fabs(Binary[Secondary].RocheFill - 1.0) <= FLT_EPSILON) {
    Xp2     = Binary[Secondary].RXCrit;
  } else {
    if (Flags.fill == OFF) {
      Xp2     = RootFind(RocheSurface, 0.00001,Binary[Secondary].RXCrit, 
			 1.0e-8,
			 "LightSetupTests", &testerr );
      if (testerr == 8) return(8);
      
    } else {
      Xp2     = Binary[Secondary].LimRad;
    }
  }
  Binary[Secondary].Xp = Xp2;

  /* ------  sanity check if F < 1.0 ------------------------------ */

  if (
      ( (Binary[Secondary].Fratio - 1.0) <= (-FLT_EPSILON)
	&& Flags.asynchron1 == ON )
      || ( (Binary[Primary].Fratio - 1.0) <= (-FLT_EPSILON) 
	   && Flags.asynchron2 == ON ) ) {
    if ((Binary[Secondary].Xp + Binary[Primary].Xp) >= (1.0+DBL_EPSILON) ) {
      WARNING(_("Fill Factor too large in Async Rotating Star --> Stars intersect"));
      return(1);
    }
  }

  if (Flags.fill == OFF) {
    CosPhiOne  = sqrt( MAX(0.0, (1.0 - SQR(Xp1 + Xp2))) );
  } else {
    CosPhiOne  = 0.1;
  }

  /* osculating cone angle at L1                                    */

  /* Chanan et al. (1972), ApJ 208, 512
   */

  if ( MIN(Binary[Primary].Mq, Binary[Secondary].Mq) > 0.4)
    CosPhiLagrange = cos (DTOR*57.88);
  else if ( MIN(Binary[Primary].Mq, Binary[Secondary].Mq) > 0.2)
    CosPhiLagrange = cos (DTOR*59.00);
  else if ( MIN(Binary[Primary].Mq, Binary[Secondary].Mq) > 0.1)
    CosPhiLagrange = cos (DTOR*60.56);
  else if ( MIN(Binary[Primary].Mq, Binary[Secondary].Mq) > 0.01)
    CosPhiLagrange = cos (DTOR*67.17);
  else 
    CosPhiLagrange = 0.0;
  
  /* exact contact system
   */
  if ( (fabs(Binary[Primary].RocheFill - 1.0) <= FLT_EPSILON) &&
       (fabs(Binary[Secondary].RocheFill - 1.0) <= FLT_EPSILON) ) { 
    CosPhiLagrange = CosPhiLagrange;
  } else {
    CosPhiLagrange = CosPhiOne;
  }

  Binary[Primary].CosPhiLagrange   = CosPhiLagrange;
  Binary[Secondary].CosPhiLagrange = CosPhiLagrange;
   
  return(0);
}
           

/********************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Eclipse Verification
 @param     (int)  j The phase index 
 @return    (void)
 @heading   Light Curve
 ********************************************************************/
void LightCurve(int j)
{
  long NElem, NElemP, NElemS;            /* # of surface elements   */ 
  long i;                                /* loop counter            */
  int       eclipsed = 1;                /* test variable           */
  const int Yes = 1;                     /* test value              */
  const int No  = 0;                     /* test value              */
  const int Invisible = 10;              /* test value              */
  int       MinErr;                      /* exit status MinFinder   */

  double   Phase, Phase2;                /* orbital phase           */
  double   OmegaDotPrimary;              /* surface rotation async  */
  double   OmegaDotSecondary;            /* surface rotation async  */
  double   lzero,  mzero,  nzero;        /* LOS vector              */
  double   lzero2, mzero2, nzero2;       /* LOS vector              */
  double   x_z, y_z, z_z;          
  double   x_tl, y_tl, z_tl;
  double   Qi, Li, QLi;                  /* for sphere test         */   
  double   t_l, t_1, t_2, t_dist;        /* intersection w/ sphere  */

  double   tmin = 0;        /* minimum along LOS                    */
  double   PotMin;          /* minimum along LOS                    */
  double   SqrXp1, SqrXp2;  /* square of (center - lagrange one)    */
  double   SqrPol1,SqrPol2; /* square of polar radius               */
  double   Sini, Cosi;      /* cos, sin of orbital inclination      */
  double   SinPhase, CosPhase; /* cos, sin of phase (primary)       */
  double   SinPhase2, CosPhase2; /* cos, sin of phase (secondary)   */
  double   Cosgamma;        /* angle between LOS and surface normal */
  double   CosPhiLagrange;  /* tangent angle at Lagrange One (cos)  */

  double   Xp1, Xp2;        /* max. distance surface-center of star */
  double   Pot_One, Pot_Two; /* surface potentials                  */

  SurfaceElement *SurfPtrP;  /* pointer to surface Primary          */
  SurfaceElement *SurfPtrS;  /* pointer to surface Secondary        */

  double   fratio1 = 1.0;    /* async rotation                      */
  double   fratio2 = 1.0;    /* async rotation                      */

  if (Flags.asynchron1 == ON) fratio1 = Binary[Primary].Fratio;
  if (Flags.asynchron2 == ON) fratio2 = Binary[Secondary].Fratio;

  OmegaDotPrimary   = fratio1   * 2.0 * PI / Orbit.TruePeriod;
  OmegaDotSecondary = fratio2   * 2.0 * PI / Orbit.TruePeriod;

  Sini  = sin(Orbit.Inclination);
  Cosi  = cos(Orbit.Inclination);
  NElem = Binary[Primary].NumElem; 
  Mq    = Binary[Primary].Mq;       /* Mq is global variable        */ 

  Pot_One        = Binary[Primary].RochePot;
  Pot_Two        = Binary[Secondary].RochePot;
  Xp1            = Binary[Primary].Xp;
  Xp2            = Binary[Secondary].Xp;

  /* CosPhiLagrange is same for Primary and Secondary               */
  CosPhiLagrange = Binary[Primary].CosPhiLagrange; 

  SqrXp1 = SQR(Xp1);
  SqrXp2 = SQR(Xp2);
  SqrPol1 = SQR(Binary[Primary].Radius);  /* square of polar radius */
  SqrPol2 = SQR(Binary[Secondary].Radius);

  /* ------------------     eclipse test         ------------------ */

  /* to allow for generalization, we dont insist on                 */
  /*  same number of surface elements on both components            */


  NElem  = IMAX(Binary[Primary].NumPlus,Binary[Secondary].NumPlus);
  NElemP = IMAX(Binary[Primary].NumElem, Binary[Primary].NumPlus);
  NElemS = IMAX(Binary[Secondary].NumElem, Binary[Secondary].NumPlus);

  /*
  NElem  = IMAX(Binary[Primary].NumElem,Binary[Secondary].NumElem);
  NElemP = Binary[Primary].NumElem;
  NElemS = Binary[Secondary].NumElem;
  */

  /* We start at 90 deg, Phase = -0.25, both stars fully visible    */

  if (Flags.elliptic == OFF) {

    Phase = (PI/2 + (2*PI/PhaseSteps) * j ); 
    FluxOut[j].Phase = (float) Phase;
    Orbit.Phase = Phase;
    Orbit.Nu    = Phase;

  } else {
    
     Phase = Orbit.Phase;
     /* Orbit.OmegaZero is mean anomaly at secondary eclipse        */
     FluxOut[j].Phase = (float) (Orbit.M + PI + Orbit.OmegaZero);
  }

  SinPhase = sin(Phase); CosPhase = cos(Phase);


  /* vector towards observer (LOS = Line Of Sight)                  */
  lzero   =  Sini * CosPhase;
  mzero   =  Sini * SinPhase;
  nzero   =  Cosi;

  /* Secondary turn by PI and mirror y->(-y)                        */
  /* this is equivalent to mirror by x->(-x)                        */

  Phase2 = (PI + Phase);
  SinPhase2 = sin(Phase2); CosPhase2 = cos(Phase2);

  /* vector towards observer (LOS = Line Of Sight)                  */
  lzero2   =  Sini * CosPhase2;
  mzero2   =  Sini * SinPhase2;
  nzero2   =  Cosi;

  /* test whether eclipse of Primary definitely impossible          */
  
  /* this flag is (Flags.InEclipse1) required for fractional visibility
   */
  if ( CosPhase <= (CosPhiLagrange-DBL_EPSILON) ) {
    Flags.InEclipse1 = OFF;
  }  else { 
    Flags.InEclipse1 = ON;
  }
  

  /* test whether eclipse of Secondary definitely impossible        */
  
  /* this flag (Flags.InEclipse2) is required for fractional visibility
   */
  if ( CosPhase2 <= (CosPhiLagrange-DBL_EPSILON) ) {
    Flags.InEclipse2 = OFF;
  }  else { 
    Flags.InEclipse2 = ON;
  }
  

  SurfPtrP = Surface[Primary];
  SurfPtrS = Surface[Secondary];

  for (i = 0; i < NElem; ++i) {      /* loop over surface elements */

    /* START WITH PRIMARY                                          */ 

    if (i < NElemP) {

      SurfPtrP->Velocity = 
	- OmegaDotPrimary * Orbit.TrueDistance * Orbit.Dist
	* (-SurfPtrP->mu     * lzero
	   + SurfPtrP->lambda * mzero);


      SurfPtrP->LOS_Pot = 0.0;
      /* angle surface normal - LOS                                */
      Cosgamma =   lzero * SurfPtrP->l
	+          mzero * SurfPtrP->m
	+          nzero * SurfPtrP->n;

      /* IF (Cosgamma > 0) THEN Visible Side, Test for Eclipse     */
      if (Cosgamma <= 0) 
	eclipsed = Invisible;  /* invisible                        */
      if (Cosgamma >= DBL_EPSILON){   /* Visible Side              */
	eclipsed = Yes;

	/* test whether eclipse definitely impossible              */
	if (CosPhase <= (CosPhiLagrange-DBL_EPSILON)) eclipsed = No;

	if (Flags.fill == ON) {
	  if (i > Binary[Primary].NumElem) eclipsed = Yes;
	}

	/* IF (eclipsed == Yes) MORE TESTING                       */
	if (eclipsed == Yes) { 

	  /* intersect with circle of rad Xp2 centered on second   */
	  /* -- minimum sphere containing the star                 */
	  x_z = SurfPtrP->lambda;
	  y_z = SurfPtrP->mu;
	  z_z = SurfPtrP->nu;
	  Qi  = 2.0*( x_z*lzero + y_z*mzero + z_z*nzero - lzero);
	  Li  = 1.0 + x_z*x_z + y_z*y_z + z_z*z_z - 2*x_z - SqrXp2;
	  QLi = Qi*Qi - 4.0*Li;
	  if (QLi <= 0.0) {    /* No intersection                  */ 
	    eclipsed = No; 
	  } else {             /* Intersection, next test          */

            /* check whether distance is greater than polar radius */
            /* -- maximum inscribed sphere                         */
	    t_l = sqrt(QLi);    /* just to hold temporary result   */
	    t_1 = (-Qi + t_l)/2.0;
	    t_2 = (-Qi - t_l)/2.0;
	    t_l = (t_1 + t_2)/2.0; 
	    /* t_l is Midpoint of ray intersecting Xp2-Sphere      */
	    x_tl = x_z + t_l * lzero;
	    y_tl = y_z + t_l * mzero;
	    z_tl = z_z + t_l * nzero;
	    t_dist = SQR(x_tl-1.0) + y_tl * y_tl + z_tl * z_tl;

	    if (Flags.fill == ON) {
	      if (i > Binary[Primary].NumElem) { 
		eclipsed = Yes;
		t_1 = MAX(t_1, t_2);
		t_2 = t_1 + 1.0;
		t_dist =  SqrPol2+1.;
	      }
	    }

	    if (t_dist <= (SqrPol2-DBL_EPSILON)) { 

	      eclipsed = Yes;       /* definitely eclipsed         */

	     } else {


	       /* Find Potential Minimum between t_1 and t_2       */
               /* final resort                                     */
               /* changed Binary[Pri].Mq to Binary[Sec].Mq Jul 6   */
               MinErr = MinFinder(&Binary[Secondary].Mq, 
				  &fratio2, 
				  &t_1, &t_2, 
				  &lzero, &mzero, &nzero,
				  &x_z, &y_z, &z_z,
				  &tmin, &PotMin);
               if (MinErr == 1 && Flags.fill == OFF) 
		 WARNING(_("LightCurve: No Minimum Found"));
               PotMin = -PotMin; 
               SurfPtrP->LOS_Pot = (float) PotMin;

	       /* added condition tmin >= 0                        */
               if (PotMin >= Pot_Two && tmin >= 0) { 
                 eclipsed = Yes; 
	       } else {
                 eclipsed = No;
	       }

	     }
	  }
	}

	SurfPtrP->EclipseFlag = eclipsed;
	SurfPtrP->CosGamma    = (float) Cosgamma;
	if (eclipsed > 0) SurfPtrP->visibility = 0;
	else SurfPtrP->visibility = 1;

      } else {
	/* invisible side                                          */
	SurfPtrP->EclipseFlag = eclipsed;
	SurfPtrP->CosGamma    = -1;
	if (eclipsed > 0) SurfPtrP->visibility = 0;
	else SurfPtrP->visibility = 1;
      }
      ++SurfPtrP;
    }
    /* END PRIMARY                                                 */

    /* HERE SECONDARY                                              */
    if (i < NElemS) {

      SurfPtrS->Velocity = 
	- OmegaDotSecondary * Orbit.TrueDistance * Orbit.Dist
	* (SurfPtrS->mu     * lzero2
	   + SurfPtrS->lambda * mzero2);

      SurfPtrS->LOS_Pot = 0.0;
      /* angle surface normal - LOS                                */
      /*  dont forget to mirror in y->(-y)                         */
      Cosgamma =   lzero2 * SurfPtrS->l
	+          mzero2 * (-SurfPtrS->m)
	+          nzero2 * SurfPtrS->n;

      /* IF (Cosgamma > 0) THEN Visible Side, Test for Eclipse     */
      if (Cosgamma <= 0) 
	eclipsed = Invisible;  /* invisible                        */
      if (Cosgamma >= DBL_EPSILON){   /* Visible Side              */

	eclipsed = Yes;

	/* test whether eclipse definitely impossible              */
	if (CosPhase2 <= (CosPhiLagrange-DBL_EPSILON)) eclipsed = No; 

	if (Flags.fill == ON) {
	  if (i > Binary[Secondary].NumElem) eclipsed = Yes;
	}

	/* IF (eclipsed == Yes) MORE TESTING                       */ 
	if (eclipsed == Yes) { 

	  /* intersect with circle of rad Xp1 centered on prim     */
	  /* -- minimum sphere containing the star                 */
	  x_z = SurfPtrS->lambda;
	  y_z = (-SurfPtrS->mu);
	  z_z = SurfPtrS->nu;
	  Qi  = 2.0*( x_z*lzero2 + y_z*mzero2 + z_z*nzero2 - lzero2);
	  Li  = 1.0 + x_z*x_z + y_z * y_z + z_z*z_z - 2*x_z - SqrXp1;
	  QLi = Qi*Qi - 4.0*Li;
	  if (QLi <= 0.0) {    /* No intersection                  */ 
	    eclipsed = No; 
	  } else {             /* Intersection, next test          */

            /* check whether distance is greater than polar radius */
            /* -- maximum inscribed sphere                         */
	    t_l = sqrt(QLi);    /* just to hold temporary result   */
	    t_1 = (-Qi + t_l)/2.0;
	    t_2 = (-Qi - t_l)/2.0;
	    t_l = (t_1 + t_2)/2.0; 
	    /* t_l is Midpoint of ray intersecting Xp2-Sphere      */
	    x_tl = x_z + t_l * lzero2;
	    y_tl = y_z + t_l * mzero2;
	    z_tl = z_z + t_l * nzero2;
	    t_dist = SQR(x_tl-1.0) + y_tl*y_tl + z_tl*z_tl;

	    if (Flags.fill == ON) {
	      if (i > Binary[Secondary].NumElem) { 
		eclipsed = Yes;
		t_1 = MAX(t_1, t_2);
		t_2 = t_1 + 1.0;
		t_dist =  SqrPol1+1.;
	      }
	    }

	    /* we do not need the sqrt, compare the squares        */
	    if (t_dist <= (SqrPol1-DBL_EPSILON)) { 

	      eclipsed = Yes;       /* definitely eclipsed         */

	    } else {

	      /* Find Potential Minimum between t_1 and t_2        */
	      /* final resort                                      */
	      /* changed Binary[Sec].Mq to Binary[Pri].Mq Jul 6    */
	      MinErr = MinFinder(&Binary[Primary].Mq, 
				 &fratio1, 
				 &t_1, &t_2, 
				 &lzero2, &mzero2, &nzero2,
				 &x_z, &y_z, &z_z,
				 &tmin, &PotMin);
	      if (MinErr == 1) WARNING(_("LightCurve: No Minimum Found"));
	      PotMin = -PotMin; 
	      SurfPtrS->LOS_Pot = (float) PotMin;

	       /* added condition tmin >= 0                        */
	      if (PotMin >= (Pot_One-DBL_EPSILON) && tmin >= 0) { 
		eclipsed = Yes; 
	      } else {
		eclipsed = No;
	      }


	    }
	  }
	}

	SurfPtrS->EclipseFlag = eclipsed;
	SurfPtrS->CosGamma    = (float) Cosgamma;
	if (eclipsed > 0) SurfPtrS->visibility = 0;
	else SurfPtrS->visibility = 1;

      } else {
	SurfPtrS->EclipseFlag = eclipsed;
	SurfPtrS->CosGamma    = -1;
	if (eclipsed > 0) SurfPtrS->visibility = 0;
	else SurfPtrS->visibility = 1;
      }
      ++SurfPtrS;
    }
    /* END  SECONDARY                                              */


  } /* end loop over surface elements                              */

} /* end LightCurve                                                */


/********************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Flux computation
 @param     (int)  Comp  The stellar component 
 @param     (int)  Phase The phase index 
 @return    (void)
 @heading   Light Curve
 ********************************************************************/
void LightFlux(int Comp, int Phase)
{
  register long           i;                 /* loop counter        */
  double                  Cosgamma;          /* LOS angle           */
  double                  Common;            /* common part         */
  double                  Weight;            /* weight factor       */
  double                  SumOfVelocity;     /* normalization       */
  double                  SumOfWeight;       /* normalization       */
  double                  OneMinCosgamma;    /* 1.0 - Cosgamma      */
  SurfaceElement          *SurfPtr;          /* pointer to surface  */
  long                    nelem;             /* loop limit          */
  PhotoOut                *OFlux;            /* pointer to outflux  */

  /* ------------------ flux computing              --------------- */

  /* We start at 90 deg, Phase = 0.25, both stars fully visible     */

  /* loops over passbands are unrolled, all if's in loop eliminated */

  SumOfVelocity = 0.0;
  SumOfWeight   = 0.0;
  nelem         = Binary[Comp].NumElem;
  OFlux         = &FluxOut[Phase];
  SurfPtr       = Surface[Comp];

  if (Flags.limb == 0) { 
 
    for (i = 0; i < nelem; ++i) {      

    /* loop over surface elements                                   */
    /* add up Flux of visible elements                              */
    /* IF NOT eclipsed THEN                                         */

      if (SurfPtr->CosGamma >= 0.0) {
	Cosgamma = SurfPtr->CosGamma;
	Common   = SurfPtr->visibility 
	  * Cosgamma;
	OneMinCosgamma = (1.0 - Cosgamma);

	Weight = SurfPtr->f_[0] * Common
	    *(1.0 - Limb[Comp][0][0]*OneMinCosgamma);
	OFlux->Mag[0] = OFlux->Mag[0] + Weight;
	Weight = SurfPtr->f_[1] * Common
	    *(1.0 - Limb[Comp][1][0]*OneMinCosgamma);
	OFlux->Mag[1] = OFlux->Mag[1] + Weight;

	Weight = SurfPtr->f_[2] * Common
	    *(1.0 - Limb[Comp][2][0]*OneMinCosgamma);
	OFlux->Mag[2] = OFlux->Mag[2] + Weight;

	SumOfVelocity        = SumOfVelocity
	  + Weight * SurfPtr->Velocity;
	SumOfWeight          = SumOfWeight + Weight;

	Weight = SurfPtr->f_[3] * Common
	    *(1.0 - Limb[Comp][3][0]*OneMinCosgamma);
	OFlux->Mag[3] = OFlux->Mag[3] + Weight;
	Weight = SurfPtr->f_[4] * Common
	    *(1.0 - Limb[Comp][4][0]*OneMinCosgamma);
	OFlux->Mag[4] = OFlux->Mag[4] + Weight;
	Weight = SurfPtr->f_[5] * Common
	    *(1.0 - Limb[Comp][5][0]*OneMinCosgamma);
	OFlux->Mag[5] = OFlux->Mag[5] + Weight;
	Weight = SurfPtr->f_[6] * Common
	    *(1.0 - Limb[Comp][6][0]*OneMinCosgamma);
	OFlux->Mag[6] = OFlux->Mag[6] + Weight;
	Weight = SurfPtr->f_[7] * Common
	    *(1.0 - Limb[Comp][7][0]*OneMinCosgamma);
	OFlux->Mag[7] = OFlux->Mag[7] + Weight;
	Weight = SurfPtr->f_[8] * Common
	    *(1.0 - Limb[Comp][8][0]*OneMinCosgamma);
	OFlux->Mag[8] = OFlux->Mag[8] + Weight;
	Weight = SurfPtr->f_[9] * Common
	    *(1.0 - Limb[Comp][9][0]*OneMinCosgamma);
	OFlux->Mag[9] = OFlux->Mag[9] + Weight;
	Weight = SurfPtr->f_[10] * Common
	    *(1.0 - Limb[Comp][10][0]*OneMinCosgamma);
	OFlux->Mag[10] = OFlux->Mag[10] + Weight;
	Weight = SurfPtr->f_[11] * Common
	    *(1.0 - Limb[Comp][10][0]*OneMinCosgamma);
	OFlux->Mag[11] = OFlux->Mag[11] + Weight;
	
      }
      ++SurfPtr;
    }
  }

  else if ( Flags.limb == 1) {
    for (i = 0; i < nelem; ++i) {      

    /* loop over surface elements                                   */
    /* add up Flux of visible elements                              */
    /* IF NOT eclipsed THEN                                         */

      if (SurfPtr->CosGamma >= 0.0) {

	Cosgamma = SurfPtr->CosGamma;
	Common   = SurfPtr->visibility 
	  * Cosgamma;
	OneMinCosgamma = (1.0 - Cosgamma);

	Weight = SurfPtr->f_[0] * Common
	    *(1.0 - Limb[Comp][0][0]*OneMinCosgamma
	      - Limb[Comp][0][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[0] = OFlux->Mag[0] + Weight;
	Weight = SurfPtr->f_[1] * Common
	    *(1.0 - Limb[Comp][1][0]*OneMinCosgamma
	      - Limb[Comp][1][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[1] = OFlux->Mag[1] + Weight;
	Weight = SurfPtr->f_[2] * Common
	    *(1.0 - Limb[Comp][2][0]*OneMinCosgamma
	      - Limb[Comp][2][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[2] = OFlux->Mag[2] + Weight;

	SumOfVelocity        = SumOfVelocity
	  + Weight * SurfPtr->Velocity;
	SumOfWeight          = SumOfWeight + Weight;

	Weight = SurfPtr->f_[3] * Common
	    *(1.0 - Limb[Comp][3][0]*OneMinCosgamma
	      - Limb[Comp][3][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[3] = OFlux->Mag[3] + Weight;
	Weight = SurfPtr->f_[4] * Common
	    *(1.0 - Limb[Comp][4][0]*OneMinCosgamma
	      - Limb[Comp][4][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[4] = OFlux->Mag[4] + Weight;
	Weight = SurfPtr->f_[5] * Common
	    *(1.0 - Limb[Comp][5][0]*OneMinCosgamma
	      - Limb[Comp][5][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[5] = OFlux->Mag[5] + Weight;
	Weight = SurfPtr->f_[6] * Common
	    *(1.0 - Limb[Comp][6][0]*OneMinCosgamma
	      - Limb[Comp][6][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[6] = OFlux->Mag[6] + Weight;
	Weight = SurfPtr->f_[7] * Common
	    *(1.0 - Limb[Comp][7][0]*OneMinCosgamma
	      - Limb[Comp][7][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[7] = OFlux->Mag[7] + Weight;
	Weight = SurfPtr->f_[8] * Common
	    *(1.0 - Limb[Comp][8][0]*OneMinCosgamma
	      - Limb[Comp][8][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[8] = OFlux->Mag[8] + Weight;
	Weight = SurfPtr->f_[9] * Common
	    *(1.0 - Limb[Comp][9][0]*OneMinCosgamma
	      - Limb[Comp][9][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[9] = OFlux->Mag[9] + Weight;
	Weight = SurfPtr->f_[10] * Common
	    *(1.0 - Limb[Comp][10][0]*OneMinCosgamma
	      - Limb[Comp][10][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[10] = OFlux->Mag[10] + Weight;
	Weight = SurfPtr->f_[11] * Common
	    *(1.0 - Limb[Comp][11][0]*OneMinCosgamma
	      - Limb[Comp][11][1]*(1.0- Cosgamma*Cosgamma));
	OFlux->Mag[11] = OFlux->Mag[11] + Weight;
	
      }
      ++SurfPtr;
    }
  }

  else if ( Flags.limb == 2) {
    for (i = 0; i < nelem; ++i) {      

    /* loop over surface elements                                   */
    /* add up Flux of visible elements                              */
    /* IF NOT eclipsed THEN                                         */

      if (SurfPtr->CosGamma >= 0.0) {

	Cosgamma = SurfPtr->CosGamma;
	Common   = SurfPtr->visibility 
	  * Cosgamma;
	OneMinCosgamma = (1.0 - Cosgamma);

	Weight = SurfPtr->f_[0] * Common
	    *(1.0 - Limb[Comp][0][0]*OneMinCosgamma
	      - Limb[Comp][0][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[0] = OFlux->Mag[0] + Weight;
	Weight = SurfPtr->f_[1] * Common
	    *(1.0 - Limb[Comp][1][0]*OneMinCosgamma
	      - Limb[Comp][1][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[1] = OFlux->Mag[1] + Weight;
	Weight = SurfPtr->f_[2] * Common
	    *(1.0 - Limb[Comp][2][0]*OneMinCosgamma
	      - Limb[Comp][2][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[2] = OFlux->Mag[2] + Weight;

	SumOfVelocity        = SumOfVelocity
	  + Weight * SurfPtr->Velocity;
	SumOfWeight          = SumOfWeight + Weight;

	Weight = SurfPtr->f_[3] * Common
	    *(1.0 - Limb[Comp][3][0]*OneMinCosgamma
	      - Limb[Comp][3][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[3] = OFlux->Mag[3] + Weight;
	Weight = SurfPtr->f_[4] * Common
	    *(1.0 - Limb[Comp][4][0]*OneMinCosgamma
	      - Limb[Comp][4][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[4] = OFlux->Mag[4] + Weight;
	Weight = SurfPtr->f_[5] * Common
	    *(1.0 - Limb[Comp][5][0]*OneMinCosgamma
	      - Limb[Comp][5][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[5] = OFlux->Mag[5] + Weight;
	Weight = SurfPtr->f_[6] * Common
	    *(1.0 - Limb[Comp][6][0]*OneMinCosgamma
	      - Limb[Comp][6][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[6] = OFlux->Mag[6] + Weight;
	Weight = SurfPtr->f_[7] * Common
	    *(1.0 - Limb[Comp][7][0]*OneMinCosgamma
	      - Limb[Comp][7][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[7] = OFlux->Mag[7] + Weight;
	Weight = SurfPtr->f_[8] * Common
	    *(1.0 - Limb[Comp][8][0]*OneMinCosgamma
	      - Limb[Comp][8][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[8] = OFlux->Mag[8] + Weight;
	Weight = SurfPtr->f_[9] * Common
	    *(1.0 - Limb[Comp][9][0]*OneMinCosgamma
	      - Limb[Comp][9][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[9] = OFlux->Mag[9] + Weight;
	Weight = SurfPtr->f_[10] * Common
	    *(1.0 - Limb[Comp][10][0]*OneMinCosgamma
	      - Limb[Comp][10][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[10] = OFlux->Mag[10] + Weight;
	Weight = SurfPtr->f_[11] * Common
	    *(1.0 - Limb[Comp][11][0]*OneMinCosgamma
	      - Limb[Comp][11][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[11] = OFlux->Mag[11] + Weight;
	
      }
      ++SurfPtr;
    }
  }
  else if ( Flags.limb == 3) {
    for (i = 0; i < nelem; ++i) {      

    /* loop over surface elements                                   */
    /* add up Flux of visible elements                              */
    /* IF NOT eclipsed THEN                                         */

      if (SurfPtr->CosGamma >= 0.0) {

	Cosgamma = SurfPtr->CosGamma;
	Common   = SurfPtr->visibility 
	  * Cosgamma;
	OneMinCosgamma = (1.0 - Cosgamma);

	Weight = SurfPtr->f_[0] * Common
	    *(1.0 - SurfPtr->LD[0][0]*OneMinCosgamma
	      - SurfPtr->LD[0][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[0] = OFlux->Mag[0] + Weight;
	Weight = SurfPtr->f_[1] * Common
	    *(1.0 - SurfPtr->LD[1][0]*OneMinCosgamma
	      - SurfPtr->LD[1][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[1] = OFlux->Mag[1] + Weight;
	Weight = SurfPtr->f_[2] * Common
	    *(1.0 - SurfPtr->LD[2][0]*OneMinCosgamma
	      - SurfPtr->LD[2][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[2] = OFlux->Mag[2] + Weight;

	SumOfVelocity        = SumOfVelocity
	  + Weight * SurfPtr->Velocity;
	SumOfWeight          = SumOfWeight + Weight;

	Weight = SurfPtr->f_[3] * Common
	    *(1.0 - SurfPtr->LD[3][0]*OneMinCosgamma
	      - SurfPtr->LD[3][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[3] = OFlux->Mag[3] + Weight;
	Weight = SurfPtr->f_[4] * Common
	    *(1.0 - SurfPtr->LD[4][0]*OneMinCosgamma
	      - SurfPtr->LD[4][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[4] = OFlux->Mag[4] + Weight;
	Weight = SurfPtr->f_[5] * Common
	    *(1.0 - SurfPtr->LD[5][0]*OneMinCosgamma
	      - SurfPtr->LD[5][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[5] = OFlux->Mag[5] + Weight;
	Weight = SurfPtr->f_[6] * Common
	    *(1.0 - SurfPtr->LD[6][0]*OneMinCosgamma
	      - SurfPtr->LD[6][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[6] = OFlux->Mag[6] + Weight;
	Weight = SurfPtr->f_[7] * Common
	    *(1.0 - SurfPtr->LD[7][0]*OneMinCosgamma
	      - SurfPtr->LD[7][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[7] = OFlux->Mag[7] + Weight;
	Weight = SurfPtr->f_[8] * Common
	    *(1.0 - SurfPtr->LD[8][0]*OneMinCosgamma
	      - SurfPtr->LD[8][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[8] = OFlux->Mag[8] + Weight;
	Weight = SurfPtr->f_[9] * Common
	    *(1.0 - SurfPtr->LD[9][0]*OneMinCosgamma
	      - SurfPtr->LD[9][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[9] = OFlux->Mag[9] + Weight;
	Weight = SurfPtr->f_[10] * Common
	    *(1.0 - SurfPtr->LD[10][0]*OneMinCosgamma
	      - SurfPtr->LD[10][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[10] = OFlux->Mag[10] + Weight;
	Weight = SurfPtr->f_[11] * Common
	    *(1.0 - SurfPtr->LD[11][0]*OneMinCosgamma
	      - SurfPtr->LD[11][1]*(1.0- sqrt(Cosgamma)));
	OFlux->Mag[11] = OFlux->Mag[11] + Weight;
	
      }
      ++SurfPtr;
    }
  }

  /* ------------  normalize velocity ----------------------        */

  if (SumOfWeight >= DBL_EPSILON) {
    FluxOut[Phase].D_RV[Comp] = (float) (SumOfVelocity / SumOfWeight);
  } else {
    FluxOut[Phase].D_RV[Comp] = 0.0;
  }

  FluxOut[Phase].RV[Comp] = 
    FluxOut[Phase].D_RV[Comp] + FluxOut[Phase].RV[Comp];

  return;
}

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Find minimum of the potential along some line-of-sight vector.
 @long      Uses Brents algorithm, which is a combination of golden section 
            search and parabolic interpolation.
            Adopted from procedure 'localmin' in: Richard Brent, Algorithms for
            minimization without derivatives, Prentice-Hall, Inc. (1973)
 @tip       Input is a parametric form of LOS vector + brackets.
 @param     (double*) q  Mass ratio
 @param     (double*) ff Async rotation
 @param     (double*) t1 Minimum bracket
 @param     (double*) t3 Minimum bracket
 @param     (double*) l0 x-component LOS vector
 @param     (double*) m0 y-component LOS vector
 @param     (double*) n0 z-component LOS vector
 @param     (double*) x0 x-component of point on LOS vector
 @param     (double*) y0 y-component of point on LOS vector
 @param     (double*) z0 z-component of point on LOS vector
 @param     (double*)  tmin The position of the minimum
 @param     (double*)  Cmin The LOS potential minimum
 @return    (int)    The error status
 @heading   Light Curve
 ****************************************************************************/
int MinFinder(double *q, double *ff, double *t1, double *t3, 
               double *l0, double *m0, double *n0,
               double *x0, double *y0, double *z0,
	      double *tmin, /*@out@*/ double *Cmin)
{
  register      int i;                 /* loop variable                     */
  double        t2;                    /* middle value                      */
  double        tmp;                   /* temp var                          */
  double        step, stepstore;       /* stepsizes                         */
  double        Fe = 0.0, Ff = 0.0;    /* function values                   */
  double        Fg = 0.0, Fh = 0.0;    /* function values                   */
  double        e = 0.0, f, g, h;      /* positions (parametric)            */
  double        delta1, delta2;        /* test convergence                  */
  double        Start, End, Middle;    /* brackets                          */
  double        nextstep = 0.0;        /* stepsize                          */
  double        x, y, z;               /* position in cartesian coordinates */
  double        P1 = 0.0, P2, P3;      /* fit parabola                      */ 
  double        precise = 0.01;        /* convergence criterion             */

  if ( (*t1) <= ((*t3)-DBL_EPSILON) ) { 
    Start = (*t1); 
    End = (*t3); 
  } else { 
    Start = (*t3); 
    End = (*t1); 
  }

  step = 0.0;

  /* we have a parametric description of a vector X(h)                      */
  /* we want to find min(func) = min(X(h)) as function of h                 */

  t2 = 0.62*(*t1) + 0.38*(*t3);

  /* initial value of X(h)                                                  */
  x = 1.0 - ((*x0) + (*l0)*(t2));  
  y =       -(*y0) - (*m0)*(t2);
  z =        (*z0) + (*n0)*(t2);

  /* NEW jul 6,98 -- change of coordinate system, calculate                 */
  /* in system of eclipsing star                                            */
  /* (i.e. x = 1.0 - x; y = -y; z = z;)                                     */

  /* t2 is the initial guess for the minimum between t1 and t3              */
  h = g = f = t2;

  /* initial value of F                                                     */
  tmp = 1.0-x; 
  tmp = sqrt(tmp*tmp + y*y + z*z );
  Ff = Fg = Fh = -(  1.0/sqrt(x * x + y * y + z * z) 
                   + (*q) * (1.0/tmp - x)
                   + ( x * x + y * y ) * (*ff)*(*ff) * (0.5*( (*q) + 1.0)));

  /* we do a maximum of MAXITER iterations - convergence usually            */
  /*  after less than ten iterations                                        */
  /*@i@*/ for (i = MAXITER; i; --i) {

    Middle = 0.5 * (Start + End);

    delta1 = precise * fabs(h) + DBL_EPSILON;
    delta2 = 2.0 * delta1;

    /* test for convergence                                                 */
    /* RETURN if converged                                                  */

    if (fabs(h-Middle) <= (delta2 - 0.5 * (End-Start) ) ) {
      *tmin = h;
      *Cmin = Fh;
      return(0);
    }

    if (fabs(step) >= delta1) {

      /* fit parabola                                                       */
      P3 = (h-g)*(Fh-Ff);
      P2 = (h-f)*(Fh-Fg);
      P1 = (h-f)*P2 - (h-g)*P3;
      P2 = 2.0 * (P2 - P3);

      if (P2 >= DBL_EPSILON) { 
        P1 = -P1; 
      } else {
        P2 = -P2;
      }

      stepstore = step;
      step = nextstep;

      if ( (fabs(P1) >= fabs(0.5 * P2 * stepstore)) 
           || (P1 <= P2*(Start-h)) 
           || (P1 >= P2*(End-h))) {

        /* a golden-section step                                            */
        if (h >= Middle) { 
          step = (Start-h); 
        } else {  
          step = (End-h);
        }
        nextstep = 0.381966 * step;

      } else {

        /* do a parabolic interpolation step                                */
        nextstep = P1/P2;
        e        = h + nextstep;

        /* f must not be evaluated too close to Start or End                */
        if ((e-Start) <= delta2 || (End-e) <= delta2) {
          if ((Middle-h) <= (-DBL_EPSILON)) { 
            nextstep = nextstep-delta1; 
          } else { 
            nextstep = nextstep+delta1;
          }
        }

      }

    } else {

      /* a golden-section step                                              */
      if (h >= Middle) { 
        step = (Start-h); 
      } else {  
        step = (End-h);
      }
      nextstep =  0.381966 * step;
      
    }

    if (fabs(nextstep) >= delta1) { 
      e = h + nextstep; 
    } else {
      /* f must not be evaluated too close                                  */
      if ( nextstep <= (-DBL_EPSILON) ) { 
        e = h - delta1; 
      } else { 
        e = h + delta1; 
      }
    }

    /* e ist the current best guess for the minimum                         */
    /* calculate X(e)                                                       */
    x = 1.0 - ((*x0) + (*l0) * e);  
    y =     -  (*y0) - (*m0) * e;  
    z =        (*z0) + (*n0) * e;

    /* NEW jul 6 -- change of coordinate system, calculate                  */
    /* in system of eclipsing star                                          */
    /* i.e x = 1.0 - x;  y = -y;  z = z;                                    */

    /* calculate f(X(e))                                                    */
    tmp = 1.0-x;
    tmp = sqrt(tmp*tmp + y*y + z*z );
    Fe         = -(  1.0/sqrt(x * x + y * y + z * z) 
                   + (*q) * (1.0/tmp - x)
                   + ( x * x + y * y )* (*ff)*(*ff) * (0.5*( (*q) + 1.0)));

    /* update                                                               */
    if (Fe <= Fh) {
      if ( e >= h ) Start = h; 
      else End = h;
      f  =  g;  g =  h;  h = e;
      Ff = Fg; Fg = Fh; Fh = Fe;
    } else {
      if ( e <= (h-DBL_EPSILON) ) Start = e; 
      else End = e;
      if ( (Fe <= Fg) || (fabs (g - x)  <= FLT_EPSILON) ) {
        f = g; g = e; 
        Ff = Fg; Fg = Fe;
      } 
      else if ( (Fe <= Ff) || (fabs (f- g) <= FLT_EPSILON) 
                || (fabs (f - h) <= FLT_EPSILON) ) { 
          f = e; Ff = Fe;
      }
    }

  /* end of main loop                                                       */
  }

  /* only reach here if no convergence                                      */
  *tmin = h;     
  *Cmin = Fh;
  return(1);
}










