/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include "Light.h"


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Simple reflection treatment (point source)
 @tip       Includes penumbral correction
 @param     (int) Comp  The stellar component
 @return    (void)   
 @heading   Light Curve
 ****************************************************************************/
void SimpleReflect(int Comp)
{

  double     penumbral_corr, penumbral_h;  /* penumbral correction          */
  double     penumbral_a, penumbral_alpha; /* penumbral correction          */
  double     penumbral_sin;                /* penumbral correction          */
  double     cosalpha, sinalpha, sumsin;   /* reflection angle              */
  double     lvec, mvec, nvec;             /* surface normal                */
  double     T1, T2, R2, Tr, rho;          /* blackbody                     */
  long       NumElem;                      /* loop counter                  */
  SurfaceElement *SurfPtr;                 /* surface pointer               */
  BinaryComponent *BinPtr;                 /* star pointer                  */

  T1 = 5000.;

  SurfPtr = Surface[Comp];
  BinPtr  = &Binary[Comp];

  if (Binary[Primary].RocheFill >= 0.3 || Binary[Secondary].RocheFill >= 0.3) 
    WARNING(_("Simple Reflection not recommended for large fill factors"));
         
  if (Comp == 0) { 
    T2 =  Binary[Comp+1].Temperature;
    R2 =  Binary[Comp+1].Radius;
  } else {
    T2 =  Binary[Comp-1].Temperature;
    R2 =  Binary[Comp-1].Radius;
  }

  for (NumElem = 0; NumElem < BinPtr->NumElem; ++NumElem) {

    rho  = SurfPtr->rho;
    lvec = SurfPtr->l;
    mvec = SurfPtr->m;
    nvec = SurfPtr->n;

    /* Surface[Comp][NumElem].lambda == rad1 * lambda                       */

    cosalpha   = rho * 
                    ( lvec * ( 1.0-SurfPtr->lambda) 
                            - SurfPtr->mu * mvec 
                            - SurfPtr->nu * nvec);
    sinalpha   = sqrt(1.0 - SQR(cosalpha));

    penumbral_sin  = R2*rho;
    penumbral_corr = sqrt(1.0-SQR(penumbral_sin));
    sumsin = sinalpha + penumbral_sin;

    if (cosalpha <= (-DBL_EPSILON) && sumsin <= (1.0-DBL_EPSILON)) {    

      /* fully invisible, do nothing                                        */
      ;

    } 
    else if (cosalpha >= 0 && sumsin <= (1.0-DBL_EPSILON)) {  

      /* fully visible                                                      */

      T1 = SurfPtr->temp;

      Tr = T2/T1; Tr = Tr * Tr * Tr * Tr;

      SurfPtr->temp = T1 * 
	sqrt(sqrt(1.0 + BinPtr->Albedo * 0.5 * cosalpha * 
		  (1.0 - sqrt(1.0 - SQR(R2*rho)))*Tr));

    } 
    else { 

      T1 = SurfPtr->temp;

      Tr = T2/T1; Tr = Tr * Tr * Tr * Tr;

      /* partially visible, apply penumbral correction                      */
      penumbral_h = 
	(sumsin - 1.)/penumbral_sin;
      if (cosalpha >= 0.0) { 

	/* we see more than half the star                                   */

	penumbral_corr = PI + PI;  /* set to full disk                      */

	/* we calculate the segment of a disk, r=1.0                        */
	penumbral_a = sqrt(penumbral_h+penumbral_h - SQR(penumbral_h));
	/* Okt 14 15:38:27 MEST 1999 -- fix rounding error */
	penumbral_a = MIN(1.0, penumbral_a);
	penumbral_a = MAX(-1.0, penumbral_a);
	penumbral_alpha = asin( CLAMP(penumbral_a, -1.0, 1.0) );

	/* subtract segment                                                 */
	penumbral_corr = penumbral_corr 
	  - (penumbral_alpha + penumbral_alpha)
	  + penumbral_alpha * (1.0 - penumbral_h);

	penumbral_corr = penumbral_corr / (2.0 * PI);

	SurfPtr->temp = T1 *
	  sqrt(sqrt(1.0 + 
		    penumbral_corr * BinPtr->Albedo * 0.5 
		    * (cosalpha + 0.5 * (sumsin - 1.)) * 
		    (1.0 - sqrt(1.0 - SQR(R2*rho)))*Tr)); 


      } else { 

	/* we see less than half the star                                  */
	/* we calculate the segment of a disk, r=1.0                       */

	penumbral_a = sqrt(2.0*penumbral_h - SQR(penumbral_h));
	/* Okt 14 15:38:27 MEST 1999 -- fix rounding error */
	penumbral_a = MIN(1.0, penumbral_a);
	penumbral_a = MAX(-1.0, penumbral_a);
	penumbral_alpha = asin( CLAMP(penumbral_a, -1.0, 1.0) );
	penumbral_corr = penumbral_alpha + penumbral_alpha
	  - penumbral_alpha * (1.0 - penumbral_h);
	penumbral_corr = penumbral_corr / (2.0 * PI);

	SurfPtr->temp = T1 *  
	  sqrt(sqrt(1.0 + 
		    penumbral_corr * BinPtr->Albedo  
		    * 0.5 * 0.5 * (sumsin - 1. ) * 
		    (1.0 - sqrt(1.0 - SQR(R2*rho)))*Tr));  
      }
    }

    if (SurfPtr->temp > DBL_MAX || SurfPtr->temp < DBL_MIN)
      SurfPtr->temp = T1;

    ++SurfPtr;
  }
}



/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Detailed reflection treatment (mutual irradiation)
 @param     (void) 
 @return    (void)   
 @heading   Light Curve
 ****************************************************************************/
void  LightReflect()
{

  static  double  SumDelta_[2][MAXELEMENTS];  /* delta flux                 */
  static  double  Temp_[2][MAXELEMENTS];      /* temperature                */


  static  double  sqtab[1001] = { 0.0 };      /* sqrt hash table            */
  static  double  u1[2], u2[2];               /* limb darkening             */
  static  double  NormLimb_[2];               /* normalization              */

  long   i, j, k;                             /* loop counter               */
  long   NumLarge, NumSmall;                  /* # of surface elements      */
  int    LargeStar, SmallStar;                /* the stars                  */
  double *Init_ptr;                           /* used to init SumDelta_     */
  double dx, dy, dz;                          /* distance                   */
  double cosgamma_ji, cosgamma_ij;            /* viewing angles             */
  double distance_ij;                         /* distance                   */
  double Delta_F, Delta_Ii, Delta_Ij;         /* deltas                     */
  double Tn, Tsmall, Tlarge;                  /* temperatures               */
  int    Nmax, Nnow;                          /* where do we stand ?        */
  SurfaceElement *SurfPtrS;                   /* surface pointer            */
  SurfaceElement *SurfPtrL;                   /* surface pointer            */
#ifdef _WITH_GTK
  char   message[64];                         /* statusbar message          */
#endif


  /* Bolometric limb darkening coefficients log_g=4.0 sqrt law              */
  /* Van Hamme W. 1993, AJ 106, 2096                                        */
static
  float  T[55] = { 3500, 3750, 4000, 4250, 4500, 4750, 5000, 5250, 5500,
                   5750, 6000, 6250, 6500, 6750, 7000, 7250, 7500, 7750,
                   8000, 8250, 8500, 8750, 9000, 9250, 9500, 9750,10000,
                  10500,11000,11500,12000,12500,13000,14000,15000,16000,
                  17000,18000,19000,20000,21000,22000,23000,24000,25000,
         	  26000,27000,28000,29000,30000,31000,32000,33000,34000,35000};
static
  float lu1[55] ={-.079,0.078,0.238,0.303,0.319,0.313,0.290,0.259,0.223,
                  0.189,0.159,0.137,0.116,0.100,0.086,0.077,0.248,0.233,
                  0.248,0.309,0.356,0.396,0.439,0.476,0.502,0.520,0.533,
                  0.549,0.558,0.565,0.572,0.580,0.587,0.597,0.601,0.599,
                  0.592,0.582,0.570,0.554,0.537,0.515,0.491,0.463,0.430,
                  0.393,0.353,0.312,0.276,0.252,0.243,0.243,0.247,0.256,0.280};
static
  float lu2[55] ={0.668,0.556,0.425,0.372,0.361,0.376,0.408,0.447,0.489,
                  0.527,0.559,0.581,0.603,0.621,0.638,0.651,0.488,0.507,
                  0.488,0.410,0.348,0.297,0.249,0.212,0.190,0.180,0.175,
                  0.173,0.177,0.179,0.177,0.174,0.171,0.170,0.174,0.184,
                  0.197,0.211,0.226,0.242,0.260,0.280,0.303,0.329,0.360,
                  0.395,0.434,0.471,0.500,0.512,0.507,0.496,0.483,0.467,0.435};

  SurfPtrS = Surface[0];
  SurfPtrL = Surface[1];

 if (Flags.first_pass == ON) {

   /* create hash table for square root                                     */
   for (i = 0; i < 1001; ++i) {
     sqtab[i] = 1.0 - (sqrt( (float)(i) / 1000. ));
   }

   /* limb darkening coefficients                                           */
   for (i = 0; i < 2; ++i) {
     Tn = Binary[i].Temperature;
     if (Tn <= T[0]) {  
          u1[i] = lu1[0]; u2[i] = lu2[0];
     } 
     else if (Tn >= T[54]) {
             u1[i] = lu1[54]; u2[i] = lu2[54];
	  } 
     else {
       for (j = 0; j < 54; ++j) { 
	 if ( Tn >= T[j]  && Tn <= (T[j+1]-DBL_EPSILON) ) { 
	   u1[i] = lu1[j]; u2[i] = lu2[j];
	 }
       }
     }
   }

  /* normalization for limb darkening coefficients */
  NormLimb_[Primary]   = 1.0/(PI*(1.0- u1[Primary]/3.0   - u2[Primary]/5.0)); 
  NormLimb_[Secondary] = 1.0/(PI*(1.0- u1[Secondary]/3.0 - u2[Secondary]/5.0));

  } /* end first_pass                                                       */


    
  /* arrange properly for simple geometric test                             */
          
  if (Binary[Primary].Radius >= Binary[Secondary].Radius) {
         LargeStar = Primary;
         SmallStar = Secondary;
  } else {
         LargeStar = Secondary;
         SmallStar = Primary;
  }             

  NumLarge = Binary[LargeStar].NumElem;
  NumSmall = Binary[SmallStar].NumElem;

  /* SmallStar is mirrored in y-z , ie. x -> (1-x)                          */
  

  /* invisible side reached after Nmax non-sightings of other star          */

  Nmax = (4*StepsPerPi);

  SurfPtrS = Surface[0];
  SurfPtrL = Surface[1];

  for (i = 0; i < MAXELEMENTS; ++i) {
    Temp_[0][i] = SurfPtrS->temp;  ++SurfPtrS;
    Temp_[1][i] = SurfPtrL->temp;  ++SurfPtrL;
  }

  for (k = 0; k < Flags.reflect; ++k) {       /* multiple reflections       */

#ifdef _WITH_GTK
    if (Flags.interactive == ON) {
      sprintf(message, _("Reflection iteration %1ld"), k); 
      my_appbar_push(message);
    }
#endif


    /* initialize the array for irradiated fluxes                           */
    Init_ptr = &SumDelta_[0][0];
    for (i = 0; i < 2*MAXELEMENTS; ++i) { 
      *Init_ptr = 0.0;
      ++Init_ptr;
    }
            
    SurfPtrL = &Surface[LargeStar][(int)(NumLarge/3)];

 
    /* -------- outer loop            ---------                             */

    for (i = (NumLarge/3); i < NumLarge; ++i) {

      /* simple geometric test, do we look towards other star ?             */
      if (SurfPtrL->lambda >= FLT_EPSILON) {  /* yes, we do                 */

	j = NumSmall - 1;  Nnow = 0;

        SurfPtrS = &Surface[SmallStar][j];

	while ( ( j >= 0 ) && ( Nnow <= Nmax ) ) {

	  dx = SurfPtrL->lambda 
	    - (1.0 - SurfPtrS->lambda);
	  dy = SurfPtrL->mu
	    - SurfPtrS->mu;
	  dz = SurfPtrL->nu
	    - SurfPtrS->nu;

	  /* we only need the correct sign, thus we do not divide           */
	  /* by the norm of (ri-rj)                                         */
	  /* thus cosgamma is really cosgamma*distance here                 */
	  cosgamma_ji = ( (-SurfPtrS->l) * dx
			  + SurfPtrS->m  * dy
			  + SurfPtrS->n  * dz  );
	  if (cosgamma_ji >= DBL_EPSILON) {

	    cosgamma_ij = (SurfPtrL->l * (-dx)
			   + SurfPtrL->m * (-dy)
			   + SurfPtrL->n * (-dz) );

	    if (cosgamma_ij >= DBL_EPSILON) { 
 
	      /* ---------  visible, proceed ----------------------------   */

	      Nnow = 0;   /* reset Nnow                                     */

	      /* we only need the square of the distance                    */
	      distance_ij = (dx*dx) + (dy*dy) + (dz*dz);

	      /* ummm .. need distance for cosgamma because of              */
	      /*          the limb darkening ...                            */
	      distance_ij = sqrt(distance_ij);
	      cosgamma_ji = cosgamma_ji/distance_ij;
	      cosgamma_ij = cosgamma_ij/distance_ij;
 
	      /* in case of elliptic orbit                                  */
	      if (Flags.elliptic == ON)
		distance_ij = distance_ij * Orbit.Dist;

	      Delta_F     = (cosgamma_ji * cosgamma_ij)
		/ SQR(distance_ij);


	      /* irradiation onto LargeStar                                 */
	      Tsmall = SurfPtrS->temp; 
	      Delta_Ii    = Delta_F  
		* SurfPtrS->area 
		* (1.0 - u1[SmallStar]*(1.0-cosgamma_ji) 
		   - u2[SmallStar]*
		   sqtab[(int)(1000*cosgamma_ji)])
		* (Tsmall*Tsmall*Tsmall*Tsmall);

	      /* irradiation on SmallStar                                   */
	      Tlarge = SurfPtrL->temp; 
	      Delta_Ij    = Delta_F 
		* SurfPtrL->area 
		* (1.0 - u1[LargeStar]*(1.0-cosgamma_ij) 
		   - u2[LargeStar]*
		   sqtab[(int)(1000*cosgamma_ij)])
		* (Tlarge*Tlarge*Tlarge*Tlarge);

	      SumDelta_[LargeStar][i] = SumDelta_[LargeStar][i] + Delta_Ii;
	      SumDelta_[SmallStar][j] = SumDelta_[SmallStar][j] + Delta_Ij;

	    } else {
	      ++Nnow;
	    }
	  } else {
	    ++Nnow;
	  }

	  --j;
	  --SurfPtrS;

	}  /* end inner loop over surface elements                          */

      }    /* end if condition                                              */
      ++SurfPtrL;
    }      /* end outer loop over surface elements                          */


    /* ----         T(new)**4  =  T(old)**4 + A * F(irr)   ---------------  */

    for (i = 0; i < Binary[Primary].NumElem; ++i) {
         Surface[Primary][i].temp = 
          sqrt(sqrt(   
             ( Temp_[Primary][i] * Temp_[Primary][i] 
	       * Temp_[Primary][i] * Temp_[Primary][i] )
               + NormLimb_[Secondary] * Binary[Primary].Albedo 
                  * SumDelta_[Primary][i]
               ));   
       }


    for (i = 0; i < Binary[Secondary].NumElem; ++i) {
         /* printf("N %4ld T %8.2f ", i, Surface[Secondary][i].temp); */
         Surface[Secondary][i].temp = 
          sqrt(sqrt(   
                  ( Temp_[Secondary][i] * Temp_[Secondary][i]
		    * Temp_[Secondary][i] * Temp_[Secondary][i] )
                   + NormLimb_[Primary] * Binary[Secondary].Albedo 
                      * SumDelta_[Secondary][i]
               )); 
    }   
  
  }  /* end multiple reflections */
}








