/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Light.h"


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Search index for temperature in array
 @param     (float)  key    temperature
 @param     (float *) array temperature array
 @param     (int)  n        size of array
 @return    (int)  the left index of the two bracketing values       
 @heading   Light Curve
 ****************************************************************************/
static int search_lkey (float key, float * r, int n )
{ 
  static int i = 0;
  static int high, low;
  
  /* speedup (?)
   */
  if (key >= r[i] && key < r[i+1])
    {
      return ( i );
    }

  /* return 0 if < r[0]; 
   */
  for ( low = (-1), high = n;  high-low > 1;  )
    {
      i = (high+low) / 2;
      if ( key < r[i])  
	high = i;
      else             
	low  = i;
    }
  i = (i < 0) ? 0 : i;
  return ( i );
}


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Compute limb darkening coefficients 
 @long      Limb darkening coefficients 
            from Diaz-Cordoves J., Claret A., Gimenez A. 1995, 
            Astronomy & Astrophysics  Supplement 110, 329 
            and Claret A., Diaz-Cordoves J., Gimenez A. 1995,
            Astronomy & Astrophysics Supplement 114, 247
 @tip       Constant log(g) = 4.0  (but not very sensitive to log g)
 @param     (int) Comp The stellar component
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/

static void FindbestLC ();

void LightLimbDark(int Comp)
{

  float    Tn, dy1, dy2, dx;                 /* temperature, delta temp     */
  register long    j, i;                     /* loop variables              */ 
  int      band;                             /* passband                    */
  int      hkey;                             /* hash key                    */
  double   log_g = 4.0;                      /* log g                       */
  SurfaceElement  *SurfPtr;                  /* pointer to surface          */

  /*  stellar temperature grid for limb darkening coefficients              */

static
  float  T[55] = { 3500, 3750, 4000, 4250, 4500, 4750, 5000, 5250, 5500,
                   5750, 6000, 6250, 6500, 6750, 7000, 7250, 7500, 7750,
                   8000, 8250, 8500, 8750, 9000, 9250, 9500, 9750,10000,
                  10500,11000,11500,12000,12500,13000,14000,15000,16000,
                  17000,18000,19000,20000,21000,22000,23000,24000,25000,
         	  26000,27000,28000,29000,30000,31000,32000,33000,34000,35000};

  /* Limb Darkening Coefficients X1/2 quadratic law, SX1/2 square root law  */ 

  /* linear                                                                 */

static
  float LR[55] ={ 0.699,0.721,0.751,0.739,0.711,0.686,0.660,0.633,0.606,
                  0.579,0.554,0.533,0.514,0.499,0.486,0.474,0.517,0.505,
                  0.492,0.475,0.458,0.443,0.431,0.421,0.414,0.407,0.402,
                  0.391,0.380,0.368,0.356,0.344,0.334,0.318,0.305,0.296,
                  0.289,0.283,0.279,0.275,0.273,0.271,0.269,0.266,0.263,
                  0.261,0.261,0.262,0.263,0.263,0.261,0.259,0.256,0.251,0.247};
static
  float LI[55] ={ 0.593,0.610,0.635,0.627,0.605,0.586,0.566,0.544,0.520,
                  0.497,0.474,0.454,0.435,0.420,0.406,0.393,0.422,0.410,
                  0.396,0.379,0.363,0.350,0.340,0.333,0.328,0.323,0.319,
                  0.312,0.304,0.295,0.286,0.276,0.268,0.255,0.245,0.238,
                  0.232,0.228,0.226,0.224,0.223,0.223,0.223,0.222,0.220,
                  0.220,0.220,0.221,0.223,0.223,0.222,0.220,0.218,0.214,0.210};
static
  float LJ[55] ={ 0.355,0.426,0.488,0.497,0.482,0.462,0.440,0.418,0.397,
                  0.377,0.358,0.340,0.325,0.311,0.299,0.287,0.313,0.302,
                  0.290,0.276,0.264,0.255,0.247,0.242,0.238,0.235,0.232,
                  0.227,0.222,0.215,0.209,0.202,0.196,0.186,0.178,0.172,
                  0.168,0.165,0.163,0.161,0.161,0.162,0.163,0.164,0.164,
                  0.164,0.165,0.166,0.167,0.168,0.167,0.166,0.165,0.164,0.161};
static
  float LH[55] ={ 0.300,0.391,0.425,0.418,0.400,0.381,0.361,0.342,0.323,
                  0.305,0.288,0.272,0.257,0.245,0.234,0.225,0.246,0.234,
                  0.224,0.214,0.204,0.196,0.191,0.187,0.184,0.182,0.180,
                  0.177,0.173,0.169,0.163,0.158,0.153,0.145,0.139,0.135,
                  0.132,0.130,0.129,0.128,0.129,0.131,0.133,0.135,0.136,
                  0.137,0.137,0.138,0.139,0.139,0.139,0.138,0.137,0.137,0.135};
static
  float LK[55] ={ 0.247,0.322,0.358,0.357,0.343,0.328,0.312,0.297,0.282,
                  0.267,0.252,0.239,0.226,0.217,0.209,0.202,0.212,0.203,
                  0.195,0.186,0.178,0.171,0.166,0.162,0.159,0.156,0.154,
                  0.150,0.147,0.142,0.138,0.133,0.129,0.123,0.118,0.114,
                  0.111,0.109,0.108,0.108,0.109,0.111,0.114,0.116,0.118,
                  0.119,0.119,0.120,0.120,0.119,0.118,0.117,0.116,0.116,0.116};


  /* quadratic                                                              */ 

static
  float  U1[55]={ 0.523,0.683,0.932,1.091,1.130,1.126,1.088,1.019,0.921,
                  0.809,0.693,0.589,0.498,0.427,0.380,0.348,0.456,0.418,
                  0.388,0.396,0.349,0.334,0.326,0.323,0.321,0.317,0.313,
                  0.304,0.291,0.277,0.261,0.250,0.239,0.222,0.209,0.202,
                  0.194,0.191,0.185,0.180,0.175,0.166,0.158,0.148,0.142,
                  0.138,0.137,0.134,0.132,0.127,0.118,0.108,0.097,0.086,0.079};
static
  float  U2[55]={ 0.282,0.101,-.145,-.270,-.285,-.272,-.226,-.153,-.058,
                  0.045,0.146,0.229,0.294,0.339,0.368,0.372,0.252,0.266,
                  0.272,0.269,0.265,0.261,0.254,0.247,0.244,0.244,0.244,
                  0.247,0.253,0.257,0.262,0.261,0.260,0.258,0.257,0.253,
                  0.254,0.251,0.254,0.255,0.258,0.265,0.271,0.277,0.280,
                  0.281,0.282,0.285,0.288,0.294,0.299,0.305,0.311,0.317,0.319};
static
  float  B1[55]={ 0.540,0.751,1.019,1.091,1.046,0.981,0.913,0.844,0.771,
                  0.703,0.637,0.580,0.529,0.483,0.447,0.414,0.476,0.431,
                  0.409,0.416,0.402,0.371,0.345,0.326,0.309,0.295,0.283,
                  0.295,0.241,0.224,0.213,0.203,0.195,0.185,0.177,0.170,
                  0.166,0.163,0.159,0.155,0.146,0.140,0.133,0.128,0.125,
                  0.124,0.122,0.120,0.116,0.108,0.102,0.089,0.079,0.072,0.065};
static
  float  B2[55]={ 0.282,0.066,-.185,-.235,-.184,-.122,-.062,-.003,-.059,
                  0.115,0.167,0.210,0.248,0.282,0.307,0.331,0.304,0.341,
                  0.353,0.332,0.329,0.343,0.353,0.359,0.364,0.369,0.372,
                  0.378,0.378,0.377,0.370,0.364,0.359,0.344,0.333,0.326,
                  0.319,0.312,0.307,0.304,0.307,0.307,0.307,0.306,0.302,
                  0.300,0.300,0.303,0.305,0.313,0.315,0.323,0.328,0.330,0.331};
static
  float  V1[55]={ 0.544,0.666,0.855,0.901,0.854,0.781,0.701,0.625,0.557,
                  0.498,0.448,0.411,0.381,0.359,0.339,0.322,0.424,0.391,
                  0.368,0.357,0.342,0.318,0.297,0.279,0.264,0.253,0.244,
                  0.226,0.212,0.199,0.189,0.181,0.174,0.165,0.158,0.152,
                  0.146,0.146,0.141,0.138,0.133,0.126,0.119,0.112,0.106,
                  0.103,0.103,0.100,0.096,0.092,0.085,0.076,0.066,0.057,0.048};
static
  float V2[55]={  0.246,0.110,-.066,-.095,-.046,0.022,0.091,0.150,0.199,
                  0.239,0.269,0.289,0.303,0.314,0.326,0.335,0.269,0.297,
                  0.311,0.308,0.307,0.313,0.320,0.326,0.330,0.332,0.332,
                  0.333,0.332,0.328,0.323,0.316,0.309,0.296,0.286,0.279,
                  0.275,0.267,0.265,0.263,0.263,0.267,0.270,0.272,0.274,
                  0.274,0.273,0.278,0.282,0.286,0.291,0.296,0.302,0.307,0.310};
static
  float u1[55] = {0.591,0.767,1.089,1.311,1.335,1.282,1.197,1.087,0.955,
                  0.824,0.698,0.591,0.499,0.435,0.396,0.375,0.519,0.491,
                  0.463,0.433,0.408,0.392,0.382,0.377,0.374,0.372,0.371,
                  0.368,0.360,0.345,0.329,0.314,0.300,0.277,0.260,0.250,
                  0.240,0.231,0.225,0.218,0.210,0.201,0.190,0.178,0.168,
                  0.161,0.159,0.157,0.154,0.148,0.139,0.125,0.113,0.101,0.091};
static
  float u2[55] = {0.242,0.040,-.279,-.460,-.455,-.393,-.303,-.194,-.069,
                  0.048,0.153,0.234,0.299,0.334,0.349,0.347,0.171,0.175,
                  0.179,0.187,0.191,0.187,0.180,0.175,0.172,0.170,0.168,
                  0.167,0.171,0.181,0.189,0.194,0.199,0.206,0.210,0.211,
                  0.213,0.218,0.220,0.225,0.232,0.241,0.250,0.259,0.264,
                  0.269,0.270,0.273,0.276,0.281,0.287,0.296,0.302,0.308,0.312};
static
  float b1[55] = {0.492,0.751,1.075,1.188,1.158,1.097,1.022,0.937,0.848,
                  0.765,0.690,0.626,0.568,0.522,0.481,0.436,0.487,0.436,
                  0.416,0.428,0.415,0.383,0.356,0.338,0.320,0.303,0.290,
                  0.264,0.244,0.229,0.216,0.207,0.199,0.188,0.180,0.175,
                  0.169,0.167,0.162,0.157,0.150,0.143,0.137,0.134,0.130,
                  0.129,0.128,0.127,0.122,0.116,0.106,0.095,0.086,0.076,0.070};
static
  float b2[55] = {0.316,0.065,-.237,-.325,-.284,-.222,-.150,-.070,0.010,
                  0.080,0.140,0.190,0.234,0.268,0.299,0.330,0.314,0.356,
                  0.365,0.339,0.336,0.350,0.362,0.366,0.372,0.379,0.382,
                  0.389,0.391,0.387,0.384,0.376,0.369,0.356,0.345,0.334,
                  0.329,0.321,0.317,0.314,0.315,0.314,0.314,0.309,0.308,
                  0.304,0.304,0.305,0.309,0.314,0.319,0.326,0.329,0.334,0.336};
static
  float v1[55] = {0.663,0.863,1.104,1.107,1.011,0.912,0.824,0.746,0.671,
                  0.605,0.548,0.507,0.473,0.447,0.423,0.401,0.470,0.429,
                  0.411,0.419,0.403,0.368,0.341,0.319,0.301,0.287,0.273,
                  0.253,0.236,0.223,0.211,0.203,0.195,0.184,0.175,0.170,
                  0.164,0.161,0.158,0.151,0.148,0.140,0.133,0.125,0.120,
                  0.119,0.116,0.114,0.107,0.101,0.092,0.082,0.071,0.063,0.057};
static
  float v2[55] = {0.214,0.001,-.228,-.222,-.131,-.044,0.030,0.091,0.149,
                  0.196,0.234,0.257,0.277,0.291,0.308,0.324,0.297,0.332,
                  0.343,0.323,0.324,0.340,0.350,0.358,0.362,0.366,0.370,
                  0.370,0.369,0.364,0.358,0.351,0.344,0.332,0.322,0.312,
                  0.307,0.301,0.295,0.295,0.292,0.293,0.295,0.296,0.295,
                  0.291,0.293,0.294,0.300,0.305,0.310,0.315,0.320,0.322,0.323};
static
  float y1[55] = {0.585,0.711,0.877,0.878,0.809,0.736,0.663,0.595,0.532,
                  0.475,0.427,0.391,0.363,0.341,0.321,0.306,0.414,0.382,
                  0.360,0.348,0.331,0.309,0.289,0.271,0.259,0.248,0.238,
                  0.221,0.208,0.196,0.186,0.177,0.172,0.162,0.155,0.148,
                  0.144,0.140,0.139,0.134,0.129,0.123,0.116,0.108,0.103,
                  0.101,0.097,0.095,0.092,0.088,0.080,0.073,0.062,0.054,0.046};
static
  float y2[55] = {0.232,0.096,-.060,-.058,0.001,0.061,0.118,0.168,0.212,
                  0.249,0.277,0.294,0.308,0.319,0.330,0.337,0.264,0.291,
                  0.304,0.303,0.302,0.308,0.313,0.320,0.321,0.323,0.324,
                  0.326,0.324,0.319,0.313,0.308,0.300,0.287,0.277,0.272,
                  0.265,0.262,0.256,0.257,0.257,0.260,0.263,0.267,0.268,
                  0.268,0.271,0.275,0.278,0.282,0.288,0.292,0.299,0.302,0.305};
static
  float R1[55] = {0.553,0.625,0.712,0.689,0.636,0.582,0.525,0.471,0.418,
                  0.370,0.329,0.296,0.268,0.246,0.227,0.211,0.327,0.305,
                  0.285,0.265,0.247,0.233,0.222,0.213,0.205,0.199,0.194,
                  0.184,0.174,0.164,0.156,0.148,0.142,0.133,0.127,0.121,
                  0.117,0.114,0.111,0.108,0.104,0.099,0.092,0.086,0.080,
                  0.076,0.074,0.072,0.070,0.068,0.064,0.059,0.051,0.042,0.034};
static
  float R2[55] = {0.189,0.126,0.050,0.065,0.098,0.136,0.176,0.212,0.244,
                  0.272,0.294,0.308,0.320,0.329,0.337,0.342,0.247,0.260,
                  0.270,0.274,0.275,0.273,0.272,0.272,0.271,0.271,0.270,
                  0.269,0.268,0.265,0.261,0.255,0.250,0.240,0.232,0.227,
                  0.223,0.220,0.219,0.219,0.220,0.225,0.230,0.235,0.238,
                  0.241,0.244,0.247,0.251,0.254,0.257,0.261,0.266,0.272,0.277};
static
  float I1[55] = {0.382,0.449,0.524,0.519,0.487,0.452,0.413,0.372,0.330,
                  0.290,0.254,0.224,0.198,0.176,0.159,0.144,0.249,0.232,
                  0.215,0.196,0.181,0.173,0.166,0.162,0.159,0.155,0.151,
                  0.145,0.139,0.131,0.125,0.119,0.114,0.105,0.100,0.095,
                  0.092,0.089,0.086,0.083,0.080,0.075,0.070,0.064,0.059,
                  0.055,0.053,0.052,0.051,0.050,0.047,0.042,0.036,0.029,0.021};
static
  float I2[55] = {0.276,0.210,0.143,0.140,0.153,0.173,0.199,0.224,0.248,
                  0.269,0.287,0.299,0.309,0.317,0.322,0.324,0.226,0.231,
                  0.235,0.238,0.237,0.231,0.226,0.222,0.220,0.219,0.218,
                  0.217,0.215,0.213,0.209,0.205,0.201,0.194,0.189,0.185,
                  0.183,0.182,0.182,0.184,0.187,0.192,0.198,0.205,0.210,
                  0.214,0.217,0.220,0.224,0.226,0.229,0.232,0.236,0.242,0.246};
static
  float J1[55] = {0.097,0.218,0.317,0.326,0.297,0.264,0.232,0.203,0.176,
                  0.152,0.130,0.113,0.097,0.084,0.073,0.065,0.140,0.131,
                  0.121,0.111,0.103,0.100,0.098,0.097,0.096,0.094,0.093,
                  0.091,0.088,0.084,0.081,0.077,0.074,0.068,0.064,0.061,
                  0.058,0.055,0.052,0.050,0.047,0.044,0.040,0.036,0.032,
                  0.030,0.029,0.028,0.028,0.028,0.027,0.024,0.021,0.016,0.010};
static
  float J2[55] = {0.335,0.271,0.222,0.223,0.240,0.257,0.271,0.280,0.288,
                  0.293,0.296,0.295,0.296,0.295,0.294,0.290,0.226,0.223,
                  0.219,0.215,0.209,0.201,0.194,0.189,0.186,0.184,0.181,
                  0.177,0.174,0.171,0.167,0.162,0.158,0.153,0.148,0.145,
                  0.144,0.143,0.144,0.145,0.149,0.154,0.160,0.167,0.172,
                  0.175,0.177,0.179,0.181,0.181,0.183,0.184,0.187,0.192,0.197};
static
  float H1[55] = {0.062,0.134,0.130,0.113,0.101,0.090,0.080,0.070,0.061,
                  0.053,0.046,0.041,0.034,0.028,0.024,0.022,0.082,0.079,
                  0.075,0.070,0.068,0.066,0.067,0.067,0.066,0.066,0.066,
                  0.065,0.064,0.061,0.058,0.055,0.053,0.048,0.044,0.041,
                  0.039,0.036,0.035,0.032,0.029,0.027,0.024,0.022,0.020,
                  0.018,0.018,0.018,0.019,0.018,0.017,0.015,0.013,0.009,0.004};
static
  float H2[55] = {0.310,0.335,0.383,0.398,0.389,0.379,0.366,0.354,0.341,
                  0.327,0.314,0.301,0.290,0.282,0.274,0.264,0.212,0.203,
                  0.194,0.187,0.178,0.169,0.162,0.157,0.153,0.151,0.149,
                  0.145,0.142,0.140,0.137,0.134,0.131,0.127,0.124,0.122,
                  0.122,0.122,0.122,0.125,0.130,0.135,0.141,0.147,0.151,
                  0.154,0.156,0.157,0.157,0.158,0.158,0.159,0.162,0.166,0.171};
static
  float K1[55] = {0.042,0.100,0.108,0.100,0.094,0.088,0.081,0.075,0.065,
                  0.055,0.048,0.042,0.037,0.032,0.029,0.028,0.081,0.076,
                  0.072,0.067,0.064,0.062,0.061,0.061,0.060,0.059,0.059,
                  0.058,0.056,0.054,0.051,0.048,0.045,0.040,0.036,0.033,
                  0.030,0.028,0.026,0.023,0.021,0.019,0.017,0.015,0.014,
                  0.014,0.014,0.014,0.015,0.015,0.013,0.011,0.008,0.005,0.001};
static
  float K2[55] = {0.267,0.289,0.325,0.334,0.324,0.312,0.300,0.290,0.282,
                  0.276,0.266,0.256,0.247,0.240,0.234,0.227,0.171,0.165,
                  0.160,0.155,0.149,0.142,0.136,0.132,0.129,0.127,0.124,
                  0.121,0.118,0.116,0.114,0.111,0.109,0.107,0.106,0.105,
                  0.105,0.106,0.107,0.111,0.115,0.120,0.126,0.132,0.135,
                  0.136,0.137,0.137,0.136,0.136,0.137,0.138,0.141,0.145,0.150};

  /* square root                                                            */

static
  float SU1[55]={ 0.297,0.582,0.989,1.233,1.290,1.287,1.229,1.119,0.961,
                  0.781,0.591,0.424,0.279,0.170,0.102,0.062,0.261,0.210,
                  0.174,0.153,0.133,0.121,0.118,0.120,0.120,0.118,0.114,
                  0.103,0.086,0.069,0.049,0.038,0.028,0.013,0.000,-.004,
                  -.013,-.013,-.021,-.028,-.035,-.050,-.064,-.079,-.089,
                  -.095,-.097,-.102,-.107,-.115,-.128,-.143,-.160,-.177,-.186};
static
  float SU2[55]={ 0.628,0.254,-.239,-.497,-.538,-.525,-.447,-.309,-.119,
                  0.090,0.303,0.484,0.631,0.734,0.789,0.810,0.550,0.584,
                  0.601,0.598,0.596,0.586,0.573,0.557,0.550,0.549,0.547,
                  0.555,0.566,0.575,0.587,0.585,0.583,0.578,0.577,0.568,
                  0.571,0.563,0.569,0.574,0.580,0.597,0.611,0.625,0.632,
                  0.637,0.639,0.646,0.653,0.663,0.675,0.689,0.703,0.719,0.723};
static
  float SB1[55]={ 0.318,0.680,1.125,1.237,1.161,1.057,0.952,0.843,0.728,
                  0.619,0.515,0.423,0.340,0.266,0.210,0.157,0.270,0.194,
                  0.156,0.171,0.154,0.109,0.072,0.047,0.026,0.006,-.009,
                  -.039,-.059,-.075,-.082,-.087,-.091,-.090,-.089,-.091,
                  -.091,-.089,-.089,-.091,-.102,-.109,-.117,-.123,-.124,
                  -.123,-.126,-.130,-.136,-.149,-.158,-.176,-.191,-.202,-.211};
static
  float SB2[55]={ 0.624,0.172,-.351,-.464,-.363,-.241,-.122,-.001,0.126,
                  0.244,0.356,0.453,0.538,0.615,0.671,0.725,0.624,0.708,
                  0.742,0.710,0.711,0.745,0.772,0.787,0.799,0.811,0.819,
                  0.834,0.838,0.835,0.823,0.809,0.797,0.766,0.741,0.726,
                  0.711,0.697,0.688,0.680,0.687,0.688,0.689,0.689,0.683,
                  0.677,0.678,0.684,0.691,0.706,0.712,0.729,0.742,0.748,0.753};
static
  float SV1[55]={ 0.353,0.596,0.881,0.951,0.877,0.761,0.635,0.516,0.412,
                  0.320,0.246,0.189,0.146,0.112,0.082,0.058,0.236,0.181,
                  0.143,0.127,0.108,0.076,0.048,0.024,0.005,-.008,-.019,
                  -.038,-.053,-.064,-.069,-.072,-.074,-.074,-.072,-.073,
                  -.076,-.072,-.075,-.077,-.082,-.093,-.104,-.113,-.121,
                  -.125,-.125,-.132,-.139,-.147,-.158,-.171,-.187,-.201,-.213};
static
  float SV2[55]={ 0.539,0.258,-.108,-.175,-.082,0.053,0.193,0.318,0.423,
                  0.513,0.581,0.629,0.665,0.693,0.719,0.740,0.559,0.622,
                  0.658,0.663,0.666,0.685,0.701,0.717,0.728,0.732,0.734,
                  0.738,0.737,0.731,0.718,0.704,0.690,0.662,0.639,0.624,
                  0.615,0.600,0.595,0.590,0.592,0.601,0.610,0.616,0.621,
                  0.623,0.622,0.631,0.641,0.650,0.661,0.673,0.689,0.700,0.709};
static
  float Su1[55]={ 0.397,0.707,1.231,1.578,1.613,1.533,1.398,1.223,1.009,
                  0.796,0.592,0.423,0.278,0.182,0.132,0.112,0.375,0.343,
                  0.311,0.275,0.246,0.232,0.228,0.227,0.227,0.227,0.230,
                  0.227,0.216,0.196,0.174,0.156,0.139,0.109,0.089,0.077,
                  0.067,0.054,0.046,0.034,0.021,0.005,-.015,-.035,-.049,
                  -.062,-.064,-.069,-.073,-.084,-.097,-.117,-.136,-.154,-.168};
static
  float Su2[55]={ 0.539,0.129,-.505,-.881,-.891,-.785,-.616,-.404,-.151,
                  0.092,0.317,0.494,0.640,0.723,0.755,0.750,0.391,0.400,
                  0.410,0.428,0.437,0.429,0.416,0.403,0.395,0.391,0.384,
                  0.381,0.390,0.408,0.425,0.436,0.445,0.463,0.472,0.474,
                  0.479,0.489,0.494,0.507,0.521,0.541,0.564,0.583,0.597,
                  0.609,0.611,0.619,0.624,0.635,0.648,0.667,0.682,0.697,0.707};
static
  float Sb1[55]={ 0.237,0.678,1.213,1.390,1.339,1.243,1.124,0.987,0.844,
                  0.710,0.589,0.485,0.391,0.317,0.252,0.182,0.277,0.191,
                  0.157,0.177,0.162,0.117,0.078,0.054,0.030,0.007,-.009,
                  -.042,-.064,-.078,-.089,-.093,-.096,-.097,-.096,-.093,
                  -.095,-.091,-.093,-.096,-.105,-.111,-.118,-.119,-.123,
                  -.121,-.121,-.125,-.132,-.142,-.156,-.173,-.185,-.200,-.209};
static
  float Sb2[55]={ 0.706,0.174,-.454,-.641,-.565,-.449,-.309,-.148,0.017,
                  0.166,0.297,0.408,0.507,0.583,0.650,0.721,0.640,0.736,
                  0.766,0.725,0.725,0.758,0.788,0.802,0.817,0.833,0.841,
                  0.859,0.864,0.858,0.851,0.835,0.821,0.792,0.768,0.745,
                  0.733,0.716,0.707,0.703,0.705,0.703,0.703,0.696,0.694,
                  0.687,0.685,0.689,0.697,0.708,0.720,0.736,0.743,0.756,0.761};
static
  float Sv1[55]={ 0.503,0.847,1.249,1.259,1.103,0.944,0.805,0.684,0.565,
                  0.461,0.373,0.310,0.258,0.219,0.182,0.147,0.271,0.202,
                  0.169,0.182,0.159,0.110,0.072,0.041,0.019,0.000,-.017,
                  -.039,-.057,-.066,-.074,-.077,-.081,-.082,-.084,-.081,
                  -.083,-.082,-.080,-.087,-.089,-.099,-.109,-.119,-.123,
                  -.123,-.127,-.129,-.142,-.151,-.163,-.179,-.195,-.206,-.214};
static
  float Sv2[55]={ 0.460,0.025,-.455,-.457,-.273,-.094,0.059,0.187,0.313,
                  0.418,0.503,0.560,0.606,0.641,0.677,0.713,0.606,0.685,
                  0.717,0.689,0.699,0.738,0.763,0.784,0.796,0.805,0.814,
                  0.817,0.818,0.807,0.796,0.780,0.767,0.739,0.718,0.698,
                  0.685,0.672,0.659,0.660,0.654,0.659,0.664,0.669,0.667,
                  0.660,0.664,0.666,0.681,0.690,0.700,0.714,0.726,0.733,0.736};
static
  float Sy1[55]={ 0.408,0.627,0.905,0.909,0.802,0.688,0.575,0.472,0.376,
                  0.290,0.218,0.166,0.124,0.091,0.061,0.039,0.228,0.175,
                  0.138,0.120,0.099,0.070,0.044,0.020,0.006,-.007,-.019,
                  -.038,-.050,-.059,-.065,-.071,-.069,-.070,-.070,-.073,
                  -.071,-.073,-.071,-.076,-.082,-.092,-.101,-.113,-.120,
                  -.124,-.130,-.134,-.141,-.148,-.161,-.172,-.188,-.201,-.213};
static
  float Sy2[55]={ 0.503,0.224,-.106,-.107,0.010,0.134,0.253,0.358,0.451,
                  0.534,0.599,0.641,0.675,0.701,0.729,0.746,0.553,0.611,
                  0.646,0.653,0.658,0.675,0.689,0.704,0.710,0.713,0.718,
                  0.722,0.719,0.709,0.698,0.687,0.668,0.642,0.621,0.610,
                  0.594,0.588,0.577,0.578,0.580,0.589,0.594,0.605,0.609,
                  0.610,0.617,0.624,0.633,0.642,0.656,0.665,0.681,0.690,0.699};
static
  float SR1[55]={ 0.407,0.522,0.663,0.630,0.556,0.476,0.392,0.312,0.235,
                  0.166,0.105,0.059,0.019,-.012,-.039,-.061,0.143,0.110,
                  0.080,0.052,0.031,0.017,0.006,-.003,-.011,-.017,-.022,
                  -.032,-.042,-.049,-.054,-.058,-.060,-.062,-.063,-.064,
                  -.066,-.067,-.070,-.073,-.079,-.088,-.099,-.111,-.121,
                  -.128,-.132,-.136,-.141,-.147,-.153,-.161,-.174,-.188,-.201};
static
  float SR2[55]={ 0.413,0.283,0.124,0.154,0.221,0.298,0.381,0.456,0.526,
                  0.587,0.637,0.672,0.703,0.725,0.744,0.759,0.531,0.561,
                  0.585,0.600,0.606,0.604,0.603,0.603,0.603,0.603,0.602,
                  0.600,0.598,0.592,0.582,0.571,0.559,0.538,0.522,0.511,
                  0.503,0.497,0.495,0.495,0.500,0.510,0.523,0.535,0.545,
                  0.552,0.558,0.565,0.574,0.581,0.588,0.597,0.609,0.624,0.636};
static
  float SI1[55]={ 0.164,0.281,0.406,0.403,0.363,0.316,0.259,0.200,0.141,
                  0.085,0.034,-.006,-.043,-.072,-.095,-.114,0.073,0.052,
                  0.030,0.006,-.009,-.013,-.016,-.018,-.019,-.022,-.025,
                  -.030,-.036,-.041,-.046,-.049,-.050,-.054,-.056,-.058,
                  -.060,-.063,-.067,-.071,-.077,-.086,-.097,-.109,-.120,
                  -.127,-.132,-.135,-.139,-.143,-.148,-.155,-.165,-.178,-.191};
static
  float SI2[55]={ 0.610,0.467,0.324,0.317,0.344,0.383,0.435,0.487,0.538,
                  0.584,0.624,0.653,0.679,0.698,0.711,0.719,0.496,0.508,
                  0.519,0.529,0.528,0.515,0.505,0.497,0.493,0.490,0.489,
                  0.485,0.482,0.477,0.470,0.461,0.452,0.439,0.427,0.420,
                  0.415,0.413,0.415,0.418,0.426,0.439,0.454,0.469,0.482,
                  0.492,0.499,0.506,0.513,0.520,0.526,0.533,0.544,0.557,0.569};
static
  float SJ1[55]={ -.183,-.007,0.140,0.152,0.111,0.065,0.023,-.014,-.048,
                  -.078,-.104,-.123,-.141,-.156,-.167,-.173,-.040,-.047,
                  -.055,-.064,-.067,-.063,-.060,-.058,-.057,-.056,-.056,
                  -.055,-.055,-.056,-.057,-.057,-.057,-.059,-.060,-.062,
                  -.064,-.066,-.071,-.075,-.080,-.089,-.098,-.108,-.116,
                  -.121,-.125,-.126,-.127,-.128,-.131,-.135,-.141,-.150,-.162};
static
  float SJ2[55]={ 0.763,0.614,0.493,0.490,0.525,0.562,0.592,0.614,0.632,
                  0.645,0.654,0.657,0.661,0.662,0.661,0.654,0.501,0.495,
                  0.489,0.482,0.469,0.451,0.437,0.426,0.419,0.413,0.408,
                  0.400,0.393,0.385,0.377,0.367,0.359,0.347,0.338,0.332,
                  0.329,0.328,0.331,0.335,0.343,0.356,0.370,0.385,0.397,
                  0.405,0.411,0.415,0.418,0.419,0.422,0.427,0.434,0.445,0.459};
static
  float SH1[55]={ -.200,-.141,-.179,-.208,-.213,-.216,-.217,-.218,-.218,
                  -.216,-.214,-.210,-.209,-.209,-.206,-.200,-.090,-.087,
                  -.084,-.083,-.079,-.073,-.067,-.063,-.060,-.058,-.057,
                  -.054,-.054,-.055,-.055,-.056,-.057,-.059,-.061,-.063,
                  -.065,-.068,-.070,-.076,-.083,-.090,-.098,-.105,-.111,
                  -.115,-.117,-.117,-.117,-.118,-.120,-.122,-.128,-.137,-.146};
static
  float SH2[55]={ 0.709,0.755,0.857,0.888,0.870,0.847,0.821,0.795,0.768,
                  0.739,0.712,0.684,0.662,0.643,0.624,0.602,0.477,0.456,
                  0.438,0.420,0.401,0.382,0.366,0.354,0.347,0.341,0.337,
                  0.328,0.322,0.317,0.310,0.304,0.298,0.290,0.285,0.281,
                  0.280,0.281,0.283,0.290,0.300,0.313,0.327,0.340,0.351,
                  0.357,0.361,0.363,0.363,0.365,0.366,0.369,0.376,0.388,0.400};
static
  float SK1[55]={ -.183,-.139,-.157,-.171,-.168,-.165,-.161,-.160,-.163,
                  -.169,-.170,-.169,-.167,-.166,-.164,-.159,-.059,-.059,
                  -.060,-.060,-.058,-.055,-.052,-.049,-.047,-.046,-.045,
                  -.043,-.042,-.043,-.045,-.046,-.047,-.052,-.055,-.058,
                  -.060,-.064,-.067,-.073,-.079,-.086,-.093,-.099,-.103,
                  -.104,-.105,-.104,-.103,-.104,-.106,-.109,-.114,-.122,-.131};
static
  float SK2[55]={ 0.611,0.655,0.730,0.749,0.725,0.699,0.672,0.648,0.632,
                  0.619,0.599,0.579,0.558,0.544,0.529,0.512,0.384,0.371,
                  0.361,0.350,0.336,0.321,0.308,0.299,0.292,0.287,0.282,
                  0.274,0.268,0.263,0.259,0.255,0.250,0.247,0.245,0.243,
                  0.244,0.246,0.248,0.257,0.267,0.280,0.294,0.306,0.313,
                  0.316,0.317,0.318,0.316,0.317,0.318,0.320,0.328,0.339,0.350};


  /* -----------  initialize ------------------------------------------ */

 SurfPtr = Surface[Comp];
  
 for (band = 0; band < NUM_MAG; ++band) {
   for (j = 0; j < 2; ++j) {
     Limb[Comp][band][j] = 0.0;
   }
 }

 /* we approximate T = Binary[Comp].Temperature for all                 */
 /* surface elements                                                    */

 Tn  = (float) Binary[Comp].Temperature;

 /* -----------   linear law ------------------------------------------ */

 if (Flags.limb == 0) {   

   /* RIJHK data                                                        */
   j = search_lkey (Tn, T, 55);

   Limb[Comp][Rmag][0] = LR[j]; 
   Limb[Comp][Imag][0] = LI[j]; 
   Limb[Comp][Jmag][0] = LJ[j];
   Limb[Comp][Hmag][0] = LH[j];
   Limb[Comp][Kmag][0] = LK[j];

   /* analytic approximations                                           */ 

   Tn = (float) log10(Binary[Comp].Temperature);

   Limb[Comp][Umag][0]       = 8.229  - 3.217 * Tn 
     - 0.017 * log_g
     + 0.326 * Tn * Tn;
   if (Limb[Comp][Umag][0]  >= 1.0) Limb[Comp][Umag][0] = 1.0;

   Limb[Comp][Bmag][0]  = 6.365 - 2.195 * Tn 
     - 0.023 * log_g
     + 0.193 * Tn * Tn;
   if (Limb[Comp][Bmag][0]  >= 1.0) Limb[Comp][Bmag][0]  = 1.0;

   Limb[Comp][Vmag][0] = 10.068 - 4.054 * Tn 
     - 0.018 * log_g
     + 0.422 * Tn * Tn;
   if (Limb[Comp][Vmag][0] >= 1.0) Limb[Comp][Vmag][0] = 1.0;

   Limb[Comp][umag][0] = 12.918 - 5.462 * Tn 
     + 0.005 * log_g
     + 0.591 * Tn * Tn;
   if (Limb[Comp][umag][0] >= 1.0) Limb[Comp][umag][0] = 1.0;

   Limb[Comp][vmag][0] =  5.863 - 1.917 * Tn 
     - 0.024 * log_g
     + 0.157 * Tn * Tn;
   if (Limb[Comp][vmag][0] >= 1.0) Limb[Comp][vmag][0] = 1.0;

   Limb[Comp][bmag][0] =  8.687 - 3.339 * Tn 
     - 0.015 * log_g
     + 0.331 * Tn * Tn;
   if (Limb[Comp][bmag][0] >= 1.0) Limb[Comp][bmag][0] = 1.0;

   Limb[Comp][ymag][0] =  9.257 - 3.729 * Tn 
     - 0.015 * log_g
     + 0.388 * Tn * Tn;
   if (Limb[Comp][ymag][0] >= 1.0) Limb[Comp][ymag][0] = 1.0;

 }

 /* -----------   quadratic law---------------------------------------- */

 else if (Flags.limb == 1) 
   { 
     j = search_lkey (Tn, T, 55);
     Limb[Comp][Vmag][0] = V1[j]; Limb[Comp][Vmag][1] = V2[j];
     Limb[Comp][Bmag][0] = B1[j]; Limb[Comp][Bmag][1] = B2[j];
     Limb[Comp][Umag][0] = U1[j]; Limb[Comp][Umag][1] = U2[j];         
     Limb[Comp][umag][0] = u1[j]; Limb[Comp][umag][1] = u2[j];
     Limb[Comp][bmag][0] = b1[j]; Limb[Comp][bmag][1] = b2[j];
     Limb[Comp][vmag][0] = v1[j]; Limb[Comp][vmag][1] = v2[j];
     Limb[Comp][ymag][0] = y1[j]; Limb[Comp][ymag][1] = y2[j];
     Limb[Comp][Rmag][0] = R1[j]; Limb[Comp][Rmag][1] = R2[j];
     Limb[Comp][Imag][0] = I1[j]; Limb[Comp][Imag][1] = I2[j];
     Limb[Comp][Jmag][0] = J1[j]; Limb[Comp][Jmag][1] = J2[j];
     Limb[Comp][Hmag][0] = H1[j]; Limb[Comp][Hmag][1] = H2[j];
     Limb[Comp][Kmag][0] = K1[j]; Limb[Comp][Kmag][1] = K2[j];
   }



 /* -----------   square root law-------------------------------------- */

 else if (Flags.limb == 2) 
   { 
     hkey = search_lkey (Tn, T, 55);

     Limb[Comp][Vmag][0] = SV1[hkey]; Limb[Comp][Vmag][1] = SV2[hkey];
     Limb[Comp][Bmag][0] = SB1[hkey]; Limb[Comp][Bmag][1] = SB2[hkey];
     Limb[Comp][Umag][0] = SU1[hkey]; Limb[Comp][Umag][1] = SU2[hkey];
     Limb[Comp][umag][0] = Su1[hkey]; Limb[Comp][umag][1] = Su2[hkey];
     Limb[Comp][bmag][0] = Sb1[hkey]; Limb[Comp][bmag][1] = Sb2[hkey];
     Limb[Comp][vmag][0] = Sv1[hkey]; Limb[Comp][vmag][1] = Sv2[hkey];
     Limb[Comp][ymag][0] = Sy1[hkey]; Limb[Comp][ymag][1] = Sy2[hkey];
     Limb[Comp][Rmag][0] = SR1[hkey]; Limb[Comp][Rmag][1] = SR2[hkey];
     Limb[Comp][Imag][0] = SI1[hkey]; Limb[Comp][Imag][1] = SI2[hkey];
     Limb[Comp][Jmag][0] = SJ1[hkey]; Limb[Comp][Jmag][1] = SJ2[hkey];
     Limb[Comp][Hmag][0] = SH1[hkey]; Limb[Comp][Hmag][1] = SH2[hkey];
     Limb[Comp][Kmag][0] = SK1[hkey]; Limb[Comp][Kmag][1] = SK2[hkey];
   }
 
 /* -----------   individual square root     -------------------------- */

 else if (Flags.limb == 3) {
 
   SurfPtr = Surface[Comp];

   for (i = 0; i < Binary[Comp].NumElem; ++i) {

     Tn = (float) SurfPtr->temp;

     if (Tn <= 10000 && Tn >= 3500)      { hkey=floor(Tn/250) - 14;  }
     else if (Tn <= 13000 && Tn >= 3500) { hkey=floor(Tn/500) + 6;   }
     else if (Tn <= 33999 && Tn >= 3500 ){ hkey=floor(Tn/1000) + 19; }
     else if (Tn <= 3500)                { hkey= 0;                  }
     else { hkey = 54; }

     if (hkey < 54 && Tn >= (T[0]+FLT_EPSILON) ) {
               
       dy1 = Tn - T[hkey];
       dy2 = T[hkey+1] - Tn;
                  
       dx = T[hkey+1] - T[hkey];
	 
       SurfPtr->LD[Vmag][0] = (SV1[hkey]*dy2 + SV1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Vmag][1] = (SV2[hkey]*dy2 + SV2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Bmag][0] = (SB1[hkey]*dy2 + SB1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Bmag][1] = (SB2[hkey]*dy2 + SB2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Umag][0] = (SU1[hkey]*dy2 + SU1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Umag][1] = (SU2[hkey]*dy2 + SU2[hkey+1]*dy1)/dx;
       SurfPtr->LD[umag][0] = (Su1[hkey]*dy2 + Su1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[umag][1] = (Su2[hkey]*dy2 + Su2[hkey+1]*dy1)/dx;
       SurfPtr->LD[bmag][0] = (Sb1[hkey]*dy2 + Sb1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[bmag][1] = (Sb2[hkey]*dy2 + Sb2[hkey+1]*dy1)/dx;
       SurfPtr->LD[vmag][0] = (Sv1[hkey]*dy2 + Sv1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[vmag][1] = (Sv2[hkey]*dy2 + Sv2[hkey+1]*dy1)/dx;
       SurfPtr->LD[ymag][0] = (Sy1[hkey]*dy2 + Sy1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[ymag][1] = (Sy2[hkey]*dy2 + Sy2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Rmag][0] = (SR1[hkey]*dy2 + SR1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Rmag][1] = (SR2[hkey]*dy2 + SR2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Imag][0] = (SI1[hkey]*dy2 + SI1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Imag][1] = (SI2[hkey]*dy2 + SI2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Jmag][0] = (SJ1[hkey]*dy2 + SJ1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Jmag][1] = (SJ2[hkey]*dy2 + SJ2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Hmag][0] = (SH1[hkey]*dy2 + SH1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Hmag][1] = (SH2[hkey]*dy2 + SH2[hkey+1]*dy1)/dx;
       SurfPtr->LD[Kmag][0] = (SK1[hkey]*dy2 + SK1[hkey+1]*dy1)/dx; 
       SurfPtr->LD[Kmag][1] = (SK2[hkey]*dy2 + SK2[hkey+1]*dy1)/dx;
	 
     } else {
          
       SurfPtr->LD[Vmag][0] = SV1[hkey];
       SurfPtr->LD[Vmag][1] = SV2[hkey];
       SurfPtr->LD[Bmag][0] = SB1[hkey];
       SurfPtr->LD[Bmag][1] = SB2[hkey];
       SurfPtr->LD[Umag][0] = SU1[hkey];
       SurfPtr->LD[Umag][1] = SU2[hkey];
       SurfPtr->LD[umag][0] = Su1[hkey];
       SurfPtr->LD[umag][1] = Su2[hkey];
       SurfPtr->LD[bmag][0] = Sb1[hkey];
       SurfPtr->LD[bmag][1] = Sb2[hkey];
       SurfPtr->LD[vmag][0] = Sv1[hkey];
       SurfPtr->LD[vmag][1] = Sv2[hkey];
       SurfPtr->LD[ymag][0] = Sy1[hkey];
       SurfPtr->LD[ymag][1] = Sy2[hkey];
       SurfPtr->LD[Rmag][0] = SR1[hkey];
       SurfPtr->LD[Rmag][1] = SR2[hkey];
       SurfPtr->LD[Imag][0] = SI1[hkey];
       SurfPtr->LD[Imag][1] = SI2[hkey];
       SurfPtr->LD[Jmag][0] = SJ1[hkey];
       SurfPtr->LD[Jmag][1] = SJ2[hkey];
       SurfPtr->LD[Hmag][0] = SH1[hkey];
       SurfPtr->LD[Hmag][1] = SH2[hkey];
       SurfPtr->LD[Kmag][0] = SK1[hkey];
       SurfPtr->LD[Kmag][1] = SK2[hkey];

     }
     ++SurfPtr;
   }
 }

 /* -----------   Modify, if using monochrome wavelengths ------------ */

 if (Flags.monochrome == ON)
   FindbestLC ();

 /* -----------   Verbose output -------------------------------------- */

 if (Flags.debug[VERBOSE] == ON && Flags.limb == 3) 
   printf(_("Limb Darkening Coefficients computed individually for each surface element \n"));
 
 if (Flags.debug[VERBOSE] == ON && Flags.limb != 3) {
   printf(_("Limb Darkening Coefficients: \n"));
   if(Comp == Primary) {
     printf(_(" Primary: \n"));
     printf("  U: %6.3f %6.3f B: %6.3f %6.3f V: %6.3f %6.3f\n",
	    Limb[Primary][Umag][0],Limb[Primary][Umag][1],
	    Limb[Primary][Bmag][0],Limb[Primary][Bmag][1],
	    Limb[Primary][Vmag][0],Limb[Primary][Vmag][1]);
     printf("  u: %6.3f %6.3f v: %6.3f %6.3f \n",
	    Limb[Primary][umag][0],Limb[Primary][umag][1],
	    Limb[Primary][vmag][0],Limb[Primary][vmag][1]);
     printf("  b: %6.3f %6.3f y: %6.3f %6.3f\n",
	    Limb[Primary][bmag][0],Limb[Primary][bmag][1],
	    Limb[Primary][ymag][0],Limb[Primary][ymag][1]);
     printf("  R: %6.3f %6.3f I: %6.3f %6.3f\n",
	    Limb[Primary][Rmag][0],Limb[Primary][Rmag][1],
	    Limb[Primary][Imag][0],Limb[Primary][Imag][1]);
     printf("  J: %6.3f %6.3f H: %6.3f %6.3f K: %6.3f %6.3f\n",
	    Limb[Primary][Jmag][0],Limb[Primary][Jmag][1],
	    Limb[Primary][Hmag][0],Limb[Primary][Hmag][1],
	    Limb[Primary][Kmag][0],Limb[Primary][Kmag][1]);
   }
   if(Comp == Secondary) {   
     printf(_(" Secondary: \n"));
     printf("  U: %6.3f %6.3f B: %6.3f %6.3f V: %6.3f %6.3f\n",
	    Limb[Secondary][Umag][0],Limb[Secondary][Umag][1],
	    Limb[Secondary][Bmag][0],Limb[Secondary][Bmag][1],
	    Limb[Secondary][Vmag][0],Limb[Secondary][Vmag][1]);
     printf("  u: %6.3f %6.3f v: %6.3f %6.3f \n",
	    Limb[Secondary][umag][0],Limb[Secondary][umag][1],
	    Limb[Secondary][vmag][0],Limb[Secondary][vmag][1]);
     printf("  b: %6.3f %6.3f y: %6.3f %6.3f\n",
	    Limb[Secondary][bmag][0],Limb[Secondary][bmag][1],
	    Limb[Secondary][ymag][0],Limb[Secondary][ymag][1]);
     printf("  R: %6.3f %6.3f I: %6.3f %6.3f\n",
	    Limb[Secondary][Rmag][0],Limb[Secondary][Rmag][1],
	    Limb[Secondary][Imag][0],Limb[Secondary][Imag][1]);
     printf("  J: %6.3f %6.3f H: %6.3f %6.3f K: %6.3f %6.3f\n",
	    Limb[Secondary][Jmag][0],Limb[Secondary][Jmag][1],
	    Limb[Secondary][Hmag][0],Limb[Secondary][Hmag][1],
	    Limb[Secondary][Kmag][0],Limb[Secondary][Kmag][1]);
   }
 }
}



/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Verbose output of wavelengths
 @param     (int) Comp The stellar component
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/
static
void VerboseMessage (int Comp)
{
  if (Flags.debug[VERBOSE] == ON) {
    printf("\n");
    printf(_("Effective Wavelengths:\n"));
    if(Comp == Primary) { 
      printf(_(" Primary: \n"));
      printf("  U: %6.4f B: %6.4f V: %6.4f R: %6.4f I: %6.4f\n",
             WaveL[Primary][Umag], WaveL[Primary][Bmag],
             WaveL[Primary][Vmag], WaveL[Primary][Rmag],
             WaveL[Primary][Imag]);
      printf("  u: %6.4f v: %6.4f b: %6.4f y: %6.4f\n",
             WaveL[Primary][umag], WaveL[Primary][vmag],
             WaveL[Primary][bmag], WaveL[Primary][ymag]);
      printf("  J: %6.4f H: %6.4f K: %6.4f\n",
             WaveL[Primary][Jmag], WaveL[Primary][Hmag],
             WaveL[Primary][Kmag]);
    }
    if(Comp == Secondary) {
      printf(_(" Secondary: \n"));
      printf("  U: %6.4f B: %6.4f V: %6.4f R: %6.4f I: %6.4f\n",
             WaveL[Secondary][Umag], WaveL[Secondary][Bmag],
             WaveL[Secondary][Vmag], WaveL[Secondary][Rmag],
             WaveL[Secondary][Imag]);
      printf("  u: %6.4f v: %6.4f b: %6.4f y: %6.4f\n",
             WaveL[Secondary][umag], WaveL[Secondary][vmag],
             WaveL[Secondary][bmag], WaveL[Secondary][ymag]);
      printf("  J: %6.4f H: %6.4f K: %6.4f\n",
             WaveL[Secondary][Jmag], WaveL[Secondary][Hmag],
             WaveL[Secondary][Kmag]);
    }
    
  }

  return;
}

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Set monochromatic wavelengths for blackbody
 @param     (int) Comp The stellar component
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/
float mono_zero[NUM_MAG] = {   
  0.3585, 0.4383, 0.5502,
  0.6400, 0.7900,
  1.25,   1.62,   2.2,
  0.3500, 0.4110, 0.4670, 0.5470
};

static
void SetMonoWavelengths(int Comp)
{
  int band;

  for (band = 0; band < NUM_MAG; ++band) 
    {
      WaveL[Comp][band] = mono_zero[band];
    }

  /* Verbose output                                                      */

  VerboseMessage(Comp);
  
  return;
}

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Define monochromatic wavelengths for blackbody
 @param     (void)
 @return    (int) Flag ON/OFF
 @heading   Light Curve
 ****************************************************************************/
int DefineMonoWavelengths ()
{
  int    band;
  int    len;
  float  fin;
  char * pp;
  char * qq;

  if (NULL == (pp = getenv("NIGHTFALL_MONO_WAVE")))
    return OFF;

  len = strlen(pp) + 1;

  qq = (char *) malloc (len);

  if (NULL == qq)
    {
      nf_error(_(errmsg[0]));
      return OFF;
    }

  strncpy (qq, pp, len);
  qq[len-1] = '\0';

  band = 0;
  pp = strtok(qq, ",");

  if (pp != NULL)
    {
      fin = atof(pp);
      mono_zero[band] = (fin > 0) ? fin : mono_zero[band];
      ++band;
    }
  else
    return OFF;

  while (pp != NULL && band < NUM_MAG)
    {
      pp = strtok(NULL, ",");
      if (pp != NULL)
	{
	  fin = atof(pp);
	  mono_zero[band] = (fin > 0) ? fin : mono_zero[band];
	  ++band;
	}
    }
     
  return ON;

}



/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Search closest value in an array
 @param     (float)   fin    A float value
 @param     (float *) farr   A float array
 @param     (int)     arrsiz Size of farr
 @return    (int)     The index of closest match
 @heading   Light Curve
 ****************************************************************************/
static
int BestIndex (float fin, float * farr, int arrsiz)
{
  register int k;
  int          i = 0;
  float        tmp;
  float        best = fabs (fin - farr[0]);
  
  for (k = 1; k < arrsiz; ++k)
    {
      tmp = fabs (fin - farr[k]);
      if (tmp < best)
	{
	  best = tmp;
	  i    = k;
	}
    }
  return i;
}
    
/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Find best limb darkening coefficient for monochrome wavelengths
 @param     (void)  
 @return    (void)
 @heading   Light Curve
 ****************************************************************************/
static double tmp_limb[2][NUM_MAG][2];

  /* effective wavelength                                               */
  /* 
            U, B, V, 
            R, I,
            J, H, K,
            u, v, b, y
	    */

  static float  l_zero[NUM_MAG] = {
    0.3585, 0.4383, 0.5502,
    0.6400, 0.7900,
    1.25,   1.62,   2.2,
    0.3500, 0.4110, 0.4670, 0.5470
  };


static
void FindbestLC ()
{
  register int i, j, k;

  for (k = 0; k < NUM_MAG; ++k)
    {
      i = BestIndex (mono_zero[k], l_zero, NUM_MAG);

      tmp_limb[0][k][0] = Limb[0][i][0];
      tmp_limb[1][k][0] = Limb[1][i][0];
      tmp_limb[0][k][1] = Limb[0][i][1];
      tmp_limb[1][k][1] = Limb[1][i][1];
    }
    

  /* copy from temporary array
   */
  for (i = 0; i < 2; ++i)
    for (j = 0; j < 2; ++j)
      for (k = 0; k < NUM_MAG; ++k)
	Limb[i][k][j] = tmp_limb[i][k][j];

  return;
}
  

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Compute temperature-dependent effective wavelengths for blackbody
 @long      Uses second moments of passbands following the prescription by:
            Young A.T. 1992, Astronomy and Astrophysics 257, 366 (Equation 12)
            Effective wavelengths 
            calculated following  Equation 3.30 in: 
            Budding E. 1993, An Introduction to Astronomic Photometry, 
            Cambridge University Press 
 @param     (int) Comp The stellar component
 @return    (void) 
 @heading   Light Curve
 ****************************************************************************/
void EffectiveWavelength(int Comp)
{


  double l_max;            /* blackbody peak                            */
  int    band;             /* passband                                  */
  double Wien = 2897.756;  /* constant for Wien's law                   */

  /* second moments of passbands                                         */
  /* approximated following the prescription by                          */
  /* Young A.T. 1992, Astronomy and Astrophysics 257, 366; Equation 12   */

  static float  M2ND[NUM_MAG]   = {
    0.000448,  0.001236,  0.001405,
    4.556e-3,  1.845e-3,  
    4.648e-3,  5.971e-3,  8.264e-3,
    1.8061e-5, 1.3514e-5, 1.6357e-5, 2.8619e-5
  };


  /* --------------------- monochromatic wavelength -------------------- */

  if (Flags.monochrome == ON)
    {
      SetMonoWavelengths(Comp);
      return;
    }

  /* --------------------- effective wavelength ------------------------ */


  /* effective wavelengths                                               */
  /* calculated following  Equation 3.30 in                              */
  /* Budding E. 1993, An Introduction to Astronomic Photometry           */
  /*      Cambridge University Press                                     */

  /* blackbody peak                                                      */

  l_max = Wien/Binary[Comp].Temperature;

  for (band = 0; band < NUM_MAG; ++band) {
    WaveL[Comp][band] = l_zero[band] 
      + 5*(l_max-l_zero[band])*M2ND[band]/SQR(l_zero[band]);   
  }

  /* Verbose output                                                      */

  VerboseMessage(Comp);

  return;
}








