/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "Light.h"

#ifdef  HAVE_UNISTD_H
#include <unistd.h> 
#endif


#ifdef _WITH_GTK


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Destroy data sheet
 @param     (GtkWidget) *widget  Discarded
 @param     (gpointer) *fs       Widget to destroy 
 @return    (void) 
 @heading   Data display
 ****************************************************************************/
void close_data( GtkWidget *widget, gpointer *fs )
       {
              gtk_widget_destroy (GTK_WIDGET (fs));
       }

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Hardcopy data sheet
 @param     (GtkWidget) *widget  Discarded
 @param     (gpointer) *fs       Discarded 
 @return    (void) 
 @heading   Data display
 ****************************************************************************/
void hard_data( GtkWidget *widget, gpointer *fs )
{
  FILE *pipe_out;
  char command[256];
  char title[32];
  
  strncpy(title, Orbit.Name, sizeof(title)-1);
  command[0] = '\0';

#ifdef HAVE_A2PS
  sprintf(command, 
	  "cat - | sed -e 's@<@ @g' | sed -e 's@>@ @g' | sed -e 's@#@ @g' | a2ps -nP -1 -nu -8 -p -nL -q -H%c%s%c > NightfallData.ps",
	  34, title, 34);
  if ( (pipe_out = popen(command, "w")) != NULL) {
    OutfileHeader(pipe_out);
    pclose(pipe_out);
  } else {
    perror(_("popen"));
  }
#else
  WARNING(_("no postscript driver (a2ps)"));
#endif
  return;
  
}


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Show data for present binary system configuration
 @param     (GtkWidget) *widget  Discarded
 @param     (GtkEvent)  *event   Discarded
 @param     (gpointer) *data     Discarded
 @return    (void) 
 @heading   Data display
 ****************************************************************************/
void showdata_event (GtkWidget *widget, gpointer *data)
{

  FILE        *inFile;                     /* filehandler for input file    */
  char        tmpfile[1024];               /* temporary file on disk        */
  char        inStrg[2];                   /* read from file, put in widget */
  int         ErrCode;                     /* exit status                   */

  GtkWidget *showdata_window;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *separator;
  GtkWidget *button;
  GtkWidget *vscrollbar;
  GtkWidget *text;
  GdkColormap *cmap;
  GdkColor colour;
  GdkFont *fixed_font; 
  
  if (getenv("TEMPDIR") != NULL) strncpy(tmpfile, getenv("TEMPDIR"), 960);
  if (getenv("TMPDIR")  != NULL) strncpy(tmpfile, getenv("TMPDIR"), 960);
  else strcpy(tmpfile, "/tmp");
  sprintf(tmpfile + strlen(tmpfile), "/nightfall.data");

  /* >>>>>>>>>>>>>>>>>  define geometry <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<     */

  /* -------- validate input data        ---------------------              */

  check_data();                 

  if (Flags.fill == OFF) {
     /* THE PRIMARY */
     ErrCode = DefineParam(Primary);
     /* THE SECONDARY */
     DefineParam(Secondary);
  } else {
     /* Overcontact system */
     ErrCode = DefineParamOver();
  }

  if (ErrCode > 0) {
    make_dialog (_(errmsg[12]));
    return;
  }

  LightSetupTests();

  /* >>>>>>>>>>>>>>>>>>>>>>>>>>  open tmp file   <<<<<<<<<<<<<<<<<<<<<<<<<< */

#ifdef  HAVE_UNISTD_H
  sprintf(tmpfile + strlen(tmpfile), ".%d", getpid() );
#endif

  if ((inFile = fopen (tmpfile, "w+")) == NULL) {
      make_dialog (_(errmsg[13]));
      return;
  }

  OutfileHeader(inFile);

  /* >>>>>>>>>>>>>>>>>>>  read tmp file   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

  /* rewind file                                                            */
  (void) fseek(inFile, 0L, SEEK_SET);

  showdata_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize (showdata_window, 640, 500);
  gtk_window_set_policy (GTK_WINDOW(showdata_window), TRUE, TRUE, FALSE);  
  gtk_signal_connect (GTK_OBJECT (showdata_window), "destroy",
                        (GtkSignalFunc) close_data, 
                         GTK_WIDGET (showdata_window));
  gtk_window_set_title (GTK_WINDOW (showdata_window), _("Data Sheet") );
  gtk_container_border_width (GTK_CONTAINER (showdata_window), 0);


  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (showdata_window), vbox);
  gtk_widget_show (vbox);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  /* Create the GtkText widget                                              */

  text = gtk_text_new (NULL, NULL);
  gtk_text_set_editable (GTK_TEXT (text), FALSE);
  gtk_box_pack_start (GTK_BOX (hbox), text, TRUE, TRUE, 0);
  gtk_widget_show (text);
  
  /* Add a vertical scrollbar to the GtkText widget                         */

  vscrollbar = gtk_vscrollbar_new (GTK_TEXT (text)->vadj);
  gtk_box_pack_start(GTK_BOX(hbox), vscrollbar, FALSE, FALSE, 0);
  gtk_widget_show (vscrollbar);
  
  /* Realizing a widget creates a window for it, ready to insert some text  */

  gtk_widget_realize (text);

  /* Freeze the text widget, ready for multiple updates                     */

  gtk_text_freeze (GTK_TEXT (text));
         
  /* >>>>>>>>>>>>>>>>>>> parse tmp file   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

  /* Get the system colour map and allocate the colour red                  */

  cmap = gdk_colormap_get_system();
  colour.red = 0xffff;
  colour.green = 0;
  colour.blue = 0;
  if (!gdk_color_alloc(cmap, &colour)) {
    WARNING (_("Could not allocate colour"));
  }

  /* Load a fixed font                                                      */

  fixed_font = gdk_font_load ("-misc-fixed-medium-r-*-*-*-140-*-*-*-*-*-*");

  inStrg[1] = '\0';

  while (1) {
    inStrg[0] = fgetc (inFile);

    /* End of file, we are done                                             */
    if (inStrg[0] == EOF) break;

    /* not special, print                                                   */
    if (inStrg[0] != '<') {
      if (inStrg[0] == '#') inStrg[0] = ' ';
      gtk_text_insert (GTK_TEXT (text), fixed_font, &text->style->black, NULL,
                          inStrg, -1);
      continue;
    }

    /* tagged word                                                          */
    while (inStrg[0] != EOF) {
     inStrg[0] = fgetc (inFile);
     if (inStrg[0] == '>') break;
     if (inStrg[0] == '#') inStrg[0] = ' ';
     gtk_text_insert (GTK_TEXT (text), fixed_font, &colour, NULL,
                          inStrg, -1);
    }
  }

  /* >>>>>>>>>>>>>>>>>>> end parse file   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

  fclose(inFile);

  remove(tmpfile);

  /* Thaw the text widget, allowing the updates to become visible           */
  
  gtk_text_thaw (GTK_TEXT (text));

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 0);
  gtk_widget_show (separator);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
  gtk_widget_show (hbox);

#ifdef HAVE_A2PS
#ifdef HAVE_GNOME
  button = gnome_pixmap_button(
	    gnome_stock_pixmap_widget(showdata_window, 
				      GNOME_STOCK_PIXMAP_PRINT), 
	    _("Postscript") );
#else
  button = gtk_button_new_with_label (_("Postscript"));
#endif
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) hard_data,
                      GTK_WIDGET (showdata_window));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 5);
  (GTK_WIDGET_FLAGS (button)  |= (GTK_CAN_DEFAULT));
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
#endif

#ifdef HAVE_GNOME
  button = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
#else
  button = gtk_button_new_with_label (_("Close") );
#endif
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) close_data,
                      GTK_WIDGET (showdata_window));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  (GTK_WIDGET_FLAGS (button)  |= (GTK_CAN_DEFAULT));
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (showdata_window);

}

#endif
