/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Light.h"

#ifdef _WITH_GTK

   static GtkWidget   *help_window;                 /* the help window */
   static GtkWidget   *help_text;                   /* the help text   */
   static int         HelpIsOn = OFF;               /* flag help open  */
   static char        my_full_helpfile[1024] = "\0";/* helpfile path   */
   static char        my_helpfile[8][16] = {        /* helpfile names  */
     "help0.txt", 
     "help1.txt", 
     "help2.txt", 
     "help3.txt", 
     "help4.txt", 
     "help5.txt",  
     "help6.txt", 
     "help7.txt"  
   };



/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Close help viewer
 @param     (GtkWidget) *widget    Discarded
 @param     (gpointer)  *fs        The window to destroy
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void close_help( GtkWidget *widget, gpointer fs )
{
  HelpIsOn = OFF;
  gtk_widget_destroy (GTK_WIDGET (fs));
  return;
}

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Delete current text in viewer
 @param     (void) 
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void delete_help( )
{
  /* Freeze the text widget, ready for multiple updates           */
  gtk_text_freeze (GTK_TEXT (help_text));
  
  /* delete everything                                            */
  gtk_text_backward_delete(GTK_TEXT (help_text),
			   gtk_text_get_length( GTK_TEXT (help_text)));


  /* Thaw the text widget, allowing the updates to become visible */  
  gtk_text_thaw (GTK_TEXT (help_text));

  return;
}


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Write the help text
 @tip       Adds a random adage using the 'fortune' program
 @param     (FILE) *inFile  The help file
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void WriteHelpText(FILE *inFile)
{

#ifdef HAVE_FORTUNE
   FILE        *wits;           /* the filehandler for fortune    */
   char        witty[256];      /* the returned fortune           */
   register int         i;      /* loop variable                  */
#endif

   GdkFont *fixed_font;         /* the fixed font                 */
   GdkFont *bold_font;          /* the bold font                  */
   GdkFont *normal_font;        /* the normal font                */ 
   GdkFont *italic_font;        /* the italic font                */ 

   int         italic = OFF;    /* the italic flag                */
   int         bold   = OFF;    /* the bold flag                  */
   int         fixed  = OFF;    /* the fixed flag                 */

   char        inStrg[2];       /* the string to put in viewer    */


  /* ------ Freeze the text widget, ready for multiple updates -- */

  gtk_text_freeze (GTK_TEXT (help_text));

  /* ---------------- parse help file --------------------------- */


  /* Load  fonts                                                  */
  fixed_font  = gdk_font_load 
               ("-misc-fixed-medium-r-*-*-*-120-*-*-*-*-*-*");
  bold_font   = gdk_font_load 
               ("*-new century schoolbook-bold-r-normal--*-120-*-*-*-*-*-*");
  italic_font = gdk_font_load 
               ("*-new century schoolbook-bold-i-normal--*-120-*-*-*-*-*-*");
  normal_font = gdk_font_load 
               ("*-new century schoolbook-medium-r-normal--*-120-*-*-*-*-*-*");

  inStrg[1] = '\0';
  while (1) {
    inStrg[0] = fgetc (inFile);

    /* End of file, we are done                                   */
    if (inStrg[0] == EOF) break;

    /* not special, print                                         */
    if (inStrg[0] != '<' && inStrg[0] != '[' && inStrg[0] != '{') {
      if (inStrg[0] == '#') inStrg[0] = ' ';
      gtk_text_insert (GTK_TEXT (help_text), normal_font, 
		       &help_text->style->black, NULL,
		       inStrg, -1);
      continue;
    }
    

    /* special, set flag                                          */
    if (inStrg[0] == '<') italic = ON;
    if (inStrg[0] == '{') fixed  = ON;
    if (inStrg[0] == '[') bold   = ON;


    /* tagged word - italic                                       */
    while (inStrg[0] != EOF && italic == ON) {
     inStrg[0] = fgetc (inFile);
     if (inStrg[0] == '>') italic = OFF;
     if (inStrg[0] == '#') inStrg[0] = ' ';
     if (italic == ON)
        gtk_text_insert (GTK_TEXT (help_text), italic_font, 
			 &help_text->style->black, NULL,
			 inStrg, -1);
    }

    /* tagged word - bold                                         */
    while (inStrg[0] != EOF && bold == ON) {
     inStrg[0] = fgetc (inFile);
     if (inStrg[0] == ']') bold = OFF; 
     if (inStrg[0] == '#') inStrg[0] = ' ';
     if (bold == ON)
        gtk_text_insert (GTK_TEXT (help_text), bold_font, 
			 &help_text->style->black, NULL,
			 inStrg, -1);
    }

    /* tagged word - fixed                                        */
    while (inStrg[0] != EOF && fixed == ON) {
     inStrg[0] = fgetc (inFile);
     if (inStrg[0] == '}') fixed = OFF; 
     if (inStrg[0] == '#') inStrg[0] = ' ';
     if (fixed == ON)
        gtk_text_insert (GTK_TEXT (help_text), fixed_font, 
			 &help_text->style->black, NULL,
			 inStrg, -1);
    }

  }


  /* ---------------- end parse help file ----------------------- */

  /* fclose(inFile); */

#ifdef HAVE_FORTUNE

  /* ---------------- add some wits ----------------------------- */

  sprintf(witty, "( %s ) 2>&1", "fortune -s | fold -w 64 -s");
  if ((wits = popen (witty, "r")) == NULL) {
      WARNING (_("No witty epigrams"));
  } else {
      inStrg[1] = '\0'; inStrg[0] = '-';
      for (i = 1; i < 50; ++i) {
        gtk_text_insert (GTK_TEXT (help_text), italic_font, 
			 &help_text->style->black, 
			 NULL, inStrg, -1);
      }
      inStrg[0] = '\n';
      gtk_text_insert (GTK_TEXT (help_text), italic_font, 
		       &help_text->style->black, 
		       NULL, inStrg, -1);
      while (1) {
        inStrg[0] = fgetc (wits);

        /* End of file, we are done */
        if (inStrg[0] == EOF) break;

        gtk_text_insert (GTK_TEXT (help_text), italic_font, 
			 &help_text->style->black, 
			 NULL, inStrg, -1);
        continue;
      }
  }
  pclose(wits);
#endif

  /* Thaw the text widget, allowing the updates to become visible */  

  gtk_text_thaw (GTK_TEXT (help_text));

  return;

}

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann
 @version   1.0
 @short     Supply alias for locales
 @param     (void) 
 @return    (static const char) *alias_locale   
 @heading   Graphical User Interface
*******************************************************************/
const char *alias_locale (const char *str)
{
  if (strncmp(str, "de", 2) == 0)
    return "de";
  if (strncmp(str, "german", 6) == 0)
    return "de";
  return str;
}

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann
 @version   1.0
 @short     Guess value of current locale from environment variables
 @param     (void) 
 @return    (static const char) *guess_locale   
 @heading   Graphical User Interface
*******************************************************************/
const char *guess_the_locale ()
{

#ifdef ENABLE_NLS
  const char *the_locale;

  /* highest priority value is the `LANGUAGE' environment         */
  /*   variable.  This is a GNU extension.                        */
  the_locale = getenv ("LANGUAGE");
  if (the_locale != NULL && the_locale[0] != '\0')
    return alias_locale (the_locale);

  /* Setting of LC_ALL overwrites all other.                      */
  the_locale = getenv ("LC_ALL");
  if (the_locale != NULL && the_locale[0] != '\0')
    return alias_locale (the_locale);

  /* next comes LC_MESSAGES                                       */
  the_locale = getenv ("LC_MESSAGES");
  if (the_locale != NULL && the_locale[0] != '\0')
    return alias_locale (the_locale);

  /* Last possibility is the LANG environment variable.           */
  the_locale = getenv ("LANG");
  if (the_locale != NULL && the_locale[0] != '\0')
    return alias_locale (the_locale);
#endif

  /* default is C                                                 */
  return "C";
}

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Replace current help text
 @param     (GtkWidget) *widget  Discarded
 @param     (gpointer)  *data    Number of help file
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void HelpNew (GtkWidget *widget, gpointer *data)
{
  FILE        *inFile;                /* filehandle for help file */
  int  i;                             /* Number of help file      */

  /* --- search helpfile, first in DATAFLS, then in current dir   */

  sscanf ((char *) data, "%d", &i);

  /* 17 = helpxx.txt + '/doc//' + '\0'                          */

  /* priority for DATAFLS directory                               */
  strncpy(my_full_helpfile, data_doc_fls(), 
	  ( 
	   sizeof(my_full_helpfile) 
	    - (17 + strlen( guess_the_locale()) ) 
	   )
	  );
  /* strcat(my_full_helpfile, "/doc/"); */
  strcat(my_full_helpfile, "/");
  strcat(my_full_helpfile, guess_the_locale() );
  strcat(my_full_helpfile, "/");
  strcat(my_full_helpfile, my_helpfile[i]);

  /* next try current directory/doc                               */
  if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {
    my_full_helpfile[0] = '.';
    my_full_helpfile[1] = '\0';
    strcat(my_full_helpfile, "/doc/");
    strcat(my_full_helpfile, guess_the_locale() );
    strcat(my_full_helpfile, "/");
    strcat(my_full_helpfile, my_helpfile[i]);

    /* not found -> error message                                 */
    if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {
      /* wrong language ? try default                             */
      if (strcmp("C", guess_the_locale() ) != 0) {
	strncpy(my_full_helpfile, data_doc_fls(), 
		( 
		 sizeof(my_full_helpfile) 
		 - (17 + strlen( guess_the_locale()) ) 
		 )
		);
	strcat(my_full_helpfile, "/C/");
	strcat(my_full_helpfile, my_helpfile[i]);
      }
      /* not found -> error message                               */
      if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {
	make_dialog (_(errmsg[11]));
	return;
      }
    }

  }

  delete_help();

  WriteHelpText(inFile);

  fclose(inFile);

  return;
}


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Callback function for Help/Context button
 @param     (GtkWidget) *widget  Discarded
 @param     (gpointer)  *data    Number of help file
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void help_event (GtkWidget *widget, gpointer *data)
{

  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *vscrollbar;
  GtkWidget *table;

  FILE   *inFile;                     /* filehandle for help file */
  int    page = 0;                    /* number of help file      */
  char   topage[3];                   /* the page                 */

  /* ------- which helpfile to display  -----------------------   */


  if ( data == NULL) {  
    page = gtk_notebook_current_page( GTK_NOTEBOOK(notebook) );
    page = page + 1;  
    if (page < 0) page = 0;
    if (page < 100) 
      sprintf(topage, "%2d", page);
    else            
      sprintf(topage, "%2d", 0);
  } else {
    sscanf((char *) data, "%d", &page);
  }

  /* ------- check whether already open  ----------------------   */

  if (HelpIsOn == ON) {
         if (data != NULL) 
	   HelpNew (NULL, (gpointer) data);
	 else 
	   HelpNew (NULL, (gpointer) topage);
         return;
  }

  /* --- search helpfile, first in DATAFLS, then in current dir   */

  /* 19 = helpxx.txt + '/doc/nn/' + '\0'                          */

  /* priority for DATAFLS directory                               */
  strncpy(my_full_helpfile, data_doc_fls(), 
	  ( 
	   sizeof(my_full_helpfile) 
	    - (17 + strlen( guess_the_locale()) ) 
	   )
	  );
  /* strcat(my_full_helpfile, "/doc/"); */
  strcat(my_full_helpfile, "/");
  strcat(my_full_helpfile, guess_the_locale() );
  strcat(my_full_helpfile, "/");
  strcat(my_full_helpfile, my_helpfile[page]);

  /* next try current directory/doc                               */
  if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {

    my_full_helpfile[0] = '.';
    my_full_helpfile[1] = '\0';
    strcat(my_full_helpfile, "/doc/");
    strcat(my_full_helpfile, guess_the_locale() );
    strcat(my_full_helpfile, "/");
    strcat(my_full_helpfile, my_helpfile[page]);


    /* not found                                                  */
    if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {
      /* wrong language ? try default                             */
      if (strcmp("C", guess_the_locale() ) != 0) {
	strncpy(my_full_helpfile, data_doc_fls(), 
		( 
		 sizeof(my_full_helpfile) 
		 - (17 + strlen( guess_the_locale()) ) 
		 )
		);
	strcat(my_full_helpfile, "/C/");
	strcat(my_full_helpfile, my_helpfile[page]);
      }
      /* not found -> error message                               */
      if ((inFile = fopen (my_full_helpfile, "r")) == NULL) {
	make_dialog (_(errmsg[11]));
	return;
      }
    }

  }

  /* ----------- open help viewer -----------------------------   */


  help_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize (help_window, 580, 500);
  gtk_window_set_policy (GTK_WINDOW(help_window), TRUE, TRUE, FALSE);  
  gtk_signal_connect (GTK_OBJECT (help_window), "destroy",
                        (GtkSignalFunc) close_help, 
                         GTK_WIDGET (help_window));
  gtk_window_set_title (GTK_WINDOW (help_window), _("Help"));
  gtk_container_border_width (GTK_CONTAINER (help_window), 0);


  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (help_window), vbox);
  gtk_widget_show (vbox);

  table = gtk_table_new (16, 5, TRUE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
  gtk_widget_show (table);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_table_attach (GTK_TABLE(table), hbox, 0, 4, 0, 16,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
  gtk_widget_show (hbox);


  /* ---------  Create the GtkText widget ----------------------  */

  help_text = gtk_text_new (NULL, NULL);
  gtk_text_set_editable (GTK_TEXT (help_text), FALSE);
  gtk_box_pack_start (GTK_BOX (hbox), help_text, TRUE, TRUE, 0);
  gtk_widget_show (help_text);
  
  /* ------- Add a vertical scrollbar to the GtkText widget ----- */

  vscrollbar = gtk_vscrollbar_new (GTK_TEXT (help_text)->vadj);
  gtk_box_pack_start(GTK_BOX(hbox), vscrollbar, FALSE, FALSE, 0);
  gtk_widget_show (vscrollbar);

  /* ----------- create buttons and add ------------------------- */

  button = gtk_button_new_with_label (_("Menu Bar"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "0");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 0, 1,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Basic"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "1");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 1, 2,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Advanced"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "2");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 2, 3,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Plot Options"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "3");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 3, 4,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Data Fitting"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "4");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 4, 5,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Spots Primary"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "5");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 5, 6,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Spots Secondary"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "6");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 6, 7,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_("Third Light"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (HelpNew), (gpointer) "7");
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 7, 8,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 0);
  gtk_widget_show (button);

#ifdef HAVE_GNOME
  button = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
#else
  button = gtk_button_new_with_label (_("Close"));
#endif
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) close_help,
                      GTK_WIDGET (help_window));
  gtk_table_attach (GTK_TABLE(table), button, 4, 5, 10, 12,
        GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 8, 8);
  (GTK_WIDGET_FLAGS (button)  |= (GTK_CAN_DEFAULT));
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  /* -------- Realizing the widget to insert some text ---------- */

  gtk_widget_realize (help_text);

  /* ------------------- set flag   ----------------------------- */

  HelpIsOn = ON;

  /* ------------------- insert text ---------------------------- */

  WriteHelpText(inFile);

  fclose(inFile);

  /* ------------- and show it ---------------------------------- */

  gtk_widget_show (help_window);

  return;
}

#endif
