/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

/* ANSI C forbids an empty source file, so put this outside                */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>

#include "Light.h"

#ifdef _WITH_GTK

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Validate input data 
 @param     (void)    
 @return    (void)   
 @heading   Graphical User Interface
 ****************************************************************************/
void check_data ()
{
  register unsigned long    i, j;              /* loop variables         */
  int                       FitDim = 0;        /* dimension of fit       */
  double                    G_SI = 6.6726e-11; /* gravitational constant */
  char                      InitString[64];    /* entry string           */
  char                      ErrMsg[256];       /* error message          */
  

  for (i = 0; i < 32; ++i) {
     if (Flags.simplex[i] == ON) ++FitDim; 
  }
  if (FitDim < 2 && Flags.WantFit == ON) {
     Flags.WantFit = OFF; 
     WARNING (_("Less than two Parameters to fit, discard Fit Option\n"));
  }


  /* no overcontact if elliptic                                          */

  if ((Flags.elliptic == ON) && (Binary[Primary].RocheFill >= (1+FLT_EPSILON)))
      {
       Binary[Primary].RocheFill = 1.0;
       sprintf(InitString, "%6.3f", Binary[Primary].RocheFill);
       gtk_entry_set_text (GTK_ENTRY (e_103), InitString);       
       WARNING (_("Eccentric Orbit --> Decrease Roche Fill Factor to One\n"));
      }

  if ((Flags.elliptic == ON) && (Binary[Secondary].RocheFill>=(1+FLT_EPSILON)))
    {
       Binary[Secondary].RocheFill = 1.0;
       sprintf(InitString, "%6.3f", Binary[Secondary].RocheFill);
       gtk_entry_set_text (GTK_ENTRY (e_104), InitString);
       WARNING (_("Eccentric Orbit --> Decrease Roche Fill Factor to One\n"));
    }



  /* common fillfactor if overcontact                                    */

  if ((Binary[Primary].RocheFill >= (1+FLT_EPSILON)) 
      || (Binary[Secondary].RocheFill >= (1+FLT_EPSILON)))
    {
      if (Binary[Primary].RocheFill != Binary[Secondary].RocheFill)
	WARNING (_("Overcontact --> Set Common Roche Fill Factor to Maximum of Both\n"));
      Binary[Primary].RocheFill = 
	MAX(Binary[Primary].RocheFill,Binary[Secondary].RocheFill);
      Binary[Secondary].RocheFill = Binary[Primary].RocheFill;
      sprintf(InitString, "%6.3f", Binary[Secondary].RocheFill);
      gtk_entry_set_text (GTK_ENTRY (e_104), InitString);
      sprintf(InitString, "%6.3f", Binary[Primary].RocheFill);
      gtk_entry_set_text (GTK_ENTRY (e_103), InitString); 
      Flags.fill = ON;
    } else {
      Flags.fill = OFF;
    }

  /* Limits for Mass Ratio                                               */

  if (Flags.fill == ON) {
    if (Binary[Primary].Mq <= (LIM_MQO_L- FLT_EPSILON) ) {
       Binary[Primary].Mq = LIM_MQO_L;
       Binary[Secondary].Mq = 1.0/Binary[Primary].Mq;
       WARNING (_("Overcontact --> Increase Mass Ratio to limit\n"));
        sprintf(InitString, "%8.5f", Binary[Primary].Mq);
        gtk_entry_set_text (GTK_ENTRY (e_101), InitString);
    }
  } else {
    if (Binary[Primary].Mq <= (LIM_MQ_L- FLT_EPSILON) ) {
       Binary[Primary].Mq = LIM_MQ_L;
       Binary[Secondary].Mq = 1.0/Binary[Primary].Mq;
       WARNING (_("Avoid Numerical Problems --> Increase Mass Ratio to limit\n"));
        sprintf(InitString, "%8.5f", Binary[Primary].Mq);
        gtk_entry_set_text (GTK_ENTRY (e_101), InitString);
    }
  }  

  /* no nonsynchroneous rotation if overcontact                          */

  if (Flags.fill == ON) {
   if ( ( fabs(Binary[Primary].Fratio-1.0) <= FLT_EPSILON 
	  && Flags.asynchron1 == ON ) || 
	( fabs(Binary[Secondary].Fratio-1.0) <= FLT_EPSILON
	  && Flags.asynchron2 == ON) ) {
      WARNING (_("Overcontact --> Reset NonSync Rotation to Unity\n"));  
      Binary[Primary].Fratio   = 1.0;
      Binary[Secondary].Fratio = 1.0;
      Flags.asynchron1 = OFF;
      Flags.asynchron2 = OFF;
      gtk_toggle_button_set_state 
	(GTK_TOGGLE_BUTTON (t_fpri_button), FALSE);
      gtk_toggle_button_set_state 
	(GTK_TOGGLE_BUTTON (t_fsec_button), FALSE);
      sprintf(InitString, "%6.3f", Binary[Primary].Fratio);
      gtk_entry_set_text (GTK_ENTRY (e_109), InitString);
      sprintf(InitString, "%6.3f", Binary[Secondary].Fratio);
      gtk_entry_set_text (GTK_ENTRY (e_110), InitString);
   }
  }


  /* -------------------  set default band to plot        -------------- */

  if (Flags.PlotBand == -1) Flags.PlotBand = Vmag;

  ComputeGravDark();

  /* >>>>>>>>>>>  consistency of mass,period, and distance <<<<<<<<<<<<< */

  if (fabs(Orbit.TruePeriod )  >= FLT_EPSILON 
	   && fabs(Orbit.TrueMass ) >= FLT_EPSILON ) {
      Orbit.TrueDistance = 
        pow((Orbit.TruePeriod*Orbit.TruePeriod) 
	    * G_SI * Orbit.TrueMass / (4.0 * (PI*PI)), 
            (1.0/3.0));
     } 
  else if (fabs(Orbit.TruePeriod)       >= FLT_EPSILON  
      && fabs(Orbit.TrueDistance ) >= FLT_EPSILON ) {
      Orbit.TrueMass = 
        (Orbit.TrueDistance * Orbit.TrueDistance) * Orbit.TrueDistance 
	* 4.0 * PI * PI 
        / (G_SI * (Orbit.TruePeriod * Orbit.TruePeriod));
  } 
  else if (fabs(Orbit.TrueDistance) >= FLT_EPSILON
	   && fabs(Orbit.TrueMass ) >= FLT_EPSILON ) {
          Orbit.TruePeriod = 
           sqrt((Orbit.TrueDistance*Orbit.TrueDistance) 
		* Orbit.TrueDistance * 4.0 * (PI*PI)
                / (G_SI * Orbit.TrueMass));
        } 
  else {

  /* ------  default ------                                              */
    Orbit.TruePeriod   = 2.2226262e7;  /* 0.7 year                       */
    Orbit.TrueDistance = 1.496e11;     /* one astronomical unit          */
    Orbit.TrueMass     = 3.978e30;     /* two solar mass                 */
                                       /* M*G/(4*PI*PI) = D*D*D/P*P      */
                                       /* (Keplers Third Law)            */
  }

    sprintf(InitString, "%8.4f", Orbit.TruePeriod/86400);
    gtk_entry_set_text (GTK_ENTRY (e_302), InitString);
    sprintf(InitString, "%8.4f", Orbit.TrueMass/1.989E30 );
    gtk_entry_set_text (GTK_ENTRY (e_303), InitString);
    sprintf(InitString, "%6.4f", Orbit.TrueDistance/6.960E8);
    gtk_entry_set_text (GTK_ENTRY (e_304), InitString);


    if(Binary[Primary].Mq <= (LIM_MQ_L-FLT_EPSILON) ) {
      sprintf(ErrMsg, 
	      _("mass ratio must be in [%f, %f]\n"), LIM_MQ_L, LIM_MQ_H);
      WARNING (ErrMsg);
      Binary[Primary].Mq = LIM_MQ_L;
      Binary[Secondary].Mq = 1.0/Binary[Primary].Mq;
    } 
    if   (Binary[Primary].Mq >= (LIM_MQ_H+FLT_EPSILON) ) {
      sprintf(ErrMsg, 
	      _("mass ratio must be in [%f, %f]\n"), LIM_MQ_L, LIM_MQ_H);
      WARNING (ErrMsg);
      Binary[Primary].Mq = LIM_MQ_H;
      Binary[Secondary].Mq = 1.0/Binary[Primary].Mq;
    }
 
    if(Orbit.Inclination <= (LIM_IN_L-FLT_EPSILON) ) {
      sprintf(ErrMsg, 
	      _("inclination must be in [%f, %f]\n"), LIM_IN_L, LIM_IN_H);
      WARNING (ErrMsg);
      Orbit.Inclination = LIM_IN_L;
    }
    if  (Orbit.Inclination >= (RTOD*LIM_IN_H + FLT_EPSILON) ) { 
      sprintf(ErrMsg, 
	      _("inclination must be in [%f, %f]\n"), LIM_IN_L, LIM_IN_H);
      WARNING (ErrMsg);
      Orbit.Inclination = LIM_IN_H;
    }

    if(Binary[Primary].RocheFill <= (LIM_RF_L-FLT_EPSILON) ) {
      sprintf(ErrMsg, 
	      _("Roche lobe filling must be in [%f, %f]\n"), 
	      LIM_RF_L, LIM_RO_H);
      WARNING (ErrMsg);
      Binary[Primary].RocheFill = LIM_RF_L;
    }
    if(Binary[Primary].RocheFill >= (LIM_RO_H + FLT_EPSILON) ) {
      sprintf(ErrMsg, _("Roche lobe filling must be in [%f, %f]\n"), 
	      LIM_RF_L, LIM_RO_H);
      WARNING (ErrMsg);
      Binary[Primary].RocheFill = LIM_RO_H;
    }

    if(Binary[Secondary].RocheFill <= (LIM_RF_L-FLT_EPSILON) ) {
      sprintf(ErrMsg, _("Roche lobe filling must be in [%f, %f]\n"), 
	      LIM_RF_L, LIM_RO_H);
      WARNING (ErrMsg);
      Binary[Secondary].RocheFill = LIM_RF_L;
    }
    if(Binary[Secondary].RocheFill >= (LIM_RO_H + FLT_EPSILON) ) {
      sprintf(ErrMsg, _("Roche lobe filling must be in [%f, %f]\n"), 
	      LIM_RF_L, LIM_RO_H);
      WARNING (ErrMsg);
      Binary[Secondary].RocheFill = LIM_RO_H;
    }

    if (Flags.blackbody == OFF) {

     if(Binary[Primary].Temperature <= (LIM_TM_L-FLT_EPSILON) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TM_L, LIM_TM_H);
       WARNING (ErrMsg);
       Binary[Primary].Temperature = LIM_TM_L;
     }
     if(Binary[Primary].Temperature >= (LIM_TM_H + FLT_EPSILON*LIM_TM_H) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TM_L, LIM_TM_H);
       WARNING (ErrMsg);
       Binary[Primary].Temperature = LIM_TM_H;
     }

     if(Binary[Secondary].Temperature <= (LIM_TM_L-FLT_EPSILON) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TM_L, LIM_TM_H);
       WARNING (ErrMsg);
       Binary[Secondary].Temperature = LIM_TM_L;
     }
     if(Binary[Secondary].Temperature >= (LIM_TM_H + FLT_EPSILON*LIM_TM_H) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TM_L, LIM_TM_H);
       WARNING (ErrMsg);
       Binary[Secondary].Temperature = LIM_TM_H;
     }

    } else {

     if(Binary[Primary].Temperature <= (LIM_TB_L-FLT_EPSILON) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TB_L, LIM_TB_H);
       WARNING (ErrMsg);
       Binary[Primary].Temperature = LIM_TB_L;
     }
     if(Binary[Primary].Temperature >= (LIM_TB_H + FLT_EPSILON*LIM_TB_H) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TB_L, LIM_TB_H);
       WARNING (ErrMsg);
       Binary[Primary].Temperature = LIM_TB_H;
     }

     if(Binary[Secondary].Temperature <= (LIM_TB_L-FLT_EPSILON) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TB_L, LIM_TB_H);
       WARNING (ErrMsg);
       Binary[Secondary].Temperature = LIM_TB_L;
     }
     if(Binary[Secondary].Temperature >= (LIM_TB_H + FLT_EPSILON*LIM_TB_H) ) {
       sprintf(ErrMsg, _("temperature must be in [%f, %f]\n"), 
	       LIM_TB_L, LIM_TB_H);
       WARNING (ErrMsg);
       Binary[Secondary].Temperature = LIM_TB_H;
     }
    }

    if (PhaseSteps > PHASESTEPS || PhaseSteps < 3) {
      sprintf(ErrMsg, 
	      _("lightcurve steps out of range [3,%3d], default (%d) used"), 
                    PHASESTEPS, MIN(PHASESTEPS, 80) );
      WARNING(ErrMsg)
      PhaseSteps = MIN(PHASESTEPS, 80);
    }

    if ( (Orbit.LambdaZero <= (LIM_PRF_L-FLT_EPSILON) )
	  || (Orbit.LambdaZero >= (LIM_PRF_H+FLT_EPSILON*LIM_PRF_H) ) ) {
       sprintf(ErrMsg,
               _("line profile rest wavelength out of range [%f, %f]"), 
	       LIM_PRF_L, LIM_PRF_H );
       WARNING(ErrMsg)
       Orbit.LambdaZero = LAMDAZERO_DEFAULT;
     } 

     if (Flags.reflect > 9 || Flags.reflect < 0) {
          WARNING(_("invalid reflection iteration count [0,9]"));
          Flags.reflect = 3;
     }

     if (Flags.elliptic == ON) {
       if (Orbit.Excentricity < LIM_EX_L){
	 sprintf(ErrMsg, _("eccentricity must be in [%f, %f]\n"), 
		 LIM_EX_L, LIM_EX_H);
	 WARNING (ErrMsg);
         Orbit.Excentricity = LIM_EX_L;
       } 
       else if ( Orbit.Excentricity >= LIM_EX_H) {
	 sprintf(ErrMsg, _("eccentricity must be in [%f, %f]\n"), 
		 LIM_EX_L, LIM_EX_H);
	 WARNING (ErrMsg);
         Orbit.Excentricity = LIM_EX_H;
       }
     } 

     if (Orbit.Omega < LIM_PA_L){
       sprintf(ErrMsg, _("periastron length must be in [%f, %f]\n"), 
	       LIM_PA_L, LIM_PA_H);
       WARNING (ErrMsg);
       Orbit.Omega  = LIM_PA_L;
     } 
     else if ( Orbit.Omega > LIM_PA_H) {
       sprintf(ErrMsg, _("periastron length must be in [%f, %f]\n"), 
	       LIM_PA_L, LIM_PA_H);
       WARNING (ErrMsg);
       Orbit.Omega = LIM_PA_H;
     } 

     if(Binary[Primary].Fratio < LIM_FR_L){
       sprintf(ErrMsg, _("rotation rate must be in [%f, %f]\n"), 
	       LIM_FR_L, LIM_FR_H);
       WARNING (ErrMsg);
       Binary[Primary].Fratio = LIM_FR_L;
     }  
     else if (Binary[Primary].Fratio > LIM_FR_H  ){ 
       sprintf(ErrMsg, _("rotation rate must be in [%f, %f]\n"), 
	       LIM_FR_L, LIM_FR_H);
       WARNING (ErrMsg);
       Binary[Primary].Fratio = LIM_FR_H;
     }  

     if(Binary[Secondary].Fratio < LIM_FR_L){
       sprintf(ErrMsg, _("rotation rate must be in [%f, %f]\n"), 
	       LIM_FR_L, LIM_FR_H);
       WARNING (ErrMsg);
       Binary[Secondary].Fratio = LIM_FR_L;
     }
     else if (Binary[Secondary].Fratio > LIM_FR_H ){ 
       sprintf(ErrMsg, _("rotation rate must be in [%f, %f]\n"), 
	       LIM_FR_L, LIM_FR_H);
       WARNING (ErrMsg);
       Binary[Secondary].Fratio = LIM_FR_H;
     }  
  
     for (i = 0; i < NUM_MAG; ++i) {
       if (Orbit.Third[i] <= LIM_3L_L)  Orbit.Third[i] = LIM_3L_L; 
       if (Orbit.Third[i] >= LIM_3L_H ) Orbit.Third[i] = LIM_3L_H; 
     }

     for (i = 0; i < 2; ++i) {
       for (j = 0; j < 2; ++j) {
	 if(Spot[i][j].radius < LIM_SRA_L) {
	   sprintf(ErrMsg, _("spot radius must be in [%f, %f]\n"), 
		   LIM_SRA_L, LIM_SRA_H);
	   WARNING (ErrMsg);
           Spot[i][j].radius = LIM_SRA_L;
	 }
	 if(Spot[i][j].radius > LIM_SRA_H) {
	   sprintf(ErrMsg, _("spot radius must be in [%f, %f]\n"), 
		   LIM_SRA_L, LIM_SRA_H);
	   WARNING (ErrMsg);
           Spot[i][j].radius = LIM_SRA_H;
	 }
         if(Spot[i][j].longitude < LIM_SLO_L) {
	   sprintf(ErrMsg, _("spot longitude must be in [%f, %f]\n"), 
		   LIM_SLO_L, LIM_SLO_H);
	   WARNING (ErrMsg);
           Spot[i][j].longitude = LIM_SLO_L;
	 }
	 if(Spot[i][j].longitude > LIM_SLO_H) {
	   sprintf(ErrMsg, _("spot longitude must be in [%f, %f]\n"), 
		   LIM_SLO_L, LIM_SLO_H);
	   WARNING (ErrMsg);
           Spot[i][j].longitude = LIM_SLO_H;
	 }
         if(Spot[i][j].latitude < LIM_SLA_L) {
	   sprintf(ErrMsg, _("spot latitude must be in [%f, %f]\n"), 
		   LIM_SLA_L, LIM_SLA_H);
	   WARNING (ErrMsg);
           Spot[i][j].latitude = LIM_SLA_L;
	 }
	 if(Spot[i][j].latitude > LIM_SLA_H) {
	   sprintf(ErrMsg, _("spot latitude must be in [%f, %f]\n"), 
		   LIM_SLA_L, LIM_SLA_H);
	   WARNING (ErrMsg);
           Spot[i][j].latitude = LIM_SLA_H;
	 }
         if(Spot[i][j].dimfactor < LIM_SDF_L) {
	   sprintf(ErrMsg, _("spot dimfactor must be in [%f, %f]\n"), 
		   LIM_SDF_L, LIM_SDF_H);
	   WARNING (ErrMsg);
           Spot[i][j].dimfactor = LIM_SDF_L;
	 }
	 if(Spot[i][j].dimfactor > LIM_SDF_H) {
	   sprintf(ErrMsg, _("spot dimfactor must be in [%f, %f]\n"), 
		   LIM_SDF_L, LIM_SDF_H);
	   WARNING (ErrMsg);
           Spot[i][j].dimfactor = LIM_SDF_H;
	 }
       }
     }

     /* >>>>>>>>>>>>>>>>>  Update the GUI <<<<<<<<<<<<<<<<<< */

     UpdateGui();
     return;
}

#endif
