/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


/* ANSI C forbids an empty source file, so put this outside                */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Light.h"

#ifdef _WITH_GTK


#include "pixmaps/mcomp.xpm"
#include "pixmaps/manim.xpm"
#include "pixmaps/mplot.xpm"
#include "pixmaps/mhelp.xpm"

#ifdef HAVE_GNOME

GnomeUIInfo toolbar[] = {
  GNOMEUIINFO_ITEM(N_("Compute"), 
		   N_("Compute a synthetic lightcurve w/ current options"), 
		   lightcurve_event, mcomp_xpm),
  GNOMEUIINFO_TOGGLEITEM(N_("Animate"), 
		   N_("Toggle animated view of binary during lightcurve computation"),
		   toggle_animate, 
		   manim_xpm),
  GNOMEUIINFO_ITEM(N_("Plot"), N_("Plot computed lightcurve"),
		   lightcurveplot_event, mplot_xpm),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM(N_("Info"), 
		   N_("Help on the active notebook page"), 
		   help_event, mhelp_xpm),
  GNOMEUIINFO_END
};

/* ---------------- status bar            ------------------ */

  GnomeAppBar   *status_bar;

#else

  GtkWidget     *status_bar;
  gint           context_id;

#endif



/*************************************************************
  
   Variables for temporary storage of data

**************************************************************/

int        TG_reflectstore;      /* the reflection flag      */
int        TG_graphstore;        /* the plot flag            */
int        TG_visualstore;       /* the visualize flag       */

/* -------- required for arbitrary ON/OFF of spots --------- */

int        TG_P1spot = 0, TG_P2spot = 0; 
int        TG_S1spot = 0, TG_S2spot = 0;
int        SpotActive[4] = { 0, 0, 0, 0 };
int        TG_1spots = 0, TG_2spots = 0;
int        FitSpotS[16] = { 0 };
SpotLight  SpotStore[2][2];            


/*************************************************************
  
   Global widgets

**************************************************************/


/* ---------------  entry fields (basic)  ------------------ */

  GtkWidget   *e_101;         /* mass ratio                  */
  GtkWidget   *e_102;         /* inclination                 */
  GtkWidget   *e_103;         /* primary rochefill           */
  GtkWidget   *e_104;         /* secondary rochefill         */
  GtkWidget   *e_105;         /* primary temperature         */
  GtkWidget   *e_106;         /* secondary temperature       */

/* ----------------  entry fields (advanced)  -------------- */

  GtkWidget   *e_107;         /* eccentricity                */
  GtkWidget   *e_108;         /* periastron length           */
  GtkWidget   *e_109;         /* primary F ratio             */
  GtkWidget   *e_110;         /* secondary F ratio           */
  GtkWidget   *e_111;         /* line rest wavelength        */
  GtkWidget   *e_112;         /* # of lightcurve steps       */
  GtkWidget   *e_113;         /* # of reflection iterations  */
		
/* ----------------  entry fields (fit)  ------------------- */

  GtkWidget   *e_name;        /* system name                 */
  GtkWidget   *e_301;         /* fit tolerance               */
  GtkWidget   *e_302;         /* period                      */
  GtkWidget   *e_303;         /* mass                        */
  GtkWidget   *e_304;         /* separation                  */
  GtkWidget   *e_305;         /* step1                       */
  GtkWidget   *e_306;         /* step2                       */
		
/* ---------------  entry fields (spots)  ------------------ */

  GtkWidget   *e_401;         /* spot1 primary   longitude   */
  GtkWidget   *e_402;         /* spot1 primary   latitude    */
  GtkWidget   *e_403;         /* spot1 primary   radius      */
  GtkWidget   *e_404;         /* spot1 primary   dimfactor   */
  GtkWidget   *e_405;         /* spot2 primary   longitude   */
  GtkWidget   *e_406;         /* spot2 primary   latitude    */
  GtkWidget   *e_407;         /* spot2 primary   radius      */
  GtkWidget   *e_408;         /* spot2 primary   dimfactor   */
  GtkWidget   *e_409;         /* spot1 secondary longitude   */
  GtkWidget   *e_410;         /* spot1 secondary latitude    */
  GtkWidget   *e_411;         /* spot1 secondary radius      */
  GtkWidget   *e_412;         /* spot1 secondary dimfactor   */
  GtkWidget   *e_413;         /* spot2 secondary longitude   */
  GtkWidget   *e_414;         /* spot2 secondary latitude    */
  GtkWidget   *e_415;         /* spot2 secondary radius      */
  GtkWidget   *e_416;         /* spot2 secondary dimfactor   */
		
/* ---------------  entry fields (third light)  ------------ */

  GtkWidget   *e_501;         /* third light U               */
  GtkWidget   *e_502;         /* third light B               */
  GtkWidget   *e_503;         /* third light V               */
  GtkWidget   *e_504;         /* third light R               */
  GtkWidget   *e_505;         /* third light I               */
  GtkWidget   *e_506;         /* third light J               */
  GtkWidget   *e_507;         /* third light H               */
  GtkWidget   *e_508;         /* third light K               */
  GtkWidget   *e_509;         /* third light u               */
  GtkWidget   *e_510;         /* third light v               */
  GtkWidget   *e_511;         /* third light b               */
  GtkWidget   *e_512;         /* third light y               */

/* ------------------  toggle buttons  --------------------- */

  GtkWidget   *t_anim_button;       /* toggle animated       */
  GtkWidget   *t_eccentric_button;  /* toggle eccentric      */
  GtkWidget   *t_fpri_button;       /* toggle asynchron pri. */
  GtkWidget   *t_fsec_button;       /* toggle asynchron sec. */
  GtkWidget   *t_ssp1_button;       /* toggle spot1 sec.     */
  GtkWidget   *t_ssp2_button;       /* toggle spot2 sec.     */
  GtkWidget   *t_psp1_button;       /* toggle spot1 pri.     */
  GtkWidget   *t_psp2_button;       /* toggle spot2 pri.     */

/* ------------------------  items to grey out  ------------ */

  GtkWidget   *menu_plot;           /* plot lightcurve       */
  GtkWidget   *menu_profile;        /* line profiles         */
  GtkWidget   *ome_button;          /* fit omega             */
  GtkWidget   *ecc_button;          /* fit eccentricity      */
  GtkWidget   *as1_button;          /* fit async primary     */
  GtkWidget   *as2_button;          /* fit async secondary   */
  GtkWidget   *ref_label;           /* reflection label      */


/* ---------------------  misc            ------------------ */

  GtkWidget   *progress_bar;        /* progress bar          */
  GtkWidget   *top_window;          /* the top window        */
  GtkWidget   *notebook;            /* the notebook          */
  GtkTooltips *tooltips;            /* the tooltips          */


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Push message on status bar
 @tip       function arguments are not used
 @param     (char)  *message       Message
 @return    (void)   
 @heading   Graphical User Interface
*******************************************************************/
void my_appbar_push (char *message)
{
#ifdef HAVE_GNOME
  gnome_appbar_clear_stack(status_bar);
  gnome_appbar_push (status_bar, message);
  while (gtk_events_pending()) gtk_main_iteration();
#else
  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), context_id);
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), context_id, message);
  while (gtk_events_pending()) gtk_main_iteration();
#endif  
  return;
}

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Main GTK routine, creates GUI and waits for input
 @param     (int) argc          The number of arguments
 @param     (char *) argv[]     The argument array
 @return    (int)               The exit status
 @heading   Graphical User Interface
*******************************************************************/
int the_gui (int argc, char *argv[])
{
  GtkWidget   *label;
  GtkWidget   *button;
  GtkWidget   *table;
  GtkWidget   *box1;
  GtkWidget   *box2;
  GtkWidget   *box3;
  GtkWidget   *box4;
  
  GSList      *group;


#ifndef HAVE_GNOME
  GtkWidget   *menu_bar;
  GtkWidget   *handlebox;
  GtkWidget   *toolbar;

  char        my_rc[] = "nightfallrc"; /* theme file                  */
  char        my_full_rc[1024] = "\0"; /* full path to theme file     */
  FILE       *test;                    /* test for file               */
#endif




  char        InitString[64];          /* init string for entry field */
  int         i;                       /* misc                        */
  char        top_title[256];          /* title string                */

  /* >>>>>>>>>>>  initialize  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */  

#ifdef HAVE_GNOME

  int lo;

  GnomeClient *client;
  gint         restarted = 0, dummy;
  gint         os_x = 20, os_y = 20, os_w = 554, os_h = 408;
  gchar        * configFile = NULL;
  gchar        * dataFile   = NULL;
  gchar        * theName    = NULL;
  gchar        *ptr;
  gchar        path[512];
  gint         old_page = 0;

  extern gint   die (GnomeClient *client, gpointer client_data);
  extern gint   save_yourself (GnomeClient        *client,
                gint                phase,
                GnomeRestartStyle   restart_style,
                gint                shutdown,
                GnomeInteractStyle  interact_style,
                gint                fast,
                gpointer            client_data);


  gnome_init("nightfall", NULL, argc, argv);
  client= gnome_master_client ();

  gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                       GTK_SIGNAL_FUNC (save_yourself), NULL);
  gtk_signal_connect (GTK_OBJECT (client), "die",
                       GTK_SIGNAL_FUNC (die), NULL);


#else

  gtk_init (&argc, &argv);

  if (getenv("HOME") != NULL) {

    /* test for $HOME/.nightfallrc     */
    strncpy(my_full_rc, getenv("HOME"), 
	    (sizeof(my_full_rc) - sizeof(my_rc) - 3));
    strcat(my_full_rc, "/.");
    strncat(my_full_rc, my_rc, sizeof(my_rc));

    /* test for $HOME/.gtkrc     */
    if ( (test = fopen(my_full_rc, "r")) == NULL) {
      strncpy(my_full_rc, getenv("HOME"), 
	      (sizeof(my_full_rc) - 8));
      strcat(my_full_rc, "/.gtkrc");
      test = fopen(my_full_rc, "r");
    }
  

    if (test != NULL) {
      /* parse $HOME/.nightfallrc or $HOME/.gtkrc */
      fclose(test);
      gtk_rc_parse(my_full_rc);
    } else { 
      /* try ./nightfallrc */
      my_full_rc[0] = '.';
      my_full_rc[1] = '/';
      my_full_rc[2] = '\0';
      strcat(my_full_rc, my_rc);
      test = fopen(my_full_rc, "r");
      if (test != NULL) {
	fclose(test);
	gtk_rc_parse(my_full_rc);
      } else {
	gtk_rc_parse("./.gtkrc");
      }
    }
  }
#endif

  /* -------------- initialize flags     ---------------------------- */

  TG_reflectstore = Flags.reflect;
  TG_graphstore   = Flags.plot; Flags.plot = OFF;
  TG_visualstore  = Flags.visualize; Flags.visualize = OFF;

  sprintf(top_title, "%10s %5s", PACKAGE, VERSION);

  /* save current spot parameters                                     */

  SpotStore[0][0] = Spot[0][0];   SpotStore[0][1] = Spot[0][1];
  SpotStore[1][0] = Spot[1][0];   SpotStore[1][1] = Spot[1][1];
  TG_1spots   = Flags.Spots1;
  if (Flags.Spots1 > 0) { TG_P1spot = ON; SpotActive[0] = ON; } 
  if (Flags.Spots1 > 1) { TG_P2spot = ON; SpotActive[1] = ON; }
  TG_2spots   = Flags.Spots1;
  if (Flags.Spots2 > 0) { TG_S1spot = ON; SpotActive[2] = ON; }
  if (Flags.Spots2 > 1) { TG_S2spot = ON; SpotActive[3] = ON; }
  for (i=0; i<16; ++i) FitSpotS[i] = Flags.simplex[10+i];


  /* >>>>>>>>>>>  top window      <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */  

  /* -------------- create new window ------------------------------  */

#ifdef HAVE_GNOME

  top_window = gnome_app_new("nightfall", top_title);

  if (GNOME_CLIENT_CONNECTED (client)) {

    if (gnome_client_get_flags (client) & GNOME_CLIENT_RESTORED) {

      restarted = 1;  /* set flag */

      gnome_config_push_prefix (gnome_client_get_config_prefix (client));

      os_x = gnome_config_get_int ("Geometry/x");
      os_y = gnome_config_get_int ("Geometry/y");
      os_w = gnome_config_get_int ("Geometry/w");
      os_h = gnome_config_get_int ("Geometry/h");

      Flags.GN_tooltips = gnome_config_get_int ("Options/HaveTooltips=1");
      Flags.GN_rempos = gnome_config_get_int ("Options/RememberPosition=1");
      Flags.GN_exitconfirm = gnome_config_get_int ("Options/PromptOnQuit=0");

      /* first, we read the config file. As the config file may have
       * commands to read data, we then clear the data, and read
       * those that were in memory when we were terminated.
       */
      configFile = gnome_config_get_string ("Configuration/File");
      if (configFile != NULL) {
	ptr = strrchr(configFile,'/');
	if (ptr == NULL) {
	  ParseConfig(configFile, &dummy);
	} else {
	  ++ptr;
	  ParseConfig(ptr, &dummy);
	}
	g_free(configFile);
      }

      /* set data count for all passbands to zero                     */
      for ( i = 0;  i < (NUM_MAG+2); ++ i) Flags.Passbands[i] = 0;

      /* clear data file list                                         */
      ClearList();

      for (i = 0; i < 12; ++i) {
	snprintf(path,  255, "Data/File%02d", i);
	dataFile = gnome_config_get_string(path);
	if (dataFile != NULL) {
	  Read ( dataFile );
	  g_free( dataFile );
	}
      }

   Flags.fill =                   gnome_config_get_int("Flags/fill");         
   Flags.elliptic =               gnome_config_get_int("Flags/elliptic");     
   Flags.asynchron1 =             gnome_config_get_int("Flags/asynchron1");   
   Flags.asynchron2 =             gnome_config_get_int("Flags/asynchron2");   
   Flags.animate =                gnome_config_get_int("Flags/animate");      
   Flags.plot =                   gnome_config_get_int("Flags/plot");         
   Flags.eps =                    gnome_config_get_int("Flags/eps");          
   Flags.lineprofile =            gnome_config_get_int("Flags/lineprofile");  
   Flags.blackbody =              gnome_config_get_int("Flags/blackbody");    
   Flags.fractional =             gnome_config_get_int("Flags/fractional");   
   Flags.limb =                   gnome_config_get_int("Flags/limb");         
   Flags.reflect =                gnome_config_get_int("Flags/reflect");      
   Flags.anneal =                 gnome_config_get_int("Flags/anneal");      
   Flags.PlotBand =               gnome_config_get_int("Flags/PlotBand");     
                                 					      
   Flags.SimplexTol =             gnome_config_get_float("Flags/STol");
   Flags.Step[0] =                gnome_config_get_float("Flags/Step0");
   Flags.Step[1] =                gnome_config_get_float("Flags/Step1");

   for (lo = 0; lo <  12; ++lo) 
     {
       snprintf(path,  255, "Flags/Dbg%02d", lo);
       Flags.debug[lo] = gnome_config_get_int (path );
   }

   TG_reflectstore = gnome_config_get_int("Flags/reflectstore");
   TG_graphstore   = gnome_config_get_int("Flags/graphstore");
   TG_visualstore  = gnome_config_get_int("Flags/visualstore");

   PhaseSteps =                   gnome_config_get_float("Misc/PhaseSteps"); 
   old_page =                     gnome_config_get_int("Misc/NoteBookPage");

   theName =                      gnome_config_get_string("Orbit/Name");
   if (theName != NULL) 
     {
       strncpy (Orbit.Name, theName, MAX_CFG_INLINE);
       free (theName);
     }

   Orbit.LambdaZero =             gnome_config_get_float("Orbit/LambdaZero"); 
   Orbit.Third[Umag] =            gnome_config_get_float("Orbit/ThirdU");     
   Orbit.Third[Bmag] =            gnome_config_get_float("Orbit/ThirdB");     
   Orbit.Third[Vmag] =            gnome_config_get_float("Orbit/ThirdV");     
   Orbit.Third[Rmag] =            gnome_config_get_float("Orbit/ThirdR");     
   Orbit.Third[Imag] =            gnome_config_get_float("Orbit/ThirdI");     
   Orbit.Third[Jmag] =            gnome_config_get_float("Orbit/ThirdJ");     
   Orbit.Third[Hmag] =            gnome_config_get_float("Orbit/ThirdH");     
   Orbit.Third[Kmag] =            gnome_config_get_float("Orbit/ThirdK");     
   Orbit.Third[umag] =            gnome_config_get_float("Orbit/Thirdu");     
   Orbit.Third[vmag] =            gnome_config_get_float("Orbit/Thirdv");     
   Orbit.Third[bmag] =            gnome_config_get_float("Orbit/Thirdb");     
   Orbit.Third[ymag] =            gnome_config_get_float("Orbit/Thirdy");     
   Orbit.TruePeriod =             gnome_config_get_float("Orbit/Period");     
   Orbit.TrueMass =               gnome_config_get_float("Orbit/Mass");       
   Orbit.TrueDistance =           gnome_config_get_float("Orbit/Distance");   
   Orbit.Excentricity =           gnome_config_get_float("Orbit/Excentr");    
   Orbit.Omega =                  gnome_config_get_float("Orbit/Omega");      
   Orbit.Inclination =            gnome_config_get_float("Orbit/Inclination");
                                 					      
   Binary[Primary].Fratio =       gnome_config_get_float("Binary/FR1");       
   Binary[Secondary].Fratio =     gnome_config_get_float("Binary/FR2");       
   Binary[Primary].Mq =           gnome_config_get_float("Binary/MQ1");       
   Binary[Secondary].Mq =         gnome_config_get_float("Binary/MQ2");       
   Binary[Primary].RocheFill =    gnome_config_get_float("Binary/RocheFill1");
   Binary[Secondary].RocheFill =  gnome_config_get_float("Binary/RocheFill2");
   Binary[Primary].Temperature =  gnome_config_get_float("Binary/Temp1");     
   Binary[Secondary].Temperature =gnome_config_get_float("Binary/Temp2");     
                                 					      
   Flags.Spots1 =                 gnome_config_get_int("Flags/Spots1");     
   Flags.Spots2 =                 gnome_config_get_int("Flags/Spots2");     

   SpotStore[Primary][0].longitude =   gnome_config_get_float("SpotP1/logitude");  
   SpotStore[Primary][0].latitude =    gnome_config_get_float("SpotP1/latitude");  
   SpotStore[Primary][0].radius =      gnome_config_get_float("SpotP1/radius");    
   SpotStore[Primary][0].dimfactor =   gnome_config_get_float("SpotP1/dimfactor"); 
                                 					      
   SpotStore[Primary][1].longitude =   gnome_config_get_float("SpotP2/logitude");  
   SpotStore[Primary][1].latitude =    gnome_config_get_float("SpotP2/latitude");  
   SpotStore[Primary][1].radius =      gnome_config_get_float("SpotP2/radius");    
   SpotStore[Primary][1].dimfactor =   gnome_config_get_float("SpotP2/dimfactor"); 
                                 					      
   SpotStore[Secondary][0].longitude = gnome_config_get_float("SpotS1/logitude");  
   SpotStore[Secondary][0].latitude =  gnome_config_get_float("SpotS1/latitude");  
   SpotStore[Secondary][0].radius =    gnome_config_get_float("SpotS1/radius");    
   SpotStore[Secondary][0].dimfactor = gnome_config_get_float("SpotS1/dimfactor"); 
                                 					      
   SpotStore[Secondary][1].longitude = gnome_config_get_float("SpotS2/logitude");  
   SpotStore[Secondary][1].latitude =  gnome_config_get_float("SpotS2/latitude");  
   SpotStore[Secondary][1].radius =    gnome_config_get_float("SpotS2/radius");    
   SpotStore[Secondary][1].dimfactor = gnome_config_get_float("SpotS2/dimfactor"); 
  
   TG_P1spot = gnome_config_get_int  ("Flags/TGP1");
   TG_P2spot = gnome_config_get_int  ("Flags/TGP2");
   TG_S1spot = gnome_config_get_int  ("Flags/TGS1");
   TG_S2spot = gnome_config_get_int  ("Flags/TGS2");

   for (lo = 0; lo <  4; ++lo) 
     {
       snprintf(path,  255, "SpotActive/%02d", lo);
       SpotActive[lo] = gnome_config_get_int (path );
   }

   for (lo = 0; lo <  16; ++lo) 
     {
       snprintf(path,  255, "SpotFit/%02d", lo);
       FitSpotS[lo] = gnome_config_get_int (path );
   }

   for (lo = 0; lo <  Flags.Spots1; ++lo) 
     {
       snprintf(path,  255, "Spot_P%02d/longitude", lo);
       Spot[Primary][lo].longitude = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_P%02d/latitude", lo);
       Spot[Primary][lo].latitude  = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_P%02d/radius", lo);
       Spot[Primary][lo].radius    = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_P%02d/dimfactor", lo);
       Spot[Primary][lo].dimfactor = gnome_config_get_float (path );
   }

   for (lo = 0; lo <  Flags.Spots2; ++lo) 
     {
       snprintf(path,  255, "Spot_S%02d/longitude", lo);
       Spot[Secondary][lo].longitude = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_S%02d/latitude", lo);
       Spot[Secondary][lo].latitude  = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_S%02d/radius", lo);
       Spot[Secondary][lo].radius    = gnome_config_get_float (path );
       snprintf(path,  255, "Spot_S%02d/dimfactor", lo);
       Spot[Secondary][lo].dimfactor = gnome_config_get_float (path );
   }

   /* fit flags
    */
   for (lo = 0; lo <  64; ++lo) 
     {
       snprintf(path,  255, "Simplex/%02d", lo);
       Flags.simplex[lo] = gnome_config_get_int (path );
   }




   gnome_config_pop_prefix ();

    } else {

      /*get this info from the config*/
      gnome_config_push_prefix ("nightfall/");
      Flags.GN_tooltips = gnome_config_get_int ("Options/HaveTooltips=1");
      Flags.GN_rempos = gnome_config_get_int ("Options/RememberPosition=1");
      Flags.GN_exitconfirm = gnome_config_get_int ("Options/PromptOnQuit=0");
      if (Flags.GN_rempos) {
	os_x = gnome_config_get_int ("Geometry/x=20");
	os_y = gnome_config_get_int ("Geometry/y=20");
	/*
	  if (strcmp("de", guess_the_locale() ) == 0)
	  os_w = gnome_config_get_int ("Geometry/w=595");
	  else
	*/
	os_w = gnome_config_get_int ("Geometry/w=554");
	os_h = gnome_config_get_int ("Geometry/h=408");
      }

      gnome_config_pop_prefix();

    }


  } else {

    Flags.GN_tooltips    = 1;
    Flags.GN_rempos      = 1;
    Flags.GN_exitconfirm = 0;

    os_w = 554;
    os_h = 408;
    os_x = 20;
    os_y = 20;

  }

  gtk_widget_set_uposition (GTK_WIDGET (top_window), os_x, os_y);
#ifndef GTK_NEW
  gtk_widget_set_usize        (GTK_WIDGET (top_window), os_w, os_h);
#else
  gtk_window_set_default_size (GTK_WINDOW (top_window), os_w, os_h);
#endif
  gtk_window_set_policy (GTK_WINDOW(top_window), TRUE, TRUE, TRUE);

#else

  top_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_border_width(GTK_CONTAINER(top_window), 0);
  gtk_window_set_title(GTK_WINDOW(top_window), top_title);
  gtk_widget_realize(top_window);

#ifdef ENABLE_NLS
  if (strcmp("de", guess_the_locale() ) == 0 ||
      strcmp("german", guess_the_locale() ) == 0)
    {
#ifndef GTK_NEW
      gtk_widget_set_usize        (GTK_WIDGET (top_window), 560, 366);
#else 
      gtk_window_set_default_size (GTK_WINDOW (top_window), 560, 366);
#endif
      gtk_window_set_policy (GTK_WINDOW(top_window), TRUE, TRUE, TRUE);
    }
  else 
    {
#endif

#ifndef GTK_NEW
      gtk_widget_set_usize        (GTK_WIDGET (top_window), 508, 366); 
#else
      gtk_window_set_default_size (GTK_WINDOW (top_window), 508, 366); 
#endif
      gtk_window_set_policy (GTK_WINDOW(top_window), TRUE, TRUE, TRUE);
#ifdef ENABLE_NLS
    }
#endif

#endif


  /* ----------- connect the exit handler --------------------------- */

  gtk_signal_connect (GTK_OBJECT (top_window), "delete_event",
                      GTK_SIGNAL_FUNC (delete_event), (gpointer) "9");

  /* --------------- set border width of window  -------------------- */

  gtk_container_border_width (GTK_CONTAINER (top_window), 0);

  /* -------------- initialize tooltips  ---------------------------- */

  tooltips = gtk_tooltips_new ();
  gtk_tooltips_set_delay  (tooltips, 1200);

#ifdef HAVE_GNOME
  if (! Flags.GN_tooltips) 
    gtk_tooltips_disable( tooltips );
#endif


  /* >>>>>>>>>>>  the main vertical box  <<<<<<<<<<<<<<<<<<<<<<<<<    */
  
  box3 = gtk_vbox_new (FALSE, 0);

#ifdef HAVE_GNOME
  gnome_app_set_contents(GNOME_APP(top_window), box3);
#else
  gtk_container_add (GTK_CONTAINER (top_window), GTK_WIDGET (box3)); 
#endif

  /* >>>>>>>>>>>  the menu bar  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */  


#ifdef HAVE_GNOME
  MakeMenu();
#else
  menu_bar = gtk_menu_bar_new(); 
  gtk_box_pack_start (GTK_BOX (box3), menu_bar, FALSE, FALSE, 0);
  MakeMenu(menu_bar);
#endif


  /* >>>>>>>>>>>  the toolbar   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */ 
 
#ifdef HAVE_GNOME

  gnome_app_create_toolbar(GNOME_APP(top_window), toolbar);
  t_anim_button = toolbar[1].widget;
  if (Flags.animate == ON)  
    gtk_toggle_button_set_state 
      (GTK_TOGGLE_BUTTON (t_anim_button), TRUE);

#else

   handlebox = gtk_handle_box_new ();
    gtk_box_pack_start ( GTK_BOX (box3) ,
			 handlebox, FALSE, TRUE, 0 );

    toolbar = gtk_toolbar_new ( GTK_ORIENTATION_HORIZONTAL,
				GTK_TOOLBAR_BOTH );
    gtk_container_border_width ( GTK_CONTAINER ( toolbar ) , 0 );
    gtk_toolbar_set_space_size ( GTK_TOOLBAR ( toolbar ), 309 );
    gtk_toolbar_set_tooltips(GTK_TOOLBAR(toolbar), TRUE);    
    gtk_container_add ( GTK_CONTAINER ( handlebox ) , toolbar );

    gtk_toolbar_append_item ( GTK_TOOLBAR (toolbar), 
			   _("Compute"),               
			   _("Compute a synthetic lightcurve w/ current options"),
			   NULL,             
			   create_pixmap(top_window, mcomp_xpm),
			   GTK_SIGNAL_FUNC (lightcurve_event), 
			   NULL );

    t_anim_button = gtk_toolbar_append_element ( GTK_TOOLBAR (toolbar), 
                           GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
                           NULL,
			   _("Animate"),               
			   _("Toggle animated view of binary during lightcurve computation"),
			   NULL,             
			   create_pixmap(top_window, manim_xpm),
			   GTK_SIGNAL_FUNC (toggle_animate), 
			   NULL );

    if (Flags.animate == ON)  
        gtk_toggle_button_set_state 
             (GTK_TOGGLE_BUTTON (t_anim_button), TRUE);


    gtk_toolbar_append_item ( GTK_TOOLBAR (toolbar), 
			   _("Plot"),               
			   _("Plot computed lightcurve"),
			   NULL,             
			   create_pixmap(top_window, mplot_xpm),
			   GTK_SIGNAL_FUNC (lightcurveplot_event), 
			   NULL );

    gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

    gtk_toolbar_append_item ( GTK_TOOLBAR (toolbar), 
			   _("Info"),               
			   _("Help on the active notebook page"),
			   NULL,             
			   create_pixmap(top_window, mhelp_xpm),
			   GTK_SIGNAL_FUNC (help_event), 
			   NULL );

    gtk_widget_show ( toolbar );
    gtk_widget_show (handlebox);

#endif


  /* >>>>>>>>>>>  the notebook  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */ 
 
   box4 = gtk_hbox_new (TRUE, 0);
   gtk_box_pack_start (GTK_BOX (box3), box4, FALSE, FALSE, 5);
   gtk_widget_show (box4);

   box4 = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (box3), box4, FALSE, FALSE, 0);

  /* --------------- initialize notebook     ------------------------ */

    notebook = gtk_notebook_new ();
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);


    gtk_box_pack_start (GTK_BOX (box4), notebook, FALSE, FALSE, 0);

    gtk_widget_show(notebook);
    gtk_widget_show(box4);

  /* --------------- First Page   ----------------------------------- */

  label = gtk_label_new (_("Basic"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);
  /* gtk_container_add (GTK_CONTAINER (top_window), box1); */ 
  gtk_widget_show (label);
  gtk_widget_show (box1);

  /* --------------------------------------------------------------   */
  /* the first vertical box holds a table with entry fields           */
  /* for the numerical arguments                                      */
  /* --------------------------------------------------------------   */

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 

    e_101 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_101), "changed",
                       GTK_SIGNAL_FUNC(enter_massratio),
                       e_101);
    sprintf(InitString, "%6.3f", Binary[Primary].Mq);
    gtk_entry_set_text (GTK_ENTRY (e_101), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_101),
                             0, GTK_ENTRY(e_101)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_101, 1, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_101);


    button = gtk_toggle_button_new_with_label (_("Mass Ratio:"));
    if (Flags.simplex[0] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "0");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for Mass Ratio"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    e_102 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_102), "changed",
                       GTK_SIGNAL_FUNC(enter_inclination),
                       e_102);
    sprintf(InitString, "%8.5f", RTOD*Orbit.Inclination);
    gtk_entry_set_text (GTK_ENTRY (e_102), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_102),
                             0, GTK_ENTRY(e_102)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_102, 3, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_102);

    button = gtk_toggle_button_new_with_label (_("Inclination:"));
    if (Flags.simplex[1] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "1");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for Inclination"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    e_103 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_103), "changed",
                       GTK_SIGNAL_FUNC(enter_rochefill1),
                       e_103);
    sprintf(InitString, "%6.3f", Binary[Primary].RocheFill);
    gtk_entry_set_text (GTK_ENTRY (e_103), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_103),
                             0, GTK_ENTRY(e_103)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_103, 1, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_103);

    button = gtk_toggle_button_new_with_label (_("Fill Factor (P):"));
    if (Flags.simplex[2] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
			GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "2");
    gtk_tooltips_set_tip (tooltips, button, 
			  _("Toggle CONST/FIT for Primary Roche Fill Factor"),
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 3, 4,
		      GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    e_104 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_104), "changed",
                       GTK_SIGNAL_FUNC(enter_rochefill2),
                       e_104);
    sprintf(InitString, "%6.3f", Binary[Secondary].RocheFill);
    gtk_entry_set_text (GTK_ENTRY (e_104), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_104),
                             0, GTK_ENTRY(e_104)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_104, 3, 4, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_104);

    button = gtk_toggle_button_new_with_label (_("Fill Factor (S):"));
    if (Flags.simplex[3] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "3");
    gtk_tooltips_set_tip(tooltips, button, 
			 _("Toggle CONST/FIT for Secondary Roche Fill Factor"),
			 NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);



    e_105 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_105), "changed",
                       GTK_SIGNAL_FUNC(enter_temp1),
                       e_105);
    sprintf(InitString, "%6.3f", Binary[Primary].Temperature);
    gtk_entry_set_text (GTK_ENTRY (e_105), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_105),
                             0, GTK_ENTRY(e_105)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_105, 1, 2, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_105);

    button = gtk_toggle_button_new_with_label (_("Temperature (P):"));
    if (Flags.simplex[4] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "4");
    gtk_tooltips_set_tip (tooltips, button, 
                   _("Toggle CONST/FIT for Primary Temperature"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);



    e_106 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_106), "changed",
                       GTK_SIGNAL_FUNC(enter_temp2),
                       e_106);
    sprintf(InitString, "%6.3f", Binary[Secondary].Temperature);
    gtk_entry_set_text (GTK_ENTRY (e_106), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_106),
                             0, GTK_ENTRY(e_106)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_106, 3, 4, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_106);

    button = gtk_toggle_button_new_with_label (_("Temperature (S):"));
    if (Flags.simplex[5] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "5");
    gtk_tooltips_set_tip (tooltips, button, 
			  _("Toggle CONST/FIT for Secondary Temperature"),
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    label = gtk_label_new (_("Terminal Output:"));
    gtk_table_attach (GTK_TABLE(table), label, 0, 1, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
    gtk_widget_show (label);   

    button = gtk_check_button_new_with_label (_("Warning"));
    if (Flags.debug[WARN] == ON)   
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_debug), (gpointer) "1");
    gtk_tooltips_set_tip (tooltips, button, 
                     _("Toggle output of warnings"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 1, 2, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    button = gtk_check_button_new_with_label (_("Verbose"));
    if (Flags.debug[VERBOSE] == ON)   
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_debug), (gpointer) "3");
    gtk_tooltips_set_tip (tooltips, button, 
                     _("Toggle verbose output"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    button = gtk_check_button_new_with_label (_("Busy"));
    if (Flags.debug[BUSY] == ON)   
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_debug), (gpointer) "2");
    gtk_tooltips_set_tip (tooltips, button, 
                     _("Toggle status output"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 3, 4, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);


  /* --------------- Second Page   ---------------------------------- */


  label = gtk_label_new (_("Advanced"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);

  gtk_widget_show (label);
  gtk_widget_show (box1);

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 

  /* ---------------  Eccentric orbit   ----------------------------- */

    t_eccentric_button = 
              gtk_check_button_new_with_label (_("Eccentric Orbit"));
    if (Flags.elliptic == ON)  
        gtk_toggle_button_set_state 
             (GTK_TOGGLE_BUTTON (t_eccentric_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (t_eccentric_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_elliptic), NULL );
    gtk_tooltips_set_tip (tooltips, t_eccentric_button, 
			  _("Switch on eccentric orbit"), NULL);
    gtk_table_attach (GTK_TABLE(table), t_eccentric_button, 0, 2, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_eccentric_button);

    e_107 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_107), "changed",
                       GTK_SIGNAL_FUNC(enter_excentricity),
                       e_107);
    sprintf(InitString, "%6.3f", Orbit.Excentricity);
    gtk_entry_set_text (GTK_ENTRY (e_107), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_107),
                             0, GTK_ENTRY(e_107)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_107, 1, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.elliptic == ON) gtk_widget_show (e_107);

    ecc_button = gtk_toggle_button_new_with_label (_("Eccentricity:"));
    if (Flags.simplex[6] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (ecc_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (ecc_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "6");
    gtk_tooltips_set_tip (tooltips, ecc_button, 
                           _("Toggle CONST/FIT for eccentricity"), NULL);
    gtk_table_attach (GTK_TABLE(table), ecc_button, 0, 1, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.elliptic == ON) gtk_widget_show (ecc_button);



    e_108 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_108), "changed",
                       GTK_SIGNAL_FUNC(enter_omega),
                       e_108);
    sprintf(InitString, "%6.3f", Orbit.Omega);
    gtk_entry_set_text (GTK_ENTRY (e_108), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_108),
                             0, GTK_ENTRY(e_108)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_108, 1, 2, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.elliptic == ON) gtk_widget_show (e_108);

    ome_button = gtk_toggle_button_new_with_label (_("Periastron Length:"));
    if (Flags.simplex[7] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (ome_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (ome_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "7");
    gtk_tooltips_set_tip (tooltips, ome_button, 
			  _("Toggle CONST/FIT for periastron length"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), ome_button, 0, 1, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.elliptic == ON) gtk_widget_show (ome_button);

  /* ---------------  Asynchroneous rotation  ----------------------- */

    e_109 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_109), "changed",
                       GTK_SIGNAL_FUNC(enter_fratio1),
                       e_109);
    sprintf(InitString, "%6.3f", Binary[Primary].Fratio);
    gtk_entry_set_text (GTK_ENTRY (e_109), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_109),
                             0, GTK_ENTRY(e_109)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_109, 3, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.asynchron1 == ON) gtk_widget_show (e_109);

    as1_button = gtk_toggle_button_new_with_label (_("F_Ratio (P):"));
    if (Flags.simplex[8] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (as1_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (as1_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "8");
    gtk_tooltips_set_tip (tooltips, button, 
			  _("Toggle CONST/FIT for Primary asynchron rotation"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), as1_button, 2, 3, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.asynchron1 == ON) gtk_widget_show (as1_button);

    e_110 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_110), "changed",
                       GTK_SIGNAL_FUNC(enter_fratio2),
                       e_110);
    sprintf(InitString, "%6.3f", Binary[Secondary].Fratio);
    gtk_entry_set_text (GTK_ENTRY (e_110), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_110),
                             0, GTK_ENTRY(e_110)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_110, 3, 4, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.asynchron2 == ON) gtk_widget_show (e_110);

    as2_button = gtk_toggle_button_new_with_label (_("F_Ratio (S):"));
    if (Flags.simplex[9] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (as2_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (as2_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "9");
    gtk_tooltips_set_tip (tooltips, button, 
                     _("Toggle CONST/FIT for Secondary Asynchron Rotation"),
			  NULL);
    gtk_table_attach (GTK_TABLE(table), as2_button, 2, 3, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.asynchron2 == ON) gtk_widget_show (as2_button);

    t_fpri_button = 
             gtk_check_button_new_with_label (_("Primary asynchron"));
    if (Flags.asynchron1 == ON)  
        gtk_toggle_button_set_state 
             (GTK_TOGGLE_BUTTON (t_fpri_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (t_fpri_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fratio1), NULL );
    gtk_tooltips_set_tip (tooltips, t_fpri_button, 
                     _("Switch on Primary asynchron rotation"),NULL);
    gtk_table_attach (GTK_TABLE(table), t_fpri_button, 2, 3, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_fpri_button);

    t_fsec_button = 
               gtk_check_button_new_with_label (_("Secondary asynchron"));
    if (Flags.asynchron2 == ON)  
      gtk_toggle_button_set_state 
	(GTK_TOGGLE_BUTTON (t_fsec_button), TRUE);
    gtk_signal_connect (GTK_OBJECT (t_fsec_button), "toggled",
			GTK_SIGNAL_FUNC (toggle_fratio2), NULL );
    gtk_tooltips_set_tip (tooltips, t_fsec_button, 
                     _("Switch On Secondary Asynchron Rotation"),NULL);
    gtk_table_attach (GTK_TABLE(table), t_fsec_button, 3, 4, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_fsec_button);

  /* ---------------  Toggle Buttons   ------------------------------ */

    button = gtk_check_button_new_with_label (_("Model Atmosphere"));
    if (Flags.blackbody == OFF)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_blackbody), NULL);
    gtk_tooltips_set_tip (tooltips, button, 
                     _("Toggle Blackbody/Model Atmosphere Flux"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    button = gtk_check_button_new_with_label (_("Fractional Visibility"));
    if (Flags.fractional == ON)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fractional), NULL);
    gtk_tooltips_set_tip (tooltips, button, 
                _("Compute Fractional Visibility to smooth Lightcurve"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 4, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

  /* ---------------  Line Profile   -------------------------------- */

    button = gtk_check_button_new_with_label (_("Line Profile at: (nm)"));
    if (Flags.lineprofile == ON)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_lineprofile), NULL );
    gtk_tooltips_set_tip (tooltips, button, 
			  _("Compute Line Profiles at different Phases for given rest wavelength"),
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    e_111 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_111), "changed",
                       GTK_SIGNAL_FUNC(enter_linewave),
                       e_111);
    sprintf(InitString, "%8.1f", Orbit.LambdaZero);
    gtk_entry_set_text (GTK_ENTRY (e_111), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_111),
                             0, GTK_ENTRY(e_111)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_111, 1, 2, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_111);

  /* ---------------  Reflection   ---------------------------------- */

    button = gtk_check_button_new_with_label (_("Detailed Reflection:"));
    if (Flags.reflect > 0)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_reflect), NULL );
    gtk_tooltips_set_tip (tooltips, button, 
        _("Toggle point source/detailed (iterative) reflection"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    ref_label = gtk_label_new (_("Reflection iterations:"));
    gtk_misc_set_alignment (GTK_MISC(ref_label), 0,0);
    gtk_tooltips_set_tip (tooltips, ref_label, 
                _("Set number of iterations for exact reflection"),NULL);
    gtk_table_attach (GTK_TABLE(table), ref_label, 0, 1, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
    if (Flags.reflect > 0) gtk_widget_show (ref_label);   

    e_113 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_113), "changed",
                       GTK_SIGNAL_FUNC(enter_reflect_iterations),
                       e_113);
    sprintf(InitString, "%3d", Flags.reflect);
    gtk_entry_set_text (GTK_ENTRY (e_113), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_113),
                             0, GTK_ENTRY(e_113)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_113, 1, 2, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    if (Flags.reflect > 0) gtk_widget_show (e_113);

  /* ---------------  Limb Darkening   ------------------------------ */

    label = gtk_label_new (_("Limb Darkening Law"));
    gtk_misc_set_alignment (GTK_MISC(label), 0,1);
    gtk_table_attach (GTK_TABLE(table), label, 2, 4, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
    gtk_widget_show (label);   


      button = gtk_radio_button_new_with_label (NULL, _("linear"));
      if (Flags.limb == 0)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (radio_limb), (gpointer) "0");
      gtk_table_attach (GTK_TABLE(table), button, 2, 3, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
      gtk_widget_show (button);

      group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
      button = gtk_radio_button_new_with_label (group, _("quadratic"));
      if (Flags.limb == 1)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (radio_limb), (gpointer) "1");
      gtk_table_attach (GTK_TABLE(table), button, 3, 4, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
      gtk_widget_show (button);  

      group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
      button = gtk_radio_button_new_with_label (group, _("square root"));
      if (Flags.limb == 2)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (radio_limb), (gpointer) "2");
      gtk_table_attach (GTK_TABLE(table), button, 2, 3, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
      gtk_widget_show (button);  

      group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
      button = gtk_radio_button_new_with_label (group, _("sqrt individual"));
      if (Flags.limb == 3)  
        gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (radio_limb), (gpointer) "3");
      gtk_table_attach (GTK_TABLE(table), button, 3, 4, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
      gtk_widget_show (button);  

  /* ---------------  Lightcurve Steps   ---------------------------- */

    label = gtk_label_new (_("Lightcurve Steps:"));
    /* gtk_misc_set_alignment (GTK_MISC(label), 0, 1); */
    gtk_tooltips_set_tip (tooltips, label, 
                _("Set number of steps for lightcurve"), NULL);
    gtk_table_attach (GTK_TABLE(table), label, 2, 3, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2); 
    gtk_widget_show (label);   

    e_112 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_112), "changed",
                       GTK_SIGNAL_FUNC(enter_phasesteps),
                       e_112);
    sprintf(InitString, "%8d", PhaseSteps);
    gtk_entry_set_text (GTK_ENTRY (e_112), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_112),
                             0, GTK_ENTRY(e_112)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_112, 3, 4, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_112);


  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);



  /* --------------- Third Page   ----------------------------------- */

  makepage3 ();


  /* --------------- Fourth Page ------------------------------------ */

  label = gtk_label_new (_("Data Fitting"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);
  /* gtk_container_add (GTK_CONTAINER (top_window), box1); */ 
  gtk_widget_show (label);
  gtk_widget_show (box1);

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 


    e_name = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_name), "changed",
                       GTK_SIGNAL_FUNC(enter_name),
                       e_name);
    gtk_entry_set_text (GTK_ENTRY (e_name), Orbit.Name);
    gtk_entry_select_region (GTK_ENTRY (e_name),
                             0, GTK_ENTRY(e_name)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_name, 2, 4, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_name);

    label = gtk_label_new (_("Name of binary:"));
     gtk_table_attach (GTK_TABLE(table), label, 0, 2, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (label);

    e_302 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_302), "changed",
                       GTK_SIGNAL_FUNC(enter_abs_period),
                       e_302);
    sprintf(InitString, "%8.4f", Orbit.TruePeriod/86400.);
    gtk_entry_set_text (GTK_ENTRY (e_302), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_302),
                             0, GTK_ENTRY(e_302)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_302, 2, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_302);

    label = gtk_label_new (_("Period (days):"));
     gtk_table_attach (GTK_TABLE(table), label, 0, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (label);

    e_303 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_303), "changed",
                       GTK_SIGNAL_FUNC(enter_abs_mass),
                       e_303);
    sprintf(InitString, "%8.4f", Orbit.TrueMass/1.989E30 );
    gtk_entry_set_text (GTK_ENTRY (e_303), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_303),
                             0, GTK_ENTRY(e_303)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_303, 2, 4, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_303);

    button = gtk_toggle_button_new_with_label (_("Mass (solar masses):"));
    if (Flags.simplex[26] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "26");
    gtk_tooltips_set_tip (tooltips, button, 
                   _("Toggle CONST/FIT for binary mass"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);


    e_304 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_304), "changed",
                       GTK_SIGNAL_FUNC(enter_abs_distance),
                       e_304);
    sprintf(InitString, "%8.4f", Orbit.TrueDistance/6.960E8);
    gtk_entry_set_text (GTK_ENTRY (e_304), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_304),
                             0, GTK_ENTRY(e_304)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_304, 2, 4, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_304);

    button = gtk_toggle_button_new_with_label(_("Separation (solar radii):"));
    if (Flags.simplex[27] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "27");
    gtk_tooltips_set_tip (tooltips, button, 
                   _("Toggle CONST/FIT for binary separation"),NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_301 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_301), "changed",
                       GTK_SIGNAL_FUNC(enter_simplextol),
                       e_301);
    sprintf(InitString, "%6.4f", Flags.SimplexTol);
    gtk_entry_set_text (GTK_ENTRY (e_301), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_301),
                             0, GTK_ENTRY(e_301)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_301, 2, 4, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_301);

    button = gtk_button_new_with_label (_("FIT with tolerance:"));
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (lightfit_event), NULL);
    gtk_tooltips_set_tip (tooltips, button, 
                         _("Determine best-fit model for observational data"),
                            NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    button = gtk_check_button_new_with_label (_("Simulated Annealing"));
    if (Flags.anneal == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_anneal), NULL );
    gtk_tooltips_set_tip (tooltips, button, 
                _("Switch on global (SA) fitting"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    button = gtk_button_new_with_label (_("MAP ChiSquare with steps:"));
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (lightmap_event), NULL);
    gtk_tooltips_set_tip (tooltips, button, 
                         _("Compute 2D chi-square map in two parameters"),
                            NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 2, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_305 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_305), "changed",
                       GTK_SIGNAL_FUNC(enter_step1),
                       e_305);
    sprintf(InitString, "%6.4f", Flags.Step[0]);
    gtk_entry_set_text (GTK_ENTRY (e_305), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_305),
                             0, GTK_ENTRY(e_305)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_305, 2, 3, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_305);

    e_306 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_306), "changed",
                       GTK_SIGNAL_FUNC(enter_step2),
                       e_306);
    sprintf(InitString, "%6.4f", Flags.Step[1]);
    gtk_entry_set_text (GTK_ENTRY (e_306), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_306),
                             0, GTK_ENTRY(e_306)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_306, 3, 4, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_306);

  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);



  /* --------------- Fifth  Page ------------------------------------ */

  label = gtk_label_new (_("Spots Primary"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);

  gtk_widget_show (label);
  gtk_widget_show (box1);

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 

    /* ------------- PRIMARY SPOT 1         ------------------------- */

    t_psp1_button = gtk_check_button_new_with_label (_("Primary Spot One"));
    if  (SpotActive[0] == ON)
          gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (t_psp1_button),TRUE);
    gtk_signal_connect (GTK_OBJECT (t_psp1_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_p_spot1), NULL);
    gtk_tooltips_set_tip (tooltips, t_psp1_button, 
                _("Toggle inclusion/exclusion of spot"), NULL);
    gtk_table_attach (GTK_TABLE(table), t_psp1_button, 0, 2, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_psp1_button);


    e_401 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_401), "changed",
                       GTK_SIGNAL_FUNC(enter_longitude_p_spot1),
                       e_401);
    sprintf(InitString, "%6.3f", SpotStore[Primary][0].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_401), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_401),
                             0, GTK_ENTRY(e_401)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_401, 1, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_401);

    button = gtk_toggle_button_new_with_label (_("Longitude:"));
    if (FitSpotS[10-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "10");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for longitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_402 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_402), "changed",
                       GTK_SIGNAL_FUNC(enter_latitude_p_spot1),
                       e_402);
    sprintf(InitString, "%6.3f", SpotStore[Primary][0].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_402), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_402),
                             0, GTK_ENTRY(e_402)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_402, 1, 2, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_402);

    button = gtk_toggle_button_new_with_label (_("Latitude:"));
    if (FitSpotS[11-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "11");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for latitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_403 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_403), "changed",
                       GTK_SIGNAL_FUNC(enter_radius_p_spot1),
                       e_403);
    if (Spot[Primary][0].radius < LIM_SRA_L)
          Spot[Primary][0].radius = LIM_SRA_L;
    if (SpotStore[Primary][0].radius < LIM_SRA_L)
          SpotStore[Primary][0].radius = LIM_SRA_L;
    sprintf(InitString, "%6.3f", SpotStore[Primary][0].radius);
    gtk_entry_set_text (GTK_ENTRY (e_403), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_403),
                             0, GTK_ENTRY(e_403)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_403, 1, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_403);

    button = gtk_toggle_button_new_with_label (_("Radius:"));
    if (FitSpotS[12-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "12");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for radius"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_404 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_404), "changed",
                       GTK_SIGNAL_FUNC(enter_dimfactor_p_spot1),
                       e_404);
    if (Spot[Primary][0].dimfactor < LIM_SDF_L) 
               Spot[Primary][0].dimfactor = LIM_SDF_L;
    if (SpotStore[Primary][0].dimfactor < LIM_SDF_L) 
               SpotStore[Primary][0].dimfactor = LIM_SDF_L;
    sprintf(InitString, "%6.3f", SpotStore[Primary][0].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_404), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_404),
                             0, GTK_ENTRY(e_404)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_404, 1, 2, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_404);

    button = gtk_toggle_button_new_with_label (_("Dimfactor:"));
    if (FitSpotS[13-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "13");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for dimfactor"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    /* ------------- PRIMARY SPOT 2         ------------------------- */

    t_psp2_button = gtk_check_button_new_with_label (_("Primary Spot Two"));
    if  (SpotActive[1] == ON)
          gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (t_psp2_button),TRUE);
    gtk_signal_connect (GTK_OBJECT (t_psp2_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_p_spot2), NULL);
    gtk_tooltips_set_tip (tooltips, t_psp2_button, 
                _("Toggle inclusion/exclusion of spot"), NULL);
    gtk_table_attach (GTK_TABLE(table), t_psp2_button, 2, 4, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_psp2_button);


    e_405 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_405), "changed",
                       GTK_SIGNAL_FUNC(enter_longitude_p_spot2),
                       e_405);
    sprintf(InitString, "%6.3f", SpotStore[Primary][1].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_405), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_405),
                             0, GTK_ENTRY(e_405)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_405, 3, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_405);

    button = gtk_toggle_button_new_with_label (_("Longitude:"));
    if (FitSpotS[14-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "14");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for longitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_406 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_406), "changed",
                       GTK_SIGNAL_FUNC(enter_latitude_p_spot2),
                       e_406);
    sprintf(InitString, "%6.3f", SpotStore[Primary][1].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_406), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_406),
                             0, GTK_ENTRY(e_406)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_406, 3, 4, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_406);

    button = gtk_toggle_button_new_with_label (_("Latitude:"));
    if (FitSpotS[15-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "15");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for latitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_407 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_407), "changed",
                       GTK_SIGNAL_FUNC(enter_radius_p_spot2),
                       e_407);
    if (Spot[Primary][1].radius < LIM_SRA_L)
                Spot[Primary][1].radius = LIM_SRA_L;
    if (SpotStore[Primary][1].radius < LIM_SRA_L)
                SpotStore[Primary][1].radius = LIM_SRA_L;
    sprintf(InitString, "%6.3f", SpotStore[Primary][1].radius);
    gtk_entry_set_text (GTK_ENTRY (e_407), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_407),
                             0, GTK_ENTRY(e_407)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_407, 3, 4, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_407);

    button = gtk_toggle_button_new_with_label (_("Radius:"));
    if (FitSpotS[16-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "16");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for radius"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_408 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_408), "changed",
                       GTK_SIGNAL_FUNC(enter_dimfactor_p_spot2),
                       e_408);
    if (Spot[Primary][1].dimfactor < LIM_SDF_L)
         Spot[Primary][1].dimfactor = LIM_SDF_L;
    if (SpotStore[Primary][1].dimfactor < LIM_SDF_L)
         SpotStore[Primary][1].dimfactor = LIM_SDF_L;
    sprintf(InitString, "%6.3f", SpotStore[Primary][1].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_408), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_408),
                             0, GTK_ENTRY(e_408)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_408, 3, 4, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_408);

    button = gtk_toggle_button_new_with_label (_("Dimfactor:"));
    if (FitSpotS[17-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "17");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for dimfactor"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   

  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);



  /* --------------- Sixth Page ------------------------------------- */


  label = gtk_label_new (_("Spots Secondary"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);

  gtk_widget_show (label);
  gtk_widget_show (box1);

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 

   
    /* ------------- SECONDARY SPOT 1        ------------------------ */

    t_ssp1_button = gtk_check_button_new_with_label (_("Secondary Spot One"));
    if  (SpotActive[2] == ON)
          gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (t_ssp1_button),TRUE);
    gtk_signal_connect (GTK_OBJECT (t_ssp1_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_s_spot1), NULL);
    gtk_tooltips_set_tip (tooltips, t_ssp1_button, 
                "Toggle inclusion/exclusion of spot ",NULL);
    gtk_table_attach (GTK_TABLE(table), t_ssp1_button, 0, 2, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_ssp1_button);


    e_409 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_409), "changed",
                       GTK_SIGNAL_FUNC(enter_longitude_s_spot1),
                       e_409);
    sprintf(InitString, "%6.3f", SpotStore[Secondary][0].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_409), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_409),
                             0, GTK_ENTRY(e_409)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_409, 1, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_409);

    button = gtk_toggle_button_new_with_label (_("Longitude:"));
    if (FitSpotS[18-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "18");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for longitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_410 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_410), "changed",
                       GTK_SIGNAL_FUNC(enter_latitude_s_spot1),
                       e_410);
    sprintf(InitString, "%6.3f", SpotStore[Secondary][0].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_410), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_410),
                             0, GTK_ENTRY(e_410)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_410, 1, 2, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_410);

    button = gtk_toggle_button_new_with_label (_("Latitude:"));
    if (FitSpotS[19-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "19");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for latitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_411 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_411), "changed",
                       GTK_SIGNAL_FUNC(enter_radius_s_spot1),
                       e_411);
    if (Spot[Secondary][0].radius < LIM_SRA_L)
         Spot[Secondary][0].radius = LIM_SRA_L;
    if (SpotStore[Secondary][0].radius < LIM_SRA_L)
         SpotStore[Secondary][0].radius = LIM_SRA_L;
    sprintf(InitString, "%6.3f", SpotStore[Secondary][0].radius);
    gtk_entry_set_text (GTK_ENTRY (e_411), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_411),
                             0, GTK_ENTRY(e_411)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_411, 1, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_411);

    button = gtk_toggle_button_new_with_label (_("Radius:"));
    if (FitSpotS[20-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "20");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for radius"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_412 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_412), "changed",
                       GTK_SIGNAL_FUNC(enter_dimfactor_s_spot1),
                       e_412);
    if (Spot[Secondary][0].dimfactor < LIM_SDF_L)
         Spot[Secondary][0].dimfactor = LIM_SDF_L;
    if (SpotStore[Secondary][0].dimfactor < LIM_SDF_L)
         SpotStore[Secondary][0].dimfactor = LIM_SDF_L;
    sprintf(InitString, "%6.3f", SpotStore[Secondary][0].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_412), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_412),
                             0, GTK_ENTRY(e_412)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_412, 1, 2, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_412);

    button = gtk_toggle_button_new_with_label (_("Dimfactor:"));
    if (FitSpotS[21-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "21");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for dimfactor"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    /* ------------- SECONDARY SPOT 2         ----------------------- */

    t_ssp2_button = gtk_check_button_new_with_label (_("Secondary Spot Two"));
    if  (SpotActive[3] == ON)
          gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (t_ssp2_button),TRUE);
    gtk_signal_connect (GTK_OBJECT (t_ssp2_button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_s_spot2), NULL );
    gtk_tooltips_set_tip (tooltips, t_ssp2_button, 
                _("Toggle inclusion/exclusion of spot"), NULL);
    gtk_table_attach (GTK_TABLE(table), t_ssp2_button, 2, 4, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (t_ssp2_button);


    e_413 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_413), "changed",
                       GTK_SIGNAL_FUNC(enter_longitude_s_spot2),
                       e_413);
    sprintf(InitString, "%6.3f", SpotStore[Secondary][1].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_413), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_413),
                             0, GTK_ENTRY(e_413)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_413, 3, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_413);

    button = gtk_toggle_button_new_with_label (_("Longitude:"));
    if (FitSpotS[22-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "22");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for longitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_414 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_414), "changed",
                       GTK_SIGNAL_FUNC(enter_latitude_s_spot2),
                       e_414);
    sprintf(InitString, "%6.3f", SpotStore[Secondary][1].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_414), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_414),
                             0, GTK_ENTRY(e_414)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_414, 3, 4, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_414);

    button = gtk_toggle_button_new_with_label (_("Latitude:"));
    if (FitSpotS[23-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "23");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for latitude"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_415 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_415), "changed",
                       GTK_SIGNAL_FUNC(enter_radius_s_spot2),
                       e_415);
    if (Spot[Secondary][1].radius < LIM_SRA_L)
         Spot[Secondary][1].radius = LIM_SRA_L;
    if (SpotStore[Secondary][1].radius < LIM_SRA_L)
         SpotStore[Secondary][1].radius = LIM_SRA_L;
    sprintf(InitString, "%6.3f", SpotStore[Secondary][1].radius);
    gtk_entry_set_text (GTK_ENTRY (e_415), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_415),
                             0, GTK_ENTRY(e_415)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_415, 3, 4, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_415);

    button = gtk_toggle_button_new_with_label (_("Radius:"));
    if (FitSpotS[24-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "24");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for radius"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
    e_416 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_416), "changed",
                       GTK_SIGNAL_FUNC(enter_dimfactor_s_spot2),
                       e_416);
    if (Spot[Secondary][1].dimfactor < LIM_SDF_L)
         Spot[Secondary][1].dimfactor = LIM_SDF_L;
    if (SpotStore[Secondary][1].dimfactor < LIM_SDF_L)
         SpotStore[Secondary][1].dimfactor = LIM_SDF_L;
    sprintf(InitString, "%6.3f", SpotStore[Secondary][1].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_416), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_416),
                             0, GTK_ENTRY(e_416)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_416, 3, 4, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_416);

    button = gtk_toggle_button_new_with_label (_("Dimfactor:"));
    if (FitSpotS[25-10] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit_spot), (gpointer) "25");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for dimfactor"), NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);
   
   
  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);



  /* --------------- Seventh Page ----------------------------------- */



  label = gtk_label_new (_("Third Light"));
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box1, label);

  gtk_widget_show (label);
  gtk_widget_show (box1);

  box2 = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box2), 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
  gtk_widget_show (box2);

  table = gtk_table_new (8, 4, TRUE); 

    e_501 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_501), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdU),
                       e_501);
    sprintf(InitString, "%6.3f", Orbit.Third[Umag]);
    gtk_entry_set_text (GTK_ENTRY (e_501), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_501),
                             0, GTK_ENTRY(e_501)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_501, 1, 2, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_501);


    button = gtk_toggle_button_new_with_label (_("U Band:"));
    if (Flags.simplex[28] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "28");
    gtk_tooltips_set_tip (tooltips, button, 
                           _("Toggle CONST/FIT for third light in U band"),
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_502 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_502), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdB),
                       e_502);
    sprintf(InitString, "%6.3f", Orbit.Third[Bmag]);
    gtk_entry_set_text (GTK_ENTRY (e_502), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_502),
                             0, GTK_ENTRY(e_502)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_502, 1, 2, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_502);


    button = gtk_toggle_button_new_with_label (_("B Band:"));
    if (Flags.simplex[29] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "29");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in B band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_503 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_503), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdV),
                       e_503);
    sprintf(InitString, "%6.3f", Orbit.Third[Vmag]);
    gtk_entry_set_text (GTK_ENTRY (e_503), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_503),
                             0, GTK_ENTRY(e_503)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_503, 1, 2, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_503);


    button = gtk_toggle_button_new_with_label (_("V Band:"));
    if (Flags.simplex[30] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "30");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in V band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_504 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_504), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdR),
                       e_504);
    sprintf(InitString, "%6.3f", Orbit.Third[Rmag]);
    gtk_entry_set_text (GTK_ENTRY (e_504), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_504),
                             0, GTK_ENTRY(e_504)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_504, 1, 2, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_504);


    button = gtk_toggle_button_new_with_label (_("R Band:"));
    if (Flags.simplex[31] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "31");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in R band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 3, 4,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_505 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_505), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdI),
                       e_505);
    sprintf(InitString, "%6.3f", Orbit.Third[Imag]);
    gtk_entry_set_text (GTK_ENTRY (e_505), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_505),
                             0, GTK_ENTRY(e_505)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_505, 1, 2, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_505);


    button = gtk_toggle_button_new_with_label (_("I Band:"));
    if (Flags.simplex[32] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "32");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in I band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 0, 1, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_506 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_506), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdJ),
                       e_506);
    sprintf(InitString, "%6.3f", Orbit.Third[Jmag]);
    gtk_entry_set_text (GTK_ENTRY (e_506), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_506),
                             0, GTK_ENTRY(e_506)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_506, 3, 4, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_506);


    button = gtk_toggle_button_new_with_label (_("J Band:") );
    if (Flags.simplex[33] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "33");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in J band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_507 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_507), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdH),
                       e_507);
    sprintf(InitString, "%6.3f", Orbit.Third[Hmag]);
    gtk_entry_set_text (GTK_ENTRY (e_507), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_507),
                             0, GTK_ENTRY(e_507)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_507, 3, 4, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_507);


    button = gtk_toggle_button_new_with_label (_("H Band:"));
    if (Flags.simplex[34] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "34");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in H band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 1, 2,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_508 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_508), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdK),
                       e_508);
    sprintf(InitString, "%6.3f", Orbit.Third[Kmag]);
    gtk_entry_set_text (GTK_ENTRY (e_508), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_508),
                             0, GTK_ENTRY(e_508)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_508, 3, 4, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_508);


    button = gtk_toggle_button_new_with_label (_("K Band:"));
    if (Flags.simplex[35] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "35");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in K band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 2, 3,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_509 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_509), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdu),
                       e_509);
    sprintf(InitString, "%6.3f", Orbit.Third[umag]);
    gtk_entry_set_text (GTK_ENTRY (e_509), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_509),
                             0, GTK_ENTRY(e_509)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_509, 3, 4, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_509);


    button = gtk_toggle_button_new_with_label (_("u Band:"));
    if (Flags.simplex[36] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "36");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in u band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 4, 5,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_510 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_510), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdv),
                       e_510);
    sprintf(InitString, "%6.3f", Orbit.Third[vmag]);
    gtk_entry_set_text (GTK_ENTRY (e_510), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_510),
                             0, GTK_ENTRY(e_510)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_510, 3, 4, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_510);


    button = gtk_toggle_button_new_with_label (_("v Band:"));
    if (Flags.simplex[37] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "37");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in v band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 5, 6,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_511 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_511), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdb),
                       e_511);
    sprintf(InitString, "%6.3f", Orbit.Third[bmag]);
    gtk_entry_set_text (GTK_ENTRY (e_511), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_511),
                             0, GTK_ENTRY(e_511)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_511, 3, 4, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_511);


    button = gtk_toggle_button_new_with_label (_("b Band:"));
    if (Flags.simplex[38] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "38");
    gtk_tooltips_set_tip (tooltips, button, 
                         _("Toggle CONST/FIT for third light in b band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 6, 7,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);

    e_512 = gtk_entry_new_with_max_length (10);
    gtk_signal_connect(GTK_OBJECT(e_512), "changed",
                       GTK_SIGNAL_FUNC(enter_thirdy),
                       e_512);
    sprintf(InitString, "%6.3f", Orbit.Third[ymag]);
    gtk_entry_set_text (GTK_ENTRY (e_512), InitString);
    gtk_entry_select_region (GTK_ENTRY (e_512),
                             0, GTK_ENTRY(e_512)->text_length);
    gtk_table_attach (GTK_TABLE(table), e_512, 3, 4, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (e_512);


    button = gtk_toggle_button_new_with_label (_("y Band:"));
    if (Flags.simplex[39] == ON)   
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
    gtk_signal_connect (GTK_OBJECT (button), "toggled",
                      GTK_SIGNAL_FUNC (toggle_fit), (gpointer) "39");
    gtk_tooltips_set_tip (tooltips, button, 
                          _("Toggle CONST/FIT for third light in y band"), 
			  NULL);
    gtk_table_attach (GTK_TABLE(table), button, 2, 3, 7, 8,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 2, 2);
    gtk_widget_show (button);



  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (box2), table, FALSE, FALSE, 0);



  /* >>>>>>>>>>>  the progress bar  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */  


#ifdef HAVE_GNOME

  status_bar = 
    GNOME_APPBAR (gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_NEVER));
  progress_bar = GTK_WIDGET(gnome_appbar_get_progress(status_bar));
  gnome_appbar_set_default(status_bar, _("Waiting"));
  gnome_app_set_statusbar(GNOME_APP(top_window), GTK_WIDGET(status_bar));

#else

  box4 = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (box3), box4, FALSE, FALSE, 0);

   table = gtk_table_new (5, 1, TRUE); 
   gtk_widget_show (table);
   gtk_box_pack_start (GTK_BOX (box4), table, FALSE, FALSE, 0);

   status_bar = gtk_statusbar_new();
   gtk_table_attach (GTK_TABLE(table), status_bar, 0, 3, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
   context_id = 
     gtk_statusbar_get_context_id( GTK_STATUSBAR(status_bar), 
				   _("Waiting"));
   gtk_statusbar_push(GTK_STATUSBAR(status_bar), context_id, _("Waiting"));  
   gtk_widget_show (status_bar);

   progress_bar = gtk_progress_bar_new ();
   gtk_table_attach (GTK_TABLE(table), progress_bar, 3, 5, 0, 1,
          GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
   gtk_widget_show (progress_bar);

   gtk_widget_show (box4);

#endif

   gtk_widget_show (box3);

  gtk_widget_show (top_window);

#ifdef HAVE_GNOME
  if (restarted == 1) gtk_notebook_set_page(GTK_NOTEBOOK(notebook), old_page);
#endif


  /* >>>>>>>>>>>  enter main loop  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<    */  

  gtk_main ();

  return 0;
}

#endif





