/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

/* --------------------------------------------------------------- */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include "Light.h"


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Blackbody flux for an element
 @param     (SurfaceElement  *) SurfPtr  the surface element
 @return    (void)        
 @heading   Flux Computation
 ****************************************************************************/
static void BBFlux(SurfaceElement  *SurfPtr, int Comp)
{
  OrbitType *OrbPtr = &Orbit;         /* orbit pointer                    */
  static  int     band;               /* the passband                     */
  static  double  Temp;               /* temperature                      */
  static  double  ScaledArea;         /* scaled surface area              */
  static  double  CC1 = 3.7418498e+15;/* 10e(5*4)      erg cm / sec       */
  static  double  CC2 = 14388.3361;   /* 10e4          cm K               */
                                      /* WaveL in 10e4 cm                 */

  /* we compute I(1) -- Intensity normal to surface                   */
  /*  in blackbody approximation                                      */  
  /* B = CC1/WaveL^5 * 1/(exp(CC2/(T*WaveL))-1) WaveL(ength) in _cm_  */

  /* -----------  surface rescaling --------------------------------- */
  
  ScaledArea = SurfPtr->area * SQR(OrbPtr->Dist); 
  
  /* -----------  flux computation  --------------------------------- */
  
  Temp = SurfPtr->temp;
  
  for (band = 0; band < NUM_MAG; ++band) 
    {

      SurfPtr->f_[band] = ScaledArea
	* CC1/
	(WaveL[Comp][band] * WaveL[Comp][band] * WaveL[Comp][band]
	 * WaveL[Comp][band] * WaveL[Comp][band])
	* 1.0/(exp(CC2/(WaveL[Comp][band] * 
			Temp))-1.0);
    }
  return;
}

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Computation of blackbody fluxes
 @param     (int)  Comp        The stellar component
 @return    (void)        
 @heading   Flux Computation
 ****************************************************************************/
void BlackbodyFlux(int Comp)
{
  SurfaceElement  *SurfPtr;           /* pointer to surface               */
  long    i;                          /* loop variable                    */

  SurfPtr = Surface[Comp];

  for (i = 0; i < Binary[Comp].NumElem; ++i) 
    { 
      BBFlux(SurfPtr, Comp);
      ++SurfPtr;
    }
}

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Search index for temperature in array
 @param     (float)  key    temperature
 @param     (float *) array temperature array
 @param     (int)  n        size of array
 @return    (int)  the left index of a centered 4-value subarray       
 @heading   Flux Computation
 ****************************************************************************/
static int search_key (float key, int * r, int n )
{ 
  static int i = 0;
  static int high, low, k;
  
  /* speedup (?)
   */
  if (key >= r[i] && key < r[i+1])
    {
      k = MAX ( (i-1),     0);
      k = MIN (     k, (n-4));
      return ( k );
    }
  
  /* return 0 if < r[0]; 
   */
  for ( low = (-1), high = n;  high-low > 1;  )
    {
      i = (high+low) / 2;
      
      if ( key < r[i])  
	high = i;
      else             
	low  = i;
    }
  
  i = (i < 0) ? 0 : i;

  k = MAX ( (i-1),     0);
  k = MIN (     k, (n-4));

  return ( k );
}


/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Computation of atmospheric model fluxes
 @param     (int)  Comp        The stellar component
 @return    (void)        
 @heading   Flux Computation
 ****************************************************************************/
void ModelFlux(int Comp)
{

#include "FluxKuruzc1991.h"
#include "FluxHauschildt.4.0.h"

  int              maxnum;            /* number of models                 */
  float            highlim;           /* upper limit                      */
  float            lowlim;            /* lower limit                      */
  int            * tarr;              /* the model temperatures           */
  float          * farr[NUM_MAG];     /* the model fluxes                 */

  SurfaceElement * SurfPtr;           /* pointer to surface               */
  OrbitType *OrbPtr = &Orbit;         /* orbit pointer                    */
  
  register int     j;                 /* loop variable                    */
  register long    i;                 /* loop variable                    */
  int              band;              /* passband                         */
  int              hkey;              /* hash key                         */
  double           T;                 /* temperature                      */
  double           Hi, Mid, Lo;       /* bracketing temperatures in model */
  double           FHi, FMid, FLo;    /* corresponding fluxes             */
  double           FLLo, LLo;         /* more bracketing temperatures     */
  double           Phi;               /* temperature                      */
  double           Cubic;             /* cubic interpolation              */
  double           ScaledArea;        /* scaled surface area              */
  double           Parabolic;         /* parabolic interpolation          */
  
  /* fudge factors for BB flux ... almost never needed
   */
  static float fudge[] = {
    871529.64, 355630.79, 539745.57, 366877.14,
    87298.90,  77997.85, 119717.59,  42864.74,
    42106.24,  25430.45,  35439.69,  69058.99
  };

  /* -----------   main loop   ---------------------------------------    */

  SurfPtr = Surface[Comp];
  
  for (i = 0; i < Binary[Comp].NumElem; ++i) {
    
    if (Flags.elliptic == ON) 
      ScaledArea = SurfPtr->area * SQR(OrbPtr->Dist);
    else
      ScaledArea = SurfPtr->area;
    
    
    T = SurfPtr->temp;

    /* h_* and k_* are defined in the include files for
     * Hauschildt & Kurucz models, respectively
     */
    if (T > h_high)
      {
	maxnum  = k_max;
	highlim = k_high;
	lowlim  = k_low;
	tarr    = k_temp_40;
	for (j = 0; j < NUM_MAG; ++j) 
	  farr[j]    = k_flux_40[j];
      }
    else if (T > h_low)
      {
	maxnum  = h_max;
	highlim = h_high;
	lowlim  = h_low;
	tarr    = h_temp_40;
	for (j = 0; j < NUM_MAG; ++j) 
	  farr[j]    = h_flux_40[j];
      }
    else
      {
	BBFlux(SurfPtr, Comp);
	for (j = 0; j < NUM_MAG; ++j)
	  SurfPtr->f_[j] /= fudge[j];
	++SurfPtr;
	continue;
      }


    
    /* -----------   get hashkey ---------------------------------------   */
    
    hkey = search_key (T, tarr, maxnum);

    /*
    if (T <= 10000 && T >= 3500)      { hkey = floor(T/250) - 14;  }
    else if (T <= 13000 && T >= 3500) { hkey = floor(T/500) + 6;   }
    else if (T <= 33999 && T >= 3500 ){ hkey = floor(T/1000) + 19; }
    else if (T <= 3500)               { hkey = 0;                  }
    else { hkey = 52; }
    */
    
    if (hkey == 0 ) {   /* quadratic interpolation */  
      
      for (band=0; band < NUM_MAG; ++band) {
	
	FLo     = farr[band][hkey];
	FMid    = farr[band][hkey+1];
	FHi     = farr[band][hkey+2];
	
	Phi     = T;
	Lo      = (double) tarr[hkey];
	Mid     = (double) tarr[hkey+1];
	Hi      = (double) tarr[hkey+2];
	
	/* ------------- do a quadratic intepolation -----------------    */
	
	Parabolic = FLo  * ((Phi-Mid)*(Phi-Hi )) / ((Lo -Mid)*(Lo -Hi ))
	  + FMid * ((Phi-Lo )*(Phi-Hi )) / ((Mid-Lo )*(Mid-Hi ))
	  + FHi  * ((Phi-Lo )*(Phi-Mid)) / ((Hi -Lo )*(Hi -Mid));
	
	SurfPtr->f_[band] = ScaledArea * Parabolic;
      }
      
    } else { /* cubic interpolation */
      
      for (band=0; band < NUM_MAG; ++band) {
	
	FLLo    = farr[band][hkey];
	FLo     = farr[band][hkey+1];
	FMid    = farr[band][hkey+2];
	FHi     = farr[band][hkey+3];
	
	Phi     = T;
	LLo     = (double) tarr[hkey];
	Lo      = (double) tarr[hkey+1];
	Mid     = (double) tarr[hkey+2];
	Hi      = (double) tarr[hkey+3];
	
	/* do a cubic intepolation */
	
	Cubic   = 
	  FLLo* ((Phi-Lo )*(Phi-Mid)*(Phi-Hi )) / 
	  ((LLo- Lo)*(LLo-Mid)*(LLo-Hi ))
	  + FLo * ((Phi-LLo)*(Phi-Mid)*(Phi-Hi )) / 
	  ((Lo -LLo)*(Lo -Mid)*(Lo -Hi ))
	  + FMid* ((Phi-LLo)*(Phi-Lo )*(Phi-Hi )) / 
	  ((Mid-LLo)*(Mid-Lo )*(Mid-Hi ))
	  + FHi * ((Phi-LLo)*(Phi-Lo )*(Phi-Mid)) / 
	  ((Hi -LLo)*(Hi -Lo )*(Hi -Mid));

	
	SurfPtr->f_[band] = ScaledArea * Cubic;

      }
    }

    ++SurfPtr;
  }

  return;
}




