/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


#include <math.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include "Light.h"

/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Divide stellar surface into grid
 @long      First, set up the grid, then compute the
            temperature distribution on the grid, including
            spots and reflection
 @param     (int)  Comp    The stellar component
 @return    (int)          The exit status
 @heading   Star Setup
 ****************************************************************************/
int DivideSurface(int Comp)
{
  register long i,j;                  /* loop variables                  */
  long n_phi;                         /* # of surface steps in phi       */
  long n_eta;                         /* # of surface steps in eta       */
  long NumElem;                       /* current surface element         */
  long NextPtr;                       /* next    surface element         */
  int  testerr = 0;                   /* exit status of RootFind         */
  
  double     eta;                     /* eta of surface element           */
  double     phi;                     /* phi of surface element           */
  double     sineta, sinphi;          /* sin,cos of phi                   */
  double     coseta, cosphi;          /* sin,cos of eta                   */
  double     rad1;                    /* radius of surface element        */
  double     sqr_rad1;                /* square of radius                 */
  double     delta_eta;               /* step in eta                      */
  double     delta_phi;               /* step in phi                      */
  double     rho;                     /* rho of surface element           */
  double     cub_rho;                 /* cube of rho                      */
  double     C_eta, C_phi, C_rad1;    /* surface gravity vector           */
                                      /* in spherical coordinates         */
  double     C_x, C_y, C_z;           /* surface gravity vector           */
                                      /* in cartesian coordinates         */
  double     grav;                    /* surface gravity                  */
  double     lvec, mvec, nvec;        /* surface normal vector            */
  double     RLag1;                   /* Lagrange 1                       */
  double     RXCrit;                  /* Critical radius                  */
  double     DSurf;                   /* surface of element               */
  double     MeanGrav = 0.0;          /* mean gravity                     */
  double     Surf = 0.0;              /* total surface                    */
  double     MeanDarkGrav = 0.0;      /* contribution to grav. darkening  */
  double     RadiusBound = 0.0;       /* inner bound for RootFind         */
  double     T1;                      /* temperature of this star         */
  double     T2;                      /* temperature of the other star    */
  double     R1;                      /* Roche fill  of this star         */
  double     R2;                      /* Roche fill  of the other star    */
  double     Tr;                      /* temperature ratio^4              */
  double     etacorr_over, eta_corr, deltaeta_corr; /* Throat adjustment  */
  double     eta_next, sineta_next;                 /* Pointer setting    */
  long       n_next, n_last, n_comp;                /* Pointer setting    */
  /*  for mean temperature calculation                                    */
  double     Area;                    /* area of element                  */
  double     Weight=0.0;              /* total area                       */
  double     TempMean=0.0;            /* mean temperature                 */

  SurfaceElement  *SurfPtr;           /* pointer to surface               */
  BinaryComponent *BinPtr;            /* pointer to binary                */

  
  /* ---------- initialize ---------------------------------------------- */

  SurfPtr      = Surface[Comp];
  BinPtr       = &Binary[Comp];

  Mq           = Binary[Comp].Mq;
  if      (Comp == Primary && Flags.asynchron1 == ON)
    F            = Binary[Comp].Fratio;
  else if (Comp == Secondary && Flags.asynchron2 == ON)
    F            = Binary[Comp].Fratio;
  else
    F            = 1.0;
  RLag1        = Binary[Comp].RLag1;
  RochePot     = Binary[Comp].RochePot; 
  RadiusBound  = 0.8*Binary[Comp].Radius;
  etacorr_over = Binary[Comp].LimEta; 


  /* ---------- Maximum Radius to search for Root ----------------        */

  if (Flags.fill == OFF) {
    RXCrit = BinPtr->RXCrit;
  } else {
    RXCrit = BinPtr->LimRad  
      + (0.07*SQR(Binary[Primary].Mq))
      +  0.02*Binary[Primary].Mq;
          
    if ( (Binary[Primary].Mq - 1.0) >= FLT_EPSILON) 
      RXCrit = BinPtr->LimRad  
	+ (0.07*SQR(Binary[Secondary].Mq))
	+  0.02*Binary[Secondary].Mq;
			 	  
    if (   (Binary[Primary].RocheFill - 1.05) <= FLT_EPSILON 
	&& (Binary[Primary].Mq - 0.2) >=  FLT_EPSILON
	&& (Binary[Primary].Mq - 5.0)  <= FLT_EPSILON   ) 
      RXCrit = BinPtr->RXCrit + Binary[Primary].RocheFill - 1.;
    
    else if ( (Binary[Primary].RocheFill - 1.02)  <= FLT_EPSILON )
      RXCrit = BinPtr->RXCrit + Binary[Primary].RocheFill - 1.;

				           
    /* critical angle w/r to z                                      */
    etacorr_over = BinPtr->RZatL1 / 
      sqrt(   BinPtr->RZatL1 * BinPtr->RZatL1 
	      + BinPtr->RLag1 *  BinPtr->RLag1);
    etacorr_over = asin( CLAMP(etacorr_over, -1.0, 1.0) );
  }


  BinPtr->Surface = 0.0; 
  BinPtr->Gravity = 0.0;

  if (Comp == 0) { 
        T1 =  BinPtr->Temperature;
        R1 =  BinPtr->Radius;
        T2 =  Binary[Comp+1].Temperature;
        R2 =  Binary[Comp+1].Radius;
  } else {
        T1 =  BinPtr->Temperature;
        R1 =  BinPtr->Radius;
        T2 =  Binary[Comp-1].Temperature;
        R2 =  Binary[Comp-1].Radius;
  }
  Tr = T2/T1;
  Tr = Tr * Tr * Tr * Tr;

  n_eta = StepsPerPi; 
  n_phi = 0;

  if (Flags.fill == OFF) BinPtr->LimEta = 0.0;

  delta_eta = (PI - BinPtr->LimEta)/n_eta; 
  eta = PI-(0.5*delta_eta);   /* start value, point opposite to companion */ 
  
  if (Flags.fill == ON) {    /* Throat adjustment                         */
     etacorr_over = (BinPtr->LimEta - etacorr_over) 
                     + 3.0 * delta_eta;
  }

  NumElem    = 0;

  /* -------------------  loop over eta  ------------------------------   */

  for (i = 0; i < n_eta; ++i) {    

    eta_corr      = eta;         /* Throat adjustment */
    deltaeta_corr = delta_eta;

    phi        = 0.0;
    sineta     = sin(eta); 
    coseta     = cos(eta);
    lambda     = coseta;


    /* n_phi proportional to sin(eta)                                     */
    if ( fabs(2 * n_eta * sineta) >= 0.0 )
            n_phi = (long) (10 + 1 + (int)(fabs(2 * n_eta * sineta)));
    else n_phi = 10;

    delta_phi = (PI+PI)/n_phi; 

    /* check if enough memory                                             */
    if ((NumElem+n_phi) >= (int) MaximumElements) {
#ifdef _WITH_GTK
      if (Flags.interactive == OFF)
             nf_error(_(errmsg[10]));
      else {
	make_dialog(_(errmsg[10]));
        return(8);
      }
#else
      nf_error(_(errmsg[10]));
#endif
    }

    BinPtr->N_PhiStep[i] = n_phi;  /* store number of phi intervals */

    /* ---------  Loop over phi ---------------------------------------- */

    for (j = 0; j < n_phi; ++j) {    

       sinphi     = sin(phi); 
       cosphi     = cos(phi);

      /* Throat adjustment for last three rings                          */
      if ((Flags.fill == ON) && (i > (n_eta-4))) {  
        if (i == n_eta-3) eta_corr = eta - ((etacorr_over/6.0)
                                            -(delta_eta/2.0))*fabs(sinphi);
        else if (i == n_eta-2) eta_corr = eta - ((etacorr_over/2.0) 
                                            -(1.5*delta_eta))*fabs(sinphi);
        else if (i == n_eta-1) eta_corr = eta - ((5.0*etacorr_over/6.0) 
					    -(5.0*delta_eta/2.0))*fabs(sinphi);

        sineta        = sin(eta_corr); 
        coseta        = cos(eta_corr);
        lambda        = coseta; 
        deltaeta_corr = delta_eta+((etacorr_over/3.0)-delta_eta)*fabs(sinphi);
      }       

       mu         = sineta*cosphi;
       nu         = sineta*sinphi;
 
       rad1       = RootFind(RocheSurface, RadiusBound, RXCrit, 1.0e-8, 
                    "DivideSurface", &testerr);
       if (testerr == 1) return(8);

       SurfPtr->eta      = (float) eta_corr;
       SurfPtr->phi      = (float) phi;
       SurfPtr->rad      = (float) rad1;
       SurfPtr->lambda   = (float) (lambda * rad1);
       SurfPtr->mu       = (float) (mu * rad1);
       SurfPtr->nu       = (float) (nu * rad1);

       sqr_rad1   = rad1*rad1;
       rho        = 1.0/sqrt(1.0+(sqr_rad1)-2.0*rad1*lambda);
       cub_rho    = rho*rho*rho;
       C_rad1     = -1.0/(sqr_rad1) + Mq*(cub_rho*(lambda-rad1)-lambda)
                      +F*F*(Mq+1.0)*(sqr_rad1)*(1.0- nu*nu);
       C_eta      = sineta
                      *(Mq*(1.0-cub_rho)-(Mq+1.0)*rad1*F*F
			*lambda*sinphi*sinphi);
       C_phi      = -(Mq + 1.0)*F*F*rad1*nu*cosphi;
       C_x        = C_rad1*lambda - C_eta*sineta;
       C_y        = C_rad1*mu + C_eta*lambda*cosphi - C_phi*sinphi;
       C_z        = C_rad1*nu + C_eta*lambda*sinphi + C_phi*cosphi;
       grav       = sqrt( C_x*C_x + C_y*C_y + C_z*C_z );
       lvec       = -C_x/grav;
       mvec       = -C_y/grav;
       nvec       = -C_z/grav;

       SurfPtr->rho    = (float) rho;
       SurfPtr->grav   = (float) grav;
       SurfPtr->l      = (float) lvec;
       SurfPtr->m      = (float) mvec;
       SurfPtr->n      = (float) nvec;

       DSurf      = (sqr_rad1) * sineta * deltaeta_corr * delta_phi;
       DSurf      = DSurf/(lambda*lvec+mu*mvec+nu*nvec);


       SurfPtr->area   = (float) DSurf;

       SurfPtr->MyNum = NumElem;

       Surf         = Surf + DSurf;
       MeanGrav     = MeanGrav + (grav*DSurf);
       MeanDarkGrav = MeanDarkGrav 
                     + exp(BinPtr->GravDarkCoeff * log(grav))*DSurf;


       /*  ---------------    INITIALIZE SURFACE TEMP  ----------------- */

       SurfPtr->temp = T1;

       phi        = phi + delta_phi;  /* increment phi                   */
       ++NumElem;                     /* increment surface element count */
       ++SurfPtr;

    }  /* -----------  end loop over phi ---------------------------     */

  eta = eta - delta_eta;

  }  /* -------------- end loop over eta ------------------------------- */


  BinPtr->NumElem = NumElem;
  BinPtr->NumPlus = NumElem;

  if (Flags.debug[VERBOSE] == ON) { 
    printf("\n"); 
    printf(_(" Divide Star %3d : %6ld Surface elements\n"),
	   Comp+1, BinPtr->NumElem);
  }

  BinPtr->Surface       = Surf;
  BinPtr->Gravity       = MeanGrav/Surf;
  BinPtr->DarkeningGrav = MeanDarkGrav/Surf;


  /* ------------------- SET next/last_ptr ---------------------------- */

  /* we need this only once                                             */

  if (Flags.first_pass == ON) { 
 
  SurfPtr      = Surface[Comp];

  n_last     = 0;
  n_next     = 0;
  NumElem    = 0;

  for (i = 0; i < n_eta; ++i) {  

    eta_next    = eta - delta_eta;                   /* pointer setting */
    sineta_next = sin(eta_next);                     /* pointer setting */


    if (i > 0) n_last = n_phi;                       /* pointer setting */
    n_phi = BinPtr->N_PhiStep[i];
    if ( i < n_eta-1) n_next = BinPtr->N_PhiStep[i+1];


   for (j = 0; j < n_phi; ++j) {

      
     /* -----------     the first ring   -----------------------------  */
     if (i == 0) { 
	       SurfPtr->last_ptr = NULL; 
               if (n_next == n_phi) {
                         SurfPtr->next_ptr = 
                                    &Surface[Comp][NumElem+n_next];
	       } else {
                 n_comp = ROUND(((float)(n_next)/(float)(n_phi))*(float)(j));
                 n_comp = n_phi - j + n_comp;
                         SurfPtr->next_ptr = 
                                    &Surface[Comp][NumElem+n_comp]; 
	       }
     }
     /* -----------     somewhere in the middle  ---------------------  */
     else if ( (i > 0) && (i < (n_eta-1))) {
               if (n_next == n_phi) {
                 NextPtr = NumElem+n_next;
                 if (NextPtr >= BinPtr->NumElem)  
                       NextPtr = BinPtr->NumElem-1;
                         SurfPtr->next_ptr = 
                                    &Surface[Comp][NextPtr];
	       } else {
                 n_comp = ROUND(((float)(n_next)/(float)(n_phi))*(float)(j));
                 n_comp = n_phi - j + n_comp;
                 NextPtr = NumElem+n_comp;
                 if (NextPtr >= BinPtr->NumElem) 
                       NextPtr = BinPtr->NumElem-1;
                         SurfPtr->next_ptr = 
                                    &Surface[Comp][NextPtr]; 
	       }
              if (n_last == n_phi) {
                 SurfPtr->last_ptr = 
                                    &Surface[Comp][NumElem-n_next];
	       } else {
                 n_comp = ROUND(((float)(n_last)/(float)(n_phi))*(float)(j));
                 n_comp = (n_last - n_comp) + j; 
                         SurfPtr->last_ptr = 
                                    &Surface[Comp][NumElem-n_comp]; 
	       }
     }

     /* -----------     the last  ring   -----------------------------  */

     /*@ignore@*/
     /* Storage SurfPtr->last_ptr is kept in one path, but live in      */
     /*    another.                                                     */

     else  if (i == (n_eta-1)) {

       if (n_last == n_phi) { 
	 NextPtr = NumElem-n_phi;
	 if (NextPtr >= BinPtr->NumElem) 
	   NextPtr = BinPtr->NumElem-1;
	 SurfPtr->last_ptr =  
	   &Surface[Comp][NextPtr];
       } else { 
	 n_comp = ROUND(((float)(n_last)/(float)(n_phi))*(float)(j));
	 n_comp = (n_last - n_comp) + j;  
	 NextPtr = NumElem-n_comp -1;
	 if (NextPtr >= BinPtr->NumElem) 
	   NextPtr = BinPtr->NumElem-1;
	 SurfPtr->last_ptr =  
	   &Surface[Comp][NextPtr];  
       }

       if (Flags.fill == OFF) {
	 n_comp = (j + (n_phi/2)) % n_phi;
	 n_comp = - j + n_comp;
	 NextPtr = NumElem+n_comp;
	 SurfPtr->next_ptr = 
	   &Surface[Comp][NextPtr];
       } else {
	 if (Comp == Primary) {
	   SurfPtr->next_ptr = 
	     &Surface[Secondary][NextPtr];
	 } else {
	   NextPtr = NumElem;
	   SurfPtr->next_ptr = 
	     &Surface[Primary][NextPtr];
	 }
       }
     }
     

     if (NumElem != 0) {
       if (j != 0) {
	 SurfPtr->phi_ptr = 
	   &Surface[Comp][NumElem-1];
       } else {
	 SurfPtr->phi_ptr =
	   &Surface[Comp][NumElem+n_phi-1];
       }
       if (j != (n_phi-1)) {
	 SurfPtr->phi1_ptr = 
	   &Surface[Comp][NumElem+1];
       } else {
	 SurfPtr->phi1_ptr =
	   &Surface[Comp][NumElem-(n_phi-1)];
       }
     } else {
       SurfPtr->phi1_ptr = 
	 &Surface[Comp][1];
       SurfPtr->phi_ptr = 
	 &Surface[Comp][1];
     }
     
     ++NumElem;
     ++SurfPtr;
     
   }
  /*@end@*/
  }

  }

  /* -------------- END SET next/last_ptr ----------------------------- */


  /**********************************************************************/
  /*                                                                    */
  /*               Temperature Distribution                             */
  /*                                                                    */
  /**********************************************************************/

  /* --------------- Gravitational Darkening  ------------------------- */

  /* T propto g**GravDarkCoeff                                          */

  SurfPtr      = Surface[Comp];

  for (i = 0; i < BinPtr->NumElem; ++i) {

   SurfPtr->temp = 
         BinPtr->Temperature 
       * (exp(BinPtr->GravDarkCoeff * log(SurfPtr->grav)) 
         / BinPtr->DarkeningGrav );

   ++SurfPtr;
 
  }

  /* ------------------------   Spots            --------------------   */


  /* make spots or initialize the dimfactor to 1.0                      */
  
  if (Flags.first_pass == ON) {
    if (Comp == Primary) {
      if (Flags.Spots1 > OFF) { 
        MakeSpots(Primary,   0); 
      } else { 
        SurfPtr      = Surface[Primary];  
        for (j = 0; j < Binary[Primary].NumElem; ++j) { 
          SurfPtr->SumDimFactor = 1.0;
          ++SurfPtr;
	}
      }
    }

    if (Comp == Secondary) {
      if (Flags.Spots2 > OFF) { 
	MakeSpots(Secondary, 0);
      } else {
        SurfPtr      = Surface[Secondary];   
	for (j = 0; j < Binary[Secondary].NumElem; ++j) { 
                       SurfPtr->SumDimFactor = 1.0;
		       ++SurfPtr;
	}
      }
    }

  } else {

    /* move spots if asynchroneous rotation or elliptic orbit           */

    if (Comp == Primary) {
      if ( (Flags.Spots1 > OFF) && 
         ( Flags.asynchron1 == ON 
	   || Flags.elliptic == ON) ) 
                   MakeSpots(Primary, Orbit.PhaseIndex);
    }

    if (Comp == Secondary) {
      if ( (Flags.Spots2 > OFF) && 
         ( Flags.asynchron2 == ON 
	   || Flags.elliptic == ON) ) 
                   MakeSpots(Secondary, Orbit.PhaseIndex);
    }

  }

  /* --------------------------------- Reflection  -------------------- */

  if ((Flags.reflect > OFF) && (Comp == Secondary)) {
       LightReflect();
  } 

  if (Flags.reflect == OFF) SimpleReflect(Comp);
  
  /* ---------------------------- mean temperature  -------------------- */

  Weight = 0.0; TempMean = 0.0;
  SurfPtr      = Surface[Comp];
  for (i = 0; i < BinPtr->NumElem; ++i) {
    Area     = SurfPtr->area;
    TempMean = TempMean + SurfPtr->temp * Area;
    Weight   = Weight + Area;
  }
  BinPtr->TempMean = TempMean/Weight;
    


  /**********************************************************************/
  /*                                                                    */
  /*               Flux intensity computation                           */
  /*                                                                    */
  /**********************************************************************/

  if (Comp == Secondary) {
     if (Flags.blackbody == ON) { 
        BlackbodyFlux(Primary);
        BlackbodyFlux(Secondary);
     } else {
        ModelFlux(Primary);
        ModelFlux(Secondary);
     }
  }

  /* ----------------           output            --------------------- */

  if (Flags.debug[VERBOSE] == ON) { 
         /* printf("\n Star: %5d\n", (Comp+1)); */
    printf(_(" Surface Area, Mean Gravity (dimensionless):  %8.6f %8.6f\n"),
	   BinPtr->Surface, BinPtr->Gravity);
    printf(_(" Temperature is %8.6f\n"), BinPtr->Temperature);
    
    printf("\n");
    printf(_(" Temperature Distribution corrected for: \n")); 
    printf(_("    Reflection and Gravity Darkening\n"));
    printf(_("    Coefficient for Gravity Darkening was: %6.4f\n"),
	   BinPtr->GravDarkCoeff);
  }

  return(0);

}

