/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <float.h>
#include "Light.h"

#define  CHECK_NONE  0
#define  CHECK_LOWER 1
#define  CHECK_UPPER 2
#define  CHECK_BOTH  3
#define  NO_FIT     -1

#ifdef _WITH_GTK

/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Update the entry fields in GUI
 @tip        - entry fields  are global variables
 @param     (void)   
 @return    (void)   
 @heading   Update GUI
*******************************************************************/
void UpdateGui()
{
  char           InitString[64];

    sprintf(InitString, "%8.5f", Binary[Primary].Mq);
    gtk_entry_set_text (GTK_ENTRY (e_101), InitString);

    sprintf(InitString, "%6.3f", Binary[Primary].RocheFill);
    gtk_entry_set_text (GTK_ENTRY (e_103), InitString);

    sprintf(InitString, "%6.3f", RTOD*Orbit.Inclination);
    gtk_entry_set_text (GTK_ENTRY (e_102), InitString);

    sprintf(InitString, "%6.3f", Binary[Secondary].RocheFill);
    gtk_entry_set_text (GTK_ENTRY (e_104), InitString);

    sprintf(InitString, "%6.3f", Binary[Primary].Temperature);
    gtk_entry_set_text (GTK_ENTRY (e_105), InitString);

    sprintf(InitString, "%6.3f", Binary[Secondary].Temperature);
    gtk_entry_set_text (GTK_ENTRY (e_106), InitString);

    sprintf(InitString, "%6.3f", Orbit.Excentricity);
    gtk_entry_set_text (GTK_ENTRY (e_107), InitString);

    sprintf(InitString, "%6.3f", Orbit.Omega);
    gtk_entry_set_text (GTK_ENTRY (e_108), InitString);

    sprintf(InitString, "%6.3f", Binary[Primary].Fratio);
    gtk_entry_set_text (GTK_ENTRY (e_109), InitString);

    sprintf(InitString, "%6.3f", Binary[Secondary].Fratio);
    gtk_entry_set_text (GTK_ENTRY (e_110), InitString);

    sprintf(InitString, "%8.1f", Orbit.LambdaZero);
    gtk_entry_set_text (GTK_ENTRY (e_111), InitString);

    sprintf(InitString, "%3d", Flags.reflect);
    gtk_entry_set_text (GTK_ENTRY (e_113), InitString);

    sprintf(InitString, "%8d", PhaseSteps);
    gtk_entry_set_text (GTK_ENTRY (e_112), InitString);

    if (SpotActive[0] == ON ) {
        sprintf(InitString, "%6.3f", Spot[Primary][0].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_401), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_402), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].radius);
        gtk_entry_set_text (GTK_ENTRY (e_403), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_404), InitString);

    }

    if (SpotActive[0] == OFF && SpotActive[1] == ON ) {
        sprintf(InitString, "%6.3f", Spot[Primary][0].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_405), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_406), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].radius);
        gtk_entry_set_text (GTK_ENTRY (e_407), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][0].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_408), InitString);
    }

    if (SpotActive[1] == ON && SpotActive[0] == ON ) {
        sprintf(InitString, "%6.3f", Spot[Primary][1].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_405), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][1].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_406), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][1].radius);
        gtk_entry_set_text (GTK_ENTRY (e_407), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Primary][1].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_408), InitString);
    }

    /**************
    fprintf(stderr, "P: %d %d S: %d %d ON=%d\n", 
	    SpotActive[0],
	    SpotActive[1],
	    SpotActive[2],
	    SpotActive[3],
	    ON);
    fprintf(stderr, "%6.3f\n", Spot[Secondary][0].longitude);
    **************/

    if (SpotActive[2] == ON ){ 
        sprintf(InitString, "%6.3f", Spot[Secondary][0].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_409), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_410), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].radius);
        gtk_entry_set_text (GTK_ENTRY (e_411), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_412), InitString);
    }

    if    (SpotActive[2] == OFF && SpotActive[3] == ON ) {
        sprintf(InitString, "%6.3f", Spot[Secondary][0].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_413), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_414), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].radius);
        gtk_entry_set_text (GTK_ENTRY (e_415), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][0].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_416), InitString);
    }

    if (SpotActive[3] == ON && SpotActive[2] == ON ) {
        sprintf(InitString, "%6.3f", Spot[Secondary][1].longitude);
        gtk_entry_set_text (GTK_ENTRY (e_413), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][1].latitude);
        gtk_entry_set_text (GTK_ENTRY (e_414), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][1].radius);
        gtk_entry_set_text (GTK_ENTRY (e_415), InitString);
        
        sprintf(InitString, "%6.3f", Spot[Secondary][1].dimfactor);
        gtk_entry_set_text (GTK_ENTRY (e_416), InitString);
    }

    gtk_entry_set_text (GTK_ENTRY (e_name), Orbit.Name);

    sprintf(InitString, "%8.4f", Orbit.TruePeriod/86400.);
    gtk_entry_set_text (GTK_ENTRY (e_302), InitString);

    sprintf(InitString, "%8.4f", Orbit.TrueMass/1.989E30 );
    gtk_entry_set_text (GTK_ENTRY (e_303), InitString);

    sprintf(InitString, "%6.4f", Orbit.TrueDistance/6.960E8);
    gtk_entry_set_text (GTK_ENTRY (e_304), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Umag]);
    gtk_entry_set_text (GTK_ENTRY (e_501), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Bmag]);
    gtk_entry_set_text (GTK_ENTRY (e_502), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Vmag]);
    gtk_entry_set_text (GTK_ENTRY (e_503), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Rmag]);
    gtk_entry_set_text (GTK_ENTRY (e_504), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Imag]);
    gtk_entry_set_text (GTK_ENTRY (e_505), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Jmag]);
    gtk_entry_set_text (GTK_ENTRY (e_506), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Hmag]);
    gtk_entry_set_text (GTK_ENTRY (e_507), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[Kmag]);
    gtk_entry_set_text (GTK_ENTRY (e_508), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[umag]);
    gtk_entry_set_text (GTK_ENTRY (e_509), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[vmag]);
    gtk_entry_set_text (GTK_ENTRY (e_510), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[bmag]);
    gtk_entry_set_text (GTK_ENTRY (e_511), InitString);

    sprintf(InitString, "%6.3f", Orbit.Third[ymag]);
    gtk_entry_set_text (GTK_ENTRY (e_512), InitString);

    return;
}


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Write out a configuration file
 @tip        - all entries set to CONST
 @param     (char) *OutputFile     The output file name
 @return    (void)   
 @heading   Output Configuration
*******************************************************************/
void WriteConfig(char *OutputFile)
{
  FILE           *out_file;
  int            spotcount;
  FileType       *dataFile = FileList;

  out_file = fopen(OutputFile, "w");

  if (out_file == NULL) {
       WARNING(_("Could not open output config file"));
       return;
  }

  /*  setlocale (LC_NUMERIC, "C"); */

  fprintf(out_file, "#####################################\n");
  fprintf(out_file, "# \n");
  fprintf(out_file, "# Nightfall configuration file \n");
  fprintf(out_file, "# \n");
  fprintf(out_file, "#####################################\n");

  fprintf(out_file, "Name                 %s\n\n", 
                                         Orbit.Name);
  fprintf(out_file, "MassRatio            CONST  %f\n\n", 
                                         Binary[Primary].Mq);
  fprintf(out_file, "Inclination          CONST  %f\n\n", 
                                         RTOD*Orbit.Inclination);
  fprintf(out_file, "PrimaryRocheFill     CONST  %f\n\n", 
                                         Binary[Primary].RocheFill);
  fprintf(out_file, "SecondaryRocheFill   CONST  %f\n\n", 
                                         Binary[Secondary].RocheFill);
  fprintf(out_file, "PrimaryTemperature   CONST  %f\n\n", 
                                         Binary[Primary].Temperature);
  fprintf(out_file, "SecondaryTemperature CONST  %f\n\n", 
                                         Binary[Secondary].Temperature);
  fprintf(out_file, "ThirdLight_U         CONST  %f\n\n", 
                                         Orbit.Third[Umag]);
  fprintf(out_file, "ThirdLight_B         CONST  %f\n\n", 
                                         Orbit.Third[Bmag]);
  fprintf(out_file, "ThirdLight_V         CONST  %f\n\n", 
                                         Orbit.Third[Vmag]);
  fprintf(out_file, "ThirdLight_R         CONST  %f\n\n", 
                                         Orbit.Third[Rmag]);
  fprintf(out_file, "ThirdLight_I         CONST  %f\n\n", 
                                         Orbit.Third[Imag]);
  fprintf(out_file, "ThirdLight_J         CONST  %f\n\n", 
                                         Orbit.Third[Jmag]);
  fprintf(out_file, "ThirdLight_H         CONST  %f\n\n", 
                                         Orbit.Third[Hmag]);
  fprintf(out_file, "ThirdLight_K         CONST  %f\n\n", 
                                         Orbit.Third[Kmag]);
  fprintf(out_file, "ThirdLight_Str_u     CONST  %f\n\n", 
                                         Orbit.Third[umag]);
  fprintf(out_file, "ThirdLight_Str_v     CONST  %f\n\n", 
                                         Orbit.Third[vmag]);
  fprintf(out_file, "ThirdLight_Str_b     CONST  %f\n\n", 
                                         Orbit.Third[bmag]);
  fprintf(out_file, "ThirdLight_Str_y     CONST  %f\n\n", 
                                         Orbit.Third[vmag]);

  if (Flags.elliptic == ON)
  fprintf(out_file, "EllipticOrbitEccentricity        CONST  %f\n\n", 
                                         Orbit.Excentricity);
  else
  fprintf(out_file, "#EllipticOrbitEccentricity        CONST  %f\n\n", 
                                         Orbit.Excentricity);

  if (Flags.elliptic == ON)
  fprintf(out_file, "EllipticOrbitPeriastronLength    CONST  %f\n\n", 
                                         Orbit.Omega);
  else
  fprintf(out_file, "#EllipticOrbitPeriastronLength    CONST  %f\n\n", 
                                         Orbit.Omega); 

  if (Flags.asynchron1 == ON)
  fprintf(out_file, "PrimaryFRatio        CONST  %f\n\n", 
                                         Binary[Primary].Fratio);
  else
  fprintf(out_file, "#PrimaryFRatio        CONST  %f\n\n", 
                                         Binary[Primary].Fratio);

  if (Flags.asynchron2 == ON)
  fprintf(out_file, "SecondaryFRatio      CONST  %f\n\n", 
                                         Binary[Secondary].Fratio);
  else
  fprintf(out_file, "#SecondaryFRatio      CONST  %f\n\n", 
                                         Binary[Secondary].Fratio);

  fprintf(out_file, "AbsoluteMass         CONST  %f\n\n", 
                                         Orbit.TrueMass/1.989E30);
  fprintf(out_file, "AbsoluteDistance     CONST  %f\n\n", 
                                         Orbit.TrueDistance/6.960E8);
  fprintf(out_file, "AbsolutePeriod       CONST  %f\n\n", 
                                         Orbit.TruePeriod/86400.);

  if (Flags.Spots1 > 0) {
    for (spotcount = 0; spotcount < Flags.Spots1; ++spotcount) {
    fprintf(out_file, "SpotPrimary  CONST CONST CONST CONST %f %f %f %f\n\n",
        Spot[Primary][spotcount].longitude,
        Spot[Primary][spotcount].latitude,
        Spot[Primary][spotcount].radius,
        Spot[Primary][spotcount].dimfactor);
    }
  }

  if (Flags.Spots2 > 0) {
    for (spotcount = 0; spotcount < Flags.Spots2; ++spotcount) {
    fprintf(out_file, "SpotSecondary  CONST CONST CONST CONST %f %f %f %f\n\n",
        Spot[Secondary][spotcount].longitude,
        Spot[Secondary][spotcount].latitude,
        Spot[Secondary][spotcount].radius,
        Spot[Secondary][spotcount].dimfactor);
    }
  }

  while (dataFile != NULL) {
    fprintf(out_file, "InputFile   %s    %d\n", 
	    dataFile->DataFile, dataFile->DataFormat);
    dataFile = dataFile->nextFile;
  }

  /*  setlocale (LC_NUMERIC, ""); */
  fclose(out_file);
}   

#endif


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Parse a line for string, do error handling
 @param     (char) *InputFile     The input file name
 @param     (char) *varname       The test string
 @param     (char) *outvar        The output variable
 @param     (int)   numline       The line number
 @param     (int)   check_flag    what to check
 @param     (float) lower         lower limit
 @param     (float) upper         upper limit
 @param     (int)   die_flag      die on failure
 @param     (int)   fit_flag      set fit for index fit_flag
 @return    (int)                 The exit status
 @heading   Read Configuration
*******************************************************************/
int parse_double(char *in_line, char *varname, double *outvar, int numline,
	     int check_flag, float lower, float upper, 
             int die_flag, int fit_flag)
{
  char           first[MAX_CFG_INLINE+1];           /* first item            */
  char           second[MAX_CFG_INLINE+1];          /* second item           */
  float          third = 0.;                        /* third item            */
  int            test;
  char           msg[84];
  int            success = OFF;

  test = sscanf(in_line, "%s %s %f", first, second, &third);

  /* fprintf(stderr, "%s\nData:  %s / %s / %s\n", 
		in_line, first, second, third);
  */

  if (test == 3) {

    if (Flags.debug[INFILE] == ON) 
       fprintf(stderr, _("search %s for %s\n"), first, varname); 

    if (strcmp(first, varname) == 0) {
 
    /* -------------------  check bounds -------------------- */

      if (check_flag == CHECK_UPPER) {
         if (third <= upper) {
             *outvar = third;
	     success = ON;
	 }
      }
      else if (check_flag == CHECK_LOWER) {
         if (third >= lower) {
             *outvar = third;
	     success = ON;
	 }
      }
      else if (check_flag == CHECK_BOTH) {
         if (third >= lower && third <= upper) {
             *outvar = third;
	     success = ON;
	 }
      }
      else if (check_flag == CHECK_NONE) {
             *outvar = third;
	     success = ON;
      }

    /* -------------------  handle error -------------------- */

      if (success == OFF) {
        sprintf(msg, 
		_("Range Error (%8.3g,%8.3g) in .cfg file: %8.3g, line %6d\n"),
		lower, upper, third, numline); 
#ifndef _WITH_GTK
        if (die_flag == ON) nf_error (msg);
        else WARNING(msg);
#else 
        if (Flags.interactive == OFF) {
           if (die_flag == ON) nf_error (msg);
           else WARNING(msg);
        } else {
           WARNING (msg);
        }
#endif
        return(OFF);
      }

    /* -------------------  set fit flag -------------------- */

      if (success == ON && fit_flag != NO_FIT) {
        if (strcmp(second, "fit") == 0) {
          Flags.simplex[fit_flag] = ON;
        } else if (strcmp(second, "const") != 0) {
          sprintf(msg,  _("Invalid line %6d in Config File\n"),
              numline);
	  WARNING (msg);
	}
      }
    
    /* -------------------  return success ------------------ */

     if (Flags.debug[INFILE] == ON) 
       fprintf(stderr, _("success \n")); 
      return(ON);
   
    }
  }

    /* -------------------  return failure ------------------ */

  return(OFF);
}


/******************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @tip       Parse a configuration file
 @param     (char) *InputFile     The input file name
 @param     (int)  *numarg        The number of mandatory params read
 @return    (void)   
 @heading   Read Configuration
*******************************************************************/
void ParseConfig(char *InputFile, int *numarg)
{

  char           fullPath[1024];                    /* full path to file     */
  char           in_line[MAX_CFG_INLINE+1];         /* input line from file  */
  char           first[MAX_CFG_INLINE+1];           /* first item            */
  char           second[MAX_CFG_INLINE+1];          /* second item           */
  char           third[MAX_CFG_INLINE+1];           /* third item            */
  char           fourth[MAX_CFG_INLINE+1];          /* fourth item           */
  char           fifth[MAX_CFG_INLINE+1];           /* fifth item            */
  float          In1, In2, In3, In4;                /* input data            */
  int            Ii1;                               /* integer input         */
  int            in_test;                           /* test variable         */
  int            test;                              /* test variable         */
  int            numline;                           /* lines read            */
  int            mandatory;                         /* mandatory args read   */
  FILE          *in_file;                           /* input filehandle      */
  char          *InPtr ;                            /* pointer to inline     */
  char           msg[64];                           /* error message         */
#ifdef _WITH_GTK
  char           InitString[64];
  int  i, j;                                        /* loop variable         */
#endif

  numline = 0; mandatory = 0;
  
  strncpy(fullPath, InputFile, sizeof(fullPath)-1);

  in_file = fopen(InputFile, "r");  

  if (in_file  == NULL) {

    strcpy(fullPath, "./cfg/");
    strncat(fullPath, InputFile, sizeof(fullPath)-strlen(InputFile)-7);
    in_file = fopen(fullPath, "r");

    if (in_file  == NULL) {

      strncpy(fullPath, data_cfg_fls(), sizeof(fullPath)-strlen(InputFile)-6);
      /* strcat(fullPath, "/cfg/"); */
      strcat(fullPath, "/");
      strcat(fullPath, InputFile);
      in_file = fopen(fullPath, "r");

    }

  }      

  /* ------    open file   -----------  */

#ifdef _WITH_GTK
  if (in_file == NULL) {
    if ( Flags.interactive == ON && Flags.parseCL == OFF ) {
       make_dialog(_(errmsg[7]));
       return;
    } else nf_error(_(errmsg[7]));
  }
#else
  if (in_file == NULL) nf_error(_(errmsg[7]));
#endif

#ifdef _WITH_GTK
  if ( Flags.interactive == ON && Flags.parseCL == OFF ) 
    my_appbar_push(_("Reading configuration"));
#endif

  if (Flags.debug[INFILE] == ON) {
      fprintf(stderr, "Configuration File is %s\n", InputFile);
  }

#ifdef _WITH_GTK
  if ( Flags.interactive == ON  && Flags.parseCL == OFF ) {

    /* ------  initialize  -----------  */

       Flags.PlotBand = -1;

       Flags.Spots1      = 0;
       Flags.Spots2      = 0;
       Flags.fill        = OFF;

       Flags.elliptic    = OFF;
       Flags.asynchron1  = OFF;
       Flags.asynchron2  = OFF;

       Flags.first_pass  = ON;
       Flags.ProComputed = OFF;
       Flags.IsComputed  = OFF;

       Orbit.Dist         = 1.0;
       Orbit.MinDist      = 1.0;
       Orbit.Excentricity = 0.0;
       Orbit.Omega        = 90.;
       Orbit.TruePeriod   = 0.0;
       Orbit.TrueDistance = 0.0;
       Orbit.TrueMass     = 0.0;
       strcpy(Orbit.Name, "(void)");

       gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_eccentric_button), FALSE);
       gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_fpri_button), FALSE);
       gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_fsec_button), FALSE);
       gtk_toggle_button_set_state 
	 (GTK_TOGGLE_BUTTON (t_psp1_button), FALSE);
       gtk_toggle_button_set_state 
	 (GTK_TOGGLE_BUTTON (t_psp2_button), FALSE);
       gtk_toggle_button_set_state 
	 (GTK_TOGGLE_BUTTON (t_ssp1_button), FALSE);
       gtk_toggle_button_set_state 
	 (GTK_TOGGLE_BUTTON (t_ssp2_button), FALSE);


       TG_P1spot = 0, TG_P2spot = 0; 
       TG_S1spot = 0, TG_S2spot = 0;
       for (i=0; i < 4; ++i) SpotActive[i] = 0;
       TG_1spots = 0, TG_2spots = 0;
       for (i=0; i < 16; ++i) FitSpotS[i] = 0;
       for (i=0; i < 2; ++i) {
	 for (j=0; j < 2; ++j) {
	   SpotStore[i][j].nowlongitude = 0.0;            
	   SpotStore[i][j].longitude    = 0.0;            
	   SpotStore[i][j].latitude     = 0.0;            
	   SpotStore[i][j].radius       = 0.0;            
	   SpotStore[i][j].dimfactor    = 0.0;
	   Spot[i][j].longitude    = 0.0;            
	   Spot[i][j].latitude     = 0.0;            
	   Spot[i][j].radius       = 0.0;            
	   Spot[i][j].dimfactor    = 0.0;
	 }
       }

       ClearList(); /* data file list    */


    /* ------  end initialize  --------  */

  }
#endif

  /*  setlocale (LC_NUMERIC, "C"); */

  in_line[0] = '\n';

  /* while (LireLigne(in_line, (int)(sizeof(in_line)), in_file) > 0) { */

  while( NULL != fgets( in_line, MAX_CFG_INLINE, in_file ))   {

        ++numline;


        /* clip off optional comment tail indicated by a semi-colon */
        if( NULL != (InPtr = strchr( in_line, ';' )))
            *InPtr = '\0';
        else
            InPtr = in_line + strlen( in_line );
        /* clip off trailing and leading white space                */
        InPtr--;
        while( isspace( *InPtr ) && InPtr >= in_line )
            *InPtr-- = '\0';
        InPtr = in_line;
        while( isspace( *InPtr ))
            InPtr++;
        /* break if empty line                                      */
        if( 0 == strlen( InPtr ))
            continue;


        /* this is a header or empty line */
        if (InPtr[0] == '#' || InPtr[0] == '\n') continue;

        in_test = OFF;
  	
	if (Flags.debug[INFILE] == ON) {
	  fprintf(stderr, "LINE: %s\n", InPtr);
	}
	
	/* -----------  the name of the system   ---------------  */
	
        test = sscanf(InPtr, "%s %s", first, second);
        /* convert to lowercase                                   */
        nf_strlwr( first );
	
        if (test == 2 && strcmp(first, "name") == 0) {
          if (Flags.debug[INFILE] == ON) 
                 fprintf(stderr, _("system name is %s\n"), second); 
	  strcpy(Orbit.Name, second);
	  continue;
	} else if (strcmp(first, "name") == 0) {
          sprintf(msg,  _("Invalid line %6d in Config File\n"),
                     numline);
          WARNING (msg);
	  continue;
        }
 
	/* -----------------  input file      ------------------  */
	
        test = sscanf(InPtr, "%s %s %d", first, second, &Ii1);
        /* convert to lowercase                                   */
        nf_strlwr( first );
	
        if (test == 2 || test == 3 ) {
	  if (strcmp(first, "inputfile") == 0) {
            if (Flags.debug[INFILE] == ON) 
                 fprintf(stderr, _("read input file %s\n"), second); 
            if (test == 2 || Ii1 == NF_NATIVE_FILE ) Read(second);
            continue;
	  } 
	}

        /* -----------  convert to lowercase -------------------  */

        nf_strlwr( InPtr );
	
	/* -----------  the mandatory parameters ---------------  */
	
	/* test input, do error handling, put in variable         */
        in_test = parse_double(InPtr, "massratio", &Binary[Primary].Mq, 
                               numline,
			       CHECK_BOTH, LIM_MQ_L, LIM_MQ_H, ON, 0); 
        if (in_test == ON) {
	  ++mandatory;
          Binary[Secondary].Mq = 1.0/Binary[Primary].Mq;
          continue;
	}
	
        in_test = parse_double(InPtr, "inclination", &Orbit.Inclination,
                               numline,
			       CHECK_BOTH, LIM_IN_L, 
			       (float) RTOD*LIM_IN_H, ON, 1);
	if (in_test == ON) {
	  ++mandatory;
	  Orbit.Inclination = DTOR*Orbit.Inclination;
          continue;
	}
          
        in_test = parse_double(InPtr, "primaryrochefill", 
			       &Binary[Primary].RocheFill,numline,
			       CHECK_BOTH, LIM_RF_L, LIM_RO_H, ON, 2);   
	if (in_test == ON) {   ++mandatory;  continue; }
	
        in_test = parse_double(InPtr, "secondaryrochefill", 
                             &Binary[Secondary].RocheFill,numline,
			       CHECK_BOTH, LIM_RF_L, LIM_RO_H, ON, 3);   
	if (in_test == ON) {   ++mandatory;  continue; }
	
        
        if (Flags.blackbody == OFF) {
          in_test = parse_double(InPtr, "primarytemperature", 
                               &Binary[Primary].Temperature,numline,
			         CHECK_BOTH, LIM_TM_L, LIM_TM_H, ON, 4);   
	  if (in_test == ON) {   ++mandatory;  continue; }
          in_test = parse_double(InPtr, "secondarytemperature", 
                               &Binary[Secondary].Temperature,numline,
			         CHECK_BOTH, LIM_TM_L, LIM_TM_H, ON, 5);   
	  if (in_test == ON) {   ++mandatory;  continue; }
	} else {
          in_test = parse_double(InPtr, "primarytemperature", 
                               &Binary[Primary].Temperature,numline,
			         CHECK_BOTH, LIM_TB_L, LIM_TB_H, ON, 4);   
	  if (in_test == ON) {   ++mandatory;  continue; }
          in_test = parse_double(InPtr, "secondarytemperature", 
                               &Binary[Secondary].Temperature,numline,
			         CHECK_BOTH, LIM_TB_L, LIM_TB_H, ON, 5);   
	  if (in_test == ON) {   ++mandatory;  continue; }
	}
	
	/* -----------------  eccentric orbit ------------------  */
        
        in_test = parse_double(InPtr, "ellipticorbiteccentricity", 
                             &Orbit.Excentricity,numline,
			       CHECK_BOTH, LIM_EX_L, LIM_EX_H, OFF, 6);
	if (Orbit.Excentricity >= FLT_EPSILON) Flags.elliptic = ON;
	if (in_test == ON) {
#ifdef _WITH_GTK
             if (Flags.interactive == ON && Flags.parseCL == OFF )
                 gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_eccentric_button), TRUE);
#endif
            continue; 
	}
	
        in_test = parse_double(InPtr, "ellipticorbitperiastronlength",
                             &Orbit.Omega,numline,
			       CHECK_BOTH, LIM_PA_L, LIM_PA_H, OFF, 7);   
	if (in_test == ON) continue;
	
	/* -------------  asyncroneous rotation ----------------  */
	
	in_test = parse_double(InPtr, "primaryfratio",
                             &Binary[Primary].Fratio,numline,
			       CHECK_BOTH, LIM_FR_L, LIM_FR_H, OFF, 8);
	if ( fabs(Binary[Primary].Fratio-1.0) >= FLT_EPSILON) 
	  Flags.asynchron1 = ON;
	if (in_test == ON) {
#ifdef _WITH_GTK
             if (Flags.interactive == ON && Flags.parseCL == OFF )
                 gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_fpri_button), TRUE);
#endif
             continue;
	}
	
	in_test = parse_double(InPtr, "secondaryfratio",
                             &Binary[Secondary].Fratio,numline,
			       CHECK_BOTH, LIM_FR_L, LIM_FR_H, OFF, 9);   
	if ( fabs(Binary[Secondary].Fratio-1.0) >= FLT_EPSILON) 
	  Flags.asynchron1 = ON;
	if (in_test == ON) {
#ifdef _WITH_GTK
             if (Flags.interactive == ON && Flags.parseCL == OFF )
                 gtk_toggle_button_set_state 
                   (GTK_TOGGLE_BUTTON (t_fsec_button), TRUE);
#endif
             continue;
	}
	
	/* -----------------  absolute values ------------------  */
	
	in_test = parse_double(InPtr, "absolutemass",
                             &Orbit.TrueMass,numline,
			       CHECK_LOWER, LIM_MASS_L,    0.0, OFF, 26);   
	if (in_test == ON) {
	  Orbit.TrueMass = Orbit.TrueMass  * 1.989E30;
	  continue;
	}
	
	in_test = parse_double(InPtr, "absolutedistance",
                             &Orbit.TrueDistance,numline,
			       CHECK_LOWER, LIM_DIST_L,     0.0, OFF, 27);   
	if (in_test == ON) {
	  Orbit.TrueDistance = Orbit.TrueDistance  * 6.960E8;
	  continue;
	}
	
	in_test = parse_double(InPtr, "absoluteperiod",
                             &Orbit.TruePeriod,numline,
			       CHECK_LOWER, LIM_PERI_L,     0.0, OFF, NO_FIT); 
	if (in_test == ON) {
	  Orbit.TruePeriod = Orbit.TruePeriod * 86400.;
	  continue;
	}
	
	
	/* -----------------  spots           ------------------  */
	
        test = sscanf(InPtr, "%s %s", first, second);
	
        if (strcmp(first, "spotprimary") == 0) {
          test = sscanf(in_line, "%s %s %s %s %s  %f %f %f %f",  
                        first, second, third, fourth, fifth,
                        &In1, &In2, &In3, &In4);
          if (test == 9) {

	    if (Flags.Spots1 == 0 ) {
#ifdef _WITH_GTK
	      if (Flags.interactive == ON && Flags.parseCL == OFF )
		gtk_toggle_button_set_state 
		  (GTK_TOGGLE_BUTTON (t_psp1_button), TRUE);
	      SpotActive[0] = ON; 
#endif 
	      Flags.Spots1 = 0;
	    }

	    if (Flags.Spots1 == 1 ) { 
#ifdef _WITH_GTK
	      if (Flags.interactive == ON && Flags.parseCL == OFF )
		gtk_toggle_button_set_state 
		  (GTK_TOGGLE_BUTTON (t_psp2_button), TRUE);
	      SpotActive[1] = ON; 
#endif 
	      Flags.Spots1 = 1;
	    }

            test = 0;
            if (In1 >= LIM_SLO_L && In1 <= LIM_SLO_H) {
              Spot[Primary][Flags.Spots1].longitude = In1; ++test; 
	    }
            if (In2 >= LIM_SLA_L && In2 <= LIM_SLA_H && test == 1){
              Spot[Primary][Flags.Spots1].latitude  = In2; ++test; 
	    }
            if (In3 >= LIM_SRA_L && In3 <= LIM_SRA_H && test == 2){
              Spot[Primary][Flags.Spots1].radius    = In3; ++test; 
	    }
            if (In4 >= LIM_SDF_L && In4 <= LIM_SDF_H && test == 3){
              Spot[Primary][Flags.Spots1].dimfactor = In4; ++test; 
	    }

            if (test == 4) {
#ifndef _WITH_GTK
              if (strcmp(second, "fit") == 0 && Flags.Spots1 < 2) {
                Flags.simplex[10+(Flags.Spots1 * 4) + 0] = ON;
              }            
              if (strcmp(third, "fit") == 0 && Flags.Spots1 < 2) {
                Flags.simplex[10+(Flags.Spots1 * 4) + 1] = ON;
              }
              if (strcmp(fourth, "fit") == 0 && Flags.Spots1 < 2) {
                Flags.simplex[10+(Flags.Spots1 * 4) + 2] = ON;
              }            
              if (strcmp(fifth, "fit") == 0 && Flags.Spots1 < 2) {
                Flags.simplex[10+(Flags.Spots1 * 4) + 3] = ON;
              }
#endif 
              ++Flags.Spots1;

	      continue;
	    } else {
              sprintf(msg,  _("Range Error in Config File, line %6d\n"),
                       numline);
	      WARNING (msg);
	    }
	  } else {
            sprintf(msg,  _("Invalid line %6d in Config File\n"),
                       numline);
	    WARNING (msg);
	  }
	  continue;
	}

	/* SECONDARY SPOTS */
	
        test = sscanf(InPtr, "%s %s", first, second);
	
        if (strcmp(first, "spotsecondary") == 0) {
          test = sscanf(in_line, "%s %s %s %s %s  %f %f %f %f", 
                        first, second, third, fourth, fifth,
                        &In1, &In2, &In3, &In4);
          if (test == 9) {

	    if (Flags.Spots2 == 0 ) {
#ifdef _WITH_GTK
	      if (Flags.interactive == ON && Flags.parseCL == OFF )
		gtk_toggle_button_set_state 
		  (GTK_TOGGLE_BUTTON (t_ssp1_button), TRUE);
	      SpotActive[2] = ON; 
#endif 
	      Flags.Spots2 = 0;
	    }

	    if (Flags.Spots2 == 1 ) { 
#ifdef _WITH_GTK
	      if (Flags.interactive == ON && Flags.parseCL == OFF )
		gtk_toggle_button_set_state 
		  (GTK_TOGGLE_BUTTON (t_ssp2_button), TRUE);
	      SpotActive[3] = ON; 
#endif 
	      Flags.Spots2 = 1;
	    }

            test = 0;
            if (In1 >= LIM_SLO_L && In1 <= LIM_SLO_H) {
              Spot[Secondary][Flags.Spots2].longitude = In1; ++test; 
	    }
            if (In2 >= LIM_SLA_L && In2 <= LIM_SLA_H && test == 1){
              Spot[Secondary][Flags.Spots2].latitude  = In2; ++test; 
	    }
            if (In3 >= LIM_SRA_L && In3 <= LIM_SRA_H && test == 2){
              Spot[Secondary][Flags.Spots2].radius    = In3; ++test; 
	    }
            if (In4 >= LIM_SDF_L && In4 <= LIM_SDF_H && test == 3){
              Spot[Secondary][Flags.Spots2].dimfactor = In4; ++test; 
	    }
	    /*
	    fprintf(stderr, "%6.3f\n", Spot[Secondary][0].longitude);
	    */

	    if (test == 4) {
#ifndef _WITH_GTK
              if (strcmp(second, "fit") == 0 && Flags.Spots2 < 2) {
                Flags.simplex[18+(Flags.Spots2 * 4) + 0] = ON;
	      }            
              if (strcmp(third, "fit") == 0 && Flags.Spots2 < 2) {
                Flags.simplex[18+(Flags.Spots2 * 4) + 1] = ON;
	      }
              if (strcmp(fourth, "fit") == 0 && Flags.Spots2 < 2) {
                Flags.simplex[18+(Flags.Spots2 * 4) + 2] = ON;
	      }            
              if (strcmp(fifth, "fit") == 0 && Flags.Spots2 < 2) {
                Flags.simplex[18+(Flags.Spots2 * 4) + 3] = ON;
	      }
#endif 

		
              ++Flags.Spots2;
	      continue;
	    } else {
              sprintf(msg,  _("Range Error in Config File, line %6d\n"),
                     numline);
	      WARNING (msg);
	    }
	  } else {
            sprintf(msg,  _("Invalid line %6d in Config File\n"),
                     numline);
	    WARNING (msg);
	  }
	  continue;
	}

	/* -----------------  third light     ------------------  */
	
	in_test = parse_double(InPtr, "thirdlight_u",
                             &Orbit.Third[Umag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 28);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_b",
                             &Orbit.Third[Bmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 29);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_v",
                             &Orbit.Third[Vmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 30);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_r",
                             &Orbit.Third[Rmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 31);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_i",
                             &Orbit.Third[Imag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 32);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_j",
                             &Orbit.Third[Jmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 33);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_h",
                             &Orbit.Third[Hmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 34);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_k",
                             &Orbit.Third[Kmag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 35);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_str_u",
                             &Orbit.Third[umag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 36);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_str_v",
                             &Orbit.Third[umag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 36);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_str_b",
                             &Orbit.Third[umag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 38);   
	if (in_test == ON) continue;
	
	in_test = parse_double(InPtr, "thirdlight_str_y",
                             &Orbit.Third[umag],numline,
			       CHECK_BOTH, LIM_3L_L, LIM_3L_L, OFF, 39);   
	if (in_test == ON) continue;
	
	
	/* -----------------  lightcurve steps -----------------  */
	
        test = sscanf(InPtr, "%s %s", first, second);
	
        if (strcmp(first, "lightcurvesteps") == 0) {
          test = sscanf(in_line, "%s %d", first,  &Ii1);
          if (test == 2) {
	       if (Ii1 >=  0 && Ii1 <  PHASESTEPS)  PhaseSteps = Ii1;
	       else {
                 sprintf(msg,  _("Range Error in Config File, line %6d\n"),
                     numline);
	         WARNING (msg);
		 }
#ifdef _WITH_GTK
               if (Flags.interactive == ON && Flags.parseCL == OFF ) {
                   sprintf(msg, "%8d", PhaseSteps);
                   gtk_entry_set_text (GTK_ENTRY (e_112), msg);
	       }
#endif 
          } else { 
            sprintf(msg,  _("Invalid line %6d in Config File\n"),
                     numline);
            WARNING (msg);
          }
        }
      
      
        /* -----------------  do we want a fit -----------------  */
      
        test = sscanf(InPtr, "%s %f", first, &In1);
      
        if (test == 2 && strcmp(first, "do_fit") == 0) {
            if ( In1 >= FLT_EPSILON ) {
#ifndef _WITH_GTK
                Flags.WantFit = ON;
                Flags.SimplexTol = In1;
                     if (fabs (Flags.SimplexTol - 0.001) <= FLT_EPSILON ) 
                                 Flags.anneal = ON;
#endif 
                continue;
	      } else {
	        sprintf(msg,
                     _("Range Error (0.0, INDEF) in Config File, line %6d\n"),
                     numline);
	        WARNING (msg);
	      }
	  } 
  }
  *numarg = mandatory;

  if (Flags.PlotBand == -1) Flags.PlotBand = Vmag;

  /*  setlocale (LC_NUMERIC, ""); */


#ifdef _WITH_GTK
  if (Flags.interactive == ON && Flags.parseCL == OFF ) {
    updateplotflagpage3 ();

    sprintf(InitString, "%6.3f", Spot[Primary][0].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_401), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][0].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_402), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][0].radius);
    gtk_entry_set_text (GTK_ENTRY (e_403), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][0].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_404), InitString);

    sprintf(InitString, "%6.3f", Spot[Primary][1].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_405), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][1].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_406), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][1].radius);
    gtk_entry_set_text (GTK_ENTRY (e_407), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Primary][1].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_408), InitString);

    sprintf(InitString, "%6.3f", Spot[Secondary][0].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_409), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Secondary][0].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_410), InitString);
    
    sprintf(InitString, "%6.3f", Spot[Secondary][0].radius);
    gtk_entry_set_text (GTK_ENTRY (e_411), InitString);
        
    sprintf(InitString, "%6.3f", Spot[Secondary][0].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_412), InitString);

    sprintf(InitString, "%6.3f", Spot[Secondary][1].longitude);
    gtk_entry_set_text (GTK_ENTRY (e_413), InitString);
        
    sprintf(InitString, "%6.3f", Spot[Secondary][1].latitude);
    gtk_entry_set_text (GTK_ENTRY (e_414), InitString);
        
    sprintf(InitString, "%6.3f", Spot[Secondary][1].radius);
    gtk_entry_set_text (GTK_ENTRY (e_415), InitString);
        
    sprintf(InitString, "%6.3f", Spot[Secondary][1].dimfactor);
    gtk_entry_set_text (GTK_ENTRY (e_416), InitString);
  }
#endif

  (void) fclose(in_file);

  strncpy(Flags.ConfFile, fullPath, 255);
  Flags.ConfFile[256] = '\0';


  return; 
}

