/* NIGHTFALL Light Curve Synthesis Program                                 */
/* Copyright (C) 1998 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include "Light.h"

#ifdef _WITH_GTK
#include <gtk/gtk.h>
#endif


/*********************************/
/* GLOBAL VARIABLES              */
/*********************************/

  /* include error messages                                         */

#include "LightMsg.h"

  /* Flags                                                          */
  FlagsHandle  Flags;
    
  /* Input Data                                                     */
  Photo3       *Observed[NUM_MAG+2];      /* [colour][datum]        */
  /*@null@*/ FileType     *FileList = NULL; /* data file list       */

  /* the profile */
  /*@out@*/ /*  float        **ProData; */
  float ProData[PHASESTEPS][PROFILE_ARRAY+1];

  /* Flux                                                           */
  /*@out@*/ PhotoOut     *FluxOut;        /* Output Flux            */

  double       F90[NUM_MAG];              /* Normalization Variable */
  int          Umag = 0, Bmag = 1, Vmag = 2;
  int          Rmag = 3, Imag = 4;
  int          Jmag = 5, Hmag = 6, Kmag = 7;
  int          umag = 8, vmag = 9, bmag =10, ymag =11;
  int          mag1 = 12, mag2 = 13;
  char         Filters[NUM_MAG+2][12] =
                       { "U Band", "B Band", "V Band", "R Band", 
                         "I Band", "J Band", "H Band", "K Band", "u Band",
                         "v Band", "b Band", "y Band", 
                         "Primary RV", "Secondary RV" };

 
  /* Limb Darkening Coefficients                                           */

  double Limb[2][NUM_MAG][2];            /* component, passband, number    */
  double WaveL[2][NUM_MAG];              /* wavelengths                    */

  /* Components                                                            */

  const int        Primary    = 0;       /* literal definition             */
  const int        Secondary  = 1;       /* literal definition             */
  BinaryComponent  Binary[2];            /* properties of the components   */
  SurfaceElement  *Surface[2];           /* surface elements               */
  SpotLight       *Spot[2];              /* surface spots                  */
  OrbitType          Orbit;              /* orbit data                     */

  /* These are Global Variables that are used (mainly) to pass             */
  /* Arguments to Functions that are called by RootFind/MinFind            */

  double   F;                  /* nonsync rotation  w/wsystem              */
  double   Mq;                 /* mass ratio, used in Roche functions      */
  double   RochePot;           /* Roche Potential, used in Roche functions */
  double   PhaseScaledVolume;  /* scaled volume                            */
  double   lambda, mu, nu;     /* coordinates                              */

  /* else */

  char   Out_Plot_File[256];               /* output plot file             */
  int    StepsPerPi;                       /* No of steps per Pi (surface) */
  double MaximumElements;                  /* max. no of surface elements  */
  int    PhaseSteps;                       /* phasesteps for lightcurve    */
  double GArgSquare;                       /* for the SQR macro            */



/****************************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     The 'main' program
 @param     (int)      argc   The number of aguments
 @param     (char)  *argv[]   The argument list
 @return    (int)    status   The exit status
 @heading   Main
 ****************************************************************************/
int main(int argc, char *argv[])
{

  double Merit = 0;  /* the merit for (observed-computed)              */
  int    i;          /* a loop variable                                */

  char   * args[12];
  int      argcount = 0;

#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, 
		  (getenv("NIGHTFALL_LOCALE_DIR") == NULL) ? 
		  LOCALEDIR : getenv("NIGHTFALL_LOCALE_DIR"));
  textdomain (PACKAGE);

  /* set this to 'C' in order not to run into problems with the datafiles */
  setlocale (LC_NUMERIC, "C"); 
#endif


  /* >>>>>>>>>>>>>>>>    USAGE INFO IF NO INPUT   <<<<<<<<<<<<<<<<<<<<<  */

  if (argc == 1) { 
         UsageInfo(); 
         exit(EXIT_FAILURE); 
  }

  args[0] = argv[0]; argcount = 1;
  for (i = 1; i < argc; ++i) {
    if (argcount < 12) {
      if ((argv[i][0] == '-') && (argv[i][1] == '-') ) {
	args[argcount] = argv[i];
	++argcount;
	if ((i+1) < argc) {
	  args[argcount] = argv[i+1];
	  ++argcount;
	}
      }
    }
  }

	
     

  /* >>>>>>>>>>>>>>>>    ALLOCATE MEMORY        <<<<<<<<<<<<<<<<<<<<<  */

  /* should be freed by OS on program exit (?)                         */

  Surface[0] = malloc(MAXELEMENTS * sizeof(SurfaceElement));
  Surface[1] = malloc(MAXELEMENTS * sizeof(SurfaceElement));
  Spot[0]    = malloc(N_SPOT * sizeof(SpotLight));
  Spot[1]    = malloc(N_SPOT * sizeof(SpotLight));
  FluxOut    = malloc(PHASESTEPS * sizeof(PhotoOut));
  /*  ProData    = matrix(PHASESTEPS,PROFILE_ARRAY+1); */

  if (Surface[0] == NULL || Surface[1] == NULL ||
      Spot[0]    == NULL || Spot[1]    == NULL ||
      FluxOut    == NULL /*  || ProData    == NULL */)
        nf_error(_(errmsg[0]));

  for (i=0; i < NUM_MAG+2; ++i) {
        Observed[i] = malloc(MAXOBS * sizeof(Photo3));
        if (Observed[i] == NULL) 
            nf_error(_(errmsg[0]));
  }

#ifdef _WITH_PGPLOT
#ifndef _WITH_GNUPLOT
  SetPgplotWindow();    /* set the PGPLOT window size                    */
#endif
#endif

  /* >>>>>>>>>>>>>>>>         INITIALIZE          <<<<<<<<<<<<<<<<<<<<<  */

  InitFlags();

  Flags.smap = SetMapPath ();

  if (Flags.smap == ON) 
    SetMapBand ();

  Flags.monochrome = DefineMonoWavelengths ();

  /* >>>>>>>>>>>>>>>>        GET ARGUMENTS        <<<<<<<<<<<<<<<<<<<<<< */

  Flags.parseCL = ON;   /* we parse the command line                     */

  GetArguments(&argc, argv);

  Flags.parseCL = OFF;  /* finished parsing cl                           */

  /* >>>>>>>>>>>>>>>>          MAIN SECTION        <<<<<<<<<<<<<<<<<<<<< */

  if (Flags.interactive == OFF) {

    if      (Flags.WantMap == ON) (void) ChiMap(); 
    else if (Flags.WantFit == ON) (void) Simplex();
    else                          (void) MainLoop(&Merit); 

    WriteOutput();

  } else {

#ifdef _WITH_GTK

    /******************************
    fprintf(stderr,
          "%f %f %f %f %f %f\n",
          Orbit.Inclination,
          Binary[Primary].Mq,
          Binary[Primary].Temperature,
          Binary[Secondary].Temperature,
          Binary[Primary].RocheFill,
          Binary[Secondary].RocheFill
          );
    *******************************/

    /* some reasonable defaults                                          */

    if (Orbit.Inclination             <= FLT_EPSILON &&
	Binary[Primary].Mq            <= 
	(Flags.fill == ON ? LIM_MQO_L : LIM_MQ_L) &&
	Binary[Primary].Temperature   <= FLT_EPSILON &&
	Binary[Secondary].Temperature <= FLT_EPSILON &&
	Binary[Primary].RocheFill     <= FLT_EPSILON &&
	Binary[Secondary].RocheFill   <= FLT_EPSILON) {
	  Orbit.Inclination             = DTOR * 80.0;
	  Binary[Primary].Mq            = 1.0;
	  Binary[Secondary].Mq          = 1.0;
	  Binary[Primary].Temperature   = 5700.0;
	  Binary[Secondary].Temperature = 5700.0;
	  Binary[Primary].RocheFill     = 0.6;
	  Binary[Secondary].RocheFill   = 0.6;
    }

    /* start the GUI                                                     */

    the_gui(argcount, args);

#else
    nf_error (_(errmsg[20]));
#endif

  }

  /* >>>>>>>>>>>>>>>>             END             <<<<<<<<<<<<<<<<<<<<<< */

  /* successful exit                                                     */

#ifdef _WITH_GNUPLOT
  if (Flags.plotOpened == ON) { Flags.plotOpened = OFF; cpgend(); }
#endif

#ifdef _WITH_GTK
  if (Flags.interactive == ON) gtk_exit (0);
#endif
  return (0);  

}



/*********************************************************************
 @package   nightfall
 @author    Rainer Wichmann (rwichman@lsw.uni-heidelberg.de)
 @version   1.0
 @short     Loop for computation of lightcurve
 @param     (double) *Merit   The merit for (observed-computed)
 @return    (int)    status   The exit status
 @heading   Main Loop
**********************************************************************/
int  MainLoop(/*@out@*/ double *Merit)
{
  int    PhaseIndex;               /* the index for the phase        */
  int    ErrCode;                  /* exit status of subroutines     */
  long   i;                        /* loop variable                  */
  float  pvalue;                   /* progress bar increment         */
  SurfaceElement  *SurfPtrP;       /* pointer to surface  primary    */
  SurfaceElement  *SurfPtrS;       /* pointer to surface  secondary  */

#ifdef _WITH_GTK
  char   message[64];              /* statusbar message              */
#endif

  SurfPtrP      = Surface[Primary];
  SurfPtrS      = Surface[Secondary];


  /* >>>>>>>>>>>>>         Initialize                 <<<<<<<<<<<<<< */

  pvalue = 0.0;
  *Merit = 0.0;

  for (i = 0; i < MAXELEMENTS; ++i) {
    SurfPtrP->SpotNum        = 0;
    SurfPtrS->SpotNum        = 0;
    SurfPtrP->SumDimFactor   = 0;
    SurfPtrS->SumDimFactor   = 0;
    ++SurfPtrP;
    ++SurfPtrS;
  }


  /* >>>>>>>>>>>>>         LightGeometry              <<<<<<<<<<<<<< */

  
#ifdef _WITH_GTK
   if (Flags.interactive == ON) 
     my_appbar_push(_("Setup geometry") );
#endif

  /* first pass (initial setup of stars)                             */
  Flags.first_pass = ON;

  /* store Roche fill factor -- needed for eccentric orbit           */
  /*   (at final phase not equal to initial value)                   */
  Binary[Primary].RocheStore   = Binary[Primary].RocheFill;
  Binary[Secondary].RocheStore = Binary[Secondary].RocheFill;


  /* elliptic orbit                                                  */
  if (Flags.elliptic == ON) FindPeriastron();

  /* Primary at x=y=z=0, Secondary at x=1, y=z=0                     */
  if (Flags.fill == OFF) {

     /* -- no overcontact --                                         */

     /* THE PRIMARY                                                  */
     ErrCode = DefineParam(Primary);
     if (ErrCode == 8) return(8);
     if (Flags.debug[VERBOSE] == ON) 
       {   printf("\n"); printf(_(" -- Primary defined   --\n") ); }

     /* THE SECONDARY                                                */
     ErrCode = DefineParam(Secondary);
     if (ErrCode == 8) return(8);
     if (Flags.debug[VERBOSE] == ON) 
       {   printf("\n"); printf(_(" -- Secondary defined --\n") ); }

  } else {

     /* -- Overcontact system --                                     */

     ErrCode = DefineParamOver();
     if (ErrCode == 8) return(8);
     if (Flags.debug[VERBOSE] == ON) 
       { printf("\n"); printf(_(" -- Overcontact Geometry defined --\n"));}

  }

  /* >>>>>>>>>>>>>  LightLimbDarkCoeff           <<<<<<<<<<<<<<<<<<< */

  /* Compute effective wavelength for filters                        */
  EffectiveWavelength(Primary);
  EffectiveWavelength(Secondary);

  /* >>>>>>>>>>>>>   LightDivide                 <<<<<<<<<<<<<<<<<<< */

  /*  -----------  DO SECONDARY SECOND !!! -----------------------   */

  /*  set up the surface grid                                        */

  /* THE PRIMARY                                                     */
  ErrCode = DivideSurface(Primary); 
  if (ErrCode == 8) return(8); 
  if (Flags.debug[VERBOSE] == ON) 
    { printf("\n"); printf(_(" Primary divided\n") );} 
  
  /* THE SECONDARY                                                   */
  ErrCode = DivideSurface(Secondary);
  if (ErrCode == 8) return(8);   
  if (Flags.debug[VERBOSE] == ON) 
    { printf("\n"); printf(_(" Secondary divided\n") );}


  /* >>>>>>>>>>>>>   LightLimbDarkCoeff          <<<<<<<<<<<<<<<<<<< */

  /* compute limb darkening coefficients                             */

  LightLimbDark(Primary);
  LightLimbDark(Secondary);

  /* >>>>>>>>>>>>>   LightSynthesize              <<<<<<<<<<<<<<<<<< */

  /* Calculate various quantities for geometric eclipse tests        */

  /*  if Asynch Rotation < 1.0, eventually the stars intersect       */
  /*  we test for this and abort                                     */

  if(LightSetupTests() > 0) {
#ifdef _WITH_GTK
      if (Flags.interactive == OFF) { 
                  nf_error(_(errmsg[5]));
      } else {
                  make_dialog(_(errmsg[5]));
                  return (8);
      }
#else
      nf_error(_(errmsg[5]));
#endif
  }

  Flags.first_pass = OFF;

  /* ---------------  LightVisualize       ------------------------- */

#ifdef _WITH_PGPLOT  
  if (Flags.visualize > OFF)  PlotGeometry();
#endif 

  /* -------------   LightOutput            ------------------------ */
  
  InitOutFlux();
 

  /* >>>>>>>>>>>>>   LightSynthesize              <<<<<<<<<<<<<<<<<< */

  if (Flags.debug[VERBOSE] == ON) 
    { printf("\n");   printf(_(" Starting Lightcurve Synthesis\n") );}

  /* **************    LOOP OVER PHASE ***************************** */

  if (Flags.debug[BUSY] == ON) { 
             printf("  0 per cent"); 
             (void) fflush(stdout); 
  }

#ifdef _WITH_GTK
   if (Flags.interactive == ON) 
     my_appbar_push(_("Start lightcurve computation"));
#endif

  for (PhaseIndex = 0; PhaseIndex < PhaseSteps; ++PhaseIndex) { 

#ifdef _WITH_GTK
    /* update the progress bar                                       */
    pvalue =  (float)(PhaseIndex)/(float)(PhaseSteps);
    if (Flags.interactive == ON) {
      gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar), pvalue);
      while (gtk_events_pending()) gtk_main_iteration();
    } 
#endif

    /* a quick hack for the MakeSpots program after shifting the     */
    /*  call to LightDivide                                          */
    /*   which was necessary to include spots correctly in the       */
    /*   reflection treatment                                        */
    Orbit.PhaseIndex = PhaseIndex;   

    if (Flags.debug[BUSY] == ON) { 
        printf("\b\b\b\b\b\b\b\b\b\b\b\b%3d per cent", 
              (int)((100.0*PhaseIndex)/PhaseSteps)); 
        (void) fflush(stdout); 
    }
 
  /* -------------   LightElliptic          ------------------------ */

    if (Flags.elliptic == ON) {

      /* update (nearly) everything                                  */

        SolveKepler(PhaseIndex);
        ErrCode = UpdateGeometry(Primary);
         if (ErrCode == 8) return(8);
        ErrCode = UpdateGeometry(Secondary);
         if (ErrCode == 8) return(8);

	/* DO SECONDARY SECOND !!!                                   */

        ErrCode = DivideSurface(Primary);
        if (ErrCode == 8) return(8);

        ErrCode = DivideSurface(Secondary);
        if (ErrCode == 8) return(8);

	/* individual treatment of surface elements                  */
        if (Flags.limb == 3) {
            LightLimbDark(Primary);
            LightLimbDark(Secondary);
	}

        ErrCode = LightSetupTests();
        if (ErrCode > 0) return(8);
      }

    /* ----------- move spots if asynchroneous rotation ------------ */

 
      if ( (Flags.Spots1 > OFF) 
	   && Flags.asynchron1 == ON
	   && Flags.elliptic == OFF )
	{ 
	      SurfPtrP      = Surface[Primary];

              for (i = 0; i < Binary[Primary].NumElem; ++i) {
                   /* Gravitational Darkening                        */
                   SurfPtrP->temp = 
                       Binary[Primary].Temperature 
                         * (exp(Binary[Primary].GravDarkCoeff 
                         * log(SurfPtrP->grav)) 
                         / Binary[Primary].DarkeningGrav );
		   ++SurfPtrP;
	      }

              MakeSpots(Primary, Orbit.PhaseIndex);
	}

      if ( (Flags.Spots2 > OFF) 
	   && Flags.asynchron1 == ON
	   && Flags.elliptic == OFF ) 
	{

	      SurfPtrS      = Surface[Secondary];

              for (i = 0; i < Binary[Secondary].NumElem; ++i) {
                   /* Gravitational Darkening                        */
                   SurfPtrS->temp = 
                       Binary[Secondary].Temperature 
                         * (exp(Binary[Secondary].GravDarkCoeff 
                         * log(SurfPtrS->grav)) 
                         / Binary[Secondary].DarkeningGrav );
		   ++SurfPtrP;
	      }

              MakeSpots(Secondary, Orbit.PhaseIndex);
	}


     if (Flags.elliptic == OFF && 
         ( (Flags.Spots2 > OFF 
	    && Flags.asynchron1 == ON) ||
           (Flags.Spots1 > OFF 
	    && Flags.asynchron2 == ON)    ) ){ 

        if (Flags.reflect > 0) {
           LightReflect();
         } 

        if (Flags.reflect == 0) { 
           SimpleReflect(Primary);
           SimpleReflect(Secondary);
	}
     }

  /* >>>>>>>>>>>>>   LightSynthesize              <<<<<<<<<<<<<<<<<< */


    if (Flags.fill == ON)  LightCopyThroat();

    /* eclipse checking                                              */

    LightCurve(PhaseIndex);
    

  /* -------------   LightFractional         ----------------------- */

    /* fractional visibility                                         */
    if (Flags.fractional == ON) {
      if (Flags.InEclipse1 == ON) { 
	LightFractionalPot(Primary);
      }
      if (Flags.InEclipse2 == ON) {
        LightFractionalPot(Secondary);
      } 
    }

  /* >>>>>>>>>>>>>   LightSynthesize              <<<<<<<<<<<<<<<<< */

    if (Flags.fill == ON) LightCopyThroatBack(); 

    RadVel(PhaseIndex);  

    /* light curve                                                 */

    LightFlux(Primary, PhaseIndex); 
    LightFlux(Secondary, PhaseIndex); 

 
  /* >>>>>>>>>>>>>   LightAnimate                <<<<<<<<<<<<<<<<< */


#ifdef  _WITH_PGPLOT
    if (Flags.animate == ON)  
      Animate(PhaseIndex);
#endif
 
  /* >>>>>>>>>>>>>   LightMap                    <<<<<<<<<<<<<<<<< */

    if (Flags.smap == ON) 
      PrintSurfaceMap (PhaseIndex);  

  }  /* END LOOP OVER PHASE */


  /* >>>>>>>>>>>>>   LightLib                    <<<<<<<<<<<<<<<<< */

  if (Flags.elliptic == ON) PhaseShift(); 

  NormalizeFlux();

  /* >>>>>>>>>>>>>   LightOptimize               <<<<<<<<<<<<<<<<< */

  /* compute the merit for (observed-computed)                     */
  *Merit = MeritFunction();

  if (Flags.debug[BUSY] == ON) {
    if (*Merit >= FLT_EPSILON) { 
      printf("\b\b\b\b\b\b\b\b\b\b\b\b100 per cent, ChiSquare = %15.5f\n",
	     *Merit); 
      (void) fflush(stdout); 
    } else {
      printf("\b\b\b\b\b\b\b\b\b\b\b\b100 per cent\n"); 
      (void) fflush(stdout); 
    }
  } 

  Flags.IsComputed = ON;

#ifdef _WITH_GTK
  if (Flags.interactive == ON) {
    sprintf(message, _("Chi Square = %15.5f"), *Merit);
    my_appbar_push(message);
    gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar), 1.0);
    while (gtk_events_pending()) gtk_main_iteration(); 
  }
#endif

#ifdef _WITH_PGPLOT
  if (Flags.plot > OFF) PlotOutput();
#endif

  /* get back Roche fill factor                                    */
  Binary[Primary].RocheFill   = Binary[Primary].RocheStore;
  Binary[Secondary].RocheFill = Binary[Secondary].RocheStore;

  return(0);

}

 











