/*

  sshadt_array.c

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 1999 SSH Communications Security, Finland
  All rights reserved.

  Created Thu Oct 21 13:35:14 1999.

  */

#include "sshincludes.h"
#include "sshadt_i.h"
#include "sshadt_array.h"
#include "sshadt_array_i.h"
#include "sshadt_std_i.h"
#include "sshdebug.h"

#define SSH_DEBUG_MODULE "SshADTArray"

#define ROOT     ((SshADTArrayRoot *)(c->container_specific))

/* We need to add and subtract one because otherwise the index zero
   would actually correspond to SSH_ADT_INVALID, which would be
   confusing. */
#define PTRTOUINT(ptr) ((((unsigned char *)(ptr)) - ((unsigned char *)0)) - 1)
#define UINTTOPTR(i) (&(((unsigned char *)0)[i + 1]))

static void init(SshADTContainer c)
{
/* Handles are of type (void *)idx. Ugly, but who cares. */
  SSH_VERIFY(sizeof(void *) >= sizeof(unsigned int));
  /* The dynamic array type has zero-sized headers so they should
     not be `contained'. */
  SSH_ASSERT(!(c->flags & SSH_ADT_FLAG_CONTAINED_HEADER));
  c->container_specific = ssh_xmalloc(sizeof(*ROOT));
  ROOT->array = NULL;
  ROOT->array_size = 0;
}

/* $$METHOD(array, container_init) */
SSH_ADT_STD_INIT(container_init, init(c);)

static void uninit(SshADTContainer c)
{
  int i;
  for (i = 0; i < ROOT->array_size; i++)
    {
      if (ROOT->array[i] != NULL)
        ssh_adt_delete(c, UINTTOPTR(i));
    }
  ssh_xfree(ROOT->array);
  ssh_xfree(ROOT);
}

/* $$METHOD(array, destroy) */
SSH_ADT_STD_DESTROY(destroy, uninit(c);)

/* $$METHOD(array, clear) */
static void clear(SshADTContainer c)
{
  int i;
  for (i = 0; i < ROOT->array_size; i++)
    {
      if (ROOT->array[i] != NULL)
        ssh_adt_delete(c, UINTTOPTR(i));
    }
}

/* After calling initialize_cell(c, n), the nth (0-based) cell is
   valid and empty. The array is expanded if necessary, and if there
   was an object at the nth cell, the object has been deleted. */
static void initialize_cell(SshADTContainer c, unsigned int idx)
{
  size_t i, old_size, new_size;
  void **array;

  if (ROOT->array_size <= idx)
    {
      old_size = ROOT->array_size;
      new_size = (idx + (idx / 4) + 1);
      ROOT->array = ssh_xrealloc(ROOT->array, new_size *
                                 sizeof(ROOT->array[0]));
      array = ROOT->array;
      for (i = old_size; i < new_size ; i++)
        {
          array[i] = NULL;
        } 
      ROOT->array_size = new_size;
      return;
    }
  else
    {
      if (ROOT->array[idx] != NULL)
        {
          ssh_adt_delete(c, UINTTOPTR(idx));
        }
    }
}

static unsigned int empty_idx(SshADTContainer c,
                              SshADTAbsoluteLocation location)
{
  unsigned int idx;
  SSH_ASSERT(SSH_ADT_IS_INDEX(location));
  idx = SSH_ADT_GET_INDEX(location);
  initialize_cell(c, idx);
  return idx;
}

/* $$METHOD(array, insert_to) */
static SshADTHandle insert_to(SshADTContainer c,
                              SshADTAbsoluteLocation location,
                              void *object)
{
  unsigned int idx;
  SshADTHandle h;
  idx = empty_idx(c, location);
  ROOT->array[idx] = object;
  c->ssh_adt_aux.num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

/* $$METHOD(array, get_handle_to_location) */
static SshADTHandle get_handle_to_location(SshADTContainer c,
                                           SshADTAbsoluteLocation location)
{
  SshADTHandle h = UINTTOPTR(SSH_ADT_GET_INDEX(location));
  return h;
}

/* $$METHOD(array, alloc_n_to) */
static SshADTHandle alloc_n_to(SshADTContainer c,
                               SshADTAbsoluteLocation location,
                               size_t size)
{
  unsigned int idx;
  SshADTHandle h;
  void *new;
  idx = empty_idx(c, location);
  ROOT->array[idx] = (new = ssh_xmalloc(size));
  c->ssh_adt_aux.num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, init, (new, size, SSH_ADT_APPCTX(c)));
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

/* $$METHOD(array, put_n_to) */
static SshADTHandle put_n_to(SshADTContainer c,
                             SshADTAbsoluteLocation location,
                             size_t size,
                             void *object)
{
  unsigned int idx;
  SshADTHandle h;
  void *new;
  idx = empty_idx(c, location);
  ROOT->array[idx] = (new = ssh_xmalloc(size));
  c->ssh_adt_aux.num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, copy, (new, size, object, SSH_ADT_APPCTX(c)));
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

/* $$METHOD(array, get) */
static void *get(SshADTContainer c, SshADTHandle h)
{
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(idx < ROOT->array_size);
  return ROOT->array[idx];
}

/* $$METHOD(array, num_objects) */
SSH_ADT_STD_NUM_OBJECTS(num_objects)

/* $$METHOD(array, get_handle_to) */
static SshADTHandle get_handle_to(SshADTContainer c, void *object)
{
  unsigned int i;
  for (i = 0; i < ROOT->array_size; i++)
    {
      if (ROOT->array[i] == object)
        return UINTTOPTR(i);
    }
  return SSH_ADT_INVALID;
}

/* $$METHOD(array, enumerate_start) */
static SshADTHandle enum_start(SshADTContainer c)
{
  SshADTHandle h = UINTTOPTR(0);
  if (ROOT->array_size == 0) return SSH_ADT_INVALID;
  return h;
}

/* $$METHOD(array, enumerate_next) */
static SshADTHandle enum_next(SshADTContainer c, SshADTHandle h)
{
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  idx++;
  if (idx >= ROOT->array_size) return SSH_ADT_INVALID;
  return UINTTOPTR(idx);
}

/* $$METHOD(array, detach) */
static void *detach(SshADTContainer c, SshADTHandle h)
{
  void *object;
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(idx < ROOT->array_size);
  SSH_ASSERT(ROOT->array[idx] != NULL);

  SSH_ADT_CALL_HOOK(c, detach, h);

  c->ssh_adt_aux.num_objects--;

  object = ROOT->array[idx];
  ROOT->array[idx] = NULL;
  return object;
}

/* $$METHOD(array, reallocate) */
static void *reallocate(SshADTContainer c, void *object, size_t new_size)
{
  ssh_fatal("Realloc does not work with arrays now.");
  return NULL;
}

/* $$METHOD(array, delete) */
static void delete(SshADTContainer c, SshADTHandle handle)
{
  void *object = ssh_adt_detach(c, handle);
  SSH_ADT_CALL_APP(c, destroy, (object, SSH_ADT_APPCTX(c)));  
  if (c->flags & SSH_ADT_FLAG_ALLOCATE)
    {
      ssh_xfree(object);
    }
}

SshADTStaticData ssh_adt_array_static_data = 
{
  {
    /* $$METHODS(array) */
    /* DO NOT EDIT THIS, edit METHODS.h and 
       the method implementations above instead. */
    container_init, /* container_init */
    clear, /* clear */
    destroy, /* destroy */
    NULL, /* insert_at */
    insert_to, /* insert_to */
    NULL, /* alloc_n_at */
    alloc_n_to, /* alloc_n_to */
    NULL, /* put_n_at */
    put_n_to, /* put_n_to */
    get, /* get */
    num_objects, /* num_objects */
    get_handle_to, /* get_handle_to */
    get_handle_to_location, /* get_handle_to_location */
    NULL, /* next */
    NULL, /* previous */
    enum_start, /* enumerate_start */
    enum_next, /* enumerate_next */
    NULL, /* get_handle_to_equal */
    reallocate, /* reallocate */
    detach, /* detach */
    delete, /* delete */
    /* $$ENDMETHODS */
  },
  0,
  0
};

const SshADTContainerType ssh_adt_array_type = &ssh_adt_array_static_data;
