/* HP95LX system manager compliant S/Key disposal key generator
 *
 * Jun-ichiro "itojun" Itoh/ESD
 *	itojun@csl.sony.co.jp
 *	itojun@mt.cs.keio.ac.jp
 * Freely Redistributable.
 */

#include <string.h>
#include <ctype.h>
#include <io.h>
#include <fcntl.h>
#include "interfac.h"
#include "event.h"
#include "menu.h"
#include "cbcodes.h"
#include "defs.h"
#include "rot.h"


/*configuration*/
#ifndef ENVPATH
# define ENVPATH "c:\\_dat\\skey.env"
#endif
#ifndef MD5FLAG
# define MD5FLAG	0
#endif


/*state constant*/
#define	CHALLENGE	0
#define	PASSWORD	1
#define	MENU		2
#define	OPTIONMENU	3
#define	QUIT		100

/*screen size*/
#define	MAXROW		16
#define	MAXCOLUMN	40

/*keys*/
#define	MENUKEY		0xc800
#define	UPKEY		0x4800
#define	DOWNKEY		0x5000
#define	PASTEKEY	0xd600


extern	int	mink(int, char *, char *, char *);
extern	char	*version;

static	void	initialize(void);
static	void	finalize(void);
static	void	disp(int, int, int, char *);
static	void	redraw(void);
static	void	redrawword(void);
static	void	awake(void);
static	void	sleep(void);
static	void	key_challenge(EVENT *);
static	void	key_passphrase(EVENT *);
static	void	key_menu(EVENT *);
static	void	key_optionmenu(EVENT *);
static	void	startmenu(void);
static	void	stopmenu(void);
static	void	startoptionmenu(int);
static	int	compute(void);
static	int	updown(int);
static	void	clrpassphrase(void);
static	int	smartpaste(void);
static	char	*getchallenge(char *, int *);
static	int	getsequence(char *);
static	char	*getseed(char *);
static	void	readopt(void);
static	void	writeopt(void);
static	char	*tonumber(int, int);
static	char	*repeat(int, char);


static	void	(*key[])(EVENT *) = {
	key_challenge, key_passphrase, key_menu, key_optionmenu,
};

static	EVENT		event;
static	MENUDATA	menu;
static	int		state = CHALLENGE;
static	int		oldstate = CHALLENGE;

#define	MAXPASSLEN	300
#define	MAXLEN		1024
static	char	challenge[MAXPASSLEN];
static	int	challengelen = 0;
static	char	passphrase[MAXPASSLEN];
static	int	passphraselen = 0;
static	char	result[MAXPASSLEN];

static	int	secureflag = 0;
	int	md5flag = MD5FLAG;
static	int	autosaveflag = 0;

static	int	redrawall = 0;
static	int	redrawmenu = 0;
static	int	redrawbottom = 0;


void	main()
{
	m_init();
	initialize();

	do {
		m_event(&event);
		switch (event.kind) {
		case E_ACTIV:
			awake();
			break;
		case E_DEACT:
			sleep();
			break;
		case E_TERM:
			state = QUIT;
			break;
		case E_KEY:
			if (key[state])
			{
				(*key[state])(&event);
			}
			break;
		}
	} while (state != QUIT);

	finalize();
	m_fini();
}


void	initialize()
{
	m_setmode(1);	/*text*/
	m_clear(0, 0, MAXROW, MAXCOLUMN);
	m_reg_app_name("S/Key");

	state = CHALLENGE;

	challengelen = 0;
	challenge[0] = '\0';
	passphraselen = 0;
	passphrase[0] = '\0';
	strcpy(result, "---");

	readopt();

	redrawall = 1;
	redraw();
}


void	finalize()
{
	if (autosaveflag)
	{
		writeopt();
	}
}


void	disp(row, column, attr, str)
	int	row;
	int	column;
	int	attr;
	char	*str;
{
	m_disp(row, column, str, strlen(str), attr, 0);
}


#define COLUMN	13

void	redraw()
{
	static int	prevstate = QUIT;	/*never happen*/
	int		redrawrange;
#define	REDRAWMENU	0x01
#define	REDRAWBOTTOM	0x02
#define	REDRAWALL	0x03

	redrawrange = 0;
	redrawrange |= (redrawall ? REDRAWALL : 0);
	redrawrange |= (redrawmenu ? REDRAWMENU : 0);
	redrawrange |= (redrawbottom ? REDRAWBOTTOM : 0);

	if (state != prevstate)
	{
		if ((prevstate == CHALLENGE || prevstate == PASSWORD)
		 && (state == CHALLENGE || state == PASSWORD))
		{
			/*OK*/
		}
		else
		{
			redrawrange |= REDRAWMENU;
		}
	}
	
	if ((redrawrange & REDRAWALL) == REDRAWALL)
	{
		drawbox("");
	}

	if (redrawrange & REDRAWMENU)
	{
		switch (state)
		{
		case CHALLENGE:
		case PASSWORD:
			disp(-3, 0, 0, (md5flag ? "S/Key challenge (MD5)"
						: "S/Key challenge"));
			disp(-2, MAXCOLUMN - strlen(version), 0, version);
			break;

		case MENU:
		case OPTIONMENU:
			menu_dis(&menu);
			break;
		}
	}

	if (redrawrange & REDRAWBOTTOM)
	{
		disp(1, 0, 0, "challenge:");
		disp(2, 0, 0, "passphrase:");
		disp(4, 0, 0, "disposal key:");
	}

	redrawword();

	redrawall = redrawmenu = redrawbottom = 0;
	prevstate = state;
}


void	redrawword()
{
	char	tmp[MAXPASSLEN];

	disp(1, COLUMN, 1, repeat(MAXCOLUMN - COLUMN, ' '));
	strcpy(tmp, challenge);
	tmp[MAXCOLUMN - COLUMN] = '\0';
	disp(1, COLUMN, 1, tmp);
	disp(2, COLUMN, 1, repeat(MAXCOLUMN - COLUMN, ' '));
	if (MAXCOLUMN - COLUMN < passphraselen)
	{
		disp(2, COLUMN, 1, repeat(MAXCOLUMN - COLUMN, '.'));
		strcpy(tmp, "[");
		strcat(tmp, tonumber(passphraselen, 10));
		strcat(tmp, "]");
		disp(2, COLUMN, 1, tmp);
	}
	else
	{
		disp(2, COLUMN, 1, repeat(passphraselen, '.'));
	}
	disp(5, 0, 0, repeat(MAXCOLUMN, ' '));
	disp(5, 2, 0, result);

	if (state == CHALLENGE)
	{
		m_setcur(1, COLUMN + challengelen);
	}
	else if (state == PASSWORD)
	{
		m_setcur(2, COLUMN + passphraselen);
	}
	else
	{
		/*invisible cursor*/
		m_setcur(0, -1);
	}
}


void	awake()
{
	m_setmode(1);	/*text*/

	redrawall = 1;
	redraw();
}


void	sleep()
{
	if (secureflag)
	{
		clrpassphrase();
	}
}


void	startmenu()
{
	menu_setup(&menu, "Quit\0Options\0Save\0", 3, 1, NULL, 0, NULL);
	menu_on(&menu);
	oldstate = state;
	state = MENU;
	redraw();
}


void	stopmenu()
{
	menu_off(&menu);
	state = oldstate;
	redraw();
}


void	startoptionmenu(active)
	int	active;
{
	char	tmp[MAXLEN];
	char	*p;

	tmp[0] = '\0';
	strcat(tmp, (secureflag ? "Secure  #" : "inSecure#"));
	strcat(tmp, (md5flag ? "MD5#" : "MD4#"));
	strcat(tmp, (autosaveflag ? "Autosave  #" : "noAutosave#"));
	for (p = tmp; *p; p++)
	{
		if (*p == '#')
		{
			*p = '\0';
		}
	}

	menu_setup(&menu, tmp, 3, 1, NULL, 0, NULL);
	menu_on(&menu);
	menu.menu_letter[0] = 'S';
	menu.menu_letter[2] = 'A';
	menu.menu_highlight = active;
	state = OPTIONMENU;
	redraw();
}


void	key_menu(ep)
	EVENT	*ep;
{
	int	item;

	switch (ep->data)
	{
	case '\033':
		stopmenu();
		break;

	default:
		menu_key(&menu, ep->data, &item);
		if (item < 0)
		{
			break;
		}
		stopmenu();

		switch (item)
		{
		case 0:
			state = QUIT;
			break;

		case 1:
			startoptionmenu(0);
			break;

		case 2:
			writeopt();
			break;
		}
		break;
	}
}


void	key_optionmenu(ep)
	EVENT	*ep;
{
	int	item;

	switch (ep->data)
	{
	case '\033':
		stopmenu();
		break;

	default:
		menu_key(&menu, ep->data, &item);
		if (item < 0)
		{
			break;
		}

		switch (item)
		{
		case 0:
			secureflag = (!secureflag);
			break;

		case 1:
			md5flag = (!md5flag);
			strcpy(result, "---");
			break;

		case 2:
			autosaveflag = (!autosaveflag);
			break;
		}
		startoptionmenu(item);
		redrawmenu = 1;
		redraw();
		break;
	}
}


void	key_challenge(ep)
	EVENT	*ep;
{
	switch (ep->data)
	{
	case MENUKEY:
		startmenu();
		break;

	case PASTEKEY:
		if (smartpaste() != 0)
		{
			m_beep();
		}
		else
		{
			redraw();
		}
		break;

	case UPKEY:
	case DOWNKEY:
		if (updown(ep->data == UPKEY) != 0)
		{
			m_beep();
		}
		else
		{
			redraw();
		}
		break;

	case '\033':
		challengelen = 0;
		challenge[0] = '\0';
		redraw();
		break;

	case '\b':
		if (challengelen > 0)
		{
			challenge[--challengelen] = '\0';
			redraw();
		}
		else
		{
			m_beep();
		}
		break;
		
	case '\r':
		strcpy(result, "---");
		state = PASSWORD;
		redraw();
		break;

	case '\t':
		strcpy(result, "---");
		state = PASSWORD;
		redraw();
		break;

	default:
		if (ep->data & 0xff00)
		{
			m_beep();
			break;
		}

		if (challengelen < sizeof(challenge)/sizeof(challenge[0]))
		{
			challenge[challengelen++] = ep->data;
			challenge[challengelen] = '\0';
			redraw();
		}
		else
		{
			m_beep();
		}
		break;
	}
}


void	key_passphrase(ep)
	EVENT	*ep;
{
	switch (ep->data)
	{
	case MENUKEY:
		startmenu();
		break;

	case '\033':
		clrpassphrase();
		redraw();
		break;

	case '\b':
		if (passphraselen > 0)
		{
			passphrase[--passphraselen] = '\0';
			redraw();
		}
		else
		{
			m_beep();
		}
		break;
		
	case '\r':
		if (challengelen && passphraselen)
		{
			if (compute() != 0)
			{
				m_beep();
				break;
			}

			if (secureflag)
			{
				clrpassphrase();
			}

			if (m_open_cb() == 0)
			{
				m_reset_cb("skey");
				m_new_rep("TEXT");
				m_cb_write(result, strlen(result));
				m_fini_rep();
				m_close_cb();
			}
		
			state = CHALLENGE;
			redraw();
		}
		else
		{
			state = CHALLENGE;
			redraw();
		}
		break;

	case '\t':
		state = CHALLENGE;
		redraw();
		break;

	default:
		if (ep->data & 0xff00)
		{
			m_beep();
			break;
		}

		if (passphraselen < sizeof(passphrase)/sizeof(passphrase[0]))
		{
			passphrase[passphraselen++] = ep->data;
			passphrase[passphraselen] = '\0';
			redraw();
		}
		else
		{
			m_beep();
		}
		break;
	}
}


int	compute()
{
	int	n;
	char	*seed;

	n = getsequence(challenge);
	seed = getseed(challenge);
	if (n < 0 || seed == (char *)NULL)
	{
		return -1;
	}

	mink(n, seed, passphrase, result);
	return 0;
}


int	updown(up)
	int	up;
{
	char	tmp[MAXPASSLEN];
	int	n;
	char	*seed;

	n = getsequence(challenge);
	seed = getseed(challenge);
	if (n < 0 || seed == (char *)NULL)
	{
		return -1;
	}

	n += (up ? 1 : -1);
	if (n < 0)
	{
		return -1;
	}

	strcpy(tmp, tonumber(n, 10));
	strcat(tmp, " ");
	strcat(tmp, seed);
	strcpy(challenge, tmp);
	challengelen = strlen(challenge);
	return 0;
}


void	clrpassphrase()
{
	memset(passphrase, 0, sizeof(passphrase));
	passphraselen = 0;
}


int	smartpaste()
{
	int		index;
	unsigned int	len;
	char		*buf;
	char		*p;
	char		*q;

	if (m_open_cb() == 0)
	{
		if (m_rep_index("TEXT", &index, &len) != 0)
		{
			m_close_cb();
			return -1;
		}
		if ((buf = m_alloc(len + 10)) == NULL)
		{
			m_close_cb();
			return -1;
		}
		m_cb_read(index, 0, buf, len);
		m_close_cb();
	}

	buf[len] = '\0';	/* safety sentinel */
	if (p = strchr(buf, '\r'))
	{
		int	md5;
		/* paste challenge only */
		p = getchallenge(buf, &md5);
		if (p == NULL)
		{
			return -1;
		}
		md5flag = md5;
		len = strlen(p);
	}
	else
	{
		/* paste as is */
		p = buf;
	}

	/* append to challenge buffer */
	q = challenge + challengelen;
	while (len-- && q - challenge < MAXPASSLEN)
	{
		*q++ = *p++;
	}
	*q++ = '\0';

	challengelen = strlen(challenge);

	m_free(buf);
	return 0;
}

/*------------------------------------------------------------*/

char	*getchallenge(buf, md5)
	char	*buf;
	int	*md5;
{
	char	*p;
	char	*q;
	char	*ret;
	char	*nextline;
	int	m;

	p = buf;
	ret = NULL;
	while (p)
	{
		nextline = strchr(p, '\r');
		if (nextline)
		{
			*nextline = '\0';
			nextline += 1;
		}
		/* The following code recognizes the regexp:
		 * /^s\/key\s*(\(md5\)\s*)?[0-9]+\s+\S+\s*$/
		 */
		if (*p == '\n')
		{
			p++;
		}

		if (strnicmp(p, "s/key", 5) != 0)
		{
			goto next;
		}
		p += 5;

		while (isspace(*p))
		{
			p++;
		}

		if (strnicmp(p, "(md5)", 5) == 0)
		{
			m = 1;
			p += 5;
			while (isspace(*p))
			{
				p++;
			}
		}
		else
		{
			m = 0;
		}

		if (!*p || !isdigit(*p))
		{
			goto next;
		}
		q = p;
		while (isdigit(*p))
		{
			p++;
		}
		if (!*p || !isspace(*p))
		{
			goto next;
		}
		while (isspace(*p))
		{
			p++;
		}
		if (!*p || isspace(*p))
		{
			goto next;
		}
		while (*p && !isspace(*p))
		{
			p++;
		}
		if (*p)	/*isspace(*p) is true*/
		{
			*p++ = '\0';
			while (*p && isspace(*p))
			{
				p++;
			}
			if (*p)	/*isspace(*p) is false here*/
			{
				goto next;
			}
		}
		ret = q;
		*md5 = m;

next:
		p = nextline;
	}

	return ret;
}


int	getsequence(challenge)
	char	*challenge;
{
	char	*p;
	int	n;

	p = challenge;
	n = 0;

	if (!isdigit(*p))
	{
		return -1;
	}
	while (isdigit(*p))
	{
		n *= 10;
		n += (*p - '0');
		p++;
	}

	return n;
}


char	*getseed(challenge)
	char	*challenge;
{
	char	*p;

	p = challenge;

	if (!isdigit(*p))
	{
		return (char *)NULL;
	}
	while (isdigit(*p))
	{
		p++;
	}

	if (! (isspace(*p) || *p == '/'))
	{
		return (char *)NULL;
	}
	p++;
	while (isspace(*p))
	{
		p++;
	}

	return p;
}

/*------------------------------------------------------------*/

static	char	*keyword[] =
{
	"md4", "md5", "secure", "insecure", "password=", "passphrase=",
	"autosave", "noautosave", NULL,
};
#define	KW_MD4		0
#define	KW_MD5		1
#define	KW_SECURE	2
#define	KW_INSECURE	3
#define	KW_PASSWORDEQ	4
#define	KW_PASSPHRASEEQ	5
#define	KW_AUTOSAVE	6
#define	KW_NOAUTOSAVE	7


void	readopt()
{
	int	fd;
	char	buf[MAXLEN];
	char	*p;
	char	*q;
	int	len;
	int	kw;

	if ((fd = open(ENVPATH, O_RDONLY)) >= 0)
	{
		len = read(fd, buf, MAXLEN);
		close(fd);

		p = buf;
		while (p-buf < len)
		{
			if (isspace(*p))
			{
				while (p-buf < len && isspace(*p))
				{
					p++;
				}
				continue;
			}

			for (kw = 0; keyword[kw]; kw++)
			{
				if (strnicmp(p, keyword[kw], strlen(keyword[kw])) == 0)
				{
					p += strlen(keyword[kw]);
					break;
				}
			}

			switch (kw)
			{
			case KW_MD4:
				md5flag = 0;
				break;

			case KW_MD5:
				md5flag = 1;
				break;

			case KW_SECURE:
				secureflag = 1;
				break;

			case KW_INSECURE:
				secureflag = 0;
				break;

			case KW_PASSWORDEQ:
			case KW_PASSPHRASEEQ:
				/* Ultra insecure: password/passphrase in file
				 *
				 * if you use the option:
				 * - you should NEVER pass your palmtop to
				 *   your friends.
				 * - you should NEVER leave your palmtop alone.
				 * - you should NEVER turn filer into server
				 *   mode.
				 *
				 * passphrase must be encoded through rot13,
				 * and terminated by EOL.
				 * passphrase will not be saved.
				 * turn autosave off (noautosave) to preserve
				 * passphrase in the configuration file.
				 */
				q = p;
				while (q-buf < len && *q != '\r' && *q != '\n')
				{
					q++;
				}
				*q++ = '\0';

				if (strlen(p) < MAXPASSLEN)
				{
					strcpy(passphrase, p);
					rot13(passphrase);
					passphraselen = strlen(passphrase);
				}

				p = q;
				break;

			case KW_AUTOSAVE:
				autosaveflag = 1;
				break;

			case KW_NOAUTOSAVE:
				autosaveflag = 0;
				break;

			default:
				while (p-buf < len && !isspace(*p))
				{
					p++;
				}
				break;
			}
		}
	}
}


void	writeopt()
{
	int	fd;
	char	buf[MAXLEN];

	buf[0] = '\0';
	strcat(buf, (secureflag ? keyword[KW_SECURE] : keyword[KW_INSECURE]));
	strcat(buf, " ");
	strcat(buf, (md5flag ? keyword[KW_MD5] : keyword[KW_MD4]));
	strcat(buf, " ");
	strcat(buf, (autosaveflag ? keyword[KW_AUTOSAVE]
				  : keyword[KW_NOAUTOSAVE]));
	strcat(buf, "\n");
	if ((fd = open(ENVPATH, O_WRONLY|O_CREAT|O_TRUNC)) >= 0)
	{
		write(fd, buf, strlen(buf));
		close(fd);
	}
}

/*------------------------------------------------------------*/

char	*tonumber(n, base)
	int	n;
	int	base;
{
	static char	*digit = "0123456789abcdef";
	static char	tmp[20];	/*maximum 16digits(if base=2)*/
	char	*p;
	int	minus;

	if (minus = (n < 0))
	{
		n *= -1;
	}

	p = tmp + sizeof(tmp)/sizeof(tmp[0]);
	*--p = '\0';
	do {
		*--p = digit[n % base];
	} while (n /= base);
	if (minus)
	{
		*--p = '-';
	}

	return p;
}


char	*repeat(len, ch)
	int	len;
	char	ch;
{
	static char	tmp[MAXPASSLEN];

	tmp[len] = '\0';
	while (0 <= --len)
	{
		tmp[len] = ch;
	}

	return tmp;
}
