/* xyview.c */
/*
 *  Stand-alone procedure for viewing the output of "xyplot"
 *  Takes input from stdin in the form of a unix-graphics protocol
 *  which it interprets (parses) as commands which it then executes.
 */

#include <stdio.h>
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include "xyview.h"

#ifndef	TITLEBAR
#define	TITLEBAR	"xyview"
#endif
#ifndef	VERSION
#define	VERSION     ""
#endif
#ifndef DEBUG
#define DEBUG 0
#endif
#ifndef SCBARS
#define SCBARS 0
#endif

int	Wxs=512,  Wys=512;	/* default window size   */
int	Wxo=0,	  Wyo=0;	/* default window origin */
static int need_erase = 0;

/* global variables */

Frame	base_frame;
Canvas	image_canvas;
Pixwin	*image_pw;


/* LABEL:
 *	put a label on the stripe
 */
void
window_label_set(s)
char	*s;
{
    window_set(base_frame,
	       FRAME_LABEL,	s,
	       0);
}

void
window_origin_set(x,y)
int x,y;
{
    window_set(base_frame,
	       WIN_X,	x,
	       WIN_Y,	y,
	       0);
	Wxo = (int)window_get(image_canvas,WIN_X);
	Wyo = (int)window_get(image_canvas,WIN_Y);
}
void
window_size_set(x,y)
int x,y;
{
    window_set(image_canvas,
	       WIN_WIDTH,	x,
	       WIN_HEIGHT,	y,
	       0);
	Wxs = (int)window_get(image_canvas,WIN_WIDTH);
	Wys = (int)window_get(image_canvas,WIN_HEIGHT);
    set_unixplot_window(Wxs,Wys);
    window_fit(base_frame);
}
void
window_canvas_set(x,y)
int x,y;
{
    window_set(image_canvas,
	       CANVAS_WIDTH,	x,
	       CANVAS_HEIGHT,	y,
	       0);
	Wxs = (int)window_get(image_canvas,CANVAS_WIDTH);
	Wys = (int)window_get(image_canvas,CANVAS_HEIGHT);
    window_fit(base_frame);
}    

/* fundamental plotting and imaging routines */
/* ----------------------------------------- */

#define	CHAR_WIDTH	9
#define	CHAR_HEIGHT	16

static	int	cur_color=255;
static	int	cur_op=PIX_SRC;
static	int	cur_x=0;
static	int	cur_y=0;

void
move_abs(x,y)
int x,y;
{
    cur_x=x, cur_y=y;
}
void
move_rel(dx,dy)
int dx,dy;
{
    cur_x += dx, cur_y += dy;
}
void
set_color(col)
int col;
{
    cur_color=col;
}
void
point()
{
    pw_vector(image_pw,cur_x,cur_y,cur_x,cur_y, cur_op,cur_color);
    
}
void
cont(x,y)
int	x,y;
{
    pw_vector(image_pw, x,y, cur_x,cur_y, cur_op, cur_color );
    move_abs(x,y);
    
}
void
vect(dx,dy)
int	dx,dy;
{
    pw_vector(image_pw, cur_x, cur_y, cur_x+dx, cur_y+dy, cur_op, cur_color );
    move_rel(dx,dy);
    
}
void
label(buf)
char	*buf;
{
    pw_text(image_pw,cur_x,cur_y,cur_op|PIX_COLOR(cur_color),(Pixfont *)0,buf);
    pw_show(image_pw);
    move_rel( CHAR_WIDTH*strlen(buf), 0 );
    
}
void erase()
{
	need_erase = 1;
}
void
do_erase()
{
    pw_writebackground(image_pw,0,0,Wxs,Wys,PIX_SRC);
	need_erase = 0;
}
void
linemod(s)
char *s;
{
	/* do nothing */
}
void
finish()
{
    window_destroy(base_frame);
    exit(0);
}


/* Process command line flags
 */
void
usage(inv)
char *inv;
{
    fprintf(stderr, "Usage: %s [-][flags] [arg] [arg] ....\n", inv);
    fprintf(stderr, "    flags:\n");
    fprintf(stderr, 
			"\t w, W: set window size in pixels, consumes two integer arguments\n");
    fprintf(stderr, 
			"\t x, X: set location of window on the screen, consumes two integer arguments\n");
    fprintf(stderr, "\t v, V: print version date\n");
}
void
options(argc,argv)
int argc;
char **argv;
{
	int j, c;
	if (argc > 1) for (j = 0; j < strlen(argv[1]); j++){
		c = argv[1][j];
		/* fprintf(stderr, "arg: %c\n", c);  */
		switch(c) {
		case '-': 
			break;
		case 'v':
		case 'V':
			fprintf(stderr, "%s %s\n", TITLEBAR, VERSION);
			exit(0);
		case 'w':
		case 'W':
			Wxs = next_int(argc, argv);
			Wys = next_int(argc, argv);
			break;
		case 'x':
		case 'X':
			Wxo = next_int(argc, argv);
			Wyo = next_int(argc, argv);
			break;
		case 'c':
		case 'C': 
			/* case 'c' retained for backward compatibility */
			/* consume two integers, do nothing with them   */
			next_int(argc, argv);
			next_int(argc, argv);
			break;
		default: 
			fprintf(stderr, "%s: ", argv[0]);
			fprintf(stderr, "Unknown flag: %c\n", c);
			usage(argv[0]);
			exit(-1);
		}
	}
}
int
next_int(argc, argv)
unsigned int argc;
char **argv;
{
    static int i = 2;
    if (i >= argc) {
		fprintf(stderr, "Missing integer command line argument");
		usage(argv[0]);
    }
    return( atoi( argv[i++]));
}


/*************************************************************************/
/*
 * This is the SunView-specific code that sets up and implements xyview
 */
    


static short	icon_image[] = {
/* Format_version=1, Width=64, Height=64, Depth=1, Valid_bits_per_item=16
 */
	0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,
	0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,
	0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,
	0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,
	0x8888,0x8888,0x8888,0x8888,0x8888,0x89FC,0x8888,0x8888,
	0x2222,0x27FF,0xA222,0x2222,0x2222,0x3FFF,0xF222,0x2222,
	0x8888,0xFFFF,0xFC88,0x8888,0x8888,0xFF88,0xBF88,0x8888,
	0x2223,0xF222,0x27E2,0x2222,0x2223,0xE222,0x23F2,0x2222,
	0x888F,0x8888,0x88FC,0x8888,0x888F,0x8888,0x889F,0x8888,
	0x223E,0x3FA2,0x7E27,0xA222,0x223A,0xFFF7,0xFFA3,0xE222,
	0x88F9,0xFFFF,0xFFE8,0xE888,0x88EB,0xFFFF,0xF9F8,0xF888,
	0x22E3,0xE7A2,0xFA7E,0x3A22,0x22A7,0xAE3E,0x3E2F,0x3E22,
	0x898F,0x89FF,0xCB8F,0x8E88,0x889E,0x99FF,0xC989,0xCB88,
	0x223E,0x33FF,0xA2A2,0xE3A2,0x223E,0x32FF,0xA3A3,0xFA22,
	0x88FF,0x88FF,0x8B9E,0xBE88,0x88F8,0xFCBC,0x8EB8,0x9F88,
	0x2262,0x3E22,0x3BFA,0x27A2,0x2222,0x2F22,0x67E2,0x2222,
	0x8888,0x8BFF,0xFF88,0x8888,0x8888,0x88FF,0xF888,0x8888,
	0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,
	0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,0x8888,
	0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,0x2222,
	0xBFFF,0xFFFF,0xFFFF,0xFFF8,0xA000,0x0000,0x0000,0x0004,
	0x2000,0x0000,0x0000,0x0006,0x2000,0x0000,0x0000,0x0007,
	0xA000,0x0000,0x0000,0x0007,0xA000,0x0000,0x0400,0x0007,
	0x2000,0x0000,0x0400,0x0007,0x2000,0x0000,0x0000,0x0007,
	0xA021,0x2121,0x1C1E,0x2087,0xA021,0x2121,0x0421,0x2087,
	0x2012,0x1121,0x0421,0x2487,0x200C,0x1212,0x043F,0x2487,
	0xA012,0x0A12,0x0420,0x2487,0xA021,0x0A0C,0x0421,0x2487,
	0x2021,0x040C,0x041E,0x1B07,0x2000,0x0400,0x0000,0x0007,
	0xA000,0x0800,0x0000,0x0007,0xA000,0x2800,0x0000,0x0007,
	0x2000,0x1000,0x0000,0x0007,0x2000,0x0000,0x0000,0x0007,
	0x9FFF,0xFFFF,0xFFFF,0xFFFF,0x8FFF,0xFFFF,0xFFFF,0xFFFF,
	0x27FF,0xFFFF,0xFFFF,0xFFFF,0x2222,0x2222,0x2222,0x2222

	};
DEFINE_ICON_FROM_IMAGE(vp_icon,icon_image)

void
act_on_mouse(cv,ev)
Canvas cv;
Event *ev;
{
	/* Any button to erase */
	switch( event_id(ev) )
	{
	case MS_LEFT:
	case MS_MIDDLE:
	case MS_RIGHT:
		if (event_is_down(ev))
			do_erase();
		break;
	}	
}

void
xyview()
{
	int	stdin_fd = 0;		/* file descriptor for stdin */
	static	Notify_value	read_input();

    setup_windows();
    set_unixplot_window(Wxs,Wys);
    notify_set_input_func(base_frame,read_input,stdin_fd);
    window_main_loop(base_frame);
}

setup_windows()
{
    base_frame = 
		window_create(NULL,FRAME,
					  FRAME_ICON,		&vp_icon,
					  WIN_X,			Wxo,
					  WIN_Y,			Wyo,
					  FRAME_NO_CONFIRM,		TRUE,
					  0);
	
	image_canvas = 
		window_create(base_frame,CANVAS,
					  CANVAS_AUTO_SHRINK,	FALSE,
					  CANVAS_RETAINED,		FALSE,
#if SCBARS
					  WIN_VERTICAL_SCROLLBAR,	scrollbar_create(0),
					  WIN_HORIZONTAL_SCROLLBAR,	scrollbar_create(0),
#endif
					  0);
	
	window_origin_set(Wxo,Wyo);
	
	window_set(image_canvas,
			   WIN_WIDTH,	Wxs,
			   WIN_HEIGHT,	Wys,
			   0);
	window_set(image_canvas,
			   CANVAS_WIDTH,	Wxs,
			   CANVAS_HEIGHT,	Wys,
			   0);
	Wxs = (int)window_get(image_canvas,WIN_WIDTH);
	Wys = (int)window_get(image_canvas,WIN_HEIGHT);
	
	window_set(image_canvas,
			   CANVAS_RETAINED,	TRUE,
			   0);

	window_set(image_canvas,
			   WIN_EVENT_PROC,	act_on_mouse,
			   0);
	
    window_fit(base_frame);
	
    image_pw = canvas_pixwin(image_canvas);
    
	window_set(base_frame,
			   FRAME_LABEL,		TITLEBAR,
			   0);

}

static	Notify_value
read_input(frame,fd)
Frame	frame;
int	fd;
{
	if (!need_erase)
	{
		if (EOF == ug_parse(stdin))
		    finish();
	}
	else
		usleep(50);
	/* else do nothing but wait for the mouse to erase for you;
     * unfortunately this waiting seems to take all of the computers
     * resources; surely there is a better way!
	 */
	/* sleep(1) does free up the resources;
	 * but the (seeingly greater than) one second delay
	 * in the response to the mouse is intolerable.
	 */
	/* however, usleep(50), which sleeps for 50 microseconds per loop
	 * seems to solve the problem!  the delay is short enough that
	 * the response time is imperceptible to humans, but long enough
	 * that the computer has time to relax
	 */
	
    return(NOTIFY_DONE);
}


main(argc,argv)
int	argc;
char	**argv;
{
	options(argc,argv);
	/* Effectively disable canvas commands */
    xyview();
}
