/*  xyplot.c */
/*
 *  The main body contains lines form the program graph(1) of the
 *  Unix distribution. Nevertheless it has been extensively
 *  modified and not much remains of graph(1). I do not know the
 *  complete hacking history, but by comparing it with existing
 *  code, it looks that graph(1) was modified in
 *
 *  june 83  - SLAC UGS added by John Richardson and Bob Pearson
 *  june 84  - options in the input and Sun viewer by Doug Toussaint
 *  may  87  - multiplots and error handling by Jim Gubernatis
 *             SunView support
 *  jan  89  - adapted xyplot to produce ascii output RM
 *  may  91  - merged with another version of xyplot developed by JT
 *             permits command line options, and include files
 *  jan  92  - some bug fixes, JT
 *  sept 92  - incorporated some RM fixes, added a few more (v 2.2)
 *             created datastyle structures (v 2.3)
 *             removed hardcoded SIZE, introduced datastring #d
 *             and #size command (v 2.4)
 */
 
char ProgName[50] = "xyplot";

int verbose = 0;

#include <stdio.h>
#include <math.h>
#include "patch.h"

#include "xyplot.h"
 
#ifdef __TURBOC__
#define POPEN_PERMITTED 0
#endif
 
#ifndef POPEN_PERMITTED
#define POPEN_PERMITTED 1
#endif
 
#ifndef XYUG
#define XYUG "xyug"
#endif
 
#ifndef XYPS
#define XYPS "xyps"
#endif
 
/* defaults */
 
#define OVERALLSIZE 1.0
#define TICKFACTOR 1.0
#define TICKSPERTICK 5
#define XTICKSPERTICK 5
#define YTICKSPERTICK 5
#define AXLWIDTH 1.0
#define LINWIDTH 0.0
#define LBLWIDTH 1.0
#define SYMSIZE 1.5
#define NUMLBLSIZE 1.5
#define TITLESIZE 1.0
#define XTITLESIZE 1.5
#define YTITLESIZE 1.5
#define GRIDSTYLE 1
#define GRIDLINESTYLE 2
#define MODE 1
 
/* ------ global variables --------------------------------- 
 */
 
struct xy         xd;     /* x-axis info */
struct xy         yd;     /* y-axis info */
struct datum     *xx;     /* array of data points */
struct datastyle *current_style=NULL;
 
/** defaults */
 
 
float   overallsize = 1.0;  	    /* set the overall scale */
 
float	overallxoff = 0.0;
float	overallyoff = 0.0;
 
#define topx (0.975 * SIZE)
#define botx (0.185 * SIZE)
#define topy (0.915 * SIZE)
#define boty (0.125 * SIZE)
 
int     leftx = botx;         /* slightly dubious variable; formerly 0.075*SIZE
							   * what does it do exactly??
							   * says where y axis label can go, it gets
							   * changed in nlabels, and used in title()
							   */
 
float           tickfactor=1.0;   /* user-scales tick size */
int             tickSize=0;         /* This is set in setticksize() */
int             tickspertick = 5;   /* sub-divisions between big ticks */
int             xtickspertick = 5;   /* sub-divisions between big ticks */
int             ytickspertick = 5;   /* sub-divisions between big ticks */
 
#define         ERRBAR_LINEWIDTH  0.5   /* how thinner error bars should be */
float           axlwidth = 1.0;         /* axes line width */
float           linwidth = 0.0;         /* standard width */
float			lblwidth = 0.0;         /* label linewidth */
 
float           symsize = 1.5;      /* symbol size */
float           numlblsize = 1.5;   /* numberlabel character size */
float           titlesize = 2.0;    /* title character size */
float           xtitlesize = 1.5;   /* x title character size */
float           ytitlesize = 1.5;   /* y title character size */
float           titlex = 0.0;
float           titley = 0.0;
float           xtitlex = 0.0;
float           xtitley = 0.0;
float           ytitlex = 0.0;
float           ytitley = 0.0;
int             titleatopf = 0;
int             titlexf = 0;
int             titleyf = 0;
int             xtitlexf = 0;
int             xtitleyf = 0;
int             ytitlexf = 0;
int             ytitleyf = 0;
int             ytitlehf = 0;       /* write y title horizontally */
 
char        titlebuf[BSIZ];
char        xtitlebuf[BSIZ];
char        ytitlebuf[BSIZ];
char		gridbuf[50];
char        datastring[50];
int         newdstringf = 0;   /* flags that datastring is newly defined */
char        xformat[50]="%g";  /* the default format for the x numbers */
char        yformat[50]="%g";  /* the default format for the y numbers */
char        *plotsymb;
 
int             n = 0;          /* no. of data points */
int             erasf = 1;      /* erase screen ? */
int             gridstyle = 1;  /* grid style 0,1,2,3 */
int             gridlinestyle = 2;  /* linemod of grid if gridstyle=2 */
int             clipf = 1;      /* clip lines to pts outsize range of plot */
int             symbf = 0;      /* default plot symbol in effect ? */
int             absf = 0;       /* auto abscissas ? */
int             auto_no = 0;    /* value of n when absf set to 1 */
float           auto_dx = 1.0;  /* delta x for auto abscissas */
float           auto_xo = 0.0;  /* init  x for auto abscissas */
int             errf = 0;       /* data has error bars ? */
char            errxy = 'y';    /* which axis are error bars on ? */
int             transf = 0;     /* transpose ? */
int             lbflag = 1;     /* break after labels ? */
int             connectf = 1;   /* connect to previous ? */
int             nextf = 1;      /* continue plotting? */
int             stdinf=1;       /* read from stdin? */
 
 
char  *modes[] = {
                  "solid", /* point plot needs to draw solid lines! */
                  "solid",
                  "dotted",
                  "dotdashed",
                  "shortdashed",
                  "longdashed",
                  "dashdotdotted"
};
int             mode = 1;
int             nmodes = 7;
 
FILE            *outputfile=stdout;
 
/*-- function declarations
 */
void prepare_plot();
void execute_plot();
void readin(/* [FILE *] */);
int  setopt();
 
char           *realloc();
char           *malloc();
char           *copystring();
 
double ident(x) double x; { return (x); }
 
 
float
modceil(f, t)
float   f, t;
{
    t = fabs((double)t);
    t = t * ceil( (double)(f/t) );
    return( (t==0.0) ? 0 : t);
}
 
float
modfloor(f, t)
float    f, t;
{
    t = fabs(t);
    t = t * floor( (double)(f/t) );
    return( (t==0.0) ? 0 : t);
}
 
void
reinit(p)
struct xy      *p;
{
    double ident();
	p->xlbf = 0;
	p->xubf = 0;
	p->xqf = 0;
    p->xlb = p->xllb = (float) (VERYBIG);
    p->xub = p->xuub = -(float) (VERYBIG);
    p->xf = ident;
	p->xlogf = 0;
    p->xmult = 1;
}
void
init(p)
struct xy      *p;
{
    p->xsize = 1;
    p->xoff = 0;
    reinit(p);
}
 
debug_xy(c,xyd)
char c;
struct xy *xyd;
{
	fprintf(stderr,
			"%c axis ----\n",c);
	fprintf(stderr,
			"Flags: xlbf=%d, xubf=%d, xqf=%d\n",
			xyd->xlbf, xyd->xubf, xyd->xqf);
	fprintf(stderr,
			"Bounds: lo=%g hi=%g quant=%g\n",
			xyd->xlb, xyd->xub, xyd->xquant);
	fprintf(stderr,
			"Scale: a=%g b=%g\n",
			xyd->xa,xyd->xb);
	fprintf(stderr,
			"Screen: offset=%g size=%g mult=%g bot=%d top=%d\n",
			xyd->xoff,xyd->xsize,xyd->xmult,xyd->xbot,xyd->xtop);
}
 
#define newDataStyle() current_style = newStyle()
 
#define isStyleCurrent() \
	((current_style != NULL ) && \
     (current_style->line_mode == mode)      && \
	 (current_style->err_flag  == errf)      && \
	 (current_style->err_xy == errxy)        && \
	 (current_style->con_flag  == connectf)  && \
	 (current_style->symsiz    == symsize)   && \
	 (current_style->symb_flag == symbf)     && \
	 (current_style->line_width  == linwidth))
 
#define isLabelCurrent(label) \
    ((label == current_style->lblptr) || \
	 ((label != NULL) && (strcmp(label,current_style->lblptr)==0) ))
 
/* allocates and updates a new datastyle */
struct datastyle *
newStyle()
{
	struct datastyle *newstyle;
	newstyle = (struct datastyle *)malloc(sizeof(struct datastyle));
	if (newstyle == NULL)
		fatal("Unable to allocate more memory for datastyle structures");
	if (verbose>1) fprintf(stderr,"New datastyle created\n");
	newstyle->line_mode = mode;
	newstyle->err_flag = errf;
	newstyle->err_xy = errxy;
	newstyle->con_flag = connectf;
	newstyle->symsiz = symsize;
	newstyle->symb_flag = symbf;
	newstyle->line_width = linwidth;
	newstyle->parent = current_style;
	return newstyle;
}
freeStyles()
{
	struct datastyle *dad, *gramps;
	if (current_style == NULL)
		return 0;
	dad = current_style->parent;
	while (dad != NULL)
	{
		gramps = dad->parent;
		/* this won't work because the funky 
		 * way that lblptr's are allocated
		 * hence it is commented out:
		 * if dad->lblptr != NULL
		 *   free((char *)lblptr);
		 */
		free((char *)dad);
		dad = gramps;
	}
	current_style->parent = dad;
	return 1;
}
 
void
prepare_plot()
{
	/* reset (some) global parameters to defaults */
    n=0;
    symbf = 0;      /* default plot symbol in effect ? */
    absf = 0;       /* supply abscissas ? */
    auto_no = 0;       /* value of n when absf set to 1 */
    errf = 0;       /* data has error bars ? */
    errxy = 0;      /* error bars are on x axis ? */
    transf = 0;     /* transpose ? */
	mode = 1;       /* default line mode (solid) */
 
	/* re-initialize the x and y axes */
    reinit(&xd);
    reinit(&yd);
    /* set title strings to empty */
    titlebuf[0]='\0';
    xtitlebuf[0]='\0';
    ytitlebuf[0]='\0';
#if 0  /* on second thought, do NOT reset x,y format strings */
	strcpy(xformat,"%g");
	strcpy(yformat,"%g");
#endif
	/* begin allocating data structure */
    xx = (struct datum *) malloc((unsigned) sizeof(struct datum));
}
void
execute_plot()
{
	if (overallsize > 0) 
	{
		xd.xsize *= overallsize;
		yd.xsize *= overallsize;
		xd.xoff *= overallsize;
		xd.xoff += overallxoff;
		yd.xoff *= overallsize;
		yd.xoff += overallyoff;
	}
	if (verbose) fprintf(stderr,"size: %g %g %g\n",
						 overallsize,overallxoff,overallyoff);
    space(0, 0, SIZE, SIZE);
    getxylim(&xd, xx, 'x');
	if (verbose) fprintf(stderr,"%c limits: %g %g\n",'x',xd.xlb,xd.xub);
    getxylim(&yd, xx, 'y');
	if (verbose) fprintf(stderr,"%c limits: %g %g\n",'y',yd.xlb,yd.xub);
    xscale(&xd);
	if (verbose) fprintf(stderr,"%c limits: %g %g\n",'x',xd.xlb,xd.xub);
    yscale(&yd);
	if (verbose) fprintf(stderr,"%c limits: %g %g\n",'y',yd.xlb,yd.xub);
	setticksize();
	if (verbose) fprintf(stderr,"before plot...");
    plot();
	if (verbose) fprintf(stderr,"ok\nbefore axes...");
    axes();
	if (verbose) fprintf(stderr,"ok\nbefore title...");
    title();
	if (verbose) fprintf(stderr,"ok\n");
	/* set bbox AFTER ticksize and titlesize has been settled */
    bbox(&xd,&yd);
    if (erasf)
        erase();
    free((char *)xx); /* perhaps there are other things to free as well! */
	freeStyles();     /* still retains the current_style */
	if (overallsize > 0) 
	{
		xd.xsize /= overallsize;
		yd.xsize /= overallsize;
		xd.xoff -= overallxoff;
		xd.xoff /= overallsize;
		yd.xoff -= overallyoff;
		yd.xoff /= overallsize;
	}
}
 
/*-- readin ---------------------------------------------------
 *  read in the inputfile,
 *      set options, read data, or execute plots as necessary;
 *-------------------------------------------------------------*/
 
void
readin(inputfile)
FILE *inputfile;
{
    char  *t;
	char  linebuf[BSIZ];
    
    for (;;) {
        if (getline(inputfile, linebuf) == -1) {
            nextf = 0;
            erasf = 1;      /* is this desired ?? */
            return;         /* End of file or garbage */
        }
 
        t = linebuf;
        while (*t == ' ' || *t == '\t')
            t++;        /* find first nonwhite character */
 
        if (*t == '#')       /* Option */
			readoptline(t);
		else if (*t != '\0') /* Data */
			readdataline(t);
	}
}
 
#define MAXARGC 32
 
/* read and parse a line that begins with '#' */
readoptline(t)
char *t;
{            
	int argc;
            char *argv[MAXARGC];
            int inarg = NO;
            int inquote = NO;
			char *tt,*index();
 
	if (*t != '#')
		fatal("internal error in readoptline(): *t==#");
 
	/* make argc and argv and then call setopt() */
	while (*(t+1) == ' ' || *(t+1) == '\t')
		t++;    /* skip white space after the hash '#' */
	*t = '#';   /* so that "#   x" becomes "#x"        */
	
	argc = 1;
	for (/*null*/; *t != '\0'; t++) {        /* scan line */
		switch (*t) {
		case ' ':    /* replace white space by nulls except */
		case '\t':   /* in quotes. inarg=0 means next nonwhite is new */
			if (!inquote) {
				*t = '\0';
				inarg = NO;
			} else {
				if (!inarg)  argv[argc++]=t;
				if (argc > MAXARGC ) break;
				inarg = YES;
            }
			break;
		case '"':       /* start or finish a quotation */
			if (!inquote)
				inquote = YES;
			else {
				*t = '\0';
				inquote = NO;
			}
			break;
		default:
			if (!inarg)
				argv[argc++] = t;
			if (argc > MAXARGC)
				break;
			inarg = YES;      /* inarg=1 means in middle of arg. */
			break;
		}
	}
 
	setopt(argc,argv);
 
} 
 
/* read and parse a data line */
int
readdataline(t)
char *t;
{
    struct datum   *temp;
	int            tmp_symbf;
	char           *lblptr=NULL;
	char           *s,*index();
 
	/* make a new structure for next value */
	temp = (struct datum *) realloc((char *) xx,
									(unsigned) (n + 1) * sizeof(struct datum));
	if (temp == NULL)
		fatal("Unable to allocate more memory for data structures");
	
	xx = temp;
 
#if 0 /* ON SECOND THOUGHT...
	   * if you are sophisticated enough to use #d, you should
	   * just beware that it over-rides errf, absf, etc.
	   */
 
	/* see if datastring has been superseded by recent changes
     * in relevant flags; eg, errf, transf, absf ...
	 */ 
	if ((errf==0 && countchar('e',datastring)) || 
		(errf==1 && !countchar('e',datastring)))
		datastring[0] = '\0';
 
	if ((absf==0 && !countchar('x',datastring)) ||
	    (absf==1 && countchar('x',datastring)))
		datastring[0] = '\0';
 
	/* basically we are going to ignore the #t flag */
#endif
	
 
	/* parse the datastring */
	if ( *datastring != '\0')
	{
		if (newdstringf) /* may have to reset absf, errf */
		{
			int nx,ne;
			nx = countchar('x',datastring);
			ne = countchar('e',datastring);
			absf = (nx == 0 ? 1 : 0);
			errf = (ne == 0 ? 1 : 0);
		}
		get_xye(&xx[n].xv,&xx[n].yv,&xx[n].yerr,t,datastring,newdstringf);
		newdstringf = 0; /* it's not new now */
	} 
	else /* if datastring is empty */
	{
		if (absf) {
			if (errf) {
				if (sscanf(t, "%f%f", &xx[n].yv, &xx[n].yerr) != 2)
					return 0;
			} else {
				if (sscanf(t, "%f", &xx[n].yv) != 1)
					return 0;
			}
		} else {
			if (errf) {
				if (sscanf(t, "%f%f%f", &xx[n].xv, &xx[n].yv,
						   &xx[n].yerr) != 3)
					return 0;
			} else {
				if (transf) {
					if (sscanf(t, "%f%f", &xx[n].yv, &xx[n].xv) != 2)
						return 0;
				} else {
					if (sscanf(t, "%f%f", &xx[n].xv, &xx[n].yv) != 2)
						return 0;
				}
			}
		}
	}
 
 
	if (absf)
			xx[n].xv = ( n - auto_no ) * auto_dx + auto_xo;
	if (!errf)
			xx[n].yerr = 0;
	
	if (verbose) fprintf(stderr,"#dat %g %g %g\n",xx[n].xv,xx[n].yv,xx[n].yerr);
 
	tmp_symbf = symbf;
	lblptr = NULL;
	if ((s = index(t, '"')) != NULL) {
		int i;
		char *tt;
		if ((tt = index(++s, '"')) != NULL) {
			*tt = '\0';
		}
		i = utou(s);
		lblptr = copystring(s, i);
		symbf = 0; /* indicate that its a label and not a symbol */
	} else if (symbf)
		lblptr = plotsymb;
	
	/*
	 * if a label break is set and there is a symbol or an error bar,
	 * then style->con_flag should be zero 
	 */
	if ((lbflag && (lblptr != NULL || errf)))
		connectf = 0;
	/* else connectf is what previous point set it to be */
 
	if ( !( isStyleCurrent() && isLabelCurrent(lblptr) ) )
	{
		newDataStyle();
		current_style->lblptr = lblptr;
	}
	xx[n].style = current_style;
	
	symbf = tmp_symbf;
	/*
	 * if a label break is set and there is a symbol or an error bar,
	 * set break for next point; otherwise default is no break
	 */
	if (lbflag && (xx[n].style->lblptr != NULL || errf))
		connectf = 0;
	else
		connectf = 1;
	
	n++;
 
	return 1; /* indicates data successfully read in */
}
 
/*-- setopt --------------------------------------------------
 *  input (argc,argv) is argument vector stripped of '#'
 *  returns 0 normally, or if error
 *  returns 1 if it is ready to be making a new plot
 *     nb, currently it always returns 0; if it's ready for
 *     a new plot, it just makes one!
 *------------------------------------------------------------*/
 
#define ISARG(i,s)          (i<argc && strcmp(argv[i],s)==0)
#define ISOPT(i,s)          (i<argc && strcmp(argv[i]+1,s)==0)
 
#define GETSTRING(i,buf) \
            p1 = buf; if (i<argc) { p2 = argv[i];\
            while (*p2) { *p1 = *p2; ++p1; ++p2; }} *p1 = 0;
 
#define GETVAL(i,fmt,xptr)  (argv[i] && sscanf(argv[i], fmt, xptr) == 1)
#define GETNUM(i,xptr)      GETVAL(i,"%e",xptr)
#define GETINT(i,xptr)      GETVAL(i,"%d",xptr)
#define GETVALCHECK(i,fmt,xptr) \
        if (!GETVAL(i,fmt,xptr)) { badarg(argc,argv); return 0; }
#define GETNUMCHECK(i,xptr)      GETVALCHECK(i,"%e",xptr)
#define GETINTCHECK(i,xptr)      GETVALCHECK(i,"%d",xptr)
 
#define ISDOT(i)            ISARG(i,".")
#define GETNUMDEFAULT(i,x,def) \
	    if (ISDOT(i)) x = def; else if ( !GETNUM(i,&x) ) badarg(argc, argv)
#define GETINTDEFAULT(i,x,def) \
	    if (ISDOT(i)) x = def; else if ( !GETINT(i,&x) ) badarg(argc, argv)
#define GETNEXTNUMDEFAULT(i,x,def) if (i+1<argc){\
        if      (ISDOT(i+1))     { ++i; x = def;   }\
        else if (GETNUM(i+1,&x)) { ++i; } else continue; }
#define GETXYLIMITS(i,xptr,xflag) if (i+1<argc){\
        if      (ISDOT(i+1))       { ++i; xflag = 0; }\
        else if (GETNUM(i+1,xptr)) { ++i; xflag = 1; }}
 
#if 0
#define INCASE(s) else if (ISOPT(i,s))
#endif
 
#define INCASE(s) if (strcmp(s,Option)==0)
#define BREAK     if(hashopt) return 0; else continue
 
int
setopt(argc, argv)
int            argc;
char           *argv[];
{
char           *p1, *p2;
register int    i, j;
int             hashopt;
 
if ( argc<2 )
	return(0);          /* nothing in argv[] to look at */
 
hashopt = ( argv[1][0] == '#' );
/* if the leading character is a '#' then do only one option */
 
/* an options line does not automatically set a line break;
   only if #k or blank # is used
*/
 
/* scan the options */
for (i = 1; i < argc; i++) {
	if (argv[i][0] != '-' && argv[i][0] != '#') { /* its a file name */
		FILE *fp;
		fp = fopen(argv[i],"r");
		if (fp==NULL) 
			fprintf(stderr,"%s: can't open file: %s\n",ProgName,argv[i]);
		else
		{
			stdinf = 0; 
			/* no need to read stdin if filename on command line */
			connectf = 0; /* a new file sets a break */
			readin(fp);
			fclose(fp);
		}
		
	}
	else /* its not a file name but a control command */
	{
		char *Option;
		Option = argv[i]+1;
 
		if (*Option=='\0') {
			/* do nothing (except set connectf=0) */
			/* though you might consider reading in stdin if
			 * leadchar=='-'
			 */
			connectf = 0;
			BREAK;
		}
 
		INCASE("v") {
			verbose += 1;
			BREAK;
		}
		INCASE("v0") {
			verbose = 0;
			BREAK;
		}
		
		INCASE("k") {  /* comment statement */
			/* do nothing except set line break */
			connectf = 0; 
			BREAK;
		} 
 
		/*
		 * First consider the data style changes 
		 */
		INCASE("e") {    errxy = 'y', errf = 1;
		BREAK;
		}
		INCASE("e0") {   errf = 0;
		BREAK;
		}
		INCASE("ex") {   errxy = 'x', errf = 1;
		BREAK;
		}
		INCASE("tc") {   clipf = 1;
		BREAK;
		}
		INCASE("tc0") {  clipf = 0;
		BREAK;
		}
		INCASE("lb") {   lbflag = 1;
		BREAK;
		}
		INCASE("lb0") {  lbflag = 0;
		BREAK;
		}
		INCASE("m") {  
			/* line style */
			++i; GETINTCHECK(i,&mode);
			if (mode >= nmodes )
				mode = 1;
		BREAK;
		}
		INCASE("lw") {  /* set linewidth */
			++i; GETNUMDEFAULT(i,linwidth,LINWIDTH);
		BREAK;
		}
		INCASE("llw") {   /* label/symbod line width */
			++i; GETNUMDEFAULT(i,lblwidth,LBLWIDTH);
		BREAK;
		}
		INCASE("c0") { /* turn off plot symbols */
			symbf = 0;
		BREAK;
		}
		INCASE("cs") { /* symbol size */
			++i; GETNUMDEFAULT(i,symsize,SYMSIZE);
		BREAK;
		}
		INCASE("c") {  /* character(s) for plotting */
			if (++i >= argc)
				symbf = 0; /* plain "#c" equivalent to "#c0" */
			else
			{
				symbf = 1;
				j = utou(argv[i]);
				plotsymb = copystring(argv[i], j);
			}
		BREAK;
		} 
		INCASE("a0") { /* automatic abscissas off */
			absf = 0;
		BREAK;
		}
		INCASE("a")  { /* automatic abscissas on  */
			absf = 1;
			auto_no = n;
			/* set auto_dx and auto_xo to defaults first */
			auto_dx = 1.0;
			auto_xo = 0.0;
			GETNEXTNUMDEFAULT(i,auto_dx,1.0);
			GETNEXTNUMDEFAULT(i,auto_xo,0.0);
		BREAK;
		}
		INCASE("d") {
			++i; GETSTRING(i,datastring);
			newdstringf=1;
		BREAK;
		}
		INCASE("d0") {
			datastring[0] = '\0';
			newdstringf=1;
		BREAK;
		}
		
		/*
		 * The following options refer to the plot as a whole
		 */
		
		INCASE("lt")   {   /* title */
			++i; GETSTRING(i,titlebuf);
		BREAK;
		}
		INCASE("lx") {   /* x label */
			++i; GETSTRING(i,xtitlebuf);
		BREAK;
		}
		INCASE("ly") {   /* vertical y label */
			++i; GETSTRING(i,ytitlebuf);
			ytitlehf = 0;
		BREAK;
		}
		INCASE("lyh"){   /* horizontal y label */
			++i; GETSTRING(i,ytitlebuf);
			ytitlehf = 1;
		BREAK;
		}
		INCASE("xf")  {   /* x format string */ 
			++i; GETSTRING(i,xformat);
		BREAK;
		}
		INCASE("yf")  {   /* y format string */
			++i; GETSTRING(i,yformat);
		BREAK;
		}
		INCASE("ltx") {       /* title x location */
			++i; GETNUMCHECK(i,&titlex);
			titlexf = 1;
		BREAK;
		}
		INCASE("lty") {       /* title y location */
			++i; 
			titleyf = 0;
			if      (ISARG(i,"^")) titleatopf=1;
			else if (ISARG(i,".")) titleatopf=0;
			else {
				GETNUMCHECK(i,&titley);
				titleyf = 1;
			}
		BREAK;
		}
		INCASE("lxx") {       /* x label  x loc. */
			++i; GETNUMCHECK(i,&xtitlex);
			xtitlexf = 1;
		BREAK;
		}
		INCASE("lxy") {       /* x label  y loc. */
			++i; GETNUMCHECK(i,&xtitley);
			xtitleyf = 1;
		BREAK;
		}
		INCASE("lyx") {       /* y label  x loc. */
			++i; GETNUMCHECK(i,&ytitlex);
			ytitlexf = 1;
		BREAK;
		}
		INCASE("lyy") {       /* y label  y loc. */
			++i; GETNUMCHECK(i,&ytitley);
			ytitleyf = 1;
		BREAK;
		}
		INCASE("lts") {       /* title size */
			++i; GETNUMDEFAULT(i,titlesize,TITLESIZE);
		BREAK;
		}
		INCASE("lxs") {       /* x label size */
			++i; GETNUMDEFAULT(i,xtitlesize,XTITLESIZE);
		BREAK;
		}
		INCASE("lys") {       /* y label size */
			++i; GETNUMDEFAULT(i,ytitlesize,YTITLESIZE);
		BREAK;
		}
		INCASE("lns") {       /* label number size */
			++i; GETNUMDEFAULT(i,numlblsize,NUMLBLSIZE);
		BREAK;
		}
		/* Grid Style */
		INCASE("g") {     /* grid style */
			/* 0 none, 1 ticks, 2 full, 3 half */
			++i; GETINTDEFAULT(i,gridstyle,GRIDSTYLE);
		BREAK;
		}
		INCASE("gs") {     /* gridstyle string */
			/* [eflt][eflt][eflt][eflt] (e)mpty,(f)rame,(l)abel,(ticks) */
			++i; GETSTRING(i,gridbuf);
			gridstyle = -1; /* to indicate use of string */
		BREAK;
		}
		INCASE("gm") {     /* grid line style */
			++i; GETINTDEFAULT(i,gridlinestyle,GRIDLINESTYLE);
		BREAK;
		}
		INCASE("nd") {   /* number of sub ticks */
			++i; GETINTDEFAULT(i,tickspertick,TICKSPERTICK);
			xtickspertick = ytickspertick = tickspertick;
		BREAK;
		}
		INCASE("ndx") {   /* number of x sub ticks */
			++i; GETINTDEFAULT(i,xtickspertick,TICKSPERTICK);
		BREAK;
		}
		INCASE("ndy") {   /* number of y sub ticks */
			++i; GETINTDEFAULT(i,ytickspertick,TICKSPERTICK);
		BREAK;
		}
		INCASE("alw") {   /* axes line width */
			++i; GETNUMDEFAULT(i,axlwidth,AXLWIDTH);
		BREAK;
		}
		INCASE("tks") {   /* ticksize */
			++i; GETNUMDEFAULT(i,tickfactor,TICKFACTOR);
		BREAK;
		}
		/* X and Y limits */
		INCASE("x") {   /* x limits */
			if ( ISARG(i+1,"l") ) {
				i++;
				xd.xf = log10;
				xd.xlogf = 1;
			}
			GETXYLIMITS(i,&xd.xlb,xd.xlbf);
			GETXYLIMITS(i,&xd.xub,xd.xubf);
			GETXYLIMITS(i,&xd.xquant,xd.xqf);
		BREAK;
		}
		INCASE("y") { /* y limits */
			if ( ISARG(i+1,"l") ) {
				i++;
				yd.xf = log10;
				yd.xlogf = 1;
			}
			GETXYLIMITS(i,&yd.xlb,yd.xlbf);
			GETXYLIMITS(i,&yd.xub,yd.xubf);
			GETXYLIMITS(i,&yd.xquant,yd.xqf);
		BREAK;
		} 
 
		/* Plot size */
 
		INCASE("h") { /* fraction of height */
			++i; GETNUMDEFAULT(i, yd.xsize, 1.0);
		BREAK;
		}
		INCASE("w") { /* fraction of width */
			++i; GETNUMDEFAULT(i, xd.xsize, 1.0);
		BREAK;
		}
		INCASE("r") { /* move to right */
			++i; GETNUMDEFAULT(i, xd.xoff, 0.0);
		BREAK;
		}
		INCASE("u") { /* move up */
			++i; GETNUMDEFAULT(i, yd.xoff, 0.0);
		BREAK;
		} 
		INCASE("size") { /* overall scale */
			float s,x,y;
			x = y = 0;
			++i; GETNUMDEFAULT(i, s, OVERALLSIZE);
			if (i+1<argc) {
				if ISDOT(i+1) {++i; x = 0.0;}
				else if (GETNUM(i+1,&x)) ++i;
			}
			if (i+1<argc) {
				if ISDOT(i+1) {++i; y = 0.0;}
				else if (GETNUM(i+1,&y)) ++i;
			}
			size(s,x,y);
			
			if (ISARG(i+1,"{")) {
					++i;
					purepush();
			}
			/* 
			 * because this is a "supercommand" the "subcommands"
			 * get set to their defaults;  thus it is important 
			 * that this command preceed #h, #u, etc.
			 */
			xd.xsize = 1.0;
			yd.xsize = 1.0;
			xd.xoff = 0.0;
			yd.xoff = 0.0;
		BREAK;
		}
		INCASE("}") { /* pop size! */
			popsize();
		BREAK;
		}
		INCASE("{") { /* push size */
			purepush();
		BREAK;
		}
		/*
         * Actually DO a plot 
		 */
 
		INCASE("s") { /* save screen, overlay plot */
			erasf = 0;
			execute_plot();
			prepare_plot();
 
			/* command "#s }" both plots graph and pops size stack */
			if (ISARG(i+1,"}")) { 
					++i;
					popsize();
			}
		BREAK;
		}
#if 0
		INCASE("s}") { /* save screen, overlay plot, AND pop stack */
			erasf = 0;
			execute_plot();
			prepare_plot();
			popsize();
		BREAK;
		}
#endif
		INCASE("n") { /* next plot flag */
			erasf = 1;
			execute_plot();
			prepare_plot();
 
			if (ISARG(i+1,"}")) {
					++i;
					popsize();
			}
		BREAK;
		}			
		INCASE("i") { /* include file */
			FILE *fp;
			fp = fopen(argv[++i],"r");
			if (fp==NULL) 
				fprintf(stderr,"%s: can't include file: %s\n",
						ProgName,argv[i]);
			else
			{
				/* note: 
				 * do not alter flag stdinf in this case 
				 */
				readin(fp);
				fclose(fp);
			}
		BREAK;
		} 
		/* 
		 * If made it here, then Option is not listed 
		 */
		fprintf(stderr,"Invalid option: %s\n",Option);
 
	} /* END if '#' or '-' ... */
	if (hashopt)
		return(0);  /* only do one option if its a #option */
  /* <-- BREAK goes to here */
} /* END for(argc... */
return(0);
} /* END setopt() */
 
getline(inputfile, p)   /* get a line from inputfile, put in string p */
FILE           *inputfile;
char           *p;
{
    register int    i, c;       /* return length of string, -1 if EOF */
    i = -1;
    while ((c = getc(inputfile)) != EOF) {
        if (c == '\n') {
            p[++i] = '\0';
            return (i);
        } else
            p[++i] = c;
    }
    if (i >= 0)
        p[++i] = '\0';
    return (i);
}
 
 
 
#if 0
#define LSIZ 1024               /* size of each label buffer */
char *
copystring(p, length)
int  length;
char *p;  /* copy a string into the label list */
{
    static char   *labstr=NULL;     /* pointer to current label  */
    static int     labsleft = 0;    /* space left in allocated buffer */
 
    register char  *temp;
    register        i;
 
    length++;                   /* length of string plus null char */
    if (length > labsleft) {    /* need more space */
        labstr = (char *) malloc(LSIZ);
        labsleft = LSIZ;
        if (labstr == NULL) 
            fatal("Unable to allocate memory for label");
    }
    temp = labstr;
    for (i = 0; i < length; i++)
        labstr[i] = p[i];
    labstr += length;
    labsleft -= length;
    return (temp);
}
#undef LSIZ
#else
char *
copystring(p, length)
char *p;  /* copy a string into the label list */
int  length;
{
	int i;
	char *t;
	t = malloc(length+1);
 
	for (i=0; i<length; ++i)
		t[i] = p[i];
	t[length] = '\0';
 
	return t;
}
#endif /* copystring */
getxylim(p, v, c)     
register struct xy *p;
register struct datum *v;
char c; /* 'x' or 'y' */
{
    register        i;
	
    if (n == 0) /* set defaults for blank plot with no points */ 
    {
        if (p->xf == log10) 
        {
            if (!p->xlbf) p->xlb = 1.0;
            if (!p->xubf) p->xub = 10.0;
        } 
        else 
        {
            if (!p->xlbf) p->xlb = 0.0;
            if (!p->xubf) p->xub = 1.0;
        }
		return;
    } 
 
	for (i = 0; i < n; i++) 
	{
		float t;
		float val;
 
		val = (c=='x' ? v[i].xv : v[i].yv );
 
		if(p->xf == log10) 
		{
			
			/* is there an error on this axis ? */
			if (v[i].style->err_flag && c == v[i].style->err_xy )
				t = exp(-v[i].yerr / v[i].yv);
			else
				t = 1;
 
			if (!p->xlbf && p->xlb > (val * t) && val > 0)
				p->xlb = val * t;
			if (!p->xubf && p->xub < (val / t) && val > 0)
				p->xub = val / t;
		}
		else
		{
			/* again, check if error bars on this axis */
			if (v[i].style->err_flag && c == v[i].style->err_xy )
				t = v[i].yerr;
			else
				t = 0;
 
			if (!p->xlbf && p->xlb > (val - t))
				p->xlb = val - t;
			if (!p->xubf && p->xub < (val + t))
				p->xub = val + t;
		}
	}
}
 
int
in_range(x,p,eps)
float x, eps;
struct xy *p;
{
	float delx;
	delx = fabs(p->xub - p->xlb);
	if ((p->xquant>0 && (x< p->xlb -eps*delx || x> p->xub +eps*delx))
      ||(p->xquant<0 && (x< p->xub -eps*delx || x> p->xlb +eps*delx)))
		return NO;
	else
		return YES;
}
int in_logrange(x,p,eps)
float x, eps;
struct xy *p;
{
	return ((1.0+eps) * p->xlb < x && (1.0-eps) * p->xub > x) ;
}
void 
outer_range(lo,hi,p)
float *lo,*hi;
struct xy *p;
{
	float q,xl,xu;
	
	q = p->xquant;
	
	if (q > 0) {
		*lo = modfloor(p->xlb - q / 6, q);
		*hi = modceil( p->xub + q / 6, q);
	} else {
		*lo = modfloor(p->xub + q / 6, -q);
		*hi = modceil( p->xlb - q / 6, -q);
	}
}
 
xscale(p)
    register struct xy *p;
{
    float           edge;
 
	tickspertick = xtickspertick;
    setlim(p);
    edge = topx - botx;
    p->xa = p->xsize * edge / ((*p->xf) (p->xub) - (*p->xf) (p->xlb));
    p->xbot = botx + edge * p->xoff;
    p->xtop = p->xbot + edge * p->xsize;
    p->xb = p->xbot - (*p->xf) (p->xlb) * p->xa +.5;
}
 
yscale(p)
    register struct xy *p;
{
    float           edge;
 
	tickspertick = ytickspertick;
    setlim(p);
    edge = topy - boty;
    p->xa = p->xsize * edge / ((*p->xf) (p->xub) - (*p->xf) (p->xlb));
    p->xbot = boty + edge * p->xoff;
    p->xtop = p->xbot + edge * p->xsize;
    p->xb = p->xbot - (*p->xf) (p->xlb) * p->xa +.5;
}
 
 
#define  TickSize  tickSize
 
plot()
{
    int             ix, iy, iy1, iy2;
    int             jx, jy, jy1, jy2;
    int             ix1, ix2;
	int             ixprev, iyprev, in, inprev;
    int             bar, dbar, delta;
    int             i;
    int             conn;
 
    bar = 3 * (tickSize>0?tickSize:-tickSize) * (n + 10) / (3 * n + 10);
    conn = 0;
 
	newm(mode);
	neww(linwidth);
 
    for (i = 0; i < n; i++) {
		int xin,yin;            /* flag if in bounds */
        conn &= xx[i].style->con_flag;             /* check for break */
        if (xx[i].style->line_mode != mode) {   /* check for line mode */
            mode = xx[i].style->line_mode;
            newm(mode);
        }
        if (xx[i].style->line_width != linwidth) {   /* check for line width */
            linwidth = xx[i].style->line_width;
            neww(linwidth);
        }
		/* first plot the point */
		xin = conv(xx[i].xv, &xd, &ix); 
		yin = conv(xx[i].yv, &yd, &iy); 
		if (verbose) fprintf(stderr,"%d: (%f %f) %d %d\n",
							 i,xx[i].xv,xx[i].yv,ix,iy);
        if (!xin || !yin) {
            /* point out of bounds */
			in = 0;
			if (clipf && conn && mode != 0 && inprev ) {
				int   ixfix,iyfix;
				clip(&ixfix,&iyfix,&xd,&yd,ix,iy,ixprev,iyprev);
				if (verbose) fprintf(stderr,"%d %d --> %d %d\n",
									 ix,iy,ixfix,iyfix);
				cont( ixfix, iyfix );
				
			}
			else if (clipf && conn && mode != 0 && !inprev ) {
				int ixfix1,ixfix2,iyfix1,iyfix2;
				if ( two_clip(&ixfix1,&iyfix1,&ixfix2,&iyfix2,&xd,&yd,
							  ix,iy,ixprev,iyprev) ) {
					move (ixfix1,iyfix1);
					cont (ixfix2,iyfix2);
				}
			}
			conn=1;
         } else {
		 	/* point in bounds */
           	in = 1;
			if (mode == 0) {
				;/* do nothing */
			} else if ( !conn || (!inprev && !clipf)) {
				move(ix,iy);/* do nothing, don't move(ix,iy); */
			} else if (inprev) {
       			neww(linwidth);
               	cont(ix,iy);
    		} else if (!inprev && clipf) {
				int ixfix,iyfix;
				clip(&ixfix,&iyfix,&xd,&yd,ixprev,iyprev,ix,iy);
				neww(linwidth);
				move(ixfix,iyfix);
				cont(ix,iy);
			}
			conn = 1;
       		symbol(ix, iy, xx[i].style );
		} /*end if in bounds*/
		/* now that point is plotted, next plot error bar */
        if (in && xx[i].style->err_flag) {
			if (xx[i].style->err_xy == 'x')
			{
				/* x-style error bar */
            jx = conv(xx[i].xv, &xd, &ix);
            jy = conv(xx[i].yv, &yd, &iy);
            if (xd.xf == log10) {
				float t;
				t = exp(xx[i].yerr / xx[i].xv);
                jy1 = conv(xx[i].xv * t, &xd, &ix1);
                jy2 = conv(xx[i].xv / t, &xd, &ix2);
            } else {
                jy1 = conv(xx[i].xv + xx[i].yerr, &xd, &ix1);
                jy2 = conv(xx[i].xv - xx[i].yerr, &xd, &ix2);
            }
            if (jx == 0 || jy == 0)
                fatal("plot: jx==0 and jy==0");
 
            if (jy1 == 0)
                ix1 = xd.xtop;
            if (jy2 == 0)
                ix2 = xd.xbot;
            delta = ix1 - ix2;
            delta = delta / 5;
			/* This give variable width error bar caps; is that good?
            dbar = (bar < delta) ? bar : delta;
			*/
			/* This gives constant width error bar caps; perhaps
             * ideally, one should run through the data and then
			 * decide on the correct cap width ?
			 */
			dbar = bar;
            iy1 = (yd.xbot < iy - dbar) ? iy - dbar : yd.xbot;
            iy2 = (yd.xtop > iy + dbar) ? iy + dbar : yd.xtop;
 
			/* for purposes of plotting error bar;
             * change line mode to solid, thin
             */
            newm(1);
			neww(ERRBAR_LINEWIDTH*xx[i].style->line_width);
 
			/* draw the (vertical) error bar itself first */
            move(ix1, iy);
            cont(ix2, iy);
			/* now draw the left vertical cap */
	        if (jy1) {
                move(ix1, iy1);
                cont(ix1, iy2);
            }
			/* and the right vertical cap */
            if (jy2) {
                move(ix2, iy1);
                cont(ix2, iy2);
            }
			/* finished plotting error bar; can
			 * change back to original line mode and width
			 */
            newm(mode);
			neww(linwidth);
 
			/* move back to the point */
	        move(ix,iy);
            /* don't need to write symbol, but we are anyway, since
               there is an xyps bug that forgets to stroke before a
               reset line width
			   symbol(ix, iy, xx[i].style );
			   xyps bug should be fixed!
             */
 
			}
			else
			{
				/* ordinary y-style error bar */
            jx = conv(xx[i].xv, &xd, &ix);
            jy = conv(xx[i].yv, &yd, &iy);
            if (yd.xf == log10) {
				float t;
				t = exp(xx[i].yerr / xx[i].yv);
                jy1 = conv(xx[i].yv * t, &yd, &iy1);
                jy2 = conv(xx[i].yv / t, &yd, &iy2);
            } else {
                jy1 = conv(xx[i].yv + xx[i].yerr, &yd, &iy1);
                jy2 = conv(xx[i].yv - xx[i].yerr, &yd, &iy2);
            }
            if (jx == 0 || jy == 0)
                fatal("plot: jx==0 and jy==0");
 
            if (jy1 == 0)
                iy1 = yd.xtop;
            if (jy2 == 0)
                iy2 = yd.xbot;
            delta = iy1 - iy2;
            delta = delta / 5;
            dbar = (bar < delta) ? bar : delta;
            ix1 = (xd.xbot < ix - dbar) ? ix - dbar : xd.xbot;
            ix2 = (xd.xtop > ix + dbar) ? ix + dbar : xd.xtop;
 
			/* for purposes of plotting error bar;
             * change line mode to solid, thin
             */
            newm(1);
			neww(ERRBAR_LINEWIDTH*xx[i].style->line_width);
 
			/* draw the (vertical) error bar itself first */
            move(ix, iy1);
            cont(ix, iy2);
			/* now draw the top horizonatl bar */
	        if (jy1) {
                move(ix1, iy1);
                cont(ix2, iy1);
            }
			/* and the bottom horizontal bar */
            if (jy2) {
                move(ix1, iy2);
                cont(ix2, iy2);
            }
			/* finished plotting error bar; can
			 * change back to original line mode and width
			 */
            newm(mode);
			neww(linwidth);
			/* move back to the point */
	        move(ix,iy);
            /* don't need to write symbol, but we are anyway, since
               there is an xyps bug that forgets to stroke before a
               reset line width
       		symbol(ix, iy, xx[i].style );
             */
 
		}
            conn = 1;
    	}/*end if err flag */
		ixprev = ix;
		iyprev = iy;
		inprev = in;
	}/*endfor*/
}
 
clip(pixfix,piyfix,pxd,pyd,ix,iy,ixp,iyp)
int *pixfix,*piyfix;
struct xy *pxd,*pyd;
int ix,iy,ixp,iyp;
{
	float xxp,yyp,xcp,ycp;
 
	if (verbose) fprintf(stderr,"clip1");
 
	xxp = ix - ixp;
	xcp = ( (ix > pxd->xtop) ? pxd->xtop :
            (ix < pxd->xbot) ? pxd->xbot : ix ) - ixp;
	yyp = iy - iyp;
	ycp = ( (iy > yd.xtop) ? yd.xtop :
            (iy < yd.xbot) ? yd.xbot : iy ) - iyp;
											
	if (verbose) fprintf(stderr," [%g*%g <=> %g*%g]",
						 xxp,ycp,yyp,xcp);
	if( fabs(xxp*ycp) > fabs(yyp*xcp)) {
		*pixfix = ixp + xcp;
		*piyfix = iyp + xcp*yyp/xxp;
    } else if( fabs(xxp*ycp) < fabs(yyp*xcp)) {
		*piyfix = iyp + ycp;
		*pixfix = ixp + ycp*xxp/yyp;
	} else {
		*pixfix = ixp + xcp;
		*piyfix = iyp + ycp;
	}
	if (verbose) fprintf(stderr," %d %d\n",*pixfix,*piyfix);
 
}
int
two_clip(pixfix1,piyfix1,pixfix2,piyfix2,pxd,pyd,ix,iy,ixp,iyp)
int *pixfix1,*piyfix1;
int *pixfix2,*piyfix2;
struct xy *pxd,*pyd;
int ix,iy,ixp,iyp;
{
	float t,xt,yt;
	int ixfix[2],iyfix[2];
	int xhi,xlo,yhi,ylo;
	int i=0;
 
	xhi = (pxd->xtop > pxd->xbot ? pxd->xtop : pxd->xbot );
	yhi = (pyd->xtop > pyd->xbot ? pyd->xtop : pyd->xbot );
	xlo = (pxd->xtop <= pxd->xbot ? pxd->xtop : pxd->xbot );
	ylo = (pyd->xtop <= pyd->xbot ? pyd->xtop : pyd->xbot );
 
	/* There are four possible crossings */
	/* If two are "good" then connect them */
 
	if (verbose) fprintf(stderr,"clip2\n");
	
	t = ix==ixp ? VERYBIG : (float)(xhi - ixp)/(ix - ixp);
	if (verbose) fprintf(stderr,"(t=%g)",t);
	if (t>0 && t<1) {
		yt = t*(iy-iyp) + iyp;
		if (yt > ylo && yt < yhi) {
			ixfix[i] = xhi;
			iyfix[i] = yt;
			++i;
		}
		
	}
	t = ix==ixp ? VERYBIG : (float)(xlo - ixp)/(ix - ixp);
	if (verbose) fprintf(stderr,"(t=%g)",t);
	if (t>0 && t<1) {
		yt = t*(iy-iyp) + iyp;
		if (yt > ylo && yt < yhi){
			ixfix[i] = xlo;
			iyfix[i] = yt;
			++i;
		}
	}
	t = iy==iyp ? VERYBIG : (float)(yhi - iyp)/(iy - iyp);
	if (verbose) fprintf(stderr,"(t=%g)",t);
	if (t>0 && t<1) {
		xt = t*(ix-ixp) + ixp;
		if (xt > xlo && xt < xhi){
			iyfix[i] = yhi;
			ixfix[i] = xt;
			++i;
		}
	}
	t = iy==iyp ? VERYBIG : (float)(ylo - iyp)/(iy - iyp);
	if (verbose) fprintf(stderr,"(t=%g)",t);
	if (t>0 && t<1) {
		xt = t*(ix-ixp) + ixp;
		if (xt > xlo && xt < xhi){
			iyfix[i] = ylo;
			ixfix[i] = xt;
			++i;
		}
	}
	if (verbose) fprintf(stderr,"\n",t);
 
	if (i==2) {
		*pixfix1 = ixfix[0];
		*piyfix1 = iyfix[0];
		*pixfix2 = ixfix[1];
		*piyfix2 = iyfix[1];
		return 1;
	}
	else 
		return 0;
}
 
conv(xv, p, ip)
    float           xv;
    register struct xy *p;
    int            *ip;
{
    long            ix;
    float           tmp;
    tmp = xv * p->xmult;
    if (p->xf == log10 && tmp <= 0) {
        ix = p->xb;
    } else {
        ix = p->xa * (*p->xf) (tmp) + p->xb;
    }
    *ip = ix;
 
	if (verbose>1) fprintf(stderr,"%g + %g * f(%g*%g) = %ld\n",
						 p->xb, p->xa, p->xmult, xv, ix);
 
 
    if (ix < p->xbot || ix > p->xtop)
		return 0;
	else
		return 1;
}
 
symbol(ix, iy, style )
    int         ix, iy;
    struct datastyle *style;
{
	float size;
	size = (style->symsiz)*overallsize*SIZE/4096;
 
    if (style->lblptr == NULL || *(style->lblptr)=='\0') {
        if (style->line_mode == 0)
            point(ix, iy);
    } else {
        newm(1);
		pushw(lblwidth);
		if (style->symb_flag)
			cascii(style->lblptr,ix,iy,size);
		else
			ascii(style->lblptr, ix, iy, size);
		popw();
        newm(mode);
		if (mode != 0) 
			move(ix,iy);
        /* why move(ix, iy); ? 
		 * because if lines are being drawn between points (mode != 0), then
		 * need to reset starting point in the center of the symbol
		 * not at the last stroke of the symbol  (this may not be
		 * necessary in .ps version)
		 */
    }
}
 
 
 
 
/*-- set and reset line width and mode ---------------------- */
static float previouswidth = 1.0;
static float currentwidth = 1.0;
pushw(w)
float w;
{
	previouswidth = currentwidth;
	neww(w);
}
popw()
{
	neww(previouswidth);
}
neww(w)
float w;
/*  change the width to w if it is not that */
{
	static int firstcall=1;
	if (firstcall || w!=currentwidth ) {
        width(w*overallsize);
    }
	currentwidth = w;
	firstcall = 0;
}
newm(m)
int m;
/*  change the mode to m if it is not that already */
{
	static int firstcall=1;
	static int currentmode = 1.0; /* the current mode at a time */
	int mm;
	mm = ( (m < nmodes) ? m : 1 );
    if (firstcall || mm!=currentmode )
	{
        linemod(modes[mm]);
    }
	currentmode = mm;
	firstcall=0;
}
 
 
 
 
 
/*-- error handling -------------------------------------------- */
 
fatal(errstring)
char *errstring;
{
    fprintf(stderr,"%s: %s\n",ProgName,errstring);
    exit(-1);
}
 
badarg(argc, argv)
int    argc;
char   *argv[];
{
    int i;
 
    fprintf(stderr, "%s: error in input stream\n",ProgName);
    for (i = 1; i < argc; i++)
        fprintf(stderr, "%s%c ", argv[i], (i < argc - 1) ? ' ' : '\n');
#if 0	/* make it nonfatal */
    exit(-1);
#endif
}
 
/*-- main ----------------------------------------------------- */
 
#define POPEN(file,rw) POPEN_PERMITTED ? popen(file,rw) : NULL	
 
main(argc, argv)
int             argc;
char           **argv;
{
    char *outPipename;
    char *getenv();
    int  filter_chosen=0;

	/* argv[0] should be "xyplot" */

	strcpy(ProgName,argv[0]);
 
	/* if "xyplot -?" typed give the version number and exit */

	if (argc>1 && strcmp(argv[1],"-?")==0) { /* version number */
		fprintf(stderr,"%s: version %s patchlevel %-d\n",
				ProgName,Version,PATCHED);
		exit(0);
	}
 
	/* To determine output filter,
     *     First:  check the command line;
     *     Second: check the environment variable XYSCAN
     *     Third:  check the #defined varible XYSCAN
     *     Fourth: default to XYUG
	 */
 
	/** The first command line argument defines the output filter **/
	if (argc>1)
	{
		if (strcmp(argv[1],"-s")==0 ||
			strcmp(argv[1],"-xyil")==0 ) { /* xy intermediate language */
			outputfile = stdout;
			filter_chosen = 1;
			--argc;	++argv;
		}
		else if (strcmp(argv[1],"-ug")==0 ) { /* unix-graphics */
			outputfile = POPEN(XYUG,"w");
			if (outputfile==NULL) 
				fatal("Cannot popen XYUG to do unix-graphics output");
			filter_chosen = 1;
			--argc;	++argv;
		}
		else if (strcmp(argv[1],"-ps")==0 ) { /* postScript */
			outputfile = POPEN(XYPS,"w");
			if (outputfile==NULL) 
				fatal("Cannot popen XYPS to do postscript");
			filter_chosen = 1;
			--argc;	++argv;
		} 
		else if (strcmp(argv[1],"-scan")==0 ) { /* general scan program */
			/* in a fuller implementation, we might
			 * want to check argv[] for various defaults: 
			 *  eg "", or "stdout", or "PS" etc. 
			 */
			outputfile = POPEN(argv[2],"w");
			if (outputfile==NULL)
				fatal("Cannot popen -scan routine");
			filter_chosen = 1;
			--argc;	++argv;
			--argc;	++argv;
		} 
	} 
 
	/** if filter not yet chosen, check XYSCAN **/
	
	if (!filter_chosen)
	{
		if ( (outPipename=getenv("XYSCAN")) && (*outPipename != '\0') )
		{
			outputfile = POPEN(outPipename,"w");
			if (outputfile == NULL ) 
				fatal("Cannot popen XYSCAN");
			filter_chosen = 1;
		}
#ifdef XYSCAN
		else
		{
			outputfile = POPEN(XYSCAN,"w");
			if (outputfile == NULL ) 
				fatal("Cannot popen XYSCAN");
			filter_chosen = 1;
		}
#endif /* def XYSCAN */
	}
 
	/** if filter still not chosen, default is XYUG **/
	/** if that fails, then ultimate default is stdout **/
 
	if (!filter_chosen)
	{
		outputfile = POPEN(XYUG,"w");
		if (outputfile==NULL) 
			outputfile = stdout;
		filter_chosen = 1;
	}
	
	nmodes = sizeof(modes)/sizeof(*modes);
	
    init(&xd);
    init(&yd);
	initstack();
    prepare_plot();
    setopt(argc,argv); 
    if (stdinf) readin(stdin);
    execute_plot();
	
#if POPEN_PERMITTED
    if (outputfile != stdout)
        pclose(outputfile);
#endif

	return 0;

}
 
