/*
 *+-----------------------------------------------------------------------+
 *| This header file contains definitions for 32-bit colour pixels and    |
 *| Z-buffer.                                                             |
 *|                                                                       |
 *| Author: Michael S. A. Robb         Version: 1.2        Date: 16/06/93 |
 *+-----------------------------------------------------------------------+
 */

typedef unsigned long ADDRESS; /* 32-bit memory address.   */
typedef unsigned int  WORD;    /* 16-bit data.             */
typedef unsigned char BYTE;    /*  8-bit data.             */
typedef unsigned long LONG;    /* 32-bit unsigned integer. */

/*
 *+-----------------------------------------------------------------------+
 *| The following data structure is used to represent a 32-bit Z-buffer   |
 *| value.                                                                |
 *+-----------------------------------------------------------------------+
 */

typedef struct zbufferhl_st
  {
  WORD zbuf_lo;
  WORD zbuf_hi;
  } ZBUFFER_HL;

typedef union zbuffer_st
  {
  long       zbuf_value;       /* Used for calculations.             */
  ZBUFFER_HL zbuf_hilo;        /* Used for read/writing TIGA memory. */
  } ZBUFFER;

#define ZBUFFER_MAX 0x7FFFFFFFL

#define MAX_BLEND   255        /* Maximum blend value         */
#define PIXEL_SIZE   32        /* 1 pixel = 32 bits           */
#define ZBUFFER_SIZE 32        /* 1 Z-buffer entry = 32 bits. */

/*+-----------------------------------------------------------------------+
 *| Hercules Video RAM is located at location 0x00000000 and is 1 Mbyte   |
 *| in size. However, the TMS34010 addresses memory using bit addresses.  |
 *| So in fact, the end of video RAM is at location 0x800000.             |
 *|                                                                       |
 *| Since each pixel is 32-bits in length, this makes it possible for the |
 *| graphics board to have two 512x256x32 bit screens. Alternatively, the |
 *| second screen can be used as a 32-bit Z-buffer. If the optional 2     |
 *| Mbytes of DRAM is present, this allows the use of two screens with    |
 *| 512x256x32 resolution, both of which have Z-buffers.                  |
 *+-----------------------------------------------------------------------+
 */

#define OFFSET_VRAM             0x0L           /* Start of VRAM       */
#define OFFSET_ZBUFFER          0x400000L      /* Second half of VRAM */

/*+-----------------------------------------------------------------------+
 *| Useful macros for converting XY addresses to linear offsets.          |
 *+-----------------------------------------------------------------------+
 */

#define ADDRESS_PIXEL(X,Y)\
  (( (ADDRESS)(X)+(ADDRESS)(Y)*screen_width)*PIXEL_SIZE+OFFSET_VRAM)

#define ADDRESS_ZBUFFER(X,Y)\
  (( (ADDRESS)(X)+(ADDRESS)(Y+256)*screen_width)*PIXEL_SIZE+OFFSET_ZBUFFER)

/*+-----------------------------------------------------------------------+
 *| The following data structure is used to represent a 32-bit colour     |
 *| pixel plus an 8-bit overlay.                                          |
 *+-----------------------------------------------------------------------+
 */

typedef struct rgb32_st
  {
  BYTE  col_blue;              /* Amount of BLUE.    */
  BYTE  col_overlay;           /* Amount of OVERLAY. */
  BYTE  col_green;             /* Amount of GREEN.   */
  BYTE  col_red;               /* Amount of RED.     */
  } RGB32;

typedef struct rgb32hl_st
  {
  WORD  col_lo;                /* Low 16 bits  */
  WORD  col_hi;                /* High 16 bits */
  } RGB32_HL;

/*+-----------------------------------------------------------------------+
 *| The following data structure is used to represent a 32-bit colour     |
 *| pixel as well as permitting Boolean operations such as AND, OR & NOT. |
 *+-----------------------------------------------------------------------+
 */

typedef union colour32
  {
  RGB32    col_rgb;            /* Structured representation. */
  RGB32_HL col_hilo;           /* Integer    representation. */
  ADDRESS  col_value;          /* Integer    representation. */
  } COLOUR32;


