/**
 ** sipp - SImple Polygon Processor
 **
 **  A general 3d graphic package
 **
 **  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 **            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 1, or any later version.
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 ** You can receive a copy of the GNU General Public License from the
 ** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **/

/**
 ** viewpoint.c - Functions that handles the viewpoint definition and
 **               calculation of viewing transformation.
 **/

#include <geometric.h>
#include <viewpoint.h>


Viewpoint camera;          /* Viewpoint of the scene  */

/*
 * Constants used in viewing transformation.
 */
double        hither;          /* Hither z-clipping plane */
double        yon;             /* Yonder z-clipping plane */


/*
 * Calculate the vector from the point of interest
 * to the viewpoint. The shaders needs this vector normalized
 * while the view coordinate transformation needs it
 * non normalized, therefore we need this routine to
 * recalculate the non normalized value.
 */
void
view_vec_eval()
{
    MakeVector(camera.vec, 
               camera.x0 - camera.x, 
               camera.y0 - camera.y, 
               camera.z0 - camera.z);
}


    
/*
 * Define the viewpoint.
 */
void
view_from(x0, y0, z0)
    double x0, y0, z0;
{
    camera.x0 = x0;
    camera.y0 = y0;
    camera.z0 = z0;
    view_vec_eval();
}


/*
 * Define the point that we are looking at.
 */
void
view_at(x, y, z)
    double x, y, z;
{
    camera.x = x;
    camera.y = y;
    camera.z = z;
    view_vec_eval();
}


/*
 * Define the "up" direction of the view (well, rather the y-z plane).
 */
void
view_up(x, y, z)
    double x, y, z;
{
    MakeVector(camera.up, x, y, z);
}


/*
 * Set the focal ratio for the "camera".
 */
void
view_focal(ratio)
    double ratio;
{
    camera.focal_ratio = ratio;
}


/*
 * Set all viewpoint parameters in one call.
 */
void
viewpoint(x0, y0, z0, x, y, z, ux, uy, uz, ratio)
    double x0, y0, z0, x, y, z, ux, uy, uz, ratio;
{
    camera.x0 = x0;
    camera.y0 = y0;
    camera.z0 = z0;
    camera.x = x;
    camera.y = y;
    camera.z = z;
    MakeVector(camera.up, ux, uy, uz);
    camera.focal_ratio = ratio;
    view_vec_eval();
}



/*
 * Build a transformation matrix for transformation
 * into view coordinates.
 */
void
get_view_transf(view_mat)
    Transf_mat *view_mat;
{
    Vector tmp;
    double transl[3];
    double vy, vz;
    int i, j;
    
    /*
     * First we need a translation so the origo
     * of the view coordinate system is placed
     * in the viewpoint.
     */
    transl[0] = -camera.x0;
    transl[1] = -camera.y0;
    transl[2] = -camera.z0;

    /*
     * Then we need a rotation that makes the
     * up-vector point up, and alignes the sightline
     * with the z-axis.
     * This code might seem magic but the algebra behind
     * it can be found in Jim Blinn's Corner in IEEE CG&A July 1988
     */
    VecCopy(tmp, camera.vec);
    VecNegate(tmp);
    vecnorm(&tmp);
    vecnorm(&camera.up);
    vz = VecDot(tmp, camera.up);
    if ((vz * vz) > 1.0) {        /* this should not happen, but... */
        vz = 1.0;
    } else {
        vy = sqrt(1.0 - vz * vz);
        if (vy == 0.0) {          /* oops, the world collapses... */
            vy = 1.0e10;
            vz = 1.0;
        } else {
            vy = 1.0 / vy;
        }
    }

    view_mat->mat[0][2] = tmp.x;
    view_mat->mat[1][2] = tmp.y;
    view_mat->mat[2][2] = tmp.z;

    VecScalMul(tmp, vz, tmp);
    VecSub(tmp, camera.up, tmp);
    view_mat->mat[0][1] = tmp.x * vy;
    view_mat->mat[1][1] = tmp.y * vy;
    view_mat->mat[2][1] = tmp.z * vy;

    view_mat->mat[0][0] = (view_mat->mat[1][1] * view_mat->mat[2][2] 
                           - view_mat->mat[1][2] * view_mat->mat[2][1]);
    view_mat->mat[1][0] = (view_mat->mat[2][1] * view_mat->mat[0][2] 
                           - view_mat->mat[2][2] * view_mat->mat[0][1]);
    view_mat->mat[2][0] = (view_mat->mat[0][1] * view_mat->mat[1][2] 
                           - view_mat->mat[0][2] * view_mat->mat[1][1]);

    /*
     * Install the translation into the matrix.
     * Note that it is PRE-multiplied into the matrix.
     */
    for (i = 0; i < 3; i++) {
        view_mat->mat[3][i] = 0.0;
        for (j = 0; j < 3; j++) {
            view_mat->mat[3][i] += transl[j] * view_mat->mat[j][i];
        }
    }

    /*
     * Since the screen coordinates are defined in a left handed
     * coordinate system, we must switch the sign of the first
     * column in the matrix to get appropriate signs of x-values.
     */
    view_mat->mat[0][0] = -view_mat->mat[0][0];
    view_mat->mat[1][0] = -view_mat->mat[1][0];
    view_mat->mat[2][0] = -view_mat->mat[2][0];
    view_mat->mat[3][0] = -view_mat->mat[3][0];

     
    /*
     * Define the perspective transformation
     * using heuristic values for hither and yon.
     */
    hither = VecLen(camera.vec) / ZCLIPF;
    yon    = VecLen(camera.vec) * ZCLIPF;
}
