/* sony.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */

/* init_video:  initialize Sony disk
   init_frame:  go to first frame to record on
   record_frame:  record frame and do simulated preroll if desired
   stop_recording:  leave record mode
   shutdown_rec:  noop
   send_to_recorder:  send command to Sony videodisk
   sony_playback:  play back sequence of frames
   sony_clear:  clear Sony status
   sony_search:  search to specified frame
   sony_blank_search:  search for first unrecorded frame    */


#include <stdio.h>
#include <bios.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

/* The following are commands controlling the Sony videodisk.
   Accompanying explanations are quoted from the Interface Manual. */

#define COMPLETION 0x1		/* Complete a specified operation */
#define NOT_TARGET 0x5		/* Target frame not found */
#define NO_FRAME_NUMBER 0x6	/* Target frame illegal */
#define	ACK 0xa			/* Effective command received */
#define NAK 0xb			/* Command execution impossible */
#define F_STEP 0x3d		/* Step forward play */
#define SONY_ENTER 0x40		/* End of parameter input */
#define SEARCH 0x43		/* Locate target address */
#define REPEAT 0x44		/* Repeat play a designated sequence */
#define STILL 0x4f		/* Still picture */
#define FRAME_NUM_MODE 0x55	/* Set to frame number mode */
#define ALL_CLEAR 0x56		/* All clear */
#define ADDR_INQ 0x60		/* Current address inquiry */
#define BLANK_AREA_SEARCH 0xbb	/* Search for recordable areas of disk */
#define CURRENT_BLANK_INQ 0xbc	/* Request for the start and end points
				   of the currently selected blank area */
#define REC_STANDBY 0xe0	/* Recording standby */
#define ASSEMBLE_REC 0xe8	/* Continuous recording on a series of
				   frames */
#define RECORD 0xe9		/* Execution of recording */
#define COLOR_MODE 0xec		/* Color mode */
#define BLANK_SEARCH_ENABLE 0xee	/* Overwrite recording prohibit */
#define BLANK_SEARCH_DISABLE 0xef	/* Entire disc recording permit */

/* end commands */

#define FRAMES 1L	/* # frames to record per picture */
#define NORMAL 30	/* normal frames per second, watching */
#define FASTEST 2	/* 2x normal as fastest speed */
#define TIME 2		/* seconds you get to rewatch in */
#define FREQUENCY 20	/* how often to rewatch */
#define PREROLL		/* rewatch at all */
#define NEXTFRAM	(in_at+1)
#define DISTANCE	NORMAL*TIME*FASTEST

long frame_zero ;	/* first frame to record on */
int recording;		/* recording enabled */

int init_video(long *);
int init_frame(long *, long);
int record_frame(long, long);
int stop_recording(void);
int shutdown_rec(void);
void send_to_recorder(char *);
void sony_playback(int, int, long);
int sony_clear(void);
void sony_search(long);
long sony_blank_search(void);




/* initialize Sony videodisk */

int
init_video(optflag)

long *optflag ;

{
    unsigned int com1_status ;

        /* server equipped with optical disk */
    *optflag = 1 ;
	/* recording is on */
    recording = 1;

    com1_status = _bios_serialcom(_COM_STATUS,0,0) ;
	/* check that on-line */
    if (!(com1_status & 0x80))
    {
        fprintf (stderr,"couldn't communicate with videodisk\n") ;
        fprintf (stderr,"try setting Sony on-line switch to remote\n") ;
        exit(0) ;
    }
    sony_clear () ;
    com1_status = _bios_serialcom(_COM_SEND,0,COLOR_MODE) ;
    while ((com1_status != ACK) && (com1_status != NAK)) 
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
    return (1) ;
}




/* go to first frame to record on */
/* This will probably not work if the disk is fragmented into
   several blank areas. */

int
init_frame (start_frame,total)

long *start_frame ;
long total ;

{
    unsigned int com1_status = 0 ;
    unsigned char in_num[5] ;
    unsigned char end_num[5] ;
    long frame_end ;
    long i ;
 
	/* search for first unrecorded area */
    sony_blank_search() ;
    com1_status = _bios_serialcom(_COM_SEND,0,CURRENT_BLANK_INQ) ;
	/* get start of unrecorded area */
    for (i = 0 ; i < 5 ; i++)
    {
        while ((com1_status = _bios_serialcom(_COM_RECEIVE,0,0)) >= 256) ;
        in_num[i] = (unsigned char) com1_status ;
    }
	/* get finish of unrecorded area */
    for (i = 0 ; i < 5 ; i++)
    {
        while ((com1_status = _bios_serialcom(_COM_RECEIVE,0,0)) >= 256) ;
        end_num[i] = (unsigned char) com1_status ;
    }
    frame_zero = atol(in_num) ;
    frame_end = atol(end_num) ;

	/* search to start of unrecorded area */
    sony_search(frame_zero) ;
    com1_status = _bios_serialcom(_COM_SEND,0,ASSEMBLE_REC) ;
    com1_status = 0 ;
    while (com1_status != ACK)
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
    printf ("ACK ASSEMBLE_REC\n") ;
#endif

    *start_frame = frame_zero ;
}




/* record frame */

int
record_frame (in_at,out_at)

long in_at, out_at ;	/* starting, ending frame numbers */

{
    unsigned char tmp[20];
    unsigned int com1_status = 0 ; 
    unsigned long temp_frame ;
    unsigned long i ;

#ifdef PREROLL
        /* if time to play back recorded frames, put
	   into rec_standby mode */
    if(((in_at - frame_zero) % FREQUENCY) == 0)
    {
        com1_status = _bios_serialcom(_COM_SEND,0,REC_STANDBY) ;
        com1_status = 0 ;
        while (com1_status != ACK)
            com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
        printf ("ACK REC_STANDBY\n") ;
#endif
        com1_status = 0 ;
        while (com1_status != COMPLETION)
            com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
        printf ("COMPLETION REC_STANDBY\n") ;
#endif
    }
#endif
    sprintf(tmp, "%05ld%c%05ld%c%c", in_at, SONY_ENTER, in_at+1, SONY_ENTER, SONY_ENTER);
    send_to_recorder(tmp);
    com1_status = 0 ;
    while (com1_status != COMPLETION)
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
    printf ("COMPLETION input of points\n") ;
#endif
    sprintf(tmp, "%c%c", RECORD,0) ;
    send_to_recorder(tmp) ;
    com1_status = 0 ;
    while (com1_status != COMPLETION)
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef PREROLL
        /* search to start of preroll area and play back */
    if(((NEXTFRAM - frame_zero) % FREQUENCY) == 0)
    {
        sony_clear () ;
        temp_frame = (((NEXTFRAM - DISTANCE) > frame_zero) ? (NEXTFRAM - DISTANCE) : frame_zero) ;
        sony_search(temp_frame) ;
        sprintf(tmp, "%c%05ld%c%c%d%c%d%c",REPEAT,in_at,F_STEP,SONY_ENTER,
                                   1,SONY_ENTER,1,SONY_ENTER) ;
        send_to_recorder(tmp) ;
        com1_status = 0 ;
        while (com1_status != COMPLETION)
            com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
        printf ("COMPLETION playback\n") ;
#endif
#endif
    }
    return(1);
}




/* leave record mode */

int
stop_recording()

{
   sony_clear() ;
}




/* noop */
int
shutdown_rec()

{
}




/* send command to Sony videodisk */

void send_to_recorder(string)

char *string;

{
    unsigned int length ;
    unsigned int i ;
    unsigned int com1_status ;

    length = strlen(string) ;
    for (i = 0 ; i < length ; i++)
    {
        com1_status = _bios_serialcom(_COM_SEND,0,(unsigned int)string[i]) ;
	while (com1_status != ACK)
            com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
    }
}




/* play back sequence of frames */

void
sony_playback (mode,speed,end_frame)

int mode ;
int speed ;
long end_frame ;

{
    int com1_status ;
    char tmp[20] ;
    long i ;
    int repeats = 1 ;

    sony_clear () ;
    sprintf(tmp, "%c%05ld%c%c%d%c%d%c",REPEAT,end_frame,F_STEP,SONY_ENTER,
                                       repeats,SONY_ENTER,speed,SONY_ENTER) ;
    send_to_recorder(tmp) ;
    com1_status = 0 ;
    while (com1_status != COMPLETION)
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
    printf ("COMPLETION playback\n") ;
#endif
}




/* clear Sony status */

int
sony_clear ()

{
    unsigned int com1_status ;

    com1_status = _bios_serialcom(_COM_SEND,0,ALL_CLEAR) ;
    while ((com1_status != ACK) && !(com1_status & 0x8000))
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
    if (com1_status & 0x8000)
    {
        fprintf (stderr,"couldn't communicate with videodisk\n") ;
        fprintf (stderr,"try setting Sony on-line switch to remote\n") ;
        exit(0) ;
    }
}




/* search to specified frame */

void
sony_search (frame)

long frame ;

{
    char tmp[20] ;
    int com1_status ;

    sprintf(tmp, "%c%c%05ld%c", SEARCH, FRAME_NUM_MODE, frame, SONY_ENTER);
#ifdef DEBUG
    printf ("tmp = %s\n",tmp) ;
#endif
    send_to_recorder(tmp);
    while ((com1_status != COMPLETION) && (com1_status != NO_FRAME_NUMBER) &&
           (com1_status != NOT_TARGET))
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
#ifdef DEBUG
    printf ("search COMPLETION\n") ;
#endif
}




/* search for first unrecorded frame */

long
sony_blank_search()

{
    unsigned int com1_status ;

    com1_status = _bios_serialcom(_COM_SEND,0,BLANK_AREA_SEARCH) ;
    while ((com1_status != ACK) && (com1_status != NAK)) 
    {
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
    }
#ifdef DEBUG
    if (com1_status == NAK)
    {
        printf ("NAK NAK\n") ;
    }
    else
        printf ("ACK\n") ;
#endif
    while (com1_status != COMPLETION)
    {
        com1_status = _bios_serialcom(_COM_RECEIVE,0,0) ;
    }
#ifdef DEBUG
    printf ("made it past BLANK_AREA_SEARCH COMPLETION\n") ;
#endif
}
