/* send.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */


#include <stdio.h>
#include <math.h>
#include <scry_image.h>
#include <scry_client.h>
#include <scry_limits.h>


/* scry_send_image:  optionally compress, transmit, and optionally record
		     an image
   sc_send_frame:    transmit image
   xdr_image_info:   XDR encode image information such as height, etc. */


struct image_stuff S_image_info ;		/* image transfer information */

int S_mapnum ;			/* number of entries in color map */
unsigned char S_map[S_MAX_COL_SIZE][3] ;	/* RGB color map */
int S_maxcol = S_MAX_COL_DISPLAY - 1 ;	/* default maximum number of entries */
int S_orig_image_total ;			/* total bytes in original image */




/* optionally compress, transmit, and optionally record an image */

scry_send_image (rawimage,comp_option,height,width,depth)

unsigned char *rawimage ;
int comp_option ;	/* compression option */
int height ;
int width ;
int depth ;		/* color depth in bytes */

{
    unsigned char *sc_conv_vis() ;	/* convert to server resolution */
    unsigned char *sc_compress() ;	/* compress image */
    unsigned char *visimage ;
    int new_ht ;	/* new resolution */
    int new_wd ;
    int new_depth ;
    unsigned char *compdata ;

    if ((sizeof(short) != 2) && (depth == 2))
    {
	fprintf (stderr,"scry_send_image:  transmitting or saving 2-byte depth\n") ;
	fprintf (stderr,"images will not work on this architecture\n") ;
	scry_close() ;
	exit(0) ;
    }
    if (rawimage == NULL)
    {
	fprintf (stderr,"scry_send_image:  space for image has not been allocated\n") ;
	scry_close() ;
	exit(0) ;
    }
    if ((comp_option > (S_CCC | S_QUANTIZE | S_LEMPEL_ZIV)) || (comp_option < 0))
    {
	fprintf (stderr,"scry_send_image:  %d is illegal compression option\n",comp_option) ;
	scry_close() ;
	exit(0) ;
    }
    if ((height < 1) || (height > S_MAX_IMAGE_HEIGHT))
    {
	fprintf (stderr,"scry_send_image:  height of %d is illegal\n",height) ;
	scry_close() ;
	exit(0) ;
    }
    if ((width < 1) || (width > S_MAX_IMAGE_WIDTH))
    {
	fprintf (stderr,"scry_send_image:  width of %d is illegal\n",width) ;
	scry_close() ;
	exit(0) ;
    }
    if ((depth < S_MIN_COLOR_DEPTH) || (depth > S_MAX_COLOR_DEPTH))
    {
	fprintf (stderr,"scry_send_image:  color depth in bytes of %d is illegal\n",depth) ;
	scry_close() ;
	exit(0) ;
    }
    sc_get_info() ;	/* get server display information */
    sc_set_compress(comp_option) ;	/* set compression option */
	/* convert image to server spatial (and color resolution,
	   unless this would result in a larger image) */
    visimage = sc_conv_vis (rawimage,height,width,depth,&new_ht,&new_wd,&new_depth) ;
	/* if some form of compression is desired */
    if (S_image_info.compression != S_NONE)
    {
	compdata = sc_compress (visimage,new_ht,new_wd,new_depth) ;
        free(visimage) ;
	   /* transmit image */
	sc_send_frame (compdata,new_ht,new_wd,new_depth) ;
	free(compdata) ;
    }
    else
    {
	   /* transmit image without compression */
	sc_send_frame (visimage,new_ht,new_wd,new_depth) ;
	free(visimage) ;
    }
}




/* transmit image */

sc_send_frame (image_data,height,width,depth)

unsigned char *image_data ;
int height ;
int width ;
int depth ;

{
    enum clnt_stat err ;	/* error status of RPC call */
    bool_t xdr_image_info() ;	/* XDR encode image information */

	/* if recording, update current frame number on videodisk */
    if (S_image_info.s_recordable != S_PREVIEW)
        sc_set_frame (1) ;
    if (S_image_info.compression == S_NONE)
        S_image_info.total = height * width * depth ;
    S_image_info.data = image_data ;
    S_image_info.c_height = height ;
    S_image_info.c_width = width ;
    S_image_info.c_depth = depth ;
    if ((depth == 1) && !(S_image_info.compression & S_CCC))
        S_image_info.compression |= S_QUANTIZE ;
    if (S_image_info.compression == S_QUANTIZE)
        S_image_info.total = height * width ;
#ifdef MESSAGES
    if (!(S_image_info.compression & S_LEMPEL_ZIV))
    {
        if (S_image_info.total != S_orig_image_total)
            fprintf(stderr,"compression ratio: original %d bytes / final %d bytes = %lf\n",
		      S_orig_image_total,S_image_info.total,
                      (double) S_orig_image_total / (double) S_image_info.total) ;
    }
    else
    {
        fprintf(stderr,"compression ratio: original %d bytes/ final %d bytes = %lf\n",
		      S_orig_image_total,S_image_info.total_lz,
                      (double) S_orig_image_total / (double) S_image_info.total_lz) ;
    }
#endif
		
    S_during_rpc = 1 ;
        /* send image info to server, where it will be displayed
	       and optionally recorded */
    if ((int) (err = clnt_call(S_client,S_DISPLAYPROC,xdr_image_info,&S_image_info,
                         xdr_void,(char *) 0,S_total_timeout)) != 0)
    {
        clnt_perrno ((int) err) ;
        fprintf (stderr,"can't make S_DISPLAYPROC call\n") ;
        clnt_destroy (S_client) ;
        exit (-1) ;
    }
    S_during_rpc = 0 ;
	/* update current frame if saving Anima file */
    ++S_image_info.current_frame_to_save ; 
    if (S_make_closing_call)
	scry_abn_exit() ;
}




/* XDR encode information for server to be able to decompress and
   record */

bool_t 
xdr_image_info (xdrsp,image_info)

XDR *xdrsp ;			/* XDR handle */
struct image_stuff *image_info ;	/* frame buffer and control information */

{
    int i ;
    int totpixs = image_info->c_height * image_info->c_width ;
    unsigned short *data_ptr ;
    int total = 0 ;		/* total bytes to send */
    unsigned short net_to_pc() ;	/* convert to PC byte order */

	    /* if saving Anima file on server, current frame number */
    if (!xdr_int(xdrsp,&image_info->current_frame_to_save))
	return (0) ;
	    /* if saving Anima file on server, total frames to save */
    if (!xdr_int(xdrsp,&image_info->total_frames))
	return (0) ;
	    /* compression type */
    if (!xdr_int(xdrsp,&image_info->compression))
	return (0) ;
    total = image_info->total ;
        /* total bytes in (compressed) image */
    if (!xdr_int(xdrsp,&image_info->total))
	return (0) ;
	    /* image height */
    if (!xdr_int(xdrsp,&image_info->c_height))
	return (0) ;
	    /* image width */
    if (!xdr_int(xdrsp,&image_info->c_width))
	return (0) ;
	    /* image color resolution in bytes */
    if (!xdr_int(xdrsp,&image_info->c_depth))
	return (0) ;
    if (image_info->compression & S_LEMPEL_ZIV)
                /* total bytes after further (Lempel-Ziv) compression */
        if (!xdr_int(xdrsp,&image_info->total_lz))
	    return (0) ;
    if (!xdr_int(xdrsp,&(image_info->in_at)))  /* start recording at */
	    return (0) ;
    if (!xdr_int(xdrsp,&(image_info->out_at))) /* finish recording at */
	    return (0) ;
    if (image_info->compression & S_LEMPEL_ZIV)
                /* total number of bytes if Lempel-Ziv compression was used */
        total = image_info->total_lz ;
    else
        total = image_info->total;
    if ((image_info->compression != S_NONE) && (image_info->compression != S_LEMPEL_ZIV))
    {
	    /* number of entries in color map */
	if (!xdr_int(xdrsp,&(S_mapnum)))
            return (0) ;
		/* RGB color map */
        if (!xdr_opaque(xdrsp,S_map,S_mapnum*3))
	    return (0) ;
    }
    if ((image_info->compression == S_NONE) && (image_info->c_depth == 2))
    {
	    /* if short format is in network byte order, convert to
	       PC byte order */
	if (htons(1000) == 1000)
	{
	    data_ptr = (unsigned short *) image_info->data ;
	    for (i = 0 ; i < totpixs ; i++, data_ptr++)
	        *data_ptr = net_to_pc(*data_ptr) ;
	}
    }
	/* transfer compressed image */
    if (!xdr_opaque(xdrsp,image_info->data,total))
        return (0) ;
    return (1) ;
}




/* if necessary, convert short into PC byte order so the slow PC won't have
   to do it */

unsigned short 
net_to_pc (var)

short var ;

{
    register unsigned char byte0 ;
    register unsigned char byte1 ;

    byte0 = ((var >> 8) & 0xff) ;
    byte1 = (var & 0xff) ;
    var = 0 ;
    var |= byte1 ;
    var <<= 8 ;
    var |= byte0 ;
    return (var) ;
}
