/*

    Copyright (c) 1990 University of Southern California

	Paul Raveling
	USC Information Sciences Institute
	4676 Admiralty Way
	Marina del Rey, California 90292


	The authors grant without fee permission to use, copy, modify,
	distribute, and sell this software for any purpose, provided:

	--  The above copyright notice appears in all copies.

	--  This disclaimer appears in all source code copies.

	--  The names of ISI and USC are not used in advertising or
	    publicity  pertaining  to distribution  of the software
	    without prior specific written permission from ISI.

	ISI provides this software freely as a public service.  It is
	NOT a commercial product,  and therefore is not subject to an
	an implied  warranty  of  merchantability  or  fitness  for a
	particular purpose.  ISI provides it as is, without warranty.

	This software  is furnished  only on the basis that any party
	who receives it indemnifies  and holds harmless  the original
	authors and their sponsors  against any claims,  demands,  or
	liabilities  connected  with  using  it,   furnishing  it  to
	others, or providing it to a third party.


	The author  of this software is  Paul Raveling.   Development
	used the author's after-work hours and ISI's facilities.

	Thanks are due to ISI for allowing  such extracurricular work
	and  to Hewlett Packard,   whose  University  Grants  program
	provided  the 9000/370  workstation  and  supporting software
	which was the platform for this work.
*/

/*0
==============================================================================
======									======
======			      Module Quantize				======
======			   Quantize Color Images			======
======									======
==============================================================================

  Function:

        This module has been modified by David Robertson of
	Lawrence Berkeley Laboratory to act as an internal
	subprogram of the Scry 1.2 client.  Modifications are
	noted by the strings "MODIFIED" and "end MODIFIED"
	in the code.

	This module contains one subprogram to quantize 24-bit
	RGB format images images.


  Private subprograms:

	quantize		Quantize 24-bit RGB image
	img_rep_node		Replenish Free Node Supply
	classify_rgb		Classify RGB pixels
	classify_mapped		Deleted
	thread_by_freq		Rethread by frequency
	synthesize_tree		Synthesize Color Tree
	assign_colors_rgb	Assign Colors to RGB Image
	assign_colors_mapped	Deleted
	find_isi_color		Find nearest color to color
				found during BTC prepass


  Module history:

	1-28-91		David Robertson
	Modified V1.3 of quantize.c to work with Scry 1.2.
	The code for dithering is still included, but has not worked
	after the modifications I made.  Some code from
	imgcredel.c is included.

	6-15-90		Paul Raveling	V1.3
	1.  Add dithering to color assignment.  Looks great so far!!!

	2.  Fixed bug in synthesize_tree, identified by Paul Close:
	    Two checks for special cases were in wrong order, could
	    sometimes produce access to a NULL pointer.

	5-11-90		Paul Raveling	V1.3
	Fixed bug in quantizing color-mapped images:  Having an initial
	color map with unused colors (frequency 0) could cause division
	by 0.

	3-7-90		Paul Raveling
	Finalized for version 1.2

	12-12-89	Paul Raveling
	Created this module for development of new quantization algorithm,
	won't log individual changes until it's up.

*/


#include <sys/types.h>
#include <stdio.h>
/* MODIFIED */
#include "imglib.h"
#include <scry_image.h>
#include <scry_limits.h>
/* end MODIFIED */

#define	reg	register




#define	HIRES_PREQUANT		/*  Use high resolution prequantization	*/

/* MODIFIED:  dithering code kept but doesn't work due to modifications
   made */
#ifdef NOT_SCRY
#define	DITHER	1		/*  1 => dither quantized image		*/
#endif
/* end MODIFIED */

/*	Distance measure to use between colors:		*/

#define	DX_0	1		/*  Distance biased for log intensity	*/
#define	DX_1	0		/*  Distance biased for intensity	*/
#define	DX_2	0		/*  Standard Euclidean 3-space distance	*/
#define	DX_3	0		/*  Distance heavily biased for itensity*/

/*	Timing:		*/

#define	TIME_OVERALL	0	/*  Report time from start to finish	*/
#define	TIME_PHASES	0	/*  Time phases				*/



#if	DITHER

/*--------------------------------------------------------------*/
/*	Data related to dithering, defined in module imgcvt	*/
/*--------------------------------------------------------------*/
/*	Tables used to accelerate processing for dithering	*/
/*	... plus one for grayscale conversion			*/
/*	Shared with other modules for other dithering		*/
/*--------------------------------------------------------------*/

extern	Rgbpix	 img_rgbbw[2] /* = {0x00000000, 0x00ffffff} */;
				/*  RGB values for black and white	*/


/*
	Data below are initialized on 1st invocation
	of a dithering procedure
*/

extern	Byte	 img_dither_tables_uninitialized;
extern	void	 img_init_dither_tables ();

extern	Rgbpix	 img_grayrgb[256];	/*  Gray value to RGB pixel	*/
					/*	Domain = [0,255]	*/
					/*	Range  = [0,255] for each
						RGB component, R=G=B	*/

extern	short	 img_errtab[256];	/*  Error in current pixel	*/
					/*	Domain = [0,255]	*/
					/*	Range  = [-127,+127]	*/

	/*   Tables below are slightly larger than required	*/
	/*   to suit their actual domains; this just makes	*/
	/*   things easier to think about as powers of 2	*/

extern	Byte	 img_cliptab[768];	/*  Clipped pixel values	*/
					/*	Domain = [-255,+512]	*/
					/*	Range  = [0,255]	*/

extern	int	 img_errw1[512];	/*  1/16 of error + 256		*/
					/*	Domain = [-255,+255]	*/

extern	int	 img_errw3[512];	/*  3/16 of error + 256		*/
					/*	Domain = [-255,+255]	*/

extern	int	 img_errw5[512];	/*  5/16 of error + 256		*/
					/*	Domain = [-255,+255]	*/

extern	int	 img_errw7[512];	/*  7/16 of error + 256		*/
					/*	Domain = [-255,+255]	*/
#endif


#ifdef	HIRES_PREQUANT
/*	Color pointer array parameters and access functions	HIRES	*/
/*	for high resolution prequantization:			HIRES	*/

#define	CPARES		6	/*  Color pointer array's
					component resolution:  n bits	*/
#define	CPA_ENTRIES	262144	/*  Number of CPA entries		*/

#define	DETAIL_ENTRIES	64	/*  Number of CPA detail map entries	*/

/*::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Look up color in color pointer array	*/
/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define	lookup_color_ptr(r,g,b) \
  (&cpa[((r&0xfc)<<10) | ((g&0xfc)<<4) | (b>>2)])

/*::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Look up color in detail color ptr array	*/
/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define lookup_detail(dcpa,r,g,b) \
  (&dcpa[((r&0x03)<<4) | ((g&0x03)<<2) | (b&0x03)])

#else
/*	Color pointer array parameters and access functions	LORES	*/
/*	for low resolution prequantization:			LORES	*/

#define	CPARES		5	/*  Color pointer array's
					component resolution:  n bits	*/
#define	CPA_ENTRIES	32768	/*  Number of CPA entries		*/

/*	No detail entries for low resolution prequantization	*/

/*::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Look up color in color pointer array	*/
/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define	lookup_color_ptr(r,g,b) \
  (&cpa[((r&0xf8)<<7) | ((g&0xf8)<<2) | (b>>3)])
#endif



/*-------------------------------------------------*/
/*	Node structure for threaded color tree	   */
/*-------------------------------------------------*/

/* MODIFIED - put in imglib.h */
#ifdef NOT_SCRY
typedef struct	imnode
	{
	struct imnode	*parent;	/*  Pointer to parent node	*/
	struct imnode	*off;		/*  Pointer to 1st offspring	*/
	struct imnode	*sibling;	/*  Pointer to next sibling	*/
	struct imnode	*thread;	/*  Linear thread		*/
	struct imnode  **cpa_entry;	/*  Ptr to color's cpa entry	*/
	struct imnode  **detail;	/*  Ptr to detail subcpa	*/
	Byte		 r;		/*  Mean red   component	*/
	Byte		 g;		/*  Mean green component	*/
	Byte		 b;		/*  Mean blue  component	*/
	Byte		 cmi;		/*  Color map index		*/
	Word		 f;		/*  Frequency (pixel count)	*/
	Word		 sumr;		/*  Sum of red components	*/
	Word		 sumg;		/*  Sum of green components	*/
	Word		 sumb;		/*  Sum of blue components	*/
	} Ctnode;
#endif
/* end MODIFIED */
				/*  Specialized NULL pointer values:	*/




/*::::::::::::::::::::::::::::::::::::::::::::::*/
/*      Macro to get a new node with all        */
/*      fields reset to 0                       */
/*::::::::::::::::::::::::::::::::::::::::::::::*/


#define get_node(np)    \
 { \
 if ( img_num_free-- > 0 )	np = img_next_node++; \
 else				np = img_rep_node (); \
 }




/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Macros to compute distance between colors	*/
/*	biased for square of log "intensity"		*/
/*							*/
/*	dx = sqrt(dr**2 + dg**2 + db**2)		*/
/*							*/
/*		dr = |log(r1)**2 - log(r2)**2|		*/
/*		dg = |log(g1)**2 - log(g2)**2|		*/
/*		dr = |log(b1)**2 - log(b2)**2|		*/
/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

#if DX_0
#define	dx(r1,g1,b1,r2,g2,b2) \
 sqdiv3tab[ (Word) dsq(r1,r2) \
	  + (Word) dsq(g1,g2) \
	  + (Word) dsq(b1,b2) ]

#define	dsq(n1,n2) \
 ( n1 >= n2 \
	? logsqtab[n1] - logsqtab[n2] \
	: logsqtab[n2] - logsqtab[n1] )
#endif	/*  DX_0	*/




/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Macros to compute distance between colors	*/
/*	biased for intensity:				*/
/*							*/
/*	dx = sqrt(dr**2 + dg**2 + db**2)		*/
/*							*/
/*		dr = |r1**2 - r2**2|			*/
/*		dg = |g1**2 - g2**2|			*/
/*		dr = |b1**2 - b2**2|			*/
/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

#if DX_1
#define	dx(r1,g1,b1,r2,g2,b2) \
 sqdiv3tab[ (Word) dsq(r1,r2) \
	  + (Word) dsq(g1,g2) \
	  + (Word) dsq(b1,b2) ]

#define	dsq(n1,n2) \
 ( n1 >= n2 \
	? exp2tab[n1] - exp2tab[n2] \
	: exp2tab[n2] - exp2tab[n1] )
#endif	/*  DX_1	*/




		/*::::::::::::::::::::::::::::::::::::::::::::::*/
		/*	Standard Euclidean 3-space distance	*/
		/*						*/
#if DX_2	/*  dx = sqrt ( (dr**2 + dg**2 + db**2)/3 )	*/
		/*						*/
		/*		dr = (r1 - r2)			*/
		/*		dg = (g1 - g2)			*/
		/*		db = (b1 - b2)			*/
		/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define	dx(r1,g1,b1,r2,g2,b2) \
 sqdiv3tab[ (Word) exp2tab[dsq(r1,r2)] \
	  + (Word) exp2tab[dsq(g1,g2)] \
	  + (Word) exp2tab[dsq(b1,b2)] ]

#define	dsq(n1,n2) \
 ( n1 >= n2 \
	? n1 - n2 \
	: n2 - n1 )
#endif	/* DX_2 */


		/*::::::::::::::::::::::::::::::::::::::::::::::*/
		/*	Distance heavily biased for intensity	*/
		/*						*/
#if DX_3	/*  dx = sqrt ( (dr**2 + dg**2 + db**2)/3 )	*/
		/*						*/
		/*		dr = (r1 - r2)**2		*/
		/*		dg = (g1 - g2)**2		*/
		/*		db = (b1 - b2)**2		*/
		/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define	dx(r1,g1,b1,r2,g2,b2) \
 sqdiv3tab[ (Word) exp2tab[dsq(r1,r2)] \
	  + (Word) exp2tab[dsq(g1,g2)] \
	  + (Word) exp2tab[dsq(b1,b2)] ]

#define	dsq(n1,n2) \
 ( n1 >= n2 \
	? exp2tab[n1 - n2] \
	: exp2tab[n2 - n1] )
#endif	/* DX_2 */



/* ++++ temp -- to revert to copy in imgcolor.c  ++++ */
	/*----------------------------------------------*/
	/*  Memory suballocation data for color trees	*/
	/*	--  Shared with color quantization	*/
	/*----------------------------------------------*/

 struct	 mem_block			/*  Memory block from malloc	*/
	 {
	 struct mem_block  *next_block;		/*  Link to next block	*/
	 Ctnode		    node_space[1000];	/*  Room for 1000 nodes	*/
	 };
 struct mem_block  *img_block_queue;	/*  Queue of malloc'd blocks	*/
 struct mem_block  *img_new_block;	/*  Pointer to new block	*/
	Ctnode	*img_next_node;		/*  Pointer to next free byte	*/
	int	 img_num_free;		/*  Remaining free space, as
						number of nodes		*/

/*----------------------------------------------------------------------*/
/*	Tables used to compute non-Euclidean distances in RGB space	*/
/*----------------------------------------------------------------------*/

/*
	exp2tab -- 	Function:	n**2
			Domain:		[0,255] representing [0.0,1.0]
			Range:		[0,255] representing [0.0,1.0]
*/


static	Byte	 exp2tab[256] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 
	4, 4, 5, 5, 5, 5, 6, 6, 6, 7, 7, 7, 8, 8, 8, 9, 
	9, 9, 10, 10, 11, 11, 11, 12, 12, 13, 13, 14, 14, 15, 15, 16, 
	16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 23, 23, 24, 24, 
	25, 26, 26, 27, 28, 28, 29, 30, 30, 31, 32, 32, 33, 34, 35, 35, 
	36, 37, 38, 38, 39, 40, 41, 42, 42, 43, 44, 45, 46, 47, 47, 48, 
	49, 50, 51, 52, 53, 54, 55, 56, 56, 57, 58, 59, 60, 61, 62, 63, 
	64, 65, 66, 67, 68, 69, 70, 71, 73, 74, 75, 76, 77, 78, 79, 80, 
	81, 82, 84, 85, 86, 87, 88, 89, 91, 92, 93, 94, 95, 97, 98, 99, 
	100, 102, 103, 104, 105, 107, 108, 109, 111, 112, 113, 115, 116, 117, 119, 120, 
	121, 123, 124, 126, 127, 128, 130, 131, 133, 134, 136, 137, 139, 140, 142, 143, 
	145, 146, 148, 149, 151, 152, 154, 155, 157, 158, 160, 162, 163, 165, 166, 168, 
	170, 171, 173, 175, 176, 178, 180, 181, 183, 185, 186, 188, 190, 192, 193, 195, 
	197, 199, 200, 202, 204, 206, 207, 209, 211, 213, 215, 217, 218, 220, 222, 224, 
	226, 228, 230, 232, 233, 235, 237, 239, 241, 243, 245, 247, 249, 251, 253, 255 };


/*
	logsqtab -- 	Function:	log (n*(e-1) + 1) ** 2
			Domain:		[0,255] representing [0.0,1.0]
			Range:		[0,255] representing [0.0,1.0]
*/


static	Byte	 logsqtab[256] = {
	0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 2, 2, 2, 2, 
	3, 3, 3, 4, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 
	10, 10, 11, 11, 12, 13, 13, 14, 15, 15, 16, 16, 17, 18, 19, 19, 
	20, 21, 21, 22, 23, 24, 25, 25, 26, 27, 28, 29, 29, 30, 31, 32, 
	33, 34, 35, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 45, 46, 
	47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 
	63, 64, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 80, 
	81, 82, 83, 84, 85, 86, 87, 88, 90, 91, 92, 93, 94, 95, 96, 97, 
	99, 100, 101, 102, 103, 104, 106, 107, 108, 109, 110, 111, 113, 114, 115, 116, 
	117, 118, 120, 121, 122, 123, 124, 126, 127, 128, 129, 130, 132, 133, 134, 135, 
	136, 138, 139, 140, 141, 142, 144, 145, 146, 147, 149, 150, 151, 152, 153, 155, 
	156, 157, 158, 160, 161, 162, 163, 165, 166, 167, 168, 170, 171, 172, 173, 174, 
	176, 177, 178, 179, 181, 182, 183, 184, 186, 187, 188, 189, 191, 192, 193, 194, 
	196, 197, 198, 199, 201, 202, 203, 205, 206, 207, 208, 210, 211, 212, 213, 215, 
	216, 217, 218, 220, 221, 222, 223, 225, 226, 227, 228, 230, 231, 232, 234, 235, 
	236, 237, 239, 240, 241, 242, 244, 245, 246, 247, 249, 250, 251, 252, 254, 255 };





/*
	sqdiv3tab -- 	Function:	sqrt(n/3)
			Domain:		[0,768] representing [0.0,3.0]
			Range:		[0,255] representing [0.0,1.0]
*/


static	Byte	 sqdiv3tab[768] = {
	0, 9, 13, 16, 18, 21, 23, 24, 26, 28, 29, 31, 32, 33, 34, 36, 
	37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 50, 51, 
	52, 53, 54, 54, 55, 56, 57, 57, 58, 59, 60, 60, 61, 62, 62, 63, 
	64, 64, 65, 66, 66, 67, 68, 68, 69, 70, 70, 71, 71, 72, 72, 73, 
	74, 74, 75, 75, 76, 76, 77, 78, 78, 79, 79, 80, 80, 81, 81, 82, 
	82, 83, 83, 84, 84, 85, 85, 86, 86, 87, 87, 88, 88, 89, 89, 90, 
	90, 91, 91, 92, 92, 93, 93, 93, 94, 94, 95, 95, 96, 96, 97, 97, 
	97, 98, 98, 99, 99, 100, 100, 100, 101, 101, 102, 102, 103, 103, 103, 104, 
	104, 105, 105, 105, 106, 106, 107, 107, 107, 108, 108, 109, 109, 109, 110, 110, 
	110, 111, 111, 112, 112, 112, 113, 113, 114, 114, 114, 115, 115, 115, 116, 116, 
	116, 117, 117, 118, 118, 118, 119, 119, 119, 120, 120, 120, 121, 121, 121, 122, 
	122, 122, 123, 123, 124, 124, 124, 125, 125, 125, 126, 126, 126, 127, 127, 127, 
	128, 128, 128, 129, 129, 129, 130, 130, 130, 131, 131, 131, 131, 132, 132, 132, 
	133, 133, 133, 134, 134, 134, 135, 135, 135, 136, 136, 136, 137, 137, 137, 137, 
	138, 138, 138, 139, 139, 139, 140, 140, 140, 141, 141, 141, 141, 142, 142, 142, 
	143, 143, 143, 144, 144, 144, 144, 145, 145, 145, 146, 146, 146, 146, 147, 147, 
	147, 148, 148, 148, 148, 149, 149, 149, 150, 150, 150, 150, 151, 151, 151, 152, 
	152, 152, 152, 153, 153, 153, 154, 154, 154, 154, 155, 155, 155, 155, 156, 156, 
	156, 157, 157, 157, 157, 158, 158, 158, 158, 159, 159, 159, 159, 160, 160, 160, 
	161, 161, 161, 161, 162, 162, 162, 162, 163, 163, 163, 163, 164, 164, 164, 164, 
	165, 165, 165, 165, 166, 166, 166, 166, 167, 167, 167, 168, 168, 168, 168, 169, 
	169, 169, 169, 170, 170, 170, 170, 171, 171, 171, 171, 172, 172, 172, 172, 172, 
	173, 173, 173, 173, 174, 174, 174, 174, 175, 175, 175, 175, 176, 176, 176, 176, 
	177, 177, 177, 177, 178, 178, 178, 178, 179, 179, 179, 179, 179, 180, 180, 180, 
	180, 181, 181, 181, 181, 182, 182, 182, 182, 183, 183, 183, 183, 183, 184, 184, 
	184, 184, 185, 185, 185, 185, 186, 186, 186, 186, 186, 187, 187, 187, 187, 188, 
	188, 188, 188, 188, 189, 189, 189, 189, 190, 190, 190, 190, 190, 191, 191, 191, 
	191, 192, 192, 192, 192, 192, 193, 193, 193, 193, 194, 194, 194, 194, 194, 195, 
	195, 195, 195, 196, 196, 196, 196, 196, 197, 197, 197, 197, 197, 198, 198, 198, 
	198, 199, 199, 199, 199, 199, 200, 200, 200, 200, 200, 201, 201, 201, 201, 202, 
	202, 202, 202, 202, 203, 203, 203, 203, 203, 204, 204, 204, 204, 204, 205, 205, 
	205, 205, 205, 206, 206, 206, 206, 206, 207, 207, 207, 207, 208, 208, 208, 208, 
	208, 209, 209, 209, 209, 209, 210, 210, 210, 210, 210, 211, 211, 211, 211, 211, 
	212, 212, 212, 212, 212, 213, 213, 213, 213, 213, 214, 214, 214, 214, 214, 215, 
	215, 215, 215, 215, 216, 216, 216, 216, 216, 217, 217, 217, 217, 217, 217, 218, 
	218, 218, 218, 218, 219, 219, 219, 219, 219, 220, 220, 220, 220, 220, 221, 221, 
	221, 221, 221, 222, 222, 222, 222, 222, 222, 223, 223, 223, 223, 223, 224, 224, 
	224, 224, 224, 225, 225, 225, 225, 225, 226, 226, 226, 226, 226, 226, 227, 227, 
	227, 227, 227, 228, 228, 228, 228, 228, 229, 229, 229, 229, 229, 229, 230, 230, 
	230, 230, 230, 231, 231, 231, 231, 231, 231, 232, 232, 232, 232, 232, 233, 233, 
	233, 233, 233, 233, 234, 234, 234, 234, 234, 235, 235, 235, 235, 235, 235, 236, 
	236, 236, 236, 236, 237, 237, 237, 237, 237, 237, 238, 238, 238, 238, 238, 238, 
	239, 239, 239, 239, 239, 240, 240, 240, 240, 240, 240, 241, 241, 241, 241, 241, 
	242, 242, 242, 242, 242, 242, 243, 243, 243, 243, 243, 243, 244, 244, 244, 244, 
	244, 244, 245, 245, 245, 245, 245, 246, 246, 246, 246, 246, 246, 247, 247, 247, 
	247, 247, 247, 248, 248, 248, 248, 248, 248, 249, 249, 249, 249, 249, 249, 250, 
	250, 250, 250, 250, 250, 251, 251, 251, 251, 251, 251, 252, 252, 252, 252, 252, 
	252, 253, 253, 253, 253, 253, 253, 254, 254, 254, 254, 254, 254, 255, 255, 255 };


#ifdef	DEBUG

dump_tree ( root, depth )

	Ctnode	*root;
	int	 depth;

{
	int	 i;

#define indent {for (i=0; i < depth; ++i) printf ( "    " );}


indent printf("(%3u %3u %3u)    %3u    %u\n",
	root -> r, root -> g, root -> b, root -> cmi, root -> f );

if ( root->off )
   dump_tree ( root->off, depth+1 );
if ( root->sibling  )
   dump_tree ( root->sibling, depth );
}

dump_thread ( root )

	Ctnode	*root;

{
	Ctnode	*np;

np = root;
while ( np )
    {
    printf("(%3u %3u %3u)    %3u    %u\n",
	np -> r, np -> g, np -> b, np -> cmi, np -> f );
    np = np -> thread;
    }
}

#endif	/* DEBUG */

/*2
******************************************************************************
******									******
******		   img_rep_node  --  Replenish Free Node Supply		******
******									******
******************************************************************************

  Invocation:

	new_node = img_rep_mode ()

	[Invoked only from get_node macro]


  Function:

	Allocate memory for a new supply of node structures
	and preset all fields to 0.



  Input:

	None from caller; img_rep_node uses shared data structures
	img_block_queue, img_new_block, img_next_node, and img_num_free

  Output:

	new_node	Ctnode	*
	<return value>	Pointer to a new color tree node

	Data structures named as input are updated.


   Notes:

	Blocks are normally allocated by the get_node macro --
	get_node(np) set node pointer np to addr of new node.
	Before use, caller must initialize memory block data as follows:

img_block_queue = NULL;		\*  Init memory block queue as empty	*\
img_num_free    = 0;		\*  Indicate no free blocks		*\

	After the color tree is no longer needed, space should be freed
	with this code fragment:

while ( img_block_queue )
      {
      img_new_block = img_block_queue -> next_block;
      free ( img_block_queue );
      img_block_queue = img_new_block;
      }
*/


Ctnode	*img_rep_node ()

{
reg	Ctnode	*cur;
reg	int	 i, n;
reg	Word	*wp;



/*------------------------------------------*/
/*	Allocate memory for a new node	    */
/*------------------------------------------*/

img_new_block = (struct mem_block *) malloc (sizeof(struct mem_block));
				/*  Link new block at head of block queue    */
if ( img_new_block == NULL )
   return NULL;

wp = (Word *) img_new_block;
n = sizeof (struct mem_block) / sizeof (Word);
for ( i=0; i < n; ++ i )		/*  Clear all node space to 0	*/
    *wp++ = 0;

img_new_block -> next_block = img_block_queue;
img_block_queue = img_new_block;
img_num_free = 999;		/*  Suballocate first node-sized piece	     */
cur = img_new_block -> node_space;	/*  while finishing block init	     */
img_next_node = &(img_new_block -> node_space[1]);

return cur;
}
/* ++++ end temp -- to revert to copy in imgcolor.c  ++++ */


/*2
******************************************************************************
******									******
******		     classify_rgb  --  Classify RGB pixels		******
******									******
******************************************************************************

  Invocation:

	node1 = classify_rgb ( image, cpa )


  Function:

	Classify RGB pixels into the given color pointer array.

	Algorithm:

	For each pixel, form an index into cpa, the color pointer
	array.  The index is formed by masking out low order bits
	of each RGB component and concatenating the reduced-precision
	components.

	If the indexed cpa entry is NULL
	   then allocate and initialize a Ctnode (color tree node)
		structure for the color.
	   else update the indexed color's existing Ctnode structure.

	Information maintained in Ctnode structures is:

	  thread pointer	Set to link Ctnodes for all
				indexed colors into a queue.

	  color frequency	Count of instances of this indexed color

	  sum of RGB values	Component sums, used later to compute mean
				actual color represented by indexed color

	  subordinate freq's	Frequencies of subordinate colors, may
				be used later to select any of the individual
				colors that were classified into a single
				indexed color

  Input:

	image		Img *
			Pointer to subject image

	cpa		Ctnode *[CPA_ENTRIES]
			Array of pointers to color nodes; at entry
			to classify_rgb, all node pointers must be NULL.


  Output:

	node1		Ctnode *
	<return value>	Pointer to first color node  (head of thread);
			NULL => insufficient memory to allocate Ctnodes.
*/


Ctnode	*classify_rgb ( image, cpa )

	Img	*image;			/*  Subject image		*/
	Ctnode **cpa;			/*  Color pointer array		*/

{
	Ctnode	*node1;			/*  Pointer to 1st node		*/
reg	int	 i;			/*  Pixel index			*/
/* MODIFIED */
reg	unsigned char	*pp;		/*  Pixel pointer		*/
/* end MODIFIED */
reg	Ctnode	*np;			/*  Node pointer		*/
reg	Ctnode	*pnp;			/*  Previous new-node pointer	*/
reg	Ctnode **cp;			/*  Pointer to node pointer	*/

#ifdef	HIRES_PREQUANT
	Ctnode	*detail_head;		/*  Head of detail thread	*/
	Ctnode	*detail_tail;		/*  Tail of detail thread	*/
	int	 npq;			/*  Number of prequantized colors  */
#endif


#if TIME_PHASES
printf ( "\nClassification:\n" );
Start_Timing
#endif


/*------------------------------*/
/*	Preset first node	*/
/*------------------------------*/

/*	This avoids a need for per-pixel special case logic	*/

i   = image -> num_pixels;		/*	Count of pixels		*/
/* MODIFIED */
pp  = (unsigned char *) image -> pixdata;	/* Pointer to 1st pixel	*/
/* end MODIFIED */

    {					/*  For 1st pixel...	*/
    reg Word	 r, g, b;	/*  Color components			*/


/* MODIFIED */
    r = (Word) *pp++;
    g = (Word) *pp++ ;  /*  Get RGB values	*/
    b = (Word) *pp++;
    -- i;
/* end MODIFIED */

    cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
    np = *cp;				/*  Get node pointer, if any	*/
    get_node(np);			/*  Get node preset to 0's	*/
    *cp = np;				/*  Set cpa pointer to it	*/
    node1 = np;				/*  Save ptr to 1st node	*/
    pnp   = np;				/*  Save ptr to prev new node	*/

    np -> cpa_entry = cp;		/*  Save ptr to cpa entry	*/
    np -> sumr = r;
    np -> sumg = g;			/*  Only set sums for now	*/
    np -> sumb = b;
    np -> f    = 1;			/*  Frequency = 1 pixel		*/
    }					/*  For 1st pixel...	*/


#ifdef	HIRES_PREQUANT	/*  For high resolution prequantization...	*/

/*------------------------------------------------------*/
/*	Classify pixels, retaining a detail node	*/
/*	for each color until number of nodes in		*/
/*	prequantized thread is at least 256		*/
/*------------------------------------------------------*/

#define	newdcpa(ptr)	{ \
  reg int	 ii; \
  reg Ctnode **pp; \
  ptr = (Ctnode **) calloc ( DETAIL_ENTRIES, sizeof(Ctnode **) ); \
  if (ptr == NULL) return NULL; \
  pp = ptr; ii = DETAIL_ENTRIES; \
  while ( ii-- > 0 ) *pp++ = NULL; }


get_node(detail_head);			/*  Set up 1st detail node	*/
*detail_head = *node1;			/*  ... copy initial node	*/
detail_tail  = detail_head;		/*  Detail tail = detail head	*/
npq          = 1;			/*  Init prequant'd color count	*/
newdcpa(node1->detail)			/*  Get detail cpa for node1	*/
cp  = lookup_detail(node1->detail,node1->sumr,node1->sumg,node1->sumb);
*cp = detail_head;			/*  Set 1st node pointer in 1st	*/
detail_head->cpa_entry = cp;		/*	detail color ptr array	*/

/*	i  = image -> num_pixels - 1;	*/
/*	pp = image -> pixdata	 + 1;	*/

while ( (i > 0) && (npq <= 256) )
    {					/*  For each pixel...	*/
    reg Word	 r, g, b;	/*  Color components			*/
    reg Ctnode *dnp;		/*  New detail node			*/
    reg Ctnode **dp;		/*  Detail cpa entry			*/

    -- i;
/* MODIFIED */
    r = (Word) *pp++;
    g = (Word) *pp++ ;  /*  Get RGB values	*/
    b = (Word) *pp++;
/* end MODIFIED */

    cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
    np = *cp;				/*  Get node pointer, if any	*/
    if ( np == NULL )
	{		/*  Initialize new node	*/
	get_node(np);			/*  Get node preset to 0's	*/
	*cp = np;			/*  Set cpa pointer to it	*/
	pnp -> thread = np;		/*  Thread prev node to new one	*/
	pnp = np;			/*  Make new one "prev new"	*/

	np -> cpa_entry = cp;		/*  Save ptr to cpa entry	*/
	np -> sumr = r;
	np -> sumg = g;			/*  Only set sums for now	*/
	np -> sumb = b;
	np -> f = 1;			/*  Frequency = 1 pixel		*/
	++ npq;

	get_node(dnp);			/*  Get node for detail thread	*/
	*dnp = *np;			/*  Copy from prequantized thread  */
	detail_tail->thread = dnp;	/*  Append to detail thread	*/
	detail_tail	    = dnp;
	newdcpa(np->detail)		/*  Get detail cpa for node	*/
	cp  = lookup_detail(np->detail,r,g,b);  /*  Set 1st node ptr in	*/
	*cp = dnp;			/*  new detail color ptr array	*/
	dnp->cpa_entry = cp;
	}		/*  Initialize new node	*/
    else
	{		/*  Count known color	*/
	np -> sumr += r;
	np -> sumg += g;		/*  Accumulate component sums	*/
	np -> sumb += b;
	++ np -> f;			/*  Increment frequency		*/

	/*  Create or update detail node	*/

	cp = lookup_detail(np->detail,r,g,b);
	dnp = *cp;
	if (dnp)
	   {
	   dnp -> sumr += r;
	   dnp -> sumg += g;		/*  Accumulate component sums	*/
	   dnp -> sumb += b;
	   ++ dnp -> f;			/*  Increment frequency		*/
	   }
	else
	   {
	   get_node(dnp);
	   *cp  = dnp;
	   *dnp = *np;
	   dnp -> cpa_entry = cp;
	   dnp -> detail = NULL;
	   dnp -> sumr	 = r;
	   dnp -> sumg	 = g;
	   dnp -> sumb	 = b;
	   dnp -> f	 = 1;
	   ++ npq;
	   detail_tail -> thread = dnp;
	   detail_tail = dnp;
	   }

	}		/*  Count known color	*/
    }					/*  For each pixel...	*/

if ( i == 0 )				/*  If all pixels were processed  */
   return detail_head;			/*	use detail thread	*/
else					/*  Else			*/
   {					/*	discard detail links	*/
   np = node1;
   while (np)
	{
/* MODIFIED */
        free(np->detail) ;
/* end MODIFIED */
	np -> detail = NULL;
	np = np -> thread;
	}
   }
#endif	/* High resolution prequantization */

/*--------------------------------------------------------------*/
/*	Classify each pixel, retaining only prequantized colors	*/
/*--------------------------------------------------------------*/

/*	i  = image -> num_pixels - 1;	*/
/*	pp = image -> pixdata	 + 1;	*/

while ( i-- > 0 )
    {					/*  For each pixel...	*/
    reg Word	 r, g, b;	/*  Color components			*/


/* MODIFIED */
    r = (Word) *pp++;
    g = (Word) *pp++ ;  /*  Get RGB values	*/
    b = (Word) *pp++;
    -- i;
/* end MODIFIED */

    cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
    np = *cp;				/*  Get node pointer, if any	*/
    if ( np == NULL )
	{		/*  Initialize new node	*/
	get_node(np);			/*  Get node preset to 0's	*/
	*cp = np;			/*  Set cpa pointer to it	*/
	pnp -> thread = np;		/*  Thread prev node to new one	*/
	pnp = np;			/*  Make new one "prev new"	*/

	np -> cpa_entry = cp;		/*  Save ptr to cpa entry	*/
	np -> sumr = r;
	np -> sumg = g;			/*  Only set sums for now	*/
	np -> sumb = b;
	np -> f = 1;			/*  Frequency = 1 pixel		*/
	}		/*  Initialize new node	*/
    else
	{		/*  Count known color	*/
	np -> sumr += r;
	np -> sumg += g;		/*  Accumulate component sums	*/
	np -> sumb += b;
	++ np -> f;			/*  Increment frequency		*/
	}		/*  Count known color	*/
    }					/*  For each pixel...	*/

#if TIME_PHASES
End_Timing	Report_Timing
#endif

return node1;				/*  Return ptr to node thread	*/
}

/* MODIFIED */
/* classify_mapped deleted */
/* end MODIFIED */

/*2
******************************************************************************
******									******
******		     thread_by_freq  --  Rethread by frequency		******
******									******
******************************************************************************

  Invocation:

	count = thread_by_freq ( thread_head )


  Function:

	Rethread a queue of color tree nodes in descending order
	by color frequency.

	While doing this, set each node's mean color based on
	its component sums and total pixel count.


  Input:

	thread_head	Ctnode **
			Pointer to head of thread.  Head of thread
			is a pointer to the 1st Ctnode in the thread


  Output:

	count		Word
	<return value>	Number of indexed colors in thread

	*thread_head	Ctnode *
			Head of thread, updated to point to Ctnode
			with highest color frequency.

*/


Word	thread_by_freq ( thread_head )

	Ctnode **thread_head;	/*  Pointer to thread's head pointer	*/

{
#define	FTSIZE	1000			/*  Size of frequency table	*/

reg	Word	 count;			/*  Count of nodes on thread	*/
reg	Ctnode	*np;			/*  Node pointer		*/
reg	Ctnode	*pnp;			/*  Previous-node pointer	*/
reg	Ctnode	*nnp;			/*  Next-node pointer		*/

reg	Word	 f;			/*  Frequency, or index into ft	*/
reg	Word	 i;			/*  Index for searching ft	*/
	Ctnode	*ft[FTSIZE];		/*  Frequency table		*/
/*
	Frequency table is a table of pointers to the last node
	in the new thread that has a given frequency,
	for frequencies < 64.
*/



#if TIME_PHASES
printf ( "Thread by frequency:\n" );
Start_Timing
#endif


/*------------------------------------------------------*/
/*	Initialize frequency table and new thread	*/
/*------------------------------------------------------*/

for ( f=0; f < FTSIZE; ++f )	/*  Initialize empty frequency table	*/
    ft[f] = NULL;

np = *thread_head;		/*  Get pointer to 1st node, old thread	*/
f  = np -> f;			/*  Get its frequency (pixel count)	*/

np -> r = np -> sumr / f;
np -> g = np -> sumg / f;	/*  Set mean color for 1st node		*/
np -> b = np -> sumb / f;

if ( f < FTSIZE )		/*  If its frequency is low enough	*/
   ft[f] = np;			/*	set a frequency table entry	*/
nnp = np -> thread;
np -> thread = 0;		/*  Terminate new thread		*/
np  = nnp;			/*  Advance to 2nd node on old thread	*/
count = 1;			/*  Initial node count is 1		*/




/*----------------------------------------------------------*/
/*	Scan oldthread from head to tail, recollating	    */
/*	each node into new thread ordered by frequency	    */
/*----------------------------------------------------------*/

while ( np )			/*  For each color in old thread...	*/
    {
    ++ count;			/*  Count a node			*/
    f = np -> f;		/*  Get color's frequency		*/
    nnp = np -> thread;		/*  Save ptr to next old-thread node	*/

    /*++++++++++++++++++++++++++++++++++*/
    /*	Derive node's mean color	*/
    /*++++++++++++++++++++++++++++++++++*/

    switch (f)			/*  Accelerate frequent special cases	*/
	{			/*  by avoiding division		*/
	case 1:		np -> r = np -> sumr;
			np -> g = np -> sumg;
			np -> b = np -> sumb;		break;

	case 2:		np -> r = np -> sumr >> 1;
			np -> g = np -> sumg >> 1;
			np -> b = np -> sumb >> 1;	break;

	case 4:		np -> r = np -> sumr >> 2;
			np -> g = np -> sumg >> 2;
			np -> b = np -> sumb >> 2;	break;

	default:	np -> r = np -> sumr / f;
			np -> g = np -> sumg / f;
			np -> b = np -> sumb / f;
	}

    /*++++++++++++++++++++++++++++++++++*/
    /*	Insert this node in new thread	*/
    /*++++++++++++++++++++++++++++++++++*/

    if ( f < FTSIZE )		/*  If frequency is low,		*/
	{			/*	check for known predecessor	*/
	pnp = ft[f];
	if ( pnp )
	   {
	   np  -> thread = pnp -> thread;
	   pnp -> thread = np;
	   ft[f] = np;
	   np = nnp;		/*  Advance to next on old thread	*/
	   continue;
	   }
	}

    /*
	Scan from head of new thread to place high-frequency color
	or color that's been encountered for first time.
    */

    pnp = *thread_head;
    if ( f > pnp -> f )		/*  Is frequency > 1st?		*/
	{			/*	Yes - insert @ head	*/
	np -> thread = *thread_head;
	*thread_head = np;
	}
    else
	{			/*	No -- collate node	*/
	while ( pnp -> thread )
	      {
	      if ( f > (pnp->thread) -> f )
		 break;
	      pnp = pnp -> thread;
	      }
	np  -> thread = pnp -> thread;
	pnp -> thread = np;
	}

    if ( f < FTSIZE )		/*  If 1st occurrence of low freq color	*/
	ft[f] = np;		/*	save pointer for future refs	*/
    np = nnp;			/*  Find next node on old thread	*/
    }

#if TIME_PHASES
End_Timing	Report_Timing
#endif

return count;
}

/*2
******************************************************************************
******									******
******		    synthesize_tree  --  Synthesize Color Tree		******
******									******
******************************************************************************

  Invocation:

	root = synthesize_tree ( thread_head, max_colors )


  Function:

	Synthesize color tree from thread of nodes, queued in
	descending order by frequency.  Rethread nodes to ease
	breadth-first scanning of the tree.


  Input:

	thread_head	Ctnode *
			Head of thread (pointer to 1st Ctnode)

	max_colors	int
			Maximum number of colors to assign;
			a synonym for maximum number of threaded
			nodes to place in tree.

  Output:

	root		Ctnode *
	<return value>	Pointer to root node of tree
*/

Ctnode	*synthesize_tree ( thread_head, max_colors )

	Ctnode	*thread_head;		/*  Head of subject thread	*/
	int	 max_colors;		/*  Maximum # colors to keep	*/

{
reg	Word	 r, g, b;		/*  Color components		*/
reg	Word	 dbn;			/*  Distance between nodes	*/
reg	Word	 mindist;		/*  Distance to nearest node	*/
reg	Word	 capdis;		/*  Current capture distance	*/
reg	Ctnode	*np;			/*  Node pointer		*/
reg	Ctnode	*nearest_color;		/*  Nearest-color node pointer	*/
reg	Ctnode	*nnp_tree;		/*  Next-node pointer		*/
reg	Ctnode	*pnp_tree;		/*  Previous-node pointer	*/
reg	Ctnode	*src_head;		/*  Ptr to head, source thread	*/
	Ctnode	*src_tail;		/*  Ptr to tail, source thread	*/
	Ctnode	*nxt_head;		/*  Ptr to head, next src thrd	*/
reg	Ctnode	*nxt_tail;		/*  Ptr to tail, next src thrd	*/
	Ctnode	*dst_head;		/*  Ptr to head, destination thread */
	Ctnode	*dst_tail;		/*  Ptr to tail, destination thread */

	Ctnode	*last_dp;	/*  Ptr to last node with dubious parentage */
				/*	on source thread for next level     */
	Ctnode	*lev_1st;		/*  Ptr to 1st node on level	    */
	Ctnode	*lev_last;		/*  Ptr to last node on level	    */

	int	 node_count;		/*  Number of threaded nodes in tree */



#if TIME_PHASES
printf ( "\nSynthesize tree:\n" );
Start_Timing
#endif

/*--------------------------------------*/
/*	Initialize root of tree		*/
/*--------------------------------------*/

np   = thread_head;		/*  Make node with highest frequency	*/
dst_head     = np;		/*    the root node			*/
dst_tail     = np;
src_head     = np -> thread;	/*  Locate next node on source thread	*/
np -> thread = NULL;		/*  Terminate destination thread	*/

lev_1st	     = np;		/*  Set pointers to 1st and last	*/
lev_last     = np;		/*  node on level both to root node	*/

node_count   = 1;		/*  1 node exists in initial tree	*/
capdis	     = 256;		/*  1st node captures everything	*/

nxt_tail = np;			/*  Trace to tail of thread and		*/
np	 = src_head;		/*  Mark entire source thread as	*/
nxt_head = src_head;		/*  as offspring of root node		*/
while ( np )
    {
    nxt_tail = np;
    np -> parent = dst_head;
    np = np -> thread;
    }


/*--------------------------------------------------*/
/*	Build successive deeper levels of tree	    */
/*--------------------------------------------------*/

capdis /= 2;			/*  Set capture radius for level	*/
				/*	below root node			*/

while ((node_count < max_colors) && nxt_head )
    {				/*  Build next deeper level	*/

    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
    /*	Generate copy of each node on previous level		*/
    /*  as head of the node's offspring queue on new level	*/
    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

    /*
	Motive:  Search for nearby colors is confined to offspring
	of parent.  Unless 1st offspring is the parent color,
	the first new offspring may be chromaticaly VERY close
	to its parent.  This avoids the highly probable case
	of assigning a nearly redundant color as the first offspring,
	virtually wasting many colormap entries.
    */

#define	copy_down { \
	get_node(nnp_tree) \
	nnp_tree -> r = np -> r; \
	nnp_tree -> g = np -> g; \
	nnp_tree -> b = np -> b; \
	nnp_tree -> f = 0;	/*  Mark with f=0 to NOT	*/ \
	nnp_tree -> cmi = 0;	/*  generate redundant color	*/ \
	nnp_tree -> cpa_entry = np -> cpa_entry; \
	nnp_tree -> parent = np; \
	np -> off = nnp_tree; \
	dst_tail -> thread = nnp_tree; \
	dst_tail  = nnp_tree; }


    lev_last = dst_tail;	/*  Last on level is old dst tail	*/
    np	     = lev_1st;		/*  Start at 1st node of level		*/
    copy_down			/*  Copy 1st node down to new level	*/
    lev_1st  = nnp_tree;	/*  Reset ptr to 1st node on new level	*/

    while ( np != lev_last )	/*  Repeat until last node is copied	*/
	{
	np = np -> thread;
	copy_down
	}


    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
    /*  Locate each node of source thread to build tree		*/
    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

    src_head = nxt_head;
    src_tail = nxt_tail;
    nxt_head = NULL;
    nxt_tail = NULL;
    last_dp  = NULL;

    np = src_head;		/*  Point to 1st source node	*/
    if ( np == NULL )		/*  If no source node exists	*/
	break;			/*	quit PDQ		*/

    while ( np )		/*  For each node on source thread...	*/
	{				/*  For nodes in thread...	*/
	src_head = np -> thread;	/*  Save pointer to next node	*/
	np -> thread = NULL;		/*  Node will end some
						destination thread	*/
	r = np -> r;
	g = np -> g;		/*  Get node's color components	*/
	b = np -> b;

	nnp_tree = (np -> parent) -> off;	/*  If parent node	*/

	/*	Find nearest sibling (offspring of parent)	*/

	mindist	= 256;
	while ( nnp_tree )
	    {
	    dbn = dx(r,g,b,nnp_tree->r,nnp_tree->g,nnp_tree->b);

	    if  ( dbn < mindist )
		{
		mindist = dbn;
		nearest_color = nnp_tree;
		}

	    pnp_tree = nnp_tree;
	    nnp_tree = nnp_tree -> sibling;
	    }

	if ( mindist < capdis )		/*  If captured by a sibling...	*/
	   {				/*  Set on next src thread	*/
	   np -> cmi = mindist;		/*  Save distance to parent	*/
	   np -> parent  = nearest_color;	/*  Define parent.	*/
	   if ( nxt_tail )
	      {
	      nxt_tail -> thread = np;
	      nxt_tail = np;
	      }
	   else
	      {
	      nxt_head = np;
	      nxt_tail = np;
	      }
	   }
	else				/*  else not captured...	*/
	   {
	   pnp_tree -> sibling = np;	/*  Append to sibling queue	*/
	   dst_tail -> thread = np;	/*  Append to destination thread*/
	   dst_tail = np;
	   lev_last = np;
	   last_dp  = nxt_tail;
	   if ( ++ node_count >= max_colors )
	      break;;
	   }

	np = src_head;
	}				/*  For nodes in thread...	*/


    capdis /= 2;		/*  Set new capture radius	*/


    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
    /*	Reparent any nodes on next source thread that are	*/
    /*	closer to a more recently classified parent		*/
    /*	than to their original parent				*/
    /*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

    if ( last_dp == NULL )	/*  Skip this if nothing happened	*/
	continue;		/*	on building last level of tree	*/

    /*	Skip this step for 1st level in tree below root;	*/
    /*  parent is guaranteed to be correct for this level	*/

    if ( nxt_head -> parent == dst_head )
	continue;

    last_dp = last_dp -> thread; /*  Make "last_dp" "1st non-dp"	*/

    np = nxt_head;
    while (np && (np != last_dp))/*  For each node up to last possible	*/
	{
	if ( np -> cmi > capdis ) /*  If near enough to surface		*/
	   {			/*	of parent's capture sphere	*/
	   /*	Recheck parentage	*/

	   mindist  = np -> cmi;
	   pnp_tree = np -> parent;
	   nnp_tree = (pnp_tree -> parent) -> off;
	   r = np -> r;
	   g = np -> g;
	   b = np -> b;

	   while ( nnp_tree )
		{
		if ( nnp_tree != pnp_tree )
		   {
		   dbn = dx(r,g,b,nnp_tree->r,nnp_tree->g,nnp_tree->b);
		   if ( dbn < mindist )
			{
			mindist		= dbn;
			np -> cmi	= dbn;
			np -> parent	= nnp_tree;
			pnp_tree	= nnp_tree;
			}
		   }
		nnp_tree = nnp_tree -> sibling;
		}
	   }

	np = np -> thread;	/*  ...next node to (possibly) reparent	*/
	}
    }				/*  Build next deeper level	*/


/*----------------------------------------------------------*/
/*	If tree reached maximum intended size		    */
/*	while nodes remain on source thread,		    */
/*	append these nodes to "next" source thread and	    */
/*	assign parentage to each such node.		    */
/*----------------------------------------------------------*/

if ( src_head )
   {				/*  Append src queue to nxt queue	*/
   if ( nxt_tail )		/*     [nxt_tail need not be reset]	*/
      nxt_tail -> thread = src_head;
   else
      nxt_head = src_head;

   np = src_head;
   while ( np )			/*  For each transferred node	*/
     {				/*	assign parentage	*/
     nnp_tree = (np->parent) -> off;
     if ( nnp_tree )		/*  If last parent has offspring	*/
	{			/*	find nearest sibling among them	*/
	r = np -> r;
	g = np -> g;
	b = np -> b;

	nearest_color = nnp_tree;
	mindist = dx(r,g,b,nnp_tree->r,nnp_tree->g,nnp_tree->b);
	nnp_tree = nnp_tree -> sibling;
	while ( nnp_tree )
	    {
	    dbn = dx(r,g,b,nnp_tree->r,nnp_tree->g,nnp_tree->b);
	    if  ( dbn < mindist )
		{
		mindist = dbn;
		nearest_color = nnp_tree;
		}
	    nnp_tree = nnp_tree -> sibling;
	    }

	np -> parent = nearest_color;
	}

     np = np -> thread;
     }				/*	assign parentage	*/
   }


/*----------------------------------------------------------*/
/*	Reset cpa entries for nodes not in tree...	    */
/*	Remap each to the parent node that captured it.	    */
/*----------------------------------------------------------*/

np = nxt_head;
while ( np )
    {
    *(np -> cpa_entry) = np -> parent;
    np = np -> thread;
    }


#if TIME_PHASES
End_Timing	Report_Timing
#endif

return dst_head;
}


/*2
******************************************************************************
******									******
******		assign_colors_rgb  --  Assign Colors to RGB Image	******
******									******
******************************************************************************

  Invocation:

	num_assigned = assign_colors_rgb (image, newimage, scry_map,
			   cpa, root, num_colors, btc);


  Function:

	Set new color map for image, then assign the selected colors
	to image pixels if a BTC prepass as part of CCC compression
	has not been performed.


  Input:

	image		Img *
			Pointer to subject image

        newimage	unsigned char *
			resulting image

 	scry_map	unsigned char[S_MAX_COL_SIZE][3]
			Scry color map

	cpa		Ctnode *[CPA_ENTRIES]
			Array of pointers to color nodes; updated
			so that entries for colors not retained
			point to colors that are retained.

	root		Ctnode *
			Root of color tree:  Pointer to 1st sibling
			at shallowest level

	num_colors	int
			Number of colors needed

        btc		int
			whether or not the image has been through
			a BTC prepass

  Output:

	num_assigned	int
	<return value>	Number of colors assigned;
			0 => insufficient memory to allocate colormap

*/

/* MODIFIED */
assign_colors_rgb (image, newimage, scry_map, cpa, root, num_colors, btc)

        Img     *image;                 /*  Pointer to subject image        */
        unsigned char *newimage ;
        unsigned char scry_map[S_MAX_COL_SIZE][3] ;	/* Scry color map */
	int btc ;	/* have done a BTC prepass as part of CCC algorithm */
/* end MODIFIED */
	Ctnode **cpa;			/*  Color pointer array		*/
	Ctnode	*root;			/*  Root pointer for color tree	*/

{
reg	int	 pi;			/*  Pixel index			*/
	int	 ci;			/*  Colormap index		*/
/* MODIFIED */
reg	unsigned char	*pp;		/*  Pixel pointer		*/
	unsigned char *newptr ;
/* end MODIFIED */
reg	Col	*cp;			/*  Color pointer		*/
reg	Ctnode	*np;			/*  Node pointer		*/
	Col	*map;			/*  New color map		*/

#if DITHER
	Rgbpix	*cslb;			/*  Current scanline buffer	*/
	Rgbpix	*nslb;			/*  Next scanline buffer	*/
	Rgbpix	*tslb;			/*  Temporary pointer		*/

reg	Rgbpix	*iocp;			/*  Ptr to current-line pixel	*/
reg	Rgbpix	*ionp;			/*  Ptr to next-line pixel	*/
reg	Rgbpix	*icp;			/*  Ptr to current-line pixel	*/
reg	Rgbpix	*inp;			/*  Ptr to next-line pixel	*/

	int	 x, y;			/*  X/Y coordinates		*/
	Col	*cme;			/*  Color map entry pointer	*/
	int	 rerr, gerr, berr;	/*  Error in one pixel		*/
#endif


#if TIME_PHASES
printf ( "\nAssign colors:\n" );
Start_Timing
#endif



/*--------------------------------------*/
/*	Set new color map for image	*/
/*--------------------------------------*/

#ifdef DEBUG
fprintf (stderr,"at start of assign_rgb %d %d %d\n",scry_map[0][0],scry_map[0][1],scry_map[0][2]) ;
#endif
map = (Col *) calloc (num_colors, sizeof(Col));	/*  Allocate new map	*/
if ( map == NULL )
   return 0;

/* MODIFIED: lines deleted */

image -> colormap = map;			/*  Supply new colormap	*/
image -> num_colors = num_colors;

/* MODIFIED: line deleted */

/*	Scan tree's thread to transfer colors from nodes to colormap	*/

np = root;
cp = map;
ci = 0;

while ( np )
    {
    if ( np -> f )			/*  If node defines a color	*/
	{
	cp -> red   = np -> r;
	cp -> green = np -> g;		/*  Set color in map entry	*/
	cp -> blue  = np -> b;
	cp -> freq  = np -> f;		/*  Set frequency in map entry	*/
	cp -> pixel = ci;		/*  Set color index in map ent	*/
	np -> cmi   = ci;		/*  Set color index in node	*/

	/* MODIFIED */
        scry_map[cp->pixel][S_RED] = cp->red ;
        scry_map[cp->pixel][S_GREEN] = cp->green ;
        scry_map[cp->pixel][S_BLUE] = cp->blue ;
	/* end MODIFIED */

	++ cp;				/*  Point to next map entry	*/
	++ ci;				/*  Increment colormap index	*/
	}
    else
        np -> cmi = (np->parent) -> cmi;

    np = np -> thread;			/*  Locate next color	*/
    }

/* MODIFIED */
if (!btc)
/* end MODIFIED */
{
#if	DITHER

/*--------------------------------------*/
/*	Initialize for dithering	*/
/*--------------------------------------*/

if ( img_dither_tables_uninitialized )	/*  If not initialized yet,	*/
   img_init_dither_tables ();		/*     initialize dither tables	*/

/*
	Allocate scanline buffers 2 pixels wider than line width
	to simplify error propagation logic
*/

cslb = (Rgbpix *) calloc ( image -> width + 2, sizeof(Rgbpix) );
nslb = (Rgbpix *) calloc ( image -> width + 2, sizeof(Rgbpix) );
if ( (cslb == NULL) || (nslb == NULL) )
   {
   fprintf (stderr, "dither_rgb:  Insufficient memory\n");
   if ( cslb != NULL ) free (cslb);
   return 0;
   }


/*
	Copy top line of image into 'next' scanline buffer,
	which will become the first 'cur' line processed.
*/

icp  = (Rgbpix *) image -> pixdata;
inp  = icp;
ionp = &nslb[1];
for ( x=0; x < image->width; ++x )
    *ionp++ = *inp++;



/*----------------------------------------------*/
/*	Assign dithered colors to pixels	*/
/*----------------------------------------------*/

/*	ionp & inp were initialized above	*/

for ( y=0; y < image->height; ++y )
    {
    /*  Switch next-line and current-line buffers	*/

    tslb = cslb;    cslb = nslb;    nslb = tslb;

    iocp = &cslb[1];		/*  Point to current-scanline buffer	*/
    ionp = &nslb[1];		/*  Point to next-scanline buffer	*/

    if  ( y < (image->height-1) )/*  If image has another scanline	*/
	{			/*	copy it into next- buffer	*/
	for ( x=0; x < image->width; ++x )
	    *ionp++ = *inp++;
	ionp = &nslb[1];
	}


    for ( x=0; x < image->width; ++x )
	{			/*  Translate each pixel		*/
	reg Word	 r, g, b;	/*  Color components		*/
	reg Rgbpix	 p;		/*  Pixel value			*/
	reg	Ctnode **cp;		/*  Pointer to node pointer	*/



	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
	/*	Get actual RGB color of current pixel		*/
	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

	p = *iocp & (pixel_red | pixel_green | pixel_blue);
	r = (p & pixel_red  ) >> pixel_red_shift;	/*  Get pixel's	*/
	g = (p & pixel_green) >> pixel_green_shift;	/*  color	*/
	b = (p & pixel_blue ) >> pixel_blue_shift;

	cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
	np = *cp;			/*  Get node pointer, if any	*/

#ifdef	HIRES_PREQUANT
	if ( np )			/*  If node exists		*/
	   if ( np -> detail )		/*     and has detail		*/
	      {				/*	look up detail entry	*/
	      cp = lookup_detail(np->detail,r,g,b);
	      np = *cp;
	      }
#endif

	if ( np == NULL )
	    {				/*  Find nearest defined color	*/
	    reg	Ctnode	*nnp;		/*  Nearest-node pointer	*/
	    reg	Word	 ndx;		/*  Nearest-node distance	*/
	    reg	Word	 cdx;		/*  Current-node distance	*/


	    np  = nnp = root;
	    ndx = dx(r,g,b,np->r,np->g,np->b);

	    while ( np && (ndx > 0) )
	        {
	        cdx = dx(r,g,b,np->r,np->g,np->b);
	        if ( cdx < ndx )
		   {
		   ndx = cdx;
		   nnp = np;
		   }
	        np = np -> thread;
	        }

	    np = nnp;
	    *cp = np;
	    }

	/*  np->cmi is index of pixel to use for this color	*/
	cme = &(image->colormap[np->cmi]);  /*  Get color's map entry ptr  */

	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
	/*	Set pixel tag to color map index for this color	*/
	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

	*icp   &= pixel_red | pixel_green | pixel_blue;
	*icp++ |= (np->cmi << pixel_tag_shift);


	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
	/*	Set error (difference between actual RGB	*/
	/*	and RGB in associated colormap entry)		*/
	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

	rerr = ((int) r) - ((int) cme -> red);
	gerr = ((int) g) - ((int) cme -> green);
	berr = ((int) b) - ((int) cme -> blue);


	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
	/*	Propagate error forward to neighboring pixels	*/
	/*::::::::::::::::::::::::::::::::::::::::::::::::::::::*/

#define	properr(pix,tab) { \
	r = (pix & pixel_red  ) >> pixel_red_shift; \
	g = (pix & pixel_green) >> pixel_green_shift; \
	b = (pix & pixel_blue ) >> pixel_blue_shift; \
	r = img_cliptab[((int) r) + tab[rerr+256]]; \
	g = img_cliptab[((int) g) + tab[gerr+256]]; \
	b = img_cliptab[((int) b) + tab[berr+256]]; \
	pix =	  (r << pixel_red_shift) \
		| (g << pixel_green_shift) \
		| (b << pixel_blue_shift); \
	}

	++ iocp;
	properr(*iocp,img_errw7)	/*  7/16 of error -> right	*/

	-- ionp;
	properr(*ionp,img_errw3)	/*  3/16 of error -> lower left	*/
	++ ionp;

	properr(*ionp,img_errw5)	/*  5/16 of error -> below	*/

	++ np;
	properr(*ionp,img_errw1)	/*  1/16 of error -> lower right*/


	/*  Logic above has advanced all pixel pointers:  icp, iocp, & ionp  */
	}
    }

free (cslb);
free (nslb);

#else	/*	Not dithered	*/




/*------------------------------------------------------*/
/*	Assign colors to pixels without dithering	*/
/*------------------------------------------------------*/

pi  = image -> num_pixels;
/* MODIFIED */
pp = (unsigned char *) image -> pixdata;
newptr = newimage ;
/* end MODIFIED */

while ( pi-- > 0 )
    {
    reg Word	 r, g, b;	/*  Color components	*/
    reg	Rgbpix	 p;		/*  Pixel value		*/
    reg	Ctnode **cp;		/*  Pointer to node pointer	*/

/* MODIFIED */
    r = (Word) *pp++;
    g = (Word) *pp++ ;  /*  Get RGB values	*/
    b = (Word) *pp++;
/* end MODIFIED */

    cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
    np = *cp;				/*  Get node pointer, if any	*/

#ifdef	HIRES_PREQUANT
    if ( np )				/*  If node exists		*/
       if ( np -> detail )		/*     and has detail		*/
	  {				/*	look up detail entry	*/
	  cp = lookup_detail(np->detail,r,g,b);
	  np = *cp;
	  }
#endif

    if ( np == NULL )
	{				/*  Find nearest defined color	*/
	reg	Ctnode	*nnp;		/*  Nearest-node pointer	*/
	reg	Word	 ndx;		/*  Nearest-node distance	*/
	reg	Word	 cdx;		/*  Current-node distance	*/


	np  = nnp = root;
	ndx = dx(r,g,b,np->r,np->g,np->b);

	while ( np && (ndx > 0) )
	    {
	    cdx = dx(r,g,b,np->r,np->g,np->b);
	    if ( cdx < ndx )
		{
		ndx = cdx;
		nnp = np;
		}
	    np = np -> thread;
	    }

	np = nnp;
	*cp = np;
	}

/* MODIFIED */
#ifdef NOT_SCRY
    p |= (np -> cmi) << pixel_tag_shift;
    *pp++ = p;
#endif
    *newptr++ = np->cmi ;
    } 	/* if !btc */
/* end MODIFIED */

#endif	/* dithered/not dithered	*/

}


if ( ci != num_colors )
   {
   image -> num_colors = ci;
   fprintf (stderr, "Warning:  Internal logic error in color quantization,\n");
   fprintf (stderr, "... found %d nodes to represent %d colors\n",
		ci, num_colors );
   }

#if TIME_PHASES
End_Timing	Report_Timing
#endif

#ifdef DEBUG
fprintf (stderr,"at end of assign_rgb %d %d %d\n",scry_map[0][0],scry_map[0][1],scry_map[0][2]) ;
#endif
return ci;			/*  Return final color count	*/
}

/* MODIFIED */
/* assign_colors_mapped deleted */
/* end MODIFIED */


/*1
******************************************************************************
******									******
******		      quantize  --  Quantize Color Image		******
******									******
******************************************************************************

  Invocation:

	actual_colors = quantize ( scry_image, comp_data, scry_map,
			    num_colors, height, width, btc )


  Function:


	Quantize colors for an image.  This selects a set of colors
	to represent an RGB image.

	Quantize generates a color map for the image.  Two cases are
	possible for the final compressed image.  In one, the
	original RGB pixels are replaced with indices into the new
	color map.
	
	In the other, only used with CCC compression,
	2 colors are first generated for each 4x4 block using
	BTC coding, and then the pixels in each block in the
	image are replaced by the two colors based on the pixel's
	luminance.  The resulting image is fed into the
	quantization algorithm.  After quantization is finished,
	the 2 colors in each block are replaced by their nearest
	colors in the color map.

	The actual number of colors in the quantized map can be
	less than the number requested if the input image contains
	relatively few colors.

  Input:

	scry_image	unsigned char *
			Pointer to image to quantize

	comp_data	unsigned char *
			resulting image

        scry_map	unsigned char[S_MAX_COL_SIZE][3]
			Scry color map

	height		int
			height of image

	width		int
			width of image

	btc		int
			whether or not the image has been through
			a BTC prepass

	num_colors	int
			Maximum number of colors to be retained
			in the image


  Output:

	actual_colors	int
	<return value>	Number of colors actually assigned:
			0 => Failed, insufficient memory, invalid
				image, or invalid number of colors requested
*/

/* MODIFIED */

int
quantize ( scry_image, compdata, scry_map, num_colors , height, width, btc)

unsigned char *scry_image;
unsigned char *compdata ;		/* compressed data */
unsigned char scry_map[S_MAX_COL_SIZE][3] ;	/* Scry color map */
int height, width ;
int btc ;	/* whether have done BTC prepass as part of CCC algorithm */
int num_colors;			/* Max # of colors to retain */

{
    Img *image ;
    Col *colormap ;
    Byte *asdata ;
    Ctnode **cpa;		/* Color pointer array */
    Ctnode *thread;		/* Head of frequency thread */
    Ctnode *root;		/* Root of color tree */
    Word indexed_color_count;	/* Number of indexed colors */

    reg	int i;			/* General purpose index */
    reg Ctnode *np;		/* Node pointer */
    Col *newmap;		/* Pointer to new color map */
    int actual_num_colors;	/* Actual # colors allocated */

/*::::::::::::::::::::::::::::::::::::::::::::::*/
/*	Macros to issue message and abort	*/
/*::::::::::::::::::::::::::::::::::::::::::::::*/

#define	abort(msg) \
	{fprintf(stderr, "Color quantization:  %s\n",msg); return 0;}

#define	abort_dpar(msg1,par,msg2) \
 {fprintf(stderr, "Color quantization:  %s%d%s\n",msg1,par,msg2); return 0;}

    colormap = NULL ;
    asdata = NULL ;
    img_block_queue = NULL;	/* Init memory block queue as empty */
    img_num_free = 0;		/* Indicate no free blocks */

        /* following lines were modified from create_img */
    image = (Img *) malloc(sizeof(Img)) ;
    image->width = width ;
    image->height = height ;
    image->num_pixels = width * height ;
    image->colormap = NULL ;
    image->pixdata = (Byte *) scry_image ;
	/* end lines from create_img */
    if ((num_colors < 2) || (num_colors > 256))
        abort_dpar("\0",num_colors," requested, only 2-256 make sense")

/*------------------------------------------------------*/
/*	Allocate pointer array and clear it to NULL	*/
/*------------------------------------------------------*/

    cpa = (Ctnode **) calloc (CPA_ENTRIES, sizeof (Ctnode *));
    if ( cpa == NULL )
        abort("Insufficient memory")

    {
        reg Ctnode **cp;	/* Ptr to color pointer */

        i = CPA_ENTRIES;
        cp = cpa;
        while ( i-- > 0 )
        *cp++ = NULL;
    }

/*------------------------------------------*/
/*	Run each phase of quantization	    */
/*------------------------------------------*/

    thread = classify_rgb (image, cpa);		/* Classify colors */
    if ( thread == NULL )
        abort("Insufficient memory")

        /* Thread by frequency */
    indexed_color_count = thread_by_freq (&thread);
    if (indexed_color_count < num_colors)
        actual_num_colors = indexed_color_count;
    else
        actual_num_colors = num_colors;
    root = synthesize_tree (thread, actual_num_colors);	/* Synthesize tree */

        /* Assign colors */
	/* If regular quantization */
    if (!btc)
        actual_num_colors = assign_colors_rgb
            (image, compdata, scry_map, cpa, root, actual_num_colors,btc);
    else
    {		/* final step of CCC algorithm */
        actual_num_colors = assign_colors_rgb
            (image, compdata, scry_map, cpa, root, actual_num_colors, btc);
		/* reassign saved BTC colors to quantized versions */
        reassign_ccc(compdata, height, width, cpa, root);
    }
    if ( actual_num_colors == 0 )
        abort("Insufficient memory")

    {
        int ctr = 0 ;
        while ( img_block_queue )
        {
            img_new_block = img_block_queue -> next_block;
            free ( img_block_queue );
            img_block_queue = img_new_block;
            ++ctr ;
        }
    }

    free (cpa) ;

        /* following 2 lines were modified from delete_img */
    free(image->colormap) ;
    free(image) ;

    return(actual_num_colors) ;
}




/* find nearest color to one in BTC block */

unsigned char
find_isi_color (ccc_r,ccc_g,ccc_b, cpa, root)

unsigned char ccc_r, ccc_g, ccc_b ;	/* original color */
Ctnode *root ;
Ctnode **cpa;		/* Color pointer array */

{
    reg Ctnode *np;		/* Node pointer */
    reg Word r, g, b;		/* Color components */
    reg Rgbpix p;		/* Pixel value */
    reg	Ctnode **cp;		/* Pointer to node pointer */
    unsigned char tmp_char ;

    r = (Word) ccc_r ;
    g = (Word) ccc_g ;
    b = (Word) ccc_b ;

    cp = lookup_color_ptr(r,g,b);	/*  Find color's cpa entry	*/
    np = *cp;				/*  Get node pointer, if any	*/

#ifdef	HIRES_PREQUANT
    if ( np )				/*  If node exists		*/
        if ( np -> detail )		/*     and has detail		*/
        {				/*	look up detail entry	*/
	    cp = lookup_detail(np->detail,r,g,b);
	    np = *cp;
        }
#endif

    if ( np == NULL )
    {				/*  Find nearest defined color	*/
        reg	Ctnode	*nnp;		/*  Nearest-node pointer	*/
	reg	Word	 ndx;		/*  Nearest-node distance	*/
	reg	Word	 cdx;		/*  Current-node distance	*/

	np  = nnp = root;
	ndx = dx(r,g,b,np->r,np->g,np->b);

	while ( np && (ndx > 0) )
        {
	    cdx = dx(r,g,b,np->r,np->g,np->b);
	    if ( cdx < ndx )
            {
		ndx = cdx;
		nnp = np;
            }
	    np = np -> thread;
        }

	np = nnp;
	*cp = np;
    }

    tmp_char = (unsigned char) np->cmi ;
	/* return entry into color map of nearest color */
    return (tmp_char) ;
}

/* end MODIFIED */
