/* Unix compress function */

/******************************************************************/
/*                                                                */
/*            Subroutine to compress input buffer                 */
/*                                                                */
/******************************************************************/

/****************************************************************/
/*                                                              */
/*                 compress() function(Lempel_Ziv)              */
/*                                                              */
/*    This routine returns the compressed data via outbuf[]     */
/*                                                              */
/*    Usage: compress(buffer, bufsize, outbuf); */
/*                                                              */
/*    note: must have the "l_zev.h" in the user dir!         */
/*                                                              */
/****************************************************************/

#include <stdio.h>     /* for standard I/O                        */


/******************************************************************/
/*                     Definition Section                         */
/******************************************************************/
#define BITS  12         /* maxinum code size                */
#if BITS == 16
# define HSIZE	69001		/* 95% occupancy */
#endif
#if BITS == 15
# define HSIZE	35023		/* 94% occupancy */
#endif
#if BITS == 14
# define HSIZE	18013		/* 91% occupancy */
#endif
#if BITS == 13
# define HSIZE	9001		/* 91% occupancy */
#endif
#if BITS <= 12
# define HSIZE	5003		/* 80% occupancy */
#endif

/* Defines for third byte of header */
#define BIT_MASK	0x1f
#define BLOCK_MASK	0x80
/* Masks 0x40 and 0x20 are free.  I think 0x20 should mean that there is
   a fourth header byte (for expansion).
*/
#define INIT_BITS 9		/* initial number of bits/code      */

/*******************************************************************/
/*                Type Definition Section                          */
/*******************************************************************/
/*
 * a code_int must be able to hold 2**BITS values of type int, and also -1
 */
#if BITS > 15
typedef long int	code_int;
#else
typedef int		code_int;
#endif
typedef long int        count_int;
typedef	unsigned char	char_type;

#define	min(a,b)        ((a>b) ? b : a)  /* line function min() */

/*******************************************************************/
/*                   Declaration Section                           */
/*******************************************************************/

char_type magic_header[] = { "\037\235" };	/* 1F 9D */

int n_bits;				/* number of bits/code             */
int maxbits = BITS;			/* user settable max # bits/code   */
code_int maxcode;			/* maximum code, given n_bits      */
code_int maxmaxcode = 1 << BITS;	/* should NEVER generate this code */
# define MAXCODE(n_bits)	((1 << (n_bits)) - 1)


count_int htab [HSIZE];
unsigned short codetab [HSIZE];

#define htabof(i)	htab[i]
#define codetabof(i)	codetab[i]
code_int hsize = HSIZE;			/* for dynamic table sizing        */

code_int free_ent = 0;			/* first unused entry              */


#define tab_prefixof(i)	codetabof(i)
#define tab_suffixof(i)	((char_type *)(htab))[i]
#define de_stack		((char_type *)&tab_suffixof(1<<BITS))

/*
 * block compression parameters -- after all codes are used up,
 * and compression rate changes, start over.
 */
int block_compress = BLOCK_MASK;
int clear_flg = 0;
/*
 * the next two codes should not be changed lightly, as they must not
 * lie within the contiguous general code space.
 */ 
#define FIRST	257	/* first free entry                     */
#define	CLEAR	256	/* table clear output code              */
static int offset;

static char buf[BITS];  /* for compress() and decompress() only */
char_type lmask[9] = {0xff, 0xfe, 0xfc, 0xf8, 0xf0, 0xe0, 0xc0, 0x80, 0x00};
char_type rmask[9] = {0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};



lzw_compress(inbuf,  bcount, outbuf) 
unsigned char inbuf[];                 /* input ram buffer      */
count_int *bcount;                     /* input buffer size ptr */
unsigned char *outbuf ;
{
    register long fcode;
    register code_int i = 0;
    register int c;
    register code_int ent;
    register int disp;
    register code_int hsize_reg;
    register int hshift;
    register count_int fsize;
    long int in_count;
    long int out_count;

    out_count = 0;
    outbuf[out_count++] = magic_header[0];
    outbuf[out_count++] = magic_header[1];
    outbuf[out_count++] = (char)(maxbits | block_compress);
    offset = 0;
    clear_flg = 0;
    n_bits = INIT_BITS;
    maxcode = MAXCODE(n_bits);
    free_ent = FIRST;
    fsize = *bcount;
    /*
     * tune hash table size for small files -- ad hoc,
     * but the sizes match earlier #defines, which
     * serve as upper bounds on the number of output codes. 
     */
    hsize = HSIZE;
    if ( fsize < (1 << 12) )
        hsize = min ( 5003, HSIZE );
    else if ( fsize < (1 << 13) )
        hsize = min ( 9001, HSIZE );
    else if ( fsize < (1 << 14) )
        hsize = min ( 18013, HSIZE );
    else if ( fsize < (1 << 15) )
        hsize = min ( 35023, HSIZE );
    else if ( fsize < 47000 )
        hsize = min ( 50021, HSIZE );

    in_count = 0;
    ent = inbuf[in_count++];
    (*bcount)--;

    hshift = 0;
    for ( fcode = (long) hsize;  fcode < 65536L; fcode *= 2L )
    	hshift++;
    hshift = 8 - hshift;		/* set hash code range bound */

    hsize_reg = hsize;
    cl_hash( (count_int) hsize_reg);		/* clear hash table */

    while ((*bcount)--)
    {
        c = inbuf[in_count++]; 
	fcode = (long) (((long) c << maxbits) + ent);
 	i = ((c << hshift) ^ ent);	           /* xor hashing */

	if ( htabof (i) == fcode ) 
	{
	    ent = codetabof (i);
	    continue;
	} else if ( (long)htabof (i) < 0 )           /* empty slot */
	    goto nomatch;
 	disp = hsize_reg - i;	/* secondary hash (after G. Knott) */
	if ( i == 0 )
	    disp = 1;
probe:
	if ( (i -= disp) < 0 )
	    i += hsize_reg;

	if ( htabof (i) == fcode ) 
	{
	    ent = codetabof (i);
	    continue;
	}
	if ( (long)htabof (i) > 0 ) 
	    goto probe;
nomatch:
	output ((code_int)ent, outbuf, &out_count);
 	ent = c;
	if ( free_ent < maxmaxcode ) 
	{
 	    codetabof (i) = free_ent++;	/* code -> hashtable */
	    htabof (i) = fcode;
	}
    }   /* end of while */
    /*
     * Put out the final code.
     */
    output((code_int)ent, outbuf, &out_count);
    output( (code_int)-1, outbuf, &out_count); 
    *bcount = out_count;      /* get new buffer size after compress() */

    return;
}

/*--------------------------------------------------------------------*/
/*                     write code to outbuf[]                         */
/*                 (this function is called by compress() only        */
/*--------------------------------------------------------------------*/
output(code, outbuf, out_count)
code_int  code;
unsigned char outbuf[];
long int *out_count;
{

    register int r_off = offset, bits= n_bits;
    register char * bp = buf;
    int i;

    if ( code >= 0 ) 
    {
	/*
	 * Get to the first byte.
	 */
	bp += (r_off >> 3);
	r_off &= 7;
	/*
	 * Since code is always >= 8 bits, only need to mask the first
	 * hunk on the left.
	 */
	*bp = (*bp & rmask[r_off]) | (code << r_off) & lmask[r_off];
	bp++;
	bits -= (8 - r_off);
	code >>= 8 - r_off;
	/* Get any 8 bit parts in the middle (<=1 for up to 16 bits). */
	if ( bits >= 8 ) {
	    *bp++ = code;
	    code >>= 8;
	    bits -= 8;
	}
	/* Last bits. */
	if(bits)
	    *bp = code;
	offset += n_bits;
	if ( offset == (n_bits << 3) ) {
	    bp = buf;
	    bits = n_bits;
	    do
	    {
		outbuf[(*out_count)++] = *bp++;
            }
	    while(--bits);
	    offset = 0;
	}

	/*
	 * If the next entry is going to be too big for the code size,
	 * then increase it, if possible.
	 */
	if ( free_ent > maxcode )
	{
	    /*
	     * Write the whole buffer, because the input side won't
	     * discover the size increase until after it has read it.
	     */
	    if ( offset > 0 ) {
		for (i=0; i<n_bits; ++i)
		    outbuf[(*out_count)++] = buf[i];
	    }
	    offset = 0;
	    n_bits++;
	    if ( n_bits == maxbits )
	        maxcode = maxmaxcode;
	    else
		maxcode = MAXCODE(n_bits);
	}
    } else {
	/*
	 * At EOF, write the rest of the buffer.
	 */
	if ( offset > 0 )
	    for (i=0; i<((offset + 7) / 8); ++i)
		outbuf[(*out_count)++] = buf[i];
	offset = 0;
    }
}


/*-----------------------------------------------------------------------*/
/*                       clear hash table                                */
/*-----------------------------------------------------------------------*/
cl_hash(hsize)		/* reset code table */
	register count_int hsize;
{
	register count_int *htab_p = htab+hsize;
	register long i;
	register long m1 = -1;

	i = hsize - 16;
 	do {				/* might use Sys V memset(3) here */
		*(htab_p-16) = m1;
		*(htab_p-15) = m1;
		*(htab_p-14) = m1;
		*(htab_p-13) = m1;
		*(htab_p-12) = m1;
		*(htab_p-11) = m1;
		*(htab_p-10) = m1;
		*(htab_p-9) = m1;
		*(htab_p-8) = m1;
		*(htab_p-7) = m1;
		*(htab_p-6) = m1;
		*(htab_p-5) = m1;
		*(htab_p-4) = m1;
		*(htab_p-3) = m1;
		*(htab_p-2) = m1;
		*(htab_p-1) = m1;
		htab_p -= 16;
	} while ((i -= 16) >= 0);
    	for ( i += 16; i > 0; i-- )
		*--htab_p = m1;
}






