/* convis.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */


/* scry_set_corner:	explicitly controls part of image to be
			sent if the image is larger than
			the server display
   scry_v_flip:		flip image about the x axis
   sc_conv_vis:		truncates image to size of server display
			and server color resolution if necessary
   convert_to_3:	converts image to 3 byte color resolution,
			and server display size if necessary
   conv_4to2:		converts 4 byte to 2 byte color resolution,
			and to server display size if necessary
   conv_3to2:		converts 3 byte to 2 byte color resolution,
			and to server display size if necessary
   conv_4to3:		converts 4 byte to 3 byte color resolution,
			and to server display size if necessary
   conv_2to3:		converts 2 byte to 3 byte color resolution,
			and to server display size if necessary
   conv_1to3:		converts 1 byte to 3 byte color resolution,
			and to server display size if necessary
   make_server_size:	converts to server display size if image
			and server color resolution same
   xdr_visin:		XDR decodes server display information
   sc_get_info:		gets server display information    */


#include <stdio.h>
#include <math.h>
#include <scry_image.h>
#include <scry_anima.h>
#include <scry_client.h>
#include <scry_limits.h>

   /* truncate to this region if image larger than the server display */
static int center_x_left, center_x_right ;
static int center_y_bottom, center_y_top ;

   /* use default truncation or not */
static int corner_set = 0 ;




/* explicitly controls part of image to be sent if image is larger
   than the server display */

scry_set_corner (x, y)

int x, y ;

{
    if (x >= 0)
    {
        center_x_left = x ;
	corner_set = 1 ;
    }
    else
	center_x_left = 0 ;
    if (y >= 0)
    {
        center_y_bottom = y ;
	corner_set = 1 ;
    }
    else
	center_y_bottom = 0 ;
}



/* flip image vertically */

scry_v_flip(image,height,width,depth)

unsigned char *image ;
int height ;
int width ;
int depth ;

{
    unsigned char *old_ptr ;
    unsigned char *new_ptr ;
    unsigned char *save_scanline ;
    int k ;


    if ((sizeof(short) != 2) && (depth == 2))
    {
	fprintf (stderr,"scry_v_flip:  2-byte depth images will not work on this architecture\n") ;
	exit(0) ;
    }
    if (image == NULL)
    {
	fprintf (stderr,"scry_v_flip:  space for image has not been allocated\n") ;
	exit(0) ;
    }
    if ((height < 1) || (height > S_MAX_IMAGE_HEIGHT))
    {
	fprintf (stderr,"scry_v_flip:  height of %d is illegal\n",height) ;
	exit(0) ;
    }
    if ((width < 1) || (width > S_MAX_IMAGE_WIDTH))
    {
	fprintf (stderr,"scry_v_flip:  width of %d is illegal\n",width) ;
	exit(0) ;
    }
    if ((depth < S_MIN_COLOR_DEPTH) || (depth > S_MAX_COLOR_DEPTH))
    {
	fprintf (stderr,"scry_v_flip:  color depth in bytes of %d is illegal\n",depth) ;
	exit(0) ;
    }
    save_scanline = (unsigned char *) malloc(depth*width) ;
    for (k = 0 ; k < height/2 ; k++)
    {
        old_ptr = &image[k*width*depth] ;
        new_ptr = &image[(height-k-1)*width*depth] ;
	memcpy(save_scanline,old_ptr,depth*width) ;
	memcpy(old_ptr,new_ptr,depth*width) ;
	memcpy(new_ptr,save_scanline,depth*width) ;
    }
    free(save_scanline) ;
}




#define ROUNDER 4


/* truncates image to size of server display, and server color
   resolution if less than the images */

unsigned char *
sc_conv_vis (image,height,width,depth,new_ht,new_wd,new_depth)

unsigned char *image ;
int height ;
int width ;
int depth ;    /* original image color resolution in bytes */
int *new_ht ;
int *new_wd ;
int *new_depth ;

{
    unsigned char *newimage = NULL ;
    unsigned char *interimage= NULL ;
    unsigned char *conv_4to2() ;
    unsigned char *conv_3to2() ;
    unsigned char *conv_1to3() ;
    unsigned char *conv_2to3() ;
    unsigned char *conv_4to3() ;
    unsigned char *make_server_size() ;
    int saveflag = 0 ;

    /* make sure compression is set properly if saving Anima file
       locally or on server */
    if (S_anima_filename[0] != '\0')
    {
	if (S_client == NULL)
	    S_image_info.compression = S_CCC ;
	else
	{
	    if (S_image_info.compression & S_LEMPEL_ZIV)
	        saveflag = S_LEMPEL_ZIV ;
	    S_image_info.compression = S_CCC | saveflag ;
	}
    }
    if (width >= S_image_info.s_width)	/* truncate image */
    {
	*new_wd = S_image_info.s_width - (S_image_info.s_width % ROUNDER) ;
        if ((!corner_set) || ((center_x_left + *new_wd) > width))
            center_x_left = (width - *new_wd) / 2 ;
        center_x_right = *new_wd + center_x_left ;
    }
    else	/* width must be multiple of 4 */
    {
	center_x_left = 0 ;
	center_x_right = *new_wd = width - (width % ROUNDER) ;
    }
    if (height >= S_image_info.s_height)	/* truncate image */
    {
	*new_ht = S_image_info.s_height - (S_image_info.s_height % ROUNDER) ;
	if ((!corner_set) || ((center_y_bottom + *new_ht) > height))
            center_y_bottom = (height - *new_ht) / 2 ;
        center_y_top = *new_ht + center_y_bottom ;
    }
    else	/* height must be multiple of 4 */
    {
	center_y_bottom = 0 ;
	center_y_top = *new_ht = height - (height % ROUNDER) ;
    }
    S_orig_image_total = *new_ht * *new_wd * depth ;
#ifdef MESSAGES
    if ((*new_ht != height) || (*new_wd != width))
	fprintf (stderr,"changing size of image to %dh %dw\n", *new_ht,*new_wd) ;
#endif
    switch (depth)
    {
	    /* if image pixel is 8 bits, send as is unless S_CCC compression
	       is desired, in which case the image must be converted
	       to 24 bits as an intermediate step */
	case 1:
		  if (!(S_image_info.compression & S_CCC))
		  {
		    *new_depth = depth ;
		    newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth) ;
		  }
		  else
		  {
		    *new_depth = 3 ;
		    newimage = conv_1to3(image,height,width,*new_ht,*new_wd) ;
		  }
            break ;
		/* if image pixel is 16 bits, only send as is if the
		   server pixel depth is also 16 bits; otherwise,
		   convert to 8 bits if server is 8 bit, or 24 bits
		   if not.  This gets rid of the byte order problem. */
	case 2:
	    switch (S_image_info.s_depth)	/* server color resolution */
	    {
		case 1:
		  if (!(S_image_info.compression & S_CCC))
		  {
		    interimage = conv_2to3(image,height,width,*new_ht,*new_wd) ;
                    newimage = (unsigned char *) malloc(*new_ht * *new_wd) ;	
#ifdef MESSAGES
	            fprintf (stderr,"quantizing image\n") ; 
#endif
		    S_mapnum = quantize (interimage,newimage,S_map,S_maxcol,*new_ht,*new_wd,0) ;
		    S_image_info.compression |= S_QUANTIZE ;
		    free(interimage) ;
		    *new_depth = 1 ;
		  }
		  else
		  {
		    newimage = conv_2to3(image,height,width,*new_ht,*new_wd) ;
		    *new_depth = 3 ;
		  }
		    break ;
		case 2:
		  if ((S_image_info.compression & S_CCC) || (S_image_info.compression & S_QUANTIZE))
		  {
		    depth = 3 ;
		    newimage = conv_2to3(image,height,width,*new_ht,*new_wd) ;
		    *new_depth = depth ;
		  }
		  else
		  {
		    *new_depth = depth ;
		    newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth) ;
		  }
		    break ;
		case 3:
		case 4:
		    newimage = conv_2to3(image,height,width,*new_ht,*new_wd) ;
		    *new_depth = 3 ;
		    break ;
	    }
            break ;
		/* image is 24 or 32 bits */
		/* in all cases, the image sent will be at most 24 bits
		   in depth, even if the server is 32 bits.  This saves
		   transmission time, but is a big loss if you are
		   interested in the alpha information */
	case 3:
	case 4:
	    switch (S_image_info.s_depth)	/* server color resolution in bytes */
	    {
		case 1:
			/* if S_CCC compression is not desired */
		    if (!(S_image_info.compression & S_CCC))
		    {
			*new_depth = depth ;
			    /* convert to server display size if necessary */
			if (depth == 4)
		            interimage = conv_4to3(image,height,width,*new_ht,*new_wd) ;
			else
			    interimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth) ;
                        newimage = (unsigned char *) malloc(*new_ht * *new_wd) ;	
			*new_depth = 1 ;
#ifdef MESSAGES
	                fprintf (stderr,"quantizing image\n") ; 
#endif
                        S_mapnum = quantize (interimage,newimage,S_map,S_maxcol,*new_ht,*new_wd,0) ;
			S_image_info.compression |= S_QUANTIZE ;
			free (interimage) ;
		    }
			/* S_CCC implementation requires image be 24 bits */
		    else
		    {
		        *new_depth = 3 ;
		        if (depth == 4)
		          newimage = conv_4to3(image,height,width,*new_ht,*new_wd) ;
		        else
			  newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth) ;
		    }
		    break ;
		case 2:
			/* convert to 16 bits if no compression used.  This
			   may introduce a byte order problem, but will save
			   transmission time over the network. */
		    if (S_image_info.compression == S_NONE)
                    {
		        *new_depth = 2 ;
			if (depth == 3)
		            newimage = conv_3to2(image,height,width,*new_ht,*new_wd) ;
			else
		            newimage = conv_4to2(image,height,width,*new_ht,*new_wd) ;
		    }
		    else	/* convert to 24 bits */
		    {
		        *new_depth = 3 ;
			if (depth == 4)
		            newimage = conv_4to3(image,height,width,*new_ht,*new_wd) ;
			else
			    newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth);
		    }
		    break ;
		case 3:
		case 4:
		    *new_depth = 3 ;
		    if (depth == 4)
		        newimage = conv_4to3(image,height,width,*new_ht,*new_wd) ;
		    else
			newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,*new_depth) ;
		    break ;
	    }
            break ;
    }
    return (newimage) ;
}




/* converts image to 3 byte color resolution, and server display
   size if necessary.  This is only used on an image that will
   be saved in an Anima file locally. */

unsigned char *
convert_to_3 (image,height,width,depth,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;
int depth ;    /* original image depth in bytes */
int *new_ht ;
int *new_wd ;

{
    unsigned char *newimage = NULL ;
    unsigned char *conv_1to3() ;
    unsigned char *conv_2to3() ;
    unsigned char *conv_4to3() ;
    unsigned char *make_server_size() ;

        /* height and width must be multiple of 4 */
    center_x_left = 0 ;
    center_x_right = *new_wd = width - (width % ROUNDER) ;
    center_y_bottom = 0 ;
    center_y_top = *new_ht = height - (height % ROUNDER) ;
#ifdef MESSAGES
    if ((*new_ht != height) || (*new_wd != width))
	fprintf (stderr,"changing size of image to %dh %dw\n", *new_ht,*new_wd) ;
#endif

    switch (depth)	/* image color resolution in bytes */
    {
	case 1:
            newimage = conv_1to3(image,height,width,*new_ht,*new_wd) ;
            break ;
	case 2:
            newimage = conv_2to3(image,height,width,*new_ht,*new_wd) ;
            break ;
	case 3:
            newimage = make_server_size(image,height,width,depth,*new_ht,*new_wd,3) ;
	    break ;
	case 4:
            newimage = conv_4to3(image,height,width,*new_ht,*new_wd) ;
            break ;
    }
    return (newimage) ;
}




/* converts 4 byte to 2 byte color resolution, and to server display
   size if necessary */

unsigned char *
conv_4to2(image,height,width,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;
int new_ht ;
int new_wd ;

{
    unsigned char *newimage = NULL ;
    unsigned short *newptr ;
    unsigned char *oldptr ;
    int i, j, k ;
    unsigned short temp2bytes ;
    int old_depth = 4 ;
    int new_depth = 2 ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
	/* truncate if necessary */
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        oldptr = (unsigned char *)&(image[((i*width)+center_x_left)*old_depth]) ;
	newptr = (unsigned short *)&(newimage[k * new_wd * new_depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	{
		 /* convert to Targa 16 bit format */
             temp2bytes =  ((*(oldptr+3)>>3) |
			      ((*(oldptr+2)>>3)<<5) |
		              ((*(oldptr+1)>>3)<<10)
			      | (0<<15)) ;
	     *newptr++ = temp2bytes ;
             oldptr += old_depth ;
        }
    }
    return (newimage) ;
}




/* converts 3 byte to 2 byte color resolution, and to server display
   size if necessary */

unsigned char *
conv_3to2(image,height,width,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;
int new_ht ;
int new_wd ;

{
    unsigned char *newimage = NULL ;
    unsigned short *newptr ;
    unsigned char *oldptr ;
    int i, j, k ;
    unsigned short temp2bytes ;
    int old_depth = 3 ;
    int new_depth = 2 ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        oldptr = (unsigned char *)&(image[((i*width)+center_x_left)*old_depth]) ;
	newptr = (unsigned short *)&(newimage[k * new_wd * new_depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	{
                temp2bytes =  ((*(oldptr+2)>>3) |
			      ((*(oldptr+1)>>3)<<5) |
		              ((*(oldptr)>>3)<<10)
			      | (0<<15)) ;
		*newptr++ = temp2bytes ;
		oldptr += old_depth ;
	}
    }
    return (newimage) ;
}




/* converts 4 byte to 3 byte color resolution, and to server
   display size if necessary */

unsigned char *
conv_4to3(image,height,width,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;
int new_ht ;
int new_wd ;

{
    unsigned char *newimage = NULL ;
    unsigned char *new_ptr ;
    unsigned char *old_ptr ;
    int new_depth = 3 ;
    int old_depth = 4 ;
    int i, j, k ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        old_ptr = (unsigned char *)&(image[((i*width)+center_x_left)*old_depth]) ;
	new_ptr = &(newimage[k*new_wd*new_depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	{
		/* convert to 3 bytes */
            *new_ptr++ = *(old_ptr+1) ;
            *new_ptr++ = *(old_ptr+2) ;
            *new_ptr++ = *(old_ptr+3) ;
            old_ptr += 4 ;
	}
    }
    return (newimage) ;
}




/* converts 2 byte to 3 byte color resolution, and to server
   display size if necessary */

unsigned char *
conv_2to3(image,height,width,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;

{
    unsigned char *newimage = NULL ;
    unsigned char *new_ptr ;
    unsigned short *old_ptr ;
    int old_depth = 2 ;
    int new_depth = 3 ;
    int i, j, k ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        old_ptr = (unsigned short *)&(image[((i*width)+center_x_left)*old_depth]) ;
        new_ptr = &(newimage[k*new_wd*new_depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	{
		/* convert Targa 16 bit format to RGB */
            *new_ptr++ = ((*old_ptr>>10)&0x1f)<<3 ;
            *new_ptr++ = ((*old_ptr>>5)&0x1f)<<3 ;
            *new_ptr++ = (*old_ptr & 0x1f)<<3 ;
            old_ptr++ ;
	}
    }
    return (newimage) ;
}




/* converts 1 byte to 3 byte color resolution, and to server
   display size if necessary */

unsigned char *
conv_1to3(image,height,width,new_ht,new_wd)

unsigned char *image ;
int height ;
int width ;
int new_ht ;
int new_wd ;

{
    unsigned char *newimage = NULL ;
    unsigned char *new_ptr ;
    unsigned char *old_ptr ;
    int old_depth = 1 ;
    int new_depth = 3 ;
    int i, j, k ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        old_ptr = (unsigned char *)&(image[((i*width)+center_x_left)*old_depth]) ;
	new_ptr = &(newimage[k*new_wd*new_depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	{
		/* converts from 1 to 3 bytes using color map which
		   must have been provided by user with scry_set_map */
            *new_ptr++ = S_map[*old_ptr][S_RED] ;
            *new_ptr++ = S_map[*old_ptr][S_GREEN] ;
            *new_ptr++ = S_map[*old_ptr][S_BLUE] ;
            ++old_ptr ;
	}
    }
    return (newimage) ;
}




/* converts to server display size if image and server color
   resolution are same */

unsigned char *
make_server_size(image,height,width,depth,new_ht,new_wd,new_depth)

unsigned char *image ;
int height ;
int width ;
int depth ;
int new_ht ;
int new_wd ;
int new_depth ;

{
    unsigned char *newimage = NULL ;
    unsigned char *oldptr ;
    unsigned char *newptr ;
    int i, j, k, m ;

    newimage = (unsigned char *) malloc (new_ht*new_wd*new_depth) ;
    for (k = 0 , i = center_y_bottom ; i < center_y_top ; i++, k++)
    {
        oldptr = (unsigned char *)&(image[((i*width)+center_x_left)*depth]) ;
        newptr = &(newimage[k*new_wd*depth]) ;
	for (j = center_x_left ; j < center_x_right ; j++)
	    for (m = 0 ; m < depth ; m++)
	        *newptr++ = *oldptr++ ;
    }
    return (newimage) ;
}




/* XDR decodes server display information */

bool_t 
xdr_visin (xdrsp,image_info)

XDR *xdrsp ;		/* XDR handle */
struct image_stuff *image_info ;

{
	    /* depth of server display in bytes */
    if (!xdr_int(xdrsp,&(image_info->s_depth)))
	return (0) ;
    if (!xdr_int(xdrsp,&(image_info->s_height)))
	return (0) ;
    if (!xdr_int(xdrsp,&(image_info->s_width)))
	return (0) ;
	    /* whether server has videodisk */
    if (!xdr_int(xdrsp,&(image_info->s_recordable)))
	return (0) ;
    return(1) ;
}




/* gets server display information */

sc_get_info()

{
    enum clnt_stat err ;
    bool_t xdr_visin() ;

    S_during_rpc = 1 ;
    if ((int) (err = clnt_call(S_client,S_VISPROC,xdr_void,(char *)0,
			 xdr_visin,&S_image_info,S_total_timeout)) != 0)
    {
        clnt_perrno ((int) err) ;
        fprintf (stderr,"can't make S_VISPROC call\n") ;
        clnt_destroy (S_client) ;
        return ;
    }
    S_during_rpc = 0 ;
    if (S_make_closing_call)
	scry_abn_exit() ;
}
