/*	Copyright (c) 1982 Michael Landy, Yoav Cohen, and George Sperling

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/*  bandrej.c - apply bandreject filter on a sequence, or compute
**		 MTF of a filter.
**
** usage -   bandrej  [-b or -r] [-f n] [-d n] <insequence >outsequence
**	or:  bandrej  [-b or -r] [-f n] [-d n] [-s] >outsequence
**
** In both calling sequences the option [-f n] which specifies a filter
** number can be replaced by the option [-p m cf la lb ha hb]
** which specifies a set of new parameters for an existing filter.
** "m" stands for the filtering method (Ideal,Butterworth or Exponential)
** -- an integer in the range 1-3. "cf" stands for the center of the
** frequency band, and is expressed as a proportion of the full frequency
** range. "la","lb","ha","hb" -- are the parameters of the filter. Note
** that all of them have to be specified, even if the filter is defined only
** by two parameters (e.g. Ideal filter); in this case the value of the
** superflous parameters is of no importance.
**
** The input sequence must be of pictures in the Fourier domain (represented
** in complex format), or of picture-spectra (represented in float format).
** If the -s option is specified no input is expected, and the program
** outputs the MTF of the filter in float format.
** The MTF is plotted on a 128X128 picture.
** Option -d followed by a number (integer or real) specifies the
** metric of the distance, where the number is the exponent of the
** Minkowsky space. (Default value: 2, for Euclidean space, results in an
** isotropic filter).
** Option -f followed by an integer specifies the filter number
** according to the following list (default filter: no.1):
**
**     number	method	 parameters
**
**	1	Ideal	  band: .167 - .50 of frequencies
**	2	Butterworth: .167 - .50, 1,1 
**	3	Exponential:       "
**	4	Ideal	  band: .25 - .75
**	5	Butterworth:       "	,1,1
**	6	Exponential:	   "
**	7	Ideal	  band: .33 - .99
**	8	Butterworth:	   "	,1,1
**	9	Exponential:	   "
**
** Three filters are implemented: Ideal,Butterworth and Exponential.
** Each filter is applied on both sides of a selected "band_center"
** which is specified in the array "ff" as a fraction of the maximum
** range of frequencies.
** To add other filtering methods, change the function "modulate",
** and add entries in the "m","ff" and "ps" arrays.
** Array "m" specifies the filtering method to be used. (method m[i] for
** filter number i ).
** Array "ff" specifies the center-frequency of the band, as described above.
** Array "ps" specifies 4 optional parameters for the filter. The first
** two parameters apply to the low-frequency  half-band, the next two
** apply to the high-frequencies portion. The first parameter of each pair
** specifies the cut-off frequency, expressed as a fraction of the
** frequencies range.  The frequency range is controlled by the options
** -b or -r.  If neither is specified, then frequency "1.0" is controlled
** by (half) the number of columns, and the rows are treated with the same
** distance metric, so that non-square pictures may have a different range of
** frequencies in the rows and columns.  Option -r specifies frequency "1.0"
** by the number of rows rather than columns.  Option -b specifies both, so
** that the distance metric for non-square pictures is different for rows and
** columns, adjusted so that both (number or rows/2) and (number of columns/2)
** are treated as frequency 1.0.
**
** To load: cc -o bandrej bandrej.c  -lhipl -lm
**
** Yoav Cohen 5/21/82
** modified by Mike Landy 7/20/84
** modified by Ann Adams & Chuck Carman 4/5/88
** created bandrej.c from bandpass.c
**
** Note: the program can be modified to run faster by computing
** 1/4 or 1/8 of the filter, storing it in an array, and then
** copying it for the rest of the array coordinates.
*/

#include <stdio.h>
#include <math.h>
#include <hipl_format.h>
char *Progname;

#define NPARAMS 4
#define NFILTERS 10

int rflag = 0;	/* rows control frequency 1.0 */
int bflag = 0;	/* both rows and columns control frequency 1.0 separately */

int m[NFILTERS] =
	{ 0,1,2,3,1,2,3,1,2,3 };
double ff[NFILTERS] =
	{ 0.0,.33,.33,.33,.5,.5,.5,.67,.67,.67 };
double ps[NFILTERS][NPARAMS] = {
		{0},
		{.167,0,.5,0},
		{.167,1.,.5,1.},
		{.167,1.,.5,1.},
		{.25,0,.75},
		{.25,1.,.75,1.},
		{.25,1.,.75,1.},
		{.33,0.,.99},
		{.33,1.,.99,1.},
		{.33,1.,.99,1.},
};

int method, nr2;
double ha, hb, la, lb, la2, ha2, band_center;

main(argc,argv)
	int argc;
	char *argv[];
{
	struct header hd;
	int	nf,nc,nr,nc2,row,col,x,y,pf;
	int	i,frame,argcc,nums;
	int	spectrum, nfilter; 
	double 	mink, invmink, powy, mval, scale, distance;
	double 	*distr, *distc, *k;
	float	*pic, *ppic;

	double modulate(), pow();

	Progname = strsave(*argv);
	if ((argv[argc-1][0]=='-') && (argv[argc-1][1]=='D')) argc--;

	spectrum = 0;
	nfilter = 1;
	mink = 2.0;
	for(argcc=1; argcc<argc; argcc++) {
		if (argv[argcc][0] != '-') 
			perr("bad argument");
		else switch (argv[argcc][1]) {
			case 'p':	nfilter=0;
					m[0]=atoi(argv[++argcc]);
					ff[0]=atof(argv[++argcc]);
					for(i=0;i<NPARAMS;i++)
						ps[0][i]=atof(argv[++argcc]);
					break;
			case 's':	spectrum=1;
					break;
			case 'r':	rflag++;
					break;
			case 'b':	bflag++;
					break;
			case 'f':	argcc++;
					nfilter=atoi(argv[argcc]);
					break;
			case 'd':	argcc++;
					mink=atof(argv[argcc]);
					break;
			default:	perr("unrecognized option");
		}
	}
	if (mink == 0.0)
		perr("exponent must not be zero");

	invmink = 1.0 / mink;

	if (rflag && bflag)
		perr("only one of -r and -b may be specified");

	if(spectrum) {
		nf=1; nr=128; nc=128; pf=PFFLOAT;
		init_header(&hd,"","",nf,"",nr,nc,32,0,pf,"");
	}
	else read_header(&hd);
	nf=hd.num_frame;
	nr=hd.rows; nc=hd.cols;
	pf=hd.pixel_format;
	nr2=nr/2; nc2=nc/2;

	if ((pf != PFFLOAT) && (pf != PFCOMPLEX))
		perr("pixel format must be float or complex");

	distr = (double *) halloc(nr2+1,sizeof(double));
	distc = (double *) halloc(nc2+1,sizeof(double));

	k = distr;
	scale = (rflag || bflag) ? nr2 : nc2;
	*k++ = 0.0;
	for(i=1; i<=nr2; i++)
		*k++ = pow(((double) i)/scale, mink);
	k = distc;
	*k++ = 0.0;
	scale = (rflag) ? nr2 : nc2;
	for(i=1; i<=nc2; i++)
		*k++ = pow(((double) i)/scale, mink);

	nums = (pf==PFFLOAT) ? 1 : 2;
	pic = (float *) halloc(nr*nc*nums,sizeof(float));

	update_header(&hd,argc,argv);
	write_header(&hd);

	select_method(nfilter);

	for(frame=0;frame<nf;frame++) {
		if (spectrum) {
			ppic = pic;
			for (i=0;i<nr*nc;i++)
				*ppic++ = 1.;
		}
		else {
			if (pread(0,pic,nr*nc*nums*sizeof(float)) !=
				nr*nc*nums*sizeof(float))
					perr("read error");
		}
		ppic = pic;
		for(row=0; row<nr; row++) {
			y = row;
			if (pf == PFCOMPLEX)
				y = (y + nr2) % nr;
			y = abs(y - nr2);
			powy = distr[y];
			for(col=0; col<nc; col++,ppic++) {
				x = col;
				if (pf == PFCOMPLEX)
					x = (x + nc2) % nc;
				x = abs(x - nc2);
				if ((powy + distc[x]) == 0.0)
					distance = 0.0;
				else
					distance = pow(powy+distc[x],invmink);
				mval = modulate(distance);
				*ppic = *ppic * mval;
				if (pf == PFCOMPLEX) {
					ppic++;
					*ppic = *ppic * mval;
				}
			}
		}
		if (write(1,pic,nr*nc*nums*sizeof(float)) !=
			nr*nc*nums*sizeof(float))
				perr("write error");
	}
	return(0);
}

select_method(nfilter) 
int	nfilter;
{
	double fabs();

	if ((nfilter >= NFILTERS) || (nfilter < 0))
		perr("bad filter number");

	method = m[nfilter];
	if (method == 0)
		perr("undefined filter");

	fprintf(stderr,"bandrej: applying filter no.%d\n",nfilter);

	band_center = ff[nfilter];
	la2 = ps[nfilter][0];
	la = fabs(band_center - la2);
	lb = ps[nfilter][1];
	if (method == 2) lb += lb;
	ha2 = ps[nfilter][2];
	ha = fabs(band_center - ha2);
	hb = ps[nfilter][3];
	if (method == 2) hb += hb;
}


double modulate(distance) 
	double distance;
{
	double d, absd;
	double pow(), exp(), fabs();

	d = distance - band_center;
	absd = fabs(d);

	switch (method) {
		case 1:	/* ideal filter */
			if ((distance <= la2) || (distance >= ha2))
				return(1.0);
			else return(0.0);
		case 2:	/* Butterworth filter */
			if (d == 0.0)
				return(0.0);
			else if (d<0)
				return(1. - 1./(1.+pow(absd/la,lb)));
			else
				return(1. - 1./(1.+pow(absd/ha,hb)));
		case 3: /* Exponential filter */
			if(d<0)
				return(1.0 - exp(-pow(absd/la,lb)));
			else
				return(1.0 - exp(-pow(absd/ha,hb)));
		default:
			perr("undefined filtering method");
	}
	return(0.0);	/* this should never execute, but lint is dumb */
}
